"""
This module contains all the 2D line primititive classes, including
the dispatcher class that returns line instances from a format string.
"""

from __future__ import generators
from __future__ import division

import pygtk
pygtk.require('2.0')
import gtk
from gtk import gdk
import Numeric as numpy

from matplotlib.utils import is_string_like
from matplotlib.colors import ColorDispatcher


def _identity_transform(x,y):
    return x,y


class Line2D:
    
    def __init__(self, x, y,
                 lineWidth=1, 
                 color=ColorDispatcher().get('b')):

        #convert sequences to numeric arrays
        try: x.shape
        except AttributeError: self._x = numpy.array(x, numpy.Float)
        else: self._x = x

        self.verticalOffset = None
        
        try: y.shape
        except AttributeError: self._y = numpy.array(y, numpy.Float)
        else: self._y = y

        if len(self._x.shape)>1 and self._x.shape[1]==1:
            self._x = numpy.resize(self._x, (len(x),))
        if len(self._y.shape)>1 and self._y.shape[1]==1:
            self._y = numpy.resize(self._y, (len(y),))
        if self._x.shape != self._y.shape:
            errs = 'x and y must have the same shape.'
            errs += '\nx.shape = %s, y.shape=%s' % (self._x.shape, self._y.shape)
            raise ValueError, errs
        self._lineWidth = lineWidth
        self._color = color
        self._xsorted = self._is_sorted(self._x)

    def set_vertical_offset(self, voff):
        self.verticalOffset = voff
        
    def _is_sorted(self, x):
        "return true if x is sorted"
        return numpy.alltrue(x[1:]-x[0:-1]>=0)
    
    def draw(self, widget, gc, transfunc=_identity_transform):
        
        gc.foreground = self._color
        gc.line_width = self._lineWidth


        # if the x or y clip is set, only plot the points in the clipping region
        try: self._xc, self._yc
        except AttributeError: x, y = self._x, self._y
        else: x, y = self._xc, self._yc
        # transform into axes coords            

        if self.verticalOffset is not None:
            #print self.verticalOffset, y.typecode()
            y = y + self.verticalOffset
            
        (self._xt, self._yt) = transfunc(x, y)
        self._draw(self._xt, self._yt, widget, gc)

    def _draw(self, *args):
        raise TypeError, 'Line2D is a pure base class.  ' + \
              'You must instantiate a derived class'



    def flush_clip(self):
        delList = ['_xmin', '_xmax', '_ymin', '_ymax', '_xc', '_yc']
        for item in delList:
            try: del self.__dict__[item]
            except KeyError: pass

    def get_x(self):
        return self._x

    def get_y(self):
        return self._y


    def get_line_width(self):
        return self._lineWidth

    def get_color(self):
        return self._color

    def _set_clip(self):

        try: self._xmin, self._xmax
        except AttributeError: indx = numpy.arange(len(self._x))
        else:
            if self._xsorted:
                # for really long signals, if we know they are sorted
                # on x we can save a lot of time using search sorted
                # since the alternative approach requires 3 O(len(x) ) ops
                inds = numpy.searchsorted(
                    self._x, numpy.array([self._xmin, self._xmax]))
                indx = numpy.arange(inds[0], inds[1])
            else:
                indx = numpy.nonzero(
                    numpy.logical_and( self._x>=self._xmin,
                                       self._x<=self._xmax ))
        

        self._xc = numpy.take(self._x, indx)
        self._yc = numpy.take(self._y, indx)

        # y data clipping for connected lines can introduce horizontal
        # line artifacts near the clip region.  If you really need y
        # clipping for efficiency, consider using plot(y,x) instead.
        # If you must have both x and y data clipping, and can live
        # with the artifacts for high gain y clipping, , do 'if 1'
        # instead of isinstance.
        if not isinstance(self, ConnectedLine2D):
            try: self._ymin, self._ymax
            except AttributeError: indy = numpy.arange(len(self._yc))
            else: indy = numpy.nonzero(
                numpy.logical_and(self._yc>=self._ymin,
                                  self._yc<=self._ymax ))
        else:
            indy = numpy.arange(len(self._yc))
            
        self._xc = numpy.take(self._xc, indy)
        self._yc = numpy.take(self._yc, indy)

    def set_color(self, color):
        if is_string_like(color):
            color = ColorDispatcher().get(color)
        self._color = color

    def set_line_width(self, w):
        self._lineWidth = w

    def set_xclip(self, xmin, xmax):
        self._xmin, self._xmax = xmin, xmax
        self._set_clip()

    def set_yclip(self, ymin, ymax):
        self._ymin, self._ymax = ymin, ymax
        self._set_clip()


class ConnectedLine2D(Line2D):
    """
    ConnectedLine2D is just a type info holder, so you can, for
    example, ask an axes for all the connected lines it contains
    """
    pass

class SolidLine2D(ConnectedLine2D):
    def __init__(self, x, y, *args, **kargs):
        Line2D.__init__(self, x, y, *args, **kargs)

    def _draw(self, xt, yt, widget, gc):
        widget.window.draw_lines(gc, zip(xt,yt))


class DashedLine2D(ConnectedLine2D):
    def __init__(self, x, y, *args, **kargs):
        Line2D.__init__(self, x, y, *args, **kargs)

    def _draw(self, xt, yt, widget, gc):
        gc.line_style = gdk.LINE_ON_OFF_DASH
        gc.cap_style =  gdk.CAP_BUTT
        gc.join_style = gdk.JOIN_MITER
        widget.window.draw_lines(gc, zip(xt, yt) )

class DashDotLine2D(ConnectedLine2D):
    def __init__(self, x, y, *args, **kargs):
        Line2D.__init__(self, x, y, *args, **kargs)
        raise RuntimeError, 'DashDotLine2D i not yet implemented.  (sorry)'        


class DottedLine2D(ConnectedLine2D):
    "this is connected because there can be dots between the x,y points"
    def __init__(self, x, y, *args, **kargs):
        Line2D.__init__(self, x, y, *args, **kargs)
        self._spacing = 2
        
    def _draw(self, xt, yt, widget, gc):
        gc.line_style = gdk.LINE_ON_OFF_DASH
        gc.cap_style =  gdk.CAP_BUTT
        gc.join_style = gdk.JOIN_ROUND
        gc.set_dashes(0,[1,self._spacing])
        widget.window.draw_lines(gc, zip(xt, yt) )

    def set_spacing(self, spacing):
        self._spacing = spacing

class SymbolLine2D(Line2D):
    def __init__(self, x, y,
                 symbolSize=5, symbolFill=0,
                 *args, **kargs):
        Line2D.__init__(self, x, y, *args, **kargs)
        self._symbolSize = symbolSize
        self._symbolFill = symbolFill
    
    def _draw(self, *args):
        raise TypeError, 'SymbolLine2D is a pure base class.  You must instantiate a derived class'

    def set_fill(self, fill):
        self._symbolFill = fill

    def set_size(self, size):
        self._symbolSize = size

    def get_fill(self):
        return self._symbolFill

    def get_size(self):
        return self._symbolSize

    

class CircleLine2D(SymbolLine2D):
    def __init__(self, x, y,
                 *args, **kargs):
        SymbolLine2D.__init__(self, x, y,
                              *args, **kargs)

    def _draw(self, xt, yt, widget, gc):
        offset = self._symbolSize/2
        for (x,y) in zip(xt, yt):
            widget.window.draw_arc(gc, self._symbolFill, x-offset, y-offset, 
                            self._symbolSize, self._symbolSize,
                            0, 360*64)



class PointLine2D(SymbolLine2D):
    def __init__(self, x, y,
                 symbolSize=4, symbolFill=1,
                 *args, **kargs):
        SymbolLine2D.__init__(self, x, y,
                              symbolSize=symbolSize,
                              symbolFill=symbolFill,
                              *args, **kargs)

    def _draw(self, xt, yt, widget, gc):
        offset = self._symbolSize/2
        for (x,y) in zip(xt, yt):
            widget.window.draw_arc(gc, self._symbolFill, x-offset, y-offset, 
                            self._symbolSize, self._symbolSize,
                            0, 360*64)


class SquareLine2D(SymbolLine2D):
    def __init__(self, x, y, *args, **kargs):
        SymbolLine2D.__init__(self, x, y, *args, **kargs)

    def _draw(self, xt, yt, widget, gc):
        offset = self._symbolSize/2        
        for (x,y) in zip(xt, yt):            
            widget.window.draw_rectangle(gc, self._symbolFill,
                                         x-offset, y-offset,
                                         self._symbolSize, self._symbolSize) 


class TriangleUpLine2D(SymbolLine2D):
    def __init__(self, x, y, *args, **kargs):
        SymbolLine2D.__init__(self, x, y, *args, **kargs)

    def _draw(self, xt, yt, widget, gc):
        offset = self._symbolSize/2        
        for (x,y) in zip(xt, yt):            
            widget.window.draw_polygon(gc, self._symbolFill,
                                ( (x, y-offset),
                                  (x-offset, y+offset),
                                  (x+offset, y+offset)
                                ))
        


class TriangleDownLine2D(SymbolLine2D):
    def __init__(self, x, y,*args, **kargs):
        SymbolLine2D.__init__(self, x, y, *args, **kargs)

    def _draw(self, xt, yt, widget, gc):
        offset = self._symbolSize/2        
        for (x,y) in zip(xt, yt):            
            widget.window.draw_polygon(gc, self._symbolFill,
                                ( (x, y+offset),
                                  (x-offset, y-offset),
                                  (x+offset, y-offset)
                                ))
    

class TriangleLeftLine2D(SymbolLine2D):
    def __init__(self, x, y,*args, **kargs):
        SymbolLine2D.__init__(self, x, y, *args, **kargs)

    def _draw(self, xt, yt, widget, gc):
        offset = self._symbolSize/2        
        for (x,y) in zip(xt, yt):            
            widget.window.draw_polygon(gc, self._symbolFill,
                                ( (x-offset, y),
                                  (x+offset, y-offset),
                                  (x+offset, y+offset)
                                ))


class TriangleRightLine2D(SymbolLine2D):
    def __init__(self, x, y,*args, **kargs):
        SymbolLine2D.__init__(self, x, y, *args, **kargs)

    def _draw(self, xt, yt, widget, gc):
        offset = self._symbolSize/2        
        for (x,y) in zip(xt, yt):            
            widget.window.draw_polygon(gc, self._symbolFill,
                                ( (x+offset, y),
                                  (x-offset, y-offset),
                                  (x-offset, y+offset)
                                ))


class PlusLine2D(SymbolLine2D):
    def __init__(self, x, y,*args, **kargs):
        SymbolLine2D.__init__(self, x, y, *args, **kargs)

    def _draw(self, xt, yt, widget, gc):
        offset = self._symbolSize/2
        for (x,y) in zip(xt, yt):
            widget.window.draw_line(gc, x-offset, y, x+offset, y)
            widget.window.draw_line(gc, x, y-offset, x, y+offset)
    


class Line2D_Dispatcher(dict):
    _dispatcher =  {'-' : SolidLine2D,
                    '--' : DashedLine2D, 
                    '-.' : DashDotLine2D, 
                    ':' : DottedLine2D, 
                    '.' : PointLine2D, 
                    'o' : CircleLine2D, 
                    '^' : TriangleUpLine2D, 
                    'v' : TriangleDownLine2D, 
                    '<' : TriangleLeftLine2D, 
                    '>' : TriangleRightLine2D, 
                    's' : SquareLine2D,
                    '+' : PlusLine2D,
                    }
    _sharedState = {}

    def __init__(self):
        self.__dict__ = self._sharedState
        self.update(self._dispatcher)    
