"""

Numerical python functions written for compatability with matlab
commands with the same names.  

  Matlab compatible functions:

    * cohere - Coherence (normalized cross spectral density)

    * corrcoef - The matrix of correlation coefficients

    * csd - Cross spectral density uing Welch's average periodogram

    * detrend -- Remove the mean or best fit line from an array
    
    * linspace -- Linear spaced array from min to max

    * hist -- Histogram
    
    * polyfit - least squares best polynomial fit of x to y

    * polyval - evaluate a vector for a vector of polynomial coeffs

    * psd - Power spectral density uing Welch's average periodogram

    * vander - the Vandermonde matrix

  Functions that don't exist in matlab, but are useful anyway:

    * cohere_pairs - Coherence over all pairs.  This is not a matlab
      function, but we compute coherence a lot in my lab, and we
      compute it for alot of pairs.  This function is optimized to do
      this efficiently by caching the direct FFTs.

Credits:

  Unless otherwise noted, these functions were written by
  Author: John D. Hunter <jdhunter@ace.bsd.uchicago.edu>

  Some others are from the Numeric documentation, or imported from
  MLab or other Numeric packages

"""

from __future__ import division
import MLab
from MLab import mean, hanning, cov, rand, randn, diff, std
from Numeric import zeros, ones, diagonal, transpose, matrixmultiply, \
     resize, sqrt, divide, array, Float, Int, Complex, concatenate, \
     convolve, dot, conjugate, absolute, arange, reshape, arctan2, \
     searchsorted, concatenate, sort, exp, pi, take, nonzero, sum

from LinearAlgebra import inverse
from FFT import fft
from Matrix import Matrix
from cbook import iterable


def mean(x, dim=None):
   if len(x)==0: return None
   elif dim is None:
      return MLab.mean(x)
   else: return MLab.mean(x, dim)
   

def linspace(xmin, xmax, N):
   dx = (xmax-xmin)/(N-1)
   return xmin + dx*arange(N)

def norm(x):
    "return sqrt(x dot x)"
    return sqrt(dot(x,x))

def window_hanning(x):
    "return x times the hanning window of len(x)"
    return hanning(len(x))*x

def window_none(x):
    "No window function; simply return x"
    return x

def detrend(x, key=None):
    if key is None or key=='constant':
        return detrend_mean(x)
    elif key=='linear':
        return detrend_linear(x)

def detrend_mean(x):
    "Return x minus the mean(x)"
    return x - mean(x)

def detrend_none(x):
    "Return x: no detrending"
    return x

def detrend_linear(x):
    "Return x minus best fit line; 'linear' detrending "

    # I'm going to regress x on xx=range(len(x)) and return x -
    # (b*xx+a).  Now that I have polyfit working, I could convert the
    # code here, but if it ain't broke, don't fix it!
    xx = arange(len(x), typecode=x.typecode())
    X = transpose(array([xx]+[x]))
    C = cov(X)
    b = C[0,1]/C[0,0]
    a = mean(x) - b*mean(xx)
    return x-(b*xx+a)


def psd(x, NFFT=256, Fs=2, detrend=detrend_none,
        window=window_hanning, noverlap=0):
    """
    The power spectral density by Welches average periodogram method.
    The vector x is divided into NFFT length segments.  Each segment
    is detrended by function detrend and windowed by function window.
    noperlap gives the length of the overlap between segments.  The
    absolute(fft(segment))**2 of each segment are averaged to compute Pxx,
    with a scaling to correct for power loss due to windowing.  Fs is
    the sampling frequency.

    -- NFFT must be a power of 2
    -- detrend and window are functions, unlike in matlab where they are
       vectors.
    -- if length x < NFFT, it will be zero padded to NFFT
    

    Returns the tuple Pxx, freqs
    
    Refs:
      Bendat & Piersol -- Random Data: Analysis and Measurement
        Procedures, John Wiley & Sons (1986)

    """

    if NFFT % 2:
        raise ValueError, 'NFFT must be a power of 2'

    # zero pad x up to NFFT if it is shorter than NFFT
    if len(x)<NFFT:
        n = len(x)
        x = resize(x, (NFFT,))
        x[n:] = 0
    

    # for real x, ignore the negative frequencies
    if x.typecode()==Complex: numFreqs = NFFT
    else: numFreqs = NFFT//2+1
        
    windowVals = window(ones((NFFT,),x.typecode()))
    step = NFFT-noverlap
    ind = range(0,len(x)-NFFT+1,step)
    n = len(ind)
    Pxx = zeros((numFreqs,n), Float)

    # do the ffts of the slices
    for i in range(n):
        thisX = x[ind[i]:ind[i]+NFFT]
        thisX = windowVals*detrend(thisX)
        fx = absolute(fft(thisX))**2
        Pxx[:,i] = fx[:numFreqs]

    # Scale the spectrum by the norm of the window to compensate for
    # windowing loss; see Bendat & Piersol Sec 11.5.2
    if n>1: Pxx = mean(Pxx,1)
    Pxx = divide(Pxx, norm(windowVals)**2)
    freqs = Fs/NFFT*arange(0,numFreqs)
    return Pxx, freqs



def csd(x, y, NFFT=256, Fs=2, detrend=detrend_none,
        window=window_hanning, noverlap=0):
    """
    The cross spectral density Pxy by Welches average periodogram
    method.  The vectors x and y are divided into NFFT length
    segments.  Each segment is detrended by function detrend and
    windowed by function window.  noverlap gives the length of the
    overlap between segments.  The product of the direct FFTs of x and
    y are averaged over each segment to compute Pxy, with a scaling to
    correct for power loss due to windowing.  Fs is the sampling
    frequency.

    NFFT must be a power of 2

    Returns the tuple Pxy, freqs

    Refs:
      Bendat & Piersol -- Random Data: Analysis and Measurement
        Procedures, John Wiley & Sons (1986)

    """

    if NFFT % 2:
        raise ValueError, 'NFFT must be a power of 2'

    # zero pad x and y up to NFFT if they are shorter than NFFT
    if len(x)<NFFT:
        n = len(x)
        x = resize(x, (NFFT,))
        x[n:] = 0
    if len(y)<NFFT:
        n = len(y)
        y = resize(y, (NFFT,))
        y[n:] = 0

    # for real x, ignore the negative frequencies
    if x.typecode()==Complex: numFreqs = NFFT
    else: numFreqs = NFFT//2+1
        
    windowVals = window(ones((NFFT,),x.typecode()))
    step = NFFT-noverlap
    ind = range(0,len(x)-NFFT+1,step)
    n = len(ind)
    Pxy = zeros((numFreqs,n), Complex)

    # do the ffts of the slices
    for i in range(n):
        thisX = x[ind[i]:ind[i]+NFFT]
        thisX = windowVals*detrend(thisX)
        thisY = y[ind[i]:ind[i]+NFFT]
        thisY = windowVals*detrend(thisY)
        fx = fft(thisX)
        fy = fft(thisY)
        Pxy[:,i] = fy[:numFreqs]*conjugate(fx[:numFreqs])

    # Scale the spectrum by the norm of the window to compensate for
    # windowing loss; see Bendat & Piersol Sec 11.5.2
    if n>1: Pxy = mean(Pxy,1)
    Pxy = divide(Pxy, norm(windowVals)**2)
    freqs = Fs/NFFT*arange(numFreqs)
    return Pxy, freqs

def cohere(x, y, NFFT=256, Fs=2, detrend=detrend_none,
           window=window_hanning, noverlap=0):
    """
    cohere the coherence between x and y.  Coherence is the normalized
    cross spectral density

    Cxy = |Pxy|^2/(Pxx*Pyy)

    The return value is (Cxy, f), where f are the frequencies of the
    coherence vector.  See the docs for psd and csd for information
    about the function arguments NFFT, detrend, windowm noverlap, as
    well as the methods used to compute Pxy, Pxx and Pyy.

    Returns the tuple Cxy, freqs

    """

    
    Pxx,f = psd(x, NFFT=NFFT, Fs=Fs, detrend=detrend,
              window=window, noverlap=noverlap)
    Pyy,f = psd(y, NFFT=NFFT, Fs=Fs, detrend=detrend,
              window=window, noverlap=noverlap)
    Pxy,f = csd(x, y, NFFT=NFFT, Fs=Fs, detrend=detrend,
              window=window, noverlap=noverlap)

    Cxy = divide(absolute(Pxy)**2, Pxx*Pyy)
    return Cxy, f

def corrcoef(*args):
    """
    
    corrcoef(X) where X is a matrix returns a matrix of correlation
    coefficients for each row of X.
    
    corrcoef(x,y) where x and y are vectors returns the matrix or
    correlation coefficients for x and y.

    Numeric arrays can be real or complex

    The correlation matrix is defined from the covariance matrix C as

    r(i,j) = C[i,j] / (C[i,i]*C[j,j])
    """

    if len(args)==2:
        X = transpose(array([args[0]]+[args[1]]))
    elif len(args==1):
        X = args[0]
    else:
        raise RuntimeError, 'Only expecting 1 or 2 arguments'

    
    C = cov(X)
    d = resize(diagonal(C), (2,1))
    r = divide(C,sqrt(matrixmultiply(d,transpose(d))))
    try: return r.real
    except AttributeError: return r




def polyfit(x,y,N):
    """

    Do a best fit polynomial of order N of y to x.  Return value is a
    vector of polynomial coefficients [pk ... p1 p0].  Eg, for N=2

      p2*x0^2 +  p1*x0 + p0 = y1
      p2*x1^2 +  p1*x1 + p0 = y1
      p2*x2^2 +  p1*x2 + p0 = y2
      .....
      p2*xk^2 +  p1*xk + p0 = yk
      
      
    Method: if X is a the Vandermonde Matrix computed from x (see
    http://mathworld.wolfram.com/VandermondeMatrix.html), then the
    polynomial least squares solution is given by the 'p' in

      X*p = y

    where X is a len(x) x N+1 matrix, p is a N+1 length vector, and y
    is a len(x) x 1 vector

    This equation can be solved as

      p = (XT*X)^-1 * XT * y

    where XT is the transpose of X and -1 denotes the inverse.

    For more info, see
    http://mathworld.wolfram.com/LeastSquaresFittingPolynomial.html,
    but note that the k's and n's in the superscripts and subscripts
    on that page.  The linear algebra is correct, however.

    See also polyval

    """

    y = reshape(y, (len(y),1))
    X = Matrix(vander(x, N+1))
    Xt = Matrix(transpose(X))
    c = array(inverse(Xt*X)*Xt*y)  # convert back to array
    c.shape = (N+1,)
    return c
    

    

def polyval(p,x):
    """
    y = polyval(p,x)

    p is a vector of polynomial coeffients and y is the polynomial
    evaluated at x.

    Example code to remove a polynomial (quadratic) trend from y:

      p = polyfit(x, y, 2)
      trend = polyval(p, x)
      resid = y - trend

    See also polyfit
    
    """

    p = reshape(p, (len(p),1))
    X = vander(x,len(p))
    y =  matrixmultiply(X,p)
    return reshape(y, x.shape)


def vander(x,N=None):
    """
    X = vander(x,N=None)

    The Vandermonde matrix of vector x.  The i-th column of X is the
    the i-th power of x.  N is the maximum power to compute; if N is
    None it defaults to len(x).

    """
    if N is None: N=len(x)
    X = ones( (len(x),N), x.typecode())
    for i in range(N-1):
        X[:,i] = x**(N-i-1)
    return X



def donothing_callback(*args):
    pass

def cohere_pairs( X, ij, NFFT=256, Fs=2, detrend=detrend_none,
                  window=window_hanning, noverlap=0,
                  preferSpeedOverMemory=1,
                  progressCallback=donothing_callback):

    """
    Cxy, Phase, freqs = cohere_pairs( X, ij, ...)
    
    Compute the coherence for all pairs in ij.  X is a
    numSamples,numCols Numeric array.  ij is a list of tuples (i,j).
    Each tuple is a pair of indexes into the columns of X for which
    you want to compute coherence.  For example, if X has 64 columns,
    and you want to compute all nonredundant pairs, define ij as

      ij = []
      for i in range(64):
          for j in range(i+1,64):
              ij.append( (i,j) )

    The other function arguments, except for 'preferSpeedOverMemory'
    (see below), are explained in the help string of 'psd'.

    Return value is a tuple (Cxy, Phase, freqs).

      Cxy -- a dictionary of (i,j) tuples -> coherence vector for that
        pair.  Ie, Cxy[(i,j) = cohere(X[:,i], X[:,j]).  Number of
        dictionary keys is len(ij)
      
      Phase -- a dictionary of phases of the cross spectral density at
        each frequency for each pair.  keys are (i,j).

      freqs -- a vector of frequencies, equal in length to either the
        coherence or phase vectors for any i,j key.  Eg, to make a coherence
        Bode plot:

          subplot(211)
          plot( freqs, Cxy[(12,19)])
          subplot(212)
          plot( freqs, Phase[(12,19)])
      
    For a large number of pairs, cohere_pairs can be much more
    efficient than just calling cohere for each pair, because it
    caches most of the intensive computations.  If N is the number of
    pairs, this function is O(N) for most of the heavy lifting,
    whereas calling cohere for each pair is O(N^2).  However, because
    of the caching, it is also more memory intensive, making 2
    additional complex arrays with approximately the same number of
    elements as X.

    The parameter 'preferSpeedOverMemory', if false, limits the
    caching by only making one, rather than two, complex cache arrays.
    This is useful if memory becomes critical.  Even when
    preferSpeedOverMemory is false, cohere_pairs will still give
    significant performace gains over calling cohere for each pair,
    and will use subtantially less memory than if
    preferSpeedOverMemory is true.  In my tests with a 43000,64 array
    over all nonredundant pairs, preferSpeedOverMemory=1 delivered a
    33% performace boost on a 1.7GHZ Athlon with 512MB RAM compared
    with preferSpeedOverMemory=0.  But both solutions were more than
    10x faster than naievly crunching all possible pairs through
    cohere.

    See test/cohere_pairs_test.py in the src tree for an example
    script that shows that this cohere_pairs and cohere give the same
    results for a given pair.

    """

    
    if NFFT % 2:
        raise ValueError, 'NFFT must be a power of 2'

    numRows, numCols = X.shape

    # zero pad if X is too short
    if numRows < NFFT:
        tmp = X
        X = zeros( (NFFT, numCols), X.typecode())
        X[:numRows,:] = tmp
        del tmp

    numRows, numCols = X.shape
    # get all the columns of X that we are interested in by checking
    # the ij tuples
    seen = {}
    for i,j in ij:
        seen[i]=1; seen[j] = 1
    allColumns = seen.keys()
    Ncols = len(allColumns)
    del seen
    
    # for real X, ignore the negative frequencies
    if X.typecode()==Complex: numFreqs = NFFT
    else: numFreqs = NFFT//2+1

    # cache the FFT of every windowed, detrended NFFT length segement
    # of every channel.  If preferSpeedOverMemory, cache the conjugate
    # as well
    windowVals = window(ones((NFFT,), X.typecode()))
    ind = range(0, numRows-NFFT+1, NFFT-noverlap)
    numSlices = len(ind)
    FFTSlices = {}
    FFTConjSlices = {}
    Pxx = {}
    slices = range(numSlices)
    normVal = norm(windowVals)**2
    for iCol in allColumns:
        progressCallback(i/Ncols, 'Cacheing FFTs')
        Slices = zeros( (numSlices,numFreqs), Complex)
        for iSlice in slices:                    
            thisSlice = X[ind[iSlice]:ind[iSlice]+NFFT, iCol]
            thisSlice = windowVals*detrend(thisSlice)
            Slices[iSlice,:] = fft(thisSlice)[:numFreqs]
            
        FFTSlices[iCol] = Slices
        if preferSpeedOverMemory:
            FFTConjSlices[iCol] = conjugate(Slices)
        Pxx[iCol] = divide(mean(absolute(Slices)**2), normVal)
    del slices, ind, windowVals    

    # compute the coherences and phases for all pairs using the
    # cached FFTs
    Cxy = {}
    Phase = {}
    count = 0
    N = len(ij)
    for i,j in ij:
        count +=1
        if count%10==0:
            progressCallback(count/N, 'Computing coherences')

        if preferSpeedOverMemory:
            Pxy = FFTSlices[i] * FFTConjSlices[j]
        else:
            Pxy = FFTSlices[i] * conjugate(FFTSlices[j])
        if numSlices>1: Pxy = mean(Pxy)
        Pxy = divide(Pxy, normVal)
        Cxy[(i,j)] = divide(absolute(Pxy)**2, Pxx[i]*Pxx[j])
        Phase[(i,j)] =  arctan2(Pxy.imag, Pxy.real)

    freqs = Fs/NFFT*arange(numFreqs)
    return Cxy, Phase, freqs



def hist(y, bins=10, normed=0):
    """
    Return the histogram of y with bins equally sized bins.  If bins
    is an array, use the bins.  Return value is
    (n,x) where n is the count for each bin in x

    If normed is False, return the counts in the first element of the
    return tuple.  If normed is True, return the probability density
    n/(len(y)*dbin)
    
    Credits: the Numeric 22 documentation


    """
    if not iterable(bins):
        ymin, ymax = min(y), max(y)
        if ymin==ymax:
            ymin -= 0.5
            ymax += 0.5
        bins = linspace(ymin, ymax, bins)

    n = searchsorted(sort(y), bins)
    n = diff(concatenate([n, [len(y)]]))
    if normed:
       db = bins[1]-bins[0]
       return 1/(len(y)*db)*n, bins
    else:
       return n, bins


def normpdf(x, *args):
   "Return the normal pdf evaluated at x; args provides mu, sigma"
   mu, sigma = args
   return 1/(sqrt(2*pi)*sigma)*exp(-0.5 * (1/sigma*(x - mu))**2)
                 

def levypdf(x, gamma, alpha):
   "Returm the levy pdf evaluated at x for params gamma, alpha"

   N = len(x)

   if N%2 != 0:
      raise ValueError, 'x must be an event length array; try\n' + \
            'x = linspace(minx, maxx, N), where N is even'
   

   dx = x[1]-x[0]

   Nyq = 1/(2*dx)
   f = 1/(N*dx)*arange(-N/2, N/2, typecode=Float)

   ind = concatenate([arange(N/2, N, typecode=Int),
                      arange(N/2,typecode=Int)])
   df = f[1]-f[0]
   cfl = exp(-gamma*absolute(2*pi*f)**alpha)

   px = fft(take(cfl,ind)*df).astype(Float)
   return take(px, ind)




      

def find(condition):
   "Return the indices where condition is true"
   return nonzero(condition)



def trapz(x, y):
   if len(x)!=len(y):
      raise ValueError, 'x and y must have the same length'
   if len(x)<2:
      raise ValueError, 'x and y must have > 1 element'
   return sum(0.5*diff(x)*(y[1:]+y[:-1]))
   
   

def longest_contiguous_ones(x):
    """
    return the indicies of the longest stretch of contiguous ones in x,
    assuming x is a vector of zeros and ones.
    """
    if len(x)==0: return array([])

    ind = find(x==0)
    if len(ind)==0:  return arange(len(x))
    if len(ind)==len(x): return array([])

    y = zeros( (len(x)+2,),  x.typecode())
    y[1:-1] = x
    dif = diff(y)
    up = find(d ==  1);
    dn = find(d == -1);
    ind = find( dn-up == max(dn - up))
    ind = arange(take(up, ind), take(dn, ind))

    return ind


def longest_ones(x):
    """
    return the indicies of the longest stretch of contiguous ones in x,
    assuming x is a vector of zeros and ones.

    If there are two equally long stretches, pick the first
    """
    try: x.shape
    except AttributeError: x = array(x)
    if len(x)==0: return array([])

    #print 'x', x
    ind = find(x==0)
    if len(ind)==0:  return arange(len(x))
    if len(ind)==len(x): return array([])

    y = zeros( (len(x)+2,), Int)
    y[1:-1] = x
    d = diff(y)
    #print 'd', d
    up = find(d ==  1);
    dn = find(d == -1);

    #print 'dn', dn, 'up', up, 
    ind = find( dn-up == max(dn - up))
    # pick the first
    if iterable(ind): ind = ind[0]
    ind = arange(up[ind], dn[ind])

    return ind
