"""
Abstract base classes definine the primitives that renderers and
graphics contexts must implement to serve as a matplotlib backend

"""

from __future__ import division
import sys
from artist import Artist
from patches import Rectangle
from cbook import is_string_like
class RendererBase:

    def draw_arc(self, gc, filled, x, y, width, height, angle1, angle2):
        """
        Draw an arc centered at x,y with width and height and angles
        from 0.0 to 360.0
        """
        pass  # derived must override

    def draw_line(self, gc, x1, y1, x2, y2):
        """
        Draw a single line from x1,y1 to x2,y2
        """
        pass  # derived must override

    def draw_lines(self, gc, x, y):
        """
        x and y are equal length arrays, draw lines connecting each
        point in x, y
        """
        pass  # derived must override

    def draw_point(self, gc, x, y):
        """
        Draw a single point at x,y
        """
        pass  # derived must override

    def draw_polygon(self, gc, filled, points):
        """
        Draw a polygon.  points is a len vertices tuple, each element
        giving the x,y coords a vertex
        """  
        pass # derived must override

    def draw_rectangle(self, gc, filled, x, y, width, height):
        """
        Draw a rectangle with lower left at x,y with width and height If
        filled=True, fill the rectangle with the gc foreground gc is a
        GraphicsContext instance
        """
        pass # derived must override

    def new_gc(self):
        """
        Return an instance of a GraphicsContextBase
        """
        return GraphicsContextBase()


def hex2color(s):
    "Convert hex string (like html uses, eg, #efefef) to a r,g,b tuple"
    if s.find('#')!=0 or len(s)!=7:
        raise ValueError, 's must be a hex string like "#efefef#'
    r,g,b = map(lambda x: int('0x' + x, 16)/256.0, (s[1:3], s[3:5], s[5:7]))
    return r,g,b

class GraphicsContextBase:

    colors = {
        'b' : (0.0, 0.0, 1.0),
        'g' : (0.0, 0.5, 0.0),
        'r' : (1.0, 0.0, 0.0),
        'c' : (0.0, 0.75, 0.75),
        'm' : (0.75, 0, 0.75),
        'y' : (0.75, 0.75, 0),
        'k' : (0.0, 0.0, 0.0),
        'w' : (1.0, 1.0, 1.0),
        }

    

    def __init__(self):
        self._rgb = (0.0, 0.0, 0.0)
        self._linewidth = 1
        self._capstyle = 'butt'
        self._joinstyle = 'miter'
        self._dashes = None, None
        self._cliprect = None
        
    def _arg_to_rgb(self, arg):
        """
        returns a tuple of three floats from 0-1.  arg can be a
        matlab format string, a html hex color string, an rgb tuple,
        or a float between 0 and 1.  In the latter case, grayscale is
        used
        """

        def looks_like_rgb():
            try: float(arg[2])
            except: return 0
            else: return 1


        if looks_like_rgb():
            return arg

        try: float(arg)
        except: 
            if is_string_like(arg) and len(arg)==7 and arg[0]=='#':
                #print 'hex string:', arg
                return  hex2color(arg)
            else: 
                return self.colors.get(arg, (0.0, 0.0, 0.0))
        else:
            if arg>=0 and arg<=1:
                print 1
                return (arg,arg,arg)
            else:
                print 2
                msg = 'Floating point color arg must be between 0 and 1\n' +\
                      'Found %1.2f' % arg
                error_msg(msg)



    def get_clip_rectangle(self):
        """
        Return the clip rectangle as (left, bottom, width, height)
        """
        return self._cliprect

    def get_dashes(self):
        """
        Return the dash information as an offset dashlist tuple The
        dash list is a even size list that gives the ink on, ink off
        in pixels.  See p107 of to postscript BLUEBOOK for more info

        Default value is None
        """
        return self._dashes

    def get_capstyle(self):
        """
        Return the capstyle as a string in ('butt', 'round', 'projecting')
        """
        return self._capstyle

    def get_joinstyle(self):
        """
        Return the line join style as one of ('miter', 'round', 'bevel')
        """
        return self._joinstyle

    def get_rgb(self):
        """
        returns a tuple of three floats from 0-1.  color can be a
        matlab format string, a html hex color string, or a rgb tuple
        """
        return self._rgb

    def set_clip_rectangle(self, rectangle):
        """
        Set the clip rectangle with sequence (left, bottom, width, height)
        """
        self._cliprect = rectangle

    def set_dashes(self, dash_offset, dash_list):
        """
        Set the dash style for the gc.  dash offset is the offset
        (usually 0).  Dash list specifies the on-off sequence as
        points
        """
        self._dashes = dash_offset, dash_list
    
    def set_foreground(self, fg):
        """
        Set the foreground color.  fg can be a matlab format string, a
        html hex color string, an rgb unit tuple, or a float between 0
        and 1.  In the latter case, grayscale is used.
        """
        self._rgb = self._arg_to_rgb(fg)

    def set_graylevel(self, frac):
        """
        Set the foreground color to be a gray level with frac frac
        """
        self._rgb = (frac, frac, frac)
        

    def set_linewidth(self, w):
        """
        Set the linewidth in points
        """
        self._linewidth = w

    def set_capstyle(self, cs):
        """
        Set the capstyle as a string in ('butt', 'round', 'projecting')
        """
        if cs not in ('butt', 'round', 'projecting'):
            error_msg('Unrecognized cap style.  Found %s' % cs)
        self._capstyle = cs

    def set_joinstyle(self, js):
        """
        Set the join style to be one of ('miter', 'round', 'bevel')
        """
        
        if js not in ('miter', 'round', 'bevel'):
            error_msg('Unrecognized join style.  Found %s' % js)
        self._joinstyle = js


    def get_linewidth(self):
        """
        Return the line width in points as a scalar
        """
        return self._linewidth

class AxisTextBase(Artist):
    """
    Handle storing and drawing of text in window or data coordinates
    """
    fontweights = {'normal' : None,  # set these constants as app for backend
                   'bold' : None,
                   'heavy' : None,
                   'light' : None,
                   'ultrabold' : None,
                   'ultralight' : None,
                   }
    fontangles = {
        'italic': None,
        'normal' : None,
        'oblique' : None,
        }

    def __init__(self, x=0, y=0, text='',
                 color='k',
                 verticalalignment='bottom',
                 horizontalalignment='left',
                 fontname='Times',
                 fontsize=10,
                 fontweight='normal',
                 fontangle='normal',
                 rotation=None,
                 ):
        Artist.__init__(self)
        self._x, self._y = x, y

        self._color = color
        self._text = text
        self._verticalalignment = verticalalignment
        self._horizontalalignment = horizontalalignment
        self._rotation = rotation
        self._fontname = fontname
        self._fontsize = fontsize
        self._fontweight = fontweight
        self._fontangle = fontangle
        self._reset=1
        self._eraseImg = None

        
    def _compute_offsets(self):
        """
        Return the (x,y) offsets to adjust for the alignment
        specifications
        """
        return 0,0  # derived must override

    def _draw(self, drawable, *args, **kwargs):
        """
        Render the text to the drawable 
        """
        pass # derived must override

    def erase(self):
        pass # derived must override (optional)

    def get_data_extent(self):
        """
        Return the ink extent of the text as (lower, bottom, width, height)
        """
        return 0,0,0,0  # derived must override

    def get_fontname(self):
        "Return the font name as string"
        return self._fontname

    def get_fontsize(self):
        "Return the font size as integer"
        return self._fontsize

    def get_fontweight(self):
        "Get the font weight as string"
        return self._fontweight

    def get_fontangle(self):
        "Get the font angle as string"
        return self._fontangle

    def get_horizontalalignment(self):
        "Return the horizontal alignment as string"
        return self._horizontalalignment

    def get_position(self):
        return self._x, self._y
    
    def get_text(self):
        "Get the text as string"
        return self._text

    def get_verticalalignment(self):
        "Return the vertical alignment as string"
        return self._verticalalignment

    def get_left_right(self):
        "get the left, right boundaries of the text in in win coords"
        ox,oy = self._compute_offsets()
        return self._x + ox, self._x + ox + self._width

    def get_bottom_top(self):
        "get the  bottom, top boundaries of the text in win coords"

        ox,oy = self._compute_offsets()
        return self._y + oy + self._height, self._y

    def get_position(self):
        "Return x, y as tuple"
        return self._x, self._y


    def wash_brushes(self):
        "Flush all state vars and prepare for a clean redraw"
        self._reset = 1
        self._eraseImg = None
        
    def set_backgroundcolor(self, color):
        "Set the background color of the text"
        if color == self._backgroundcolor: return 
        self._state_change()
        self._backgroundcolor = color

        
    def set_color(self, color):
        "Set the foreground color of the text"
        if self._color == color: return 
        self._state_change()
        self._color = color

    def set_horizontalalignment(self, align):
        """
        Set the horizontal alignment to one of
        'center', 'right', or 'left'
        """
        legal = ('center', 'right', 'left')
        if align not in legal:
            raise ValueError,\
                  'Horizontal alignment must be one of %s' % legal
        if self._horizontalalignment == align: return     
        self._state_change()
        self._horizontalalignment = align


    def _set_font(self):
        "Update the font object"
        pass # derived must override

    def set_fontname(self, fontname):
        """
        Set the font name, eg, 'Sans', 'Courier', 'Helvetica'
        """
        if self._fontname == fontname: return
        self._state_change()
        self._fontname = fontname
        

    def set_fontsize(self, fontsize):
        """
        Set the font size, eg, 8, 10, 12, 14...
        """
        if self._fontsize == fontsize: return
        self._state_change()
        self._fontsize = fontsize
        
        
    def set_fontweight(self, weight):
        """
        Set the font weight, one of:
        'normal', 'bold', 'heavy', 'light', 'ultrabold',  'ultralight'
        """
        if self._fontweight == weight: return
        self._state_change()
        self._fontweight = weight
        
        
    def set_fontangle(self, angle):
        """
        Set the font angle, one of 'normal', 'italic', 'oblique'
        """
        if self._fontangle == angle: return  
        self._state_change()
        self._fontangle = angle
        
        
    def set_position(self, xy):
        if xy == (self._x, self._y): return
        self._state_change()
        self._x, self._y = xy        
        
    def set_rotation(self, s):
        "Currently only s='vertical', or s='horizontal' are supported"
        if s==self._rotation: return
        self._state_change()
        self._rotation = s
        
        
    def set_verticalalignment(self, align):
        """
        Set the vertical alignment to one of
        'center', 'top', or 'bottom'
        """

        legal = ('top', 'bottom', 'center')
        if align not in legal:
            raise ValueError,\
                  'Vertical alignment must be one of %s' % legal

        if self._verticalalignment == align: return
        self._state_change()
        self._verticalalignment = align
        
        
    def set_drawing_area(self, da):
        "Update the drawing area the widget renders into"
        # todo: get rid of set drawing area 
        #print 'Setting da for', self._text
        Artist.set_drawing_area(self, da)

        self._state_change()

    def set_text(self, text):
        "Set the text"
        if self._text == text: return
        self._state_change()
        self._text = text
        
    def _state_change(self):
        self._resets = 1
    

    def update_properties(self, d):
        "Update the font attributes with the dictionary in d"
        #check the keys
        
        legal = ('color', 'verticalalignment', 'horizontalalignment',
                 'fontname', 'fontsize', 'fontweight',
                 'fontangle', 'rotation')
        for k,v in d.items():
            if k not in legal:
                raise ValueError, 'Illegal key %s.  Legal values are %s' % (
                    k, legal)
            self.__dict__['_' + k] = v
            #print self.__dict__
        self._state_change()
        
    def __del__(self):
        "Bye bye"
        self.erase()

class FigureBase:
    def __init__(self, figsize, dpi, facecolor=0.75, edgecolor='k'):
        """
        paper size is a w,h tuple in inches
        DPI is dots per inch 
        """
        self.figsize = figsize
        self.dpi = dpi
        self.axes = []
        self.drawable = None
        self._text=[]
        self._width = figsize[0]*dpi
        self._height = figsize[1]*dpi
        
        self._figurePatch = Rectangle(
            (0,0), self._width, self._height,
            facecolor=facecolor, edgecolor=edgecolor)
        

    def add_axis(self, a):
        self.axes.append(a)
        a.set_size(self.figsize, self.dpi)
                
    def clear(self):
        self.axes = []
        self.draw()
        
    def draw(self, drawable=None, *args, **kwargs):
        """
        Render the figure using Renderer instance drawable
        """
        pass  # derived must override 

    def get_axes(self):
        return self.axes

    def realize(self, *args):
        """
        This method will be called when the system is ready to draw,
        eg when a GUI window is realized
        """
        pass   # derived must override
    
    def text(self, x, y, text, *args, **kwargs):
        """
        Add text to figure at location x,y (relative 0-1 coords) See
        the help for Axis text for the meaning of the other arguments
        """
        pass

def _process_text_args(override, *args, **kwargs):
    "Return an override dict.  See 'text' docstring for info"

    if len(args)>1:
        raise ValueError, 'Only a single optional arg can be supplied to text'
    if len(args)==1:
        if not isinstance(args[0], dict):
            msg = 'The optional nonkeyword argument to text must be a dict'
            raise TypeError, msg

        override.update(args[0])
    override.update(kwargs)
    return override



def error_msg(msg, *args, **kwargs):
    """
    Alert an error condition with message
    """
    print >>sys.stderr, msg
    sys.exit()
