"""
This module contains all the 2D line class which can draw with a
variety of line styles, markers and colors
"""
#TODO readd vline
from __future__ import generators
from __future__ import division

from Numeric import Float, alltrue, arange, array, logical_and,\
     nonzero, reshape, resize, searchsorted, take, Int

from artist import Artist

True = 1
False = 0

lineStyles = {'-':1, '--':1, '-.':1, ':':1}
lineMarkers =    {'.':1, ',':1, 'o':1, '^':1,
                  'v':1, '<':1, '>':1, 's':1, '+':1}

class Line2D(Artist):

    def __init__(self, xdata, ydata,
                 linewidth=0.5, linestyle='-',
                 color='b', 
                 marker=None, markersize=6,
                 markeredgecolor=None, markerfacecolor=None):
        Artist.__init__(self)
        #convert sequences to numeric arrays

        self._linestyle = linestyle
        self._linewidth = linewidth
        self._color = color
        self._markersize = markersize
        if markeredgecolor is None: markeredgecolor=color
        
        self._markerfacecolor = markerfacecolor
        self._markeredgecolor = markeredgecolor

        self.verticalOffset = None        
        self.set_data(xdata, ydata)


        self._lineStyles =  {
            '-'  : self._draw_solid,
            '--' : self._draw_dashed, 
            '-.' : self._draw_dash_dot, 
            ':'  : self._draw_dotted,
            None : self._draw_nothing}

        self._markers =  {
            '.'  : self._draw_point,
            ','  : self._draw_pixel, 
            'o'  : self._draw_circle, 
            'v'  : self._draw_triangle_down, 
            '^'  : self._draw_triangle_up, 
            '<'  : self._draw_triangle_left, 
            '>'  : self._draw_triangle_right, 
            's'  : self._draw_square,
            '+'  : self._draw_plus,
            None : self._draw_nothing
            }

        if not self._lineStyles.has_key(linestyle):
            print 'oops style', linestyle
            pass # todo: warn
        if not self._markers.has_key(marker):
            print 'oops marker', marker
            pass # todo: warn
        
        self._lineFunc = self._lineStyles.get(linestyle, self._draw_nothing)
        self._markerFunc = self._markers.get(marker, self._draw_nothing)
        
    def set_data(self, x, y):
        try: x.shape
        except AttributeError: self._x = array(x, Float)
        else: self._x = x
        
        try: y.shape
        except AttributeError: self._y = array(y, Float)
        else: self._y = y

        N = max(self._x.shape)
        #if 1 in self._x.shape:
        #    self._x.shape  = (N,)
        #if 1 in self._y.shape:
        #    self._y.shape  = (N,)

        self._xsorted = self._is_sorted(self._x)

    def set_vertical_offset(self, voff):
        self.verticalOffset = voff
        
    def _is_sorted(self, x):
        "return true if x is sorted"
        if len(x)<2: return 1
        return alltrue(x[1:]-x[0:-1]>=0)
    
    def _get_numeric_clipped_data_in_range(self):
        # if the x or y clip is set, only plot the points in the
        # clipping region
        try: self._xc, self._yc
        except AttributeError: x, y = self._x, self._y
        else: x, y = self._xc, self._yc
        # transform into axes coords            

        if self.verticalOffset is not None:
            #print self.verticalOffset, y.typecode()
            y = y + self.verticalOffset
            
        return x, y

    def _draw(self, drawable, *args, **kwargs):
        
        x, y = self._get_numeric_clipped_data_in_range()
        if len(x)==0: return 

        xt, yt = self.transform_xy_to_display(x, y)

        gc = drawable.new_gc()
        gc.set_foreground(self._color)
        gc.set_linewidth(self._linewidth)
        self.clip_gc(gc)
        self._lineFunc(drawable, gc, xt, yt)

        gc = drawable.new_gc()
        gc.set_foreground(self._color)
        gc.set_linewidth(self._linewidth)
        self.clip_gc(gc)

        self._markerFunc(drawable, gc, xt, yt)

    def _derived_draw(self, drawable, gc, x, y):
        raise NotImplementedError, 'Line2D is a pure base class.  ' + \
              'You must instantiate a derived class'

    def flush_clip(self):
        delList = ['_xmin', '_xmax', '_ymin', '_ymax', '_xc', '_yc']
        for item in delList:
            try: del self.__dict__[item]
            except KeyError: pass

    def get_xdata(self): return self._x
    def get_ydata(self):  return self._y
    def get_linewidth(self): return self._linewidth
    def get_linestyle(self): return self._linestyle
    def get_markersize(self): return self._markersize
    def get_markerfacecolor(self): return self._markerfacecolor
    def get_markeredgecolor(self): return self._markeredgecolor
    def get_color(self): return self._color

    def get_data_extent(self):
        """
        Return the bounding box as left, bottom, width, height in
        window coords
        """
        x, y = self._get_numeric_clipped_data_in_range()
        minx, maxx = min(x), max(x)
        miny, maxy = min(y), max(y)
        return minx, maxy, maxx-minx, maxy-miny
        
    def _set_clip(self):

        try: self._xmin, self._xmax
        except AttributeError: indx = arange(len(self._x))
        else:
            if len(self._x)==1:
                indx = 0
            elif self._xsorted:
                # for really long signals, if we know they are sorted
                # on x we can save a lot of time using search sorted
                # since the alternative approach requires 3 O(len(x) ) ops
                inds = searchsorted(
                    self._x, array([self._xmin, self._xmax]))

                skip = 0
                if self._lod:
                    # if level of detail is on, decimate the data
                    # based on pixel width
                    l, b, w, h = self.get_window_extent()
                    skip = int((inds[1]-inds[0])/w)                    
                if skip>0:  indx = arange(inds[0], inds[1], skip)
                else: indx = arange(inds[0], inds[1])
            else:
                indx = nonzero(
                    logical_and( self._x>=self._xmin,
                                       self._x<=self._xmax ))
        

        self._xc = take(self._x, indx)
        self._yc = take(self._y, indx)

        # y data clipping for connected lines can introduce horizontal
        # line artifacts near the clip region.  If you really need y
        # clipping for efficiency, consider using plot(y,x) instead.
        if ( self._yc.shape==self._xc.shape and 
             self._linestyle is None):
            try: self._ymin, self._ymax
            except AttributeError: indy = arange(len(self._yc))
            else: indy = nonzero(
                logical_and(self._yc>=self._ymin,
                                  self._yc<=self._ymax ))
        else:
            indy = arange(len(self._yc))
            
        self._xc = take(self._xc, indy)
        self._yc = take(self._yc, indy)

    def set_color(self, color):
        self._color = color

    def set_linewidth(self, w):
        self._linewidth = w

    def set_linestyle(self, s):
        self._linestyle = s
        self._drawFunc = self._lineStyles.get(self._linestyle, '-')

    def set_markeredgecolor(self, ec):
        self._markeredgecolor = ec

    def set_markerfacecolor(self, fc):
        self._markerfacecolor = fc

    def set_markersize(self, sz):
        self._markersize = sz

    def set_xdata(self, x):
        self.set_data(x, self._y)

    def set_ydata(self, y):
        self.set_data(self._x, y)
        
    def set_xclip(self, xmin, xmax):
        self._xmin, self._xmax = xmin, xmax
        self._set_clip()

    def set_yclip(self, ymin, ymax):
        self._ymin, self._ymax = ymin, ymax
        self._set_clip()

    def _draw_nothing(self, drawable, gc, xt, yt):
        pass
    
    def _draw_solid(self, drawable, gc, xt, yt):
        drawable.draw_lines(gc, xt,yt)

    def _draw_dashed(self, drawable, gc, xt, yt):
        dashes = self._dpi/72.0*array([6.0, 6.0])
        gc.set_dashes(0, dashes)  #  gdk.LINE_ON_OFF_DASH
        gc.set_capstyle('butt')   #  gdk.CAP_BUTT
        gc.set_joinstyle('miter')  #  gdk.JOIN_MITER
        drawable.draw_lines(gc, xt, yt)

    def _draw_dash_dot(self, drawable, gc, xt, yt):
        dashes = self._dpi/72.0*array([3.0, 5.0, 1.0, 5.0])
        gc.set_dashes(0, dashes)  #3 on, 5 off, 1 on, 5 off
        gc.set_capstyle('butt')   
        gc.set_joinstyle('miter') 
        drawable.draw_lines(gc, xt, yt)

    def _draw_dotted(self, drawable, gc, xt, yt):
        dashes = self._dpi/72.0*array([1.0, 2.0])
        gc.set_dashes(0, dashes)  
        gc.set_capstyle('butt')   
        gc.set_joinstyle('miter') 
        drawable.draw_lines(gc, xt, yt)

    def _draw_point(self, drawable, gc, xt, yt):
        offset = 1
        for (x,y) in zip(xt, yt):
            drawable.draw_arc(gc, True, x, y, 1, 1, 0.0, 360.0)


    def _draw_pixel(self, drawable, gc, xt, yt):
        for (x,y) in zip(xt, yt):
            drawable.draw_point(gc, x, y)


    def _draw_circle(self, drawable, gc, xt, yt):
        radx = rady = 0.5*self._markersize*self._dpi/72.0
        for (x,y) in zip(xt, yt):
            gc.set_foreground(self._markeredgecolor)
            drawable.draw_arc(gc, False, x, y, radx, rady, 0.0, 360.0)
            if self._markerfacecolor is not None:
                gc.set_foreground(self._markerfacecolor)
                drawable.draw_arc(gc, True, x, y, radx, rady, 0.0, 360.0)

    def _draw_triangle_up(self, drawable, gc, xt, yt):
        offset = 0.5*self._markersize*self._dpi/72.0
        
        for (x,y) in zip(xt, yt):
            verts = ( (x, y+offset),
                      (x-offset, y-offset),
                      (x+offset, y-offset) )
            gc.set_foreground(self._markeredgecolor)
            drawable.draw_polygon(gc, False, verts)
            if self._markerfacecolor is not None:
                gc.set_foreground(self._markerfacecolor)
                drawable.draw_polygon(gc, True, verts)


    def _draw_triangle_down(self, drawable, gc, xt, yt):
        offset = 0.5*self._markersize*self._dpi/72.0
        for (x,y) in zip(xt, yt):            
            verts = ( (x-offset, y+offset),
                      (x+offset, y+offset),
                      (x, y-offset))
            gc.set_foreground(self._markeredgecolor)
            drawable.draw_polygon(gc, False, verts)
            if self._markerfacecolor is not None:
                gc.set_foreground(self._markerfacecolor)
                drawable.draw_polygon(gc, True, verts)

    def _draw_triangle_left(self, drawable, gc, xt, yt):
        offset = 0.5*self._markersize*self._dpi/72.0
        for (x,y) in zip(xt, yt):            
            verts = ( (x-offset, y),
                      (x+offset, y-offset),
                      (x+offset, y+offset))
            gc.set_foreground(self._markeredgecolor)
            drawable.draw_polygon(gc, False, verts)
            if self._markerfacecolor is not None:
                gc.set_foreground(self._markerfacecolor)
                drawable.draw_polygon(gc, True, verts)


    def _draw_triangle_right(self, drawable, gc, xt, yt):
        offset = 0.5*self._markersize*self._dpi/72.0        
        for (x,y) in zip(xt, yt):            
            verts = ( (x+offset, y),
                      (x-offset, y-offset),
                      (x-offset, y+offset))
            gc.set_foreground(self._markeredgecolor)
            drawable.draw_polygon(gc, False, verts)
            if self._markerfacecolor is not None:
                gc.set_foreground(self._markerfacecolor)
                drawable.draw_polygon(gc, True, verts)

            

    def _draw_square(self, drawable, gc, xt, yt):
        side = self._dpi/72.0*self._markersize
        offset = side*0.5
        for (x,y) in zip(xt, yt):            
            gc.set_foreground(self._markeredgecolor)
            drawable.draw_rectangle(gc, False,
                                    x-offset, y-offset,
                                    side, side) 
            if self._markerfacecolor is not None:
                gc.set_foreground(self._markerfacecolor)
                drawable.draw_rectangle(gc, True,
                                        x-offset, y-offset,
                                        self._markersize, self._markersize) 

    def _draw_plus(self, drawable, gc, xt, yt):
        offset = self._markersize*self._dpi/72.0
        # todo: unify marker size args; use default vals for line
        # sytles in dict?
        for (x,y) in zip(xt, yt):
            drawable.draw_line(gc, x-offset, y, x+offset, y)
            drawable.draw_line(gc, x, y-offset, x, y+offset)

    def copy_properties(self, line):
        self._linestyle = line._linestyle
        self._linewidth = line._linewidth
        self._color = line._color
        self._markersize = line._markersize        
        self._markerfacecolor = line._markerfacecolor
        self._markeredgecolor = line._markeredgecolor
        self._lineFunc = line._lineFunc
        self._markerFunc = line._markerFunc
