from __future__ import division 
"""

 backend_wx.py

 A wxPython backend for matplotlib, based (very heavily) on
 backend_template.py and backend_gtk.py

 Author: Jeremy O'Donoghue (jeremy@o-donoghue.com)
 
 Derived from original copyright work by John Hunter
 (jdhunter@ace.bsd.uchicago.edu)

 Copyright (C) Jeremy O'Donoghue & John Hunter, 2003
 
 License: This work is licensed under the PSF. A copy should be included
 with this source code, and is also available at
 http://www.python.org/psf/license.html

"""
# CVS log information
#
# $Log: backend_wx.py,v $
# Revision 1.9  2003/11/11 22:23:51  jdh2358
# refactored axes
#
# Revision 1.8  2003/11/11 16:22:02  jodonoghue
# Added status bar. Added mousewheel support. Fixed script session does not exit cleanly, fixed legend not displayed, fixed no clipping on text labels, fixed vertical text renders improperly
#
#
"""
KNOWN BUGS -

- Under Windows 2000, the Figure window is larger than the figure
  (seems OK on Linux, however)

- pcolor almost works!  See pcolor_demo.  But there appears to be a
  rounding error in the locations of the patches, shich gives a
  banded appearance to the pcolor (I have a similar problem in GTK
  which I haven't figured out how to deal with)

- xlabel location is off (eg in errorbar_demo.py and legend_demo).
  This may mean that the bounding box of the text class is not
  correct

- we need to come up with a way of standardizing font names between
  backends.  I've thought about this but don't have a good solution.

- Display of axes menu under wxGTK is ugly (this is more of a wxGTK
  implementation issue!)
  
- No axes selected when menu initially called - have to select by hand.

- Using Y axis zoom causes vertical axis text to become misplaced.

- Vertical text renders incorrectly if you use a non TrueType font
  on Windows. This is a known wxPython issue.

JDH's MUSINGS

- I try to maintain a consistent naming convention -- mixedCase for
  non-callable variables, underscore_separated for functions,
  UpperCase for classes.  In several instances, you use underscore
  separated for variables, eg, ul_x and ul_y in draw_arc or
  wx_fontname in _set_font.  When you don't have anything better to
  do <wink>, consider replacing these with names consistent with the
  matplotlib naming convention.

Not implemented:

- Printing

Ugly hacks:

- Dotted lines - I have a nasty hack to make this work sufficiently
  for demonstration purposes. The problem is that you currently
  implement dotted lines in a rather GTK-specific way. I'd prefer,
  given that there are only a few 'dotted' styles, for these to have
  symbolic names (e.g. in a dictionary), and to do all of the
  decisions on how to draw the 'dotted' style in the back-end.

- As wxWindows has no 'stock' icons, I have converted GTK icons
  similar to those used in backend_gtk into XPM format and put them
  into an 'images' directory under backends. I'm open to your
  preference as to whether it would be better to put the images into
  python code or leave them as pixmaps.

Fixed this release:

- Added a status bar for tooltips and selected zoom function
- Added mousewheel support: mousewheel now executes the last
  selected interactive function (which is displayed on the toolbar
- On Windows 2000 script session does not exit cleanly [fixed - JOD]
- Legend not displayed [fixed - JOD]
- No clipping performed on text labels [fixed - JOD]
- Interactive session does not exit cleanly
- Vertical text renders improperly, eg, ylabels [fixed - JOD]
  (default fonts now all TrueType - non-TT doesn't work on Windows)

Examples which work on this release:

---------------------------------------------------------------
                        | Windows 2000    |  Linux            |
                        | wxPython 2.3.3  |  wxPython 2.4.2.4 |
--------------------------------------------------------------|
- arctest.py            |     OK          |
- axes_demo.py          |     OK  (1)     |
- color_demo.py         |     OK          |
- dynamic_demo.py       |     N/A (2)     |
- embedding_in_gtk.py   |     N/A (2)     |
- errorbar_demo.py      |     OK          |
- figtext.py            |     OK          |
- histogram_demo.py     |     OK          |
- interactive.py        |     N/A (2)     |
- interactive2.py       |     N/A (2)     |
- legend_demo.py        |     OK          |
- line_styles.py        |     OK          |
- log_demo.py           |     OK          |
- logo.py               |    FAIL (3)     |
- mpl_with_glade.py     |     N/A (2)     |
- mri_demo.py           |    FAIL (4)     |
- mri_demo_with_eeg.py  |    FAIL (4)     |
- multiple_figs_demo.py |     OK          |
- pcolor_demo.py        |     OK          |
- scatter_demo.py       |     OK          |
- simple_plot.py        |     OK          |
- stock_demo.py         |     OK          |
- subplot_demo.py       |     OK          |
- system_monitor.py     |     N/A (2)     |
- text_handles.py       |     OK          |
- text_themes.py        |     OK          |
- vline_demo.py         |     OK          |
---------------------------------------------------------------

(1) - Do not see data in top right axis until zoom approx x5
      (probably a bug)
(2) - Script uses GTK-specific features - cannot not run,
      but wxPython equivalent should be written.
(3) - Script fails with: No such file or directory:
      'data/membrane.dat' - get the data file!
      (probably not a bug)
(4) - Script fails with: ValueError: string size must be a multiple
      of element size. (probably not a bug)

"""

cvs_id = '$id$'
# Debug level
_DEBUG = 2
_DEBUG_lvls = {1 : 'Low ', 2 : 'Med ', 3 : 'High', 4 : 'Error' }

def DEBUG_MSG(string, lvl=3, o=None):
    if lvl >= _DEBUG:
        cls = o.__class__
        print >>sys.stderr, "%s - %s in %s" % (_DEBUG_lvls[lvl], string, cls)
 

import sys, os, os.path

try:
    from wxPython.wx import *
except:
    print >>sys.stderr, "Matplotlib backend_wx requires wxPython be installed"
    sys.exit()    

import distutils.sysconfig

from matplotlib.backend_bases import RendererBase, GraphicsContextBase,\
     FigureBase, AxisTextBase, _process_text_args, error_msg
from matplotlib._matlab_helpers import FigureManagerBase, GcfBase
from matplotlib.artist import Artist

wxapp = wxPySimpleApp()

defaultDPI = 75     # the default dots per inch resolution
defaultFigSize = (5,4)   # the default canvas size in inches


def error_msg_wx(msg, parent=None):
    """
    Signal an error condition -- in a GUI, popup a error dialog
    """
    dialog = wxMessageDialog(parent  = parent,
                             message = msg,
                             caption = 'Matplotlib backend_wx error',
                             style   = wxOK | wxCENTRE)
    dialog.ShowModal()
    dialog.Destroy()
    return None

    
class RendererWx(RendererBase):
    """
    The renderer handles all the drawing primitives using a graphics
    context instance that controls the colors/styles. It acts as the
    'drawable' instance used by many classes in the hierarchy.
    """
    #In wxPython, drawing is performed on a wxDC instance, which will
    #generally be mapped to the client aread of the window displaying
    #the plot. Under wxPython, the wxDC instance has a wxPen which
    #describes the colour and weight of any lines drawn, and a wxBrush
    #which describes the fill colour of any closed polygon.
    
    def __init__(self, win):
        """
        Initialise a wxWindows renderer instance.
        """
        DEBUG_MSG("__init__()", 1, self)
        self.width, self.height = win.GetSizeTuple()
        self.win = win
        self.gc = None
    
    def draw_arc(self, gc, filled, x, y, width, height, angle1, angle2):
        """
        Draw an arc centered at x,y with width and height and angles
        from 0.0 to 360.0.
        """
        DEBUG_MSG("draw_arc()", 1, self)
		# wxPython requires upper left corner of bounding rectange for ellipse
		# Theoretically you don't need the int() below, but it seems to make 
		# rounding of arc centre point more accurate in screen co-ordinates
        ulX = x - int(width/2)
        ulY = self.height - int(y + (height/2))
        if not filled:
            old_brush = gc.wxDC.GetBrush()
            gc.wxDC.SetBrush(wxTRANSPARENT_BRUSH)
        gc.wxDC.DrawEllipticArc(int(ulX), int(ulY), int(width), int(height),
                                int(angle1), int(angle2))
        if not filled:
            gc.wxDC.SetBrush(old_brush)
    
    def draw_line(self, gc, x1, y1, x2, y2):
        """
        Draw a single line from x1,y1 to x2,y2
        """
        DEBUG_MSG("draw_line()", 1, self)
        gc.wxDC.DrawLine(int(x1), self.height - int(y1),
                         int(x2), self.height - int(y2))
    
    def draw_lines(self, gc, x, y):
        """
        x and y are equal length arrays, draw lines connecting each
        point in x, y
        """
        DEBUG_MSG("draw_lines()", 1, self)
        assert len(x) == len(y), "draw_lines() x and y must be of equal length"
        gc.wxDC.DrawLines([wxPoint(int(x[i]), self.height - int(y[i])) for i in range(len(x))])
    
    def draw_polygon(self, gc, filled, points):
        """
        Draw a polygon.  points is a len vertices tuple, each element
        giving the x,y coords a vertex
        """  
        DEBUG_MSG("draw_polygon()", 1, self)
        points = [(int(x), self.height - int(y)) for x,y in points]
        if not filled:
            oldBrush = gc.wxDC.GetBrush()
            gc.wxDC.SetBrush(wxTRANSPARENT_BRUSH)
        gc.wxDC.DrawPolygon(points)
        if not filled:
            gc.wxDC.SetBrush(oldBrush)

    def draw_rectangle(self, gc, filled, x, y, width, height):
        """
        Draw a rectangle at lower left x,y with width and height
        If filled=True, fill the rectangle with the gc foreground
        gc is a GraphicsContext instance
        """
        # wxPython uses rectangle from TOP left!
        DEBUG_MSG("draw_rectangle()", 1, self)
        if not filled:
            oldBrush = gc.wxDC.GetBrush()
            gc.wxDC.SetBrush(wxTRANSPARENT_BRUSH)
        gc.wxDC.DrawRectangle(int(x), self.height - int(height + y), int(width), int(height))
        if not filled:
            gc.wxDC.SetBrush(oldBrush)

    def draw_point(self, gc, x, y):
        """
        Draw a single point at x,y
        """
        DEBUG_MSG("draw_point()", 1, self)
        gc.wxDC.DrawPoint(int(x), self.height - int(y))
        
    # Note that these text operations, unlike the drawing operations,
    # have already been translated to wxPython co-ordinates before calling
    # in the AxisTextWx class - and in other backends AxisText does not
    # perform calls via RendererXX - I believe that in the long term the
    # renderer is the 'logical' place to perform text operations, but for now
    # this is a reasonable solution for wxWindows.
    def draw_text(self, gc, t, x, y):
        """
        Draw text at x, y using attributes from gc
        """
        gc.wxDC.DrawText(t, x, y)
        
    def draw_rotated_text(self, gc, t, x, y, a=90):
        """
        Draw text at (x,y), rotated by a degrees, using attributes from gc
        """
        gc.wxDC.DrawRotatedText(t, x, y, a)

    def new_gc(self, wxDC=None):
        """
        Return an instance of a GraphicsContextWx, and sets the current gc copy
        """
        DEBUG_MSG("new_gc()", 1, self)
        if wxDC == None: wxDC = wxClientDC(self.win)
        self.gc = GraphicsContextWx(wxDC, self.win, self)
        return self.gc
        
    def set_gc(self, gc):
        """
        Set the locally cached gc.
        """
        # This is a dirty hack to allow anything with access to a drawable to
        # access the current graphics context
        self.gc = gc
        
    def get_gc(self):
        """
        Fetch the locally cached gc, or create a new one if there is not one
        already.
        """
        # This is a dirty hack to allow anything with access to a drawable to
        # access the current graphics context
        if self.gc == None:
            return self.new_gc()
        else:
            return self.gc

class GraphicsContextWx(GraphicsContextBase):
    """
    The graphics context provides the color, line styles, etc...
    
    In wxPython this is done by wrapping a wxDC object and forwarding the
    appropriate calls to it. Notice also that colour and line styles are
    mapped on the wxPen() member of the wxDC. This means that we have some
    rudimentary pen management here.
    
    The base GraphicsContext stores colors as a RGB tuple on the unit
    interval, eg, (0.5, 0.0, 1.0).  wxPython uses an int interval, but
    since wxPython colour management is rather simple, I have not chosen
    to implement a separate colour manager class.
    """
    _capd = { 'butt':       wxCAP_BUTT,
              'projecting': wxCAP_PROJECTING,
              'round':      wxCAP_ROUND }
    
    _joind = { 'bevel':     wxJOIN_BEVEL,
               'miter':     wxJOIN_MITER,
               'round':     wxJOIN_ROUND }
			   
    _dashd = { 6:           wxSHORT_DASH,
               3:           wxDOT_DASH,
               1:           wxDOT }
               
    def __init__(self, wxDC, win, drawable):
        GraphicsContextBase.__init__(self)
        DEBUG_MSG("__init__()", 1, self)
        self.wxDC = wxDC
        self.win = win
        self.wxDC.SetPen(wxPen('BLACK', 1, wxSOLID))
        self._style = wxSOLID
        self.drawable = drawable
        
    def set_clip_rectangle(self, rect):
        """
        Destroys previous clipping region and defines a new one.
        """
        DEBUG_MSG("set_clip_rectangle()", 1, self)
        l,b,w,h = rect
        self.wxDC.SetClippingRegion(int(l), self.drawable.height - int(b+h),
                                    int(w), int(h))
        
    def set_dashes(self, dashOffset, dashList):
        """
        Allows the mark to space ratio of a dashed line to be defined - this
        is given as an offset into an (even size) list giving on-off
        information in pixels.
        """
        DEBUG_MSG("set_dashes()", 1, self)
        # TODO: dash specification needs to be more device independent?
        GraphicsContextBase.set_dashes(self, dashOffset, dashList)
        # NOTE: this implementation is very dirty, and depends (in an
        # unacceptable way) on knowledge of the values in Line2D._draw_xxx()
        # functions - which may change at any time
        dv = int(dashList[0])
        try:
            self._style = GraphicsContextWx._dashd[dv]
        except KeyError:
            self._style = wxLONG_DASH # Style not used elsewhere...
        
        # On MS Windows platform, only line width of 1 allowed for dash lines
        if wxPlatform == '__WXMSW__':
            self.set_linewidth(1)
            
        pen = self.wxDC.GetPen()
        pen.SetStyle(self._style)
        self.wxDC.SetPen(pen)

    def set_foreground(self, fg):
        """
        Set the foreground color.  fg can be a matlab format string, a
        html hex color string, an rgb unit tuple, or a float between 0
        and 1.  In the latter case, grayscale is used.
        """
		# Implementation note: wxPython has a separate concept of pen and
		# brush - the brush fills any outline trace left by the pen.
		# Here we set both to the same colour - if a figure is not to be 
		# filled, the renderer will set the brush to be transparent
        # Same goes for text foreground...
        DEBUG_MSG("set_foreground()", 1, self)
        GraphicsContextBase.set_foreground(self, fg)
        # I do not understand why self.wxDC.GetPen().SetColour(self._get_wxcolour())
        # does not work here, but it does not - must explicitly call SetPen()
        pen = self.wxDC.GetPen()
        pen.SetColour(self.get_wxcolour())
        self.wxDC.SetPen(pen)
        brush = wxBrush(self.get_wxcolour(), wxSOLID)
        self.wxDC.SetBrush(brush)
        self.wxDC.SetTextForeground(self.get_wxcolour())
        
    def set_graylevel(self, frac):
        """
        Set the foreground color.  fg can be a matlab format string, a
        html hex color string, an rgb unit tuple, or a float between 0
        and 1.  In the latter case, grayscale is used.
        """
        DEBUG_MSG("set_graylevel()", 1, self)
        GraphicsContextBase.set_graylevel(self, frac)
        pen = self.wxDC.GetPen()
        pen.SetColour(self.get_wxcolour())
        self.wxDC.SetPen(pen)
        brush = wxBrush(self.get_wxcolour(), wxSOLID)
        self.wxDC.SetBrush(brush)

    def set_linewidth(self, w):
        """
        Set the line width.
        """
        DEBUG_MSG("set_linewidth()", 1, self)
        if w>0 and w<1: w = 1
        GraphicsContextBase.set_linewidth(self, w)
        pen = self.wxDC.GetPen()
        pen.SetWidth(self._linewidth)
        self.wxDC.SetPen(pen)
       
    def set_capstyle(self, cs):
        """
        Set the capstyle as a string in ('butt', 'round', 'projecting')
        """
        DEBUG_MSG("set_capstyle()", 1, self)
        GraphicsContextBase.set_capstyle(self, cs)
        pen = self.wxDC.GetPen()
        pen.SetCap(GraphicsContextWx._capd[self._capstyle])
        self.wxDC.SetPen(pen)
        
    def set_joinstyle(self, js):
        """
        Set the join style to be one of ('miter', 'round', 'bevel')
        """
        DEBUG_MSG("set_joinstyle()", 1, self)
        GraphicsContextBase.set_joinstyle(self, js)
        pen = self.wxDC.GetPen()
        pen.SetJoin(GraphicsContextWx._joind[self._joinstyle])
        self.wxDC.SetPen(pen)

    def get_wxcolour(self):
        """return a wxColour from RGB format"""
        r, g, b = self.get_rgb()
        r *= 255
        g *= 255
        b *= 255
        return wxColour(red=int(r), green=int(g), blue=int(b))

class FigureWx(FigureBase, wxPanel):
    """
    The figure is the main object -- it contains the Axes and Text,
    which in turn contain lines, patches, and more text.
    
    In the wxPython backend, it is derived from wxPanel, and (usually) lives
    inside a frame instantiated by a FigureManagerWx. The parent window
    implements a wxSizer to control the displayed control size - but we give a
    hint as to our preferred minimum size.
    """
    def __init__(self, parent, id=-1, figsize=defaultFigSize, dpi=defaultDPI):
        """
        Initialise a FigureWx instance.
        
        - Initialise the FigureBase and wxPanel parents.
        - Set event handlers for:
          EVT_SIZE  (Resize event)
          EVT_PAINT (Paint event)
        """
        FigureBase.__init__(self, figsize, dpi)
        DEBUG_MSG("__init__()", 1, self)

        # Set preferred window size hint - helps the sizer (if one is connected)
        w = figsize[0] * dpi  # figsize is in inches
        h = figsize[1] * dpi
        wxPanel.__init__(self, parent, id, size=wxSize(w, h))        
        self.SetSizeHints(minW=w, minH=h)
        
        # TODO: Add support for 'point' inspection and plot navigation.
        self._isRealized = False
        self._isConfigured = False
        self._printQueued = []
        self._gray = (0.7, 0.7, 0.7)
		
        # Event handlers
        EVT_SIZE(self, self._onSize)
        EVT_PAINT(self, self._onPaint)
        EVT_RIGHT_DOWN(self, self._onRightButtonDown)
        EVT_RIGHT_UP(self, self._onRightButtonUp)
        EVT_MOUSEWHEEL(self, self._onMouseWheel)
        EVT_LEFT_DOWN(self, self._onLeftButtonDown)
        EVT_LEFT_UP(self, self._onLeftButtonUp)

    def add_axis(self, a):
        """
        Add an instance of Axis to the figure. 
        """
        FigureBase.add_axis(self, a)
        DEBUG_MSG("add_axis()", 1, self)
        a.set_fig_bgcolor(self._gray)
        a.set_renderer(self.drawable)
               
    def draw(self, drawable=None, *args, **kwargs):
        """
        Render the figure using RendererWx instance drawable, or using a
        previously defined drawable if none is specified.
        """


        DEBUG_MSG("draw()", 1, self)
        if drawable is None: drawable = self.drawable
        if drawable is None: return
        self.drawable = drawable
        if not self._isRealized: return

        # draw the figure bounding box, perhaps none for white figure
        self._figurePatch.draw(drawable)
        # render the axes
        for a in self.axes:
            a.wash_brushes()
            a._update_legend_positions()
            a.draw(drawable)
        # render the figure text
        for t in self._text:
            t.draw(drawable)

    
    def print_figure(self, filename, dpi=300):
        """
        Render the figure to hardcopy using self.drawable as the
        renderer if necessary
        """
        # TODO: add printing support to print_figure() - currently does nothing
        DEBUG_MSG("print_figure()", 1, self)
        pass

    def realize(self):
        """
        This method will be called when the system is ready to draw,
        eg when a GUI window is realized
        """

        DEBUG_MSG("realize()", 1, self)
        if not self._isRealized:
            self._update_renderer()
        
        # Print any queued figures
        # TODO: add print queue support to realize
        #for fname, dpi in self.printQued:
        #    self.print_figure(fname, dpi)
        #self.printQued = []
        self._isRealized = True
            
    def text(self, x, y, s, *args, **kwargs):
        """
        Add text to figure at location x,y (relative 0-1 coords) See
        the help for Axis text command for the meaning of the other arguments
        """
        DEBUG_MSG("text()", 1, self)
        def to_display(x,y):
            """Convert plot co-ordinates to display co-ordinates."""
            w, h = self.GetClientSize()
            return w * x, h * y 
            
        clip_gc = lambda gc: gc.set_clip_rectangle((0,0,self._width,self._height))
        scale_disp = lambda x, y: x, y

        override = _process_text_args({}, *args, **kwargs)
        t = AxisTextWx( x=x, y=y, text=s, **override)
        t.transform_xy_to_display = to_display
        t.transform_xyscale_to_display = scale_disp
        t.clip_gc = clip_gc
        self._text.append(t)
        return t

    def _update_renderer(self):
        """
        Set a new wxRenderer instance for Axes and Text in this figure.
        """
        DEBUG_MSG("update_renderer()", 1, self)
        self.drawable = RendererWx(self)
        for a in self.axes:
            a.set_renderer(self.drawable)
        for t in self._text:
            t.set_renderer(self.drawable)


    def _onPaint(self, evt):
        """
        Called when wxPaintEvt is generated
        """

        DEBUG_MSG("_onPaint()", 1, self)
        if not self._isRealized:
            self.realize()
            
        # Must use wxPaintDC during paint event
        gc = self.drawable.new_gc(wxPaintDC(self))
        self.draw()
        evt.Skip()
        
    def _onSize(self, evt):
        """
        Called when wxEventSize is generated.
        
        In this application we attempt to resize to fit the window, so it
        is better to take the performance hit and redraw the whole window.
        """

        DEBUG_MSG("_onSize()", 1, self)
        self._width, self._height = self.GetClientSize()
        if self._width <= 1 or self._height <= 1: return # Empty figure
        
        # Scale the displayed image (but don't update self.figsize)
        if not self._isConfigured:
            self._orig_width, self._orig_height = self._width, self._height
            self._isConfigured = True
        scalew = self._width / self._orig_width
        scaleh = self._height / self._orig_height
        figsize = self.figsize[0] * scalew, self.figsize[1] * scaleh
        self._update_renderer()
        for a in self.axes:
            a.set_size(figsize, self.dpi)
        clip_gc = lambda gc: gc.set_clip_rectangle((0, 0, self._width, self._height))
        for t in self._text:
            t.clip_gc = clip_gc
            t.set_size(figsize, self.dpi)
        if self._isRealized:
            self.draw()
        evt.Skip()
        
    def _onRightButtonDown(self, evt):
        # TODO: implement right button down handler - show motion menu
        pass
        
    def _onRightButtonUp(self, evt):
        # TODO: implement right button up handler - remove motion menu?
        pass
        
    def _onLeftButtonDown(self, evt):
        """Start measuring on an axis."""
        evt.Skip()
        
    def _onLeftButtonUp(self, evt):
        """End measuring on an axis."""
        evt.Skip()
        
    def _onMouseWheel(self, evt):
        # TODO: implement mouse wheel handler
        pass

class AxisTextWx(AxisTextBase):
    """
    Handle storing and drawing of text in window or data co-ordinates.
    """
    # wxPython only supports three standard font weights - I have mapped them
    # onto the same weights as used in the GTK back-end
    fontweights = {'normal'     : wxNORMAL,
                   'bold'       : wxBOLD,
                   'heavy'      : wxBOLD,
                   'light'      : wxLIGHT,
                   'ultrabold'  : wxBOLD,
                   'ultralight' : wxLIGHT }
                   
    fontangles = { 'italic'  : wxITALIC,
                   'normal'  : wxNORMAL,
                   'oblique' : wxSLANT }
                   
    # wxPython allows for portable font styles, choosing them appropriately
    # for the target platform. Map some standard font names to the portable
    # styles
    # QUESTION: Is it be wise to agree standard fontnames across all backends?
    fontnames = { 'Sans'       : wxSWISS,
                  'Roman'      : wxROMAN,
                  'Script'     : wxSCRIPT,
                  'Decorative' : wxDECORATIVE,
                  'Modern'     : wxMODERN,
                  'Courier'    : wxMODERN,
                  'courier'    : wxMODERN }
                   
    def __init__(self, x=0, y=0, text='', color='k', verticalalignment='bottom',
                  horizontalalignment='left', fontname='Sans', fontsize=10,
                  fontweight='normal', fontangle='normal', rotation=None):
        AxisTextBase.__init__(self, x, y, text, color, verticalalignment,
                               horizontalalignment, fontname, fontsize,
                               fontweight, fontangle, rotation)
        DEBUG_MSG("__init__()", 1, self)
    
    def _compute_offsets(self):
        """
        Return the (x,y) offsets to adjust for the alignment specifications.
        """
        DEBUG_MSG("_compute_offsets()", 1, self)
        # Identical to GTK version
        try: self._width
        except AttributeError: self._set_font()
        
        if self._rotation=='vertical':
            w, h = self._height, self._width
            if self._horizontalalignment=='center': offsetx = -w/2
            elif self._horizontalalignment=='right': offsetx = -w
            else: offsetx = 0

            if self._verticalalignment=='center': offsety = -h/2
            elif self._verticalalignment=='top': offsety = 0
            else: offsety = -h
        else:
            if self._horizontalalignment=='center': offsetx = -self._width/2
            elif self._horizontalalignment=='right': offsetx = -self._width
            else: offsetx = 0

            if self._verticalalignment=='center': offsety = self._height/2
            elif self._verticalalignment=='top': offsety = 0
            else: offsety = self._height

        return (offsetx, offsety)
    
    def _draw(self, drawable, *args, **kwargs):
        """
        Render the text to the drawable (or the default if drawable is None)
        """
        DEBUG_MSG("_draw()", 1, self)
            
        if (drawable is None and self._drawable is None): return 
        if drawable is not None: self._drawable = drawable

        gc = drawable.new_gc()
        if self._text=='': return
        if self._reset:
            self._set_font()
            self._reset = 0

        x, y = self.transform_xy_to_display(self._x, self._y)
        ox, oy = self._compute_offsets()
        xox = int(x+ox)
        yoy = drawable.height - int(y+oy)

        if y+oy<0:
            print >>sys.stderr, 'Warning: text label "%s" is outside window extent' % self._text
        if x+ox<0:
            print >>sys.stderr, 'Warning: text label "%s" is outside window extent' % self._text

        self.clip_gc(gc)
        gc.set_foreground(self._color)
        
        if self._rotation=='vertical':            
            drawable.draw_rotated_text(gc, self._text, xox, yoy)            
        else:            
            drawable.draw_text(gc, self._text, xox, yoy)
         
    def get_data_extent(self):
        """
        Return the ink extent of the text as (left, bottom, width, height)
        """
        DEBUG_MSG("get_data_extent()", 1, self)
        
        try: self._width
        except AttributeError: self._set_font()
        
        ox, oy = self._compute_offsets()
        if self._rotation == 'vertical':
            return (self._x + ox, self._y + oy - self._width, self._height, self._width)
        else:
            return (self._x + ox, self._y + oy, self._width, self._height)
    
    def _set_font(self):
        """
        Update any font information; this function will be called if
        font state (eg, fontsize, fontweight) has been changed
        """
        DEBUG_MSG("_set_font()", 1, self)
        if self._drawable is None: return
        
        # Allow use of platform independent and dependent font names
        try:
            wxFontname = self.fontnames[self._fontname]
            wxFacename = '' # Empty => wxPython chooses based on wx_fontname
        except KeyError:
            wxFontname = wxDEFAULT
            wxFacename = self._fontname
            
        # Font colour is determined by the active wxPen
        gc = self._drawable.get_gc()

        # TODO: It may be wise to cache font information
        self._font = wxFont(self._fontsize,                     # Size
                            wxFontname,                        # 'Generic' name
                            self.fontangles[self._fontangle],   # Angle
                            self.fontweights[self._fontweight], # Weight
                            False,                              # Underline
                            wxFacename)                        # Platform font name
                            
        # Now work out the size of our text on the device context

        wxDC = gc.wxDC
        wxDC.SetFont(self._font)
        self._width, self._height = wxDC.GetTextExtent(self._text)
        #return True
        
    def set_size(self, figsize, dpi):
        """
        Perform sizing (or re-sizing) of the text axes
        """
        DEBUG_MSG("set_size()", 1, self)
        Artist.set_size(self, figsize, dpi)
        self._set_font()

########################################################################
#    
# The following functions and classes are for matlab compatibility
# mode (matplotlib.matlab) and implement figure managers, etc...
#
########################################################################

class ShowOn:
    """
    Manage whether frames are shown or not.
    
    This is a helper class to show() and draw_if_interactive()
    """
    show = 0

    def set(self, on):
        DEBUG_MSG("set()", 1, self)
        # show all the fig wins
        
        if not ShowOn.show:
            for figwin in GcfWx.get_all_figwins():
                figwin.figure.realize()
                figwin.frame.Show()
            
        ShowOn.show = on
    
    def get(self):
        DEBUG_MSG("get()", 1, self)
        return ShowOn.show

    def realize_windows(self):

        DEBUG_MSG("realize()", 1, self)
        self.set(1)
        wxapp.MainLoop()

def draw_if_interactive():
    """
    This should be overriden in a windowing environment if drawing
    should be done in interactive python mode
    """
    if ShowOn().get():
        current_fig = Gcf().get_current_figwin().figure
        current_fig.draw()
        current_fig.queue_draw()

def show():
    """
    Current implementation assumes that matplotlib is executed in a PyCrust
    shell. It appears to be possible to execute wxPython applications from
    within a PyCrust without having to ensure that wxPython has been created
    in a secondary thread (e.g. SciPy gui_thread).
    
    Unfortunately, gui_thread seems to introduce a number of further
    dependencies on SciPy modules, which I do not wish to introduce
    into the backend at this point. If there is a need I will look
    into this in a later release.
    """
    ShowOn().realize_windows()

class _FigureFrame(wxFrame):
    def __init__(self, figsize, dpi, num):
        wxFrame.__init__(self, parent=None, id=-1, title="Figure %d" % num)

        DEBUG_MSG("__init__()", 1, self)
        self.fig = FigureWx(figsize=figsize, dpi=dpi, parent=self)
        self.statusbar = StatusBarWx(self)
        self.SetStatusBar(self.statusbar)
        self.toolbar = NavigationToolbarWx(self.fig, self)
        self.sizer = wxBoxSizer(wxVERTICAL)
        self.sizer.Add(self.fig, 0, wxTOP | wxLEFT)
        # By adding toolbar in sizer, we are able to put it at the bottom
        # of the frame - so appearance is closer to GTK version
        self.sizer.Add(self.toolbar, 0, wxLEFT)
        self.SetSizer(self.sizer)
        self.Fit()
        self.figmgr = FigureManagerWx(self.fig, num, self)
        self.fig.draw(RendererWx(self.fig))
        # Event handlers
        EVT_CLOSE(self, self._onClose)
        
    def get_figure_manager(self):
        DEBUG_MSG("get_figure_manager()", 1, self)
        return self.figmgr
        
    def _onClose(self, evt):
        self.Destroy()
        
    def GetToolBar(self):
        """Override wxFrame::GetToolBar as we don't have managed toolbar"""
        return self.toolbar

class GcfWx(GcfBase):
    """
    This class wraps all of the FigureWx instances which have been
    instantiated, and provides methods by which they may be managed.
    
    If there is no figure in existance, a new one is created.
    
    NB: GcfBase is found in _matlab_helpers.
    """
    def __init__(self, num=None, figsize=(5,4), dpi=150):
        GcfBase.__init__(self, num, figsize, dpi)
        DEBUG_MSG("__init__()", 1, self)
		
    def destroy(num):
        if not GcfBase.has_fignum(num): return        
        GcfWx.figs[num].figure.drawable = None
        GcfBase.destroy(num)
        # TODO: deal with case where no more figures after destroy?
        # - this is handled in the GTK version - see for inspiration!
    destroy = staticmethod(destroy)
        
    def newfig(self, num=None, figsize=defaultFigSize, dpi=defaultDPI):
        """
        Add a new figure num (default autoincrement).
        
        For now, the new figure is created in a wxFrame, and has size
        managed by a wxSizer.
        """
        DEBUG_MSG("newfig()", 1, self)
        if num is None:
            if len(self.figs)>0:
                num = max(self.figs.keys())+1
            else:
                num = 1
        frame = _FigureFrame(figsize, dpi, num)        
        self.figs[num] = frame.get_figure_manager()
        return self.figs[num]

class FigureManagerWx(FigureManagerBase):
    """
    This class manages the axes and subplots within a single figure.
    
    It is instantiated by GcfWx whenever a new figure is created. GcfWx is
    responsible for managing multiple instances of FigureManagerWx.
    
    NB: FigureManagerBase is found in _matlab_helpers
    """
    def __init__(self, figure, num, frame):
        DEBUG_MSG("__init__()", 1, self)
        FigureManagerBase.__init__(self, figure, num)
        self.frame = frame
        self.tb = frame.GetToolBar()
        
    def add_subplot(self, *args, **kwargs):
        DEBUG_MSG("add_subplot()", 1, self)
        a = FigureManagerBase.add_subplot(self, *args, **kwargs)
        self.tb.update()
        return a
        
    def add_axes(self, rect, axisbg):
        DEBUG_MSG("add_axes()", 1, self)
        a = FigureManagerBase.add_axes(self, rect, axisbg)
        self.tb.update()
        return a
    
    def set_current_axes(self, a):
        DEBUG_MSG("set_current_axes()", 1, self)
        if a not in self.axes.values():
            error_msg_gtk('Axes is not in current figure')
        FigureManagerBase.set_current_axes(self, a)

# Identifiers for toolbar controls - images_wx contains bitmaps for the images
# used in the controls. wxWindows does not provide any stock images, so I've
# 'stolen' those from GTK2, and transformed them into the appropriate format.
#import images_wx

_NTB_AXISMENU        = wxNewId()
_NTB_AXISMENU_BUTTON = wxNewId()
_NTB_X_PAN_LEFT      = wxNewId()
_NTB_X_PAN_RIGHT     = wxNewId()
_NTB_X_ZOOMIN        = wxNewId()
_NTB_X_ZOOMOUT       = wxNewId()
_NTB_Y_PAN_UP        = wxNewId()
_NTB_Y_PAN_DOWN      = wxNewId()
_NTB_Y_ZOOMIN        = wxNewId()
_NTB_Y_ZOOMOUT       = wxNewId()
_NTB_REDRWAW         = wxNewId()
_NTB_SAVE            = wxNewId()
_NTB_CLOSE           = wxNewId()

def _load_bitmap(filename):
    """
    Load a bitmap file from the backends/images subdirectory in which the
    matplotlib library is installed. The filename parameter should not
    contain any path information as this is determined automatically.
    
    Bitmaps should be in XPM format, and of size 16x16 (unless you change
    the code!). I have converted the stock GTK2 16x16 icons to XPM format.
    
    Returns a wxBitmap object
    """
    basedir = os.path.join(distutils.sysconfig.PREFIX, 'share', 'matplotlib')

    bmpFilename = os.path.normpath(os.path.join(basedir, filename))
    if not os.path.exists(bmpFilename):
        print >>sys.stderr, 'Could not find bitmap file; dying', bmpFilename
        sys.exit()
    bmp = wxBitmap(bmpFilename, wxBITMAP_TYPE_XPM)
    return  bmp

class MenuButtonWx(wxButton):
    """
    wxPython does not permit a menu to be incorporated directly into a toolbar.
    This class simulates the effect by associating a pop-up menu with a button
    in the toolbar, and managing this as though it were a menu.
    """
    def __init__(self, parent):
        wxButton.__init__(self, parent, _NTB_AXISMENU_BUTTON, "Axis Menu",
                          style=wxBU_EXACTFIT)
        self._toolbar = parent
        self._menu = wxMenu()
        self._axisId = []
        # First two menu items never change...
        self._allId = wxNewId()
        self._invertId = wxNewId()
        self._menu.Append(self._allId, "All", "Select all axes", False)
        self._menu.Append(self._invertId, "Invert", "Invert axes selected", False)
        self._menu.AppendSeparator()
        EVT_BUTTON(self, _NTB_AXISMENU_BUTTON, self._onMenuButton)
        EVT_MENU(self, self._allId, self._handleSelectAllAxes)
        EVT_MENU(self, self._invertId, self._handleInvertAxesSelected)
        
    def Destroy(self):
        self._menu.Destroy()
        self.Destroy()
        
    def _onMenuButton(self, evt):
        """Handle menu button pressed."""
        x, y = self.GetPositionTuple()
        w, h = self.GetSizeTuple()
        # Need below to ensure that axes active first time menu activated
        self._toolbar.set_active(self.getActiveAxes())
        self.PopupMenuXY(self._menu, x, y+h-4)

    def _handleSelectAllAxes(self, evt):
        """Called when the 'select all axes' menu item is selected."""
        if len(self._axisId) == 0:
            return
        for i in range(len(self._axisId)):
            self._menu.Check(self._axisId[i], True)
        self._toolbar.set_active(self.getActiveAxes())
        
    def _handleInvertAxesSelected(self, evt):
        """Called when the invert all menu item is selected"""
        if len(self._axisId) == 0: return
        for i in range(len(self._axisId)):
            if self._menu.IsChecked(self._axisId[i]):
                self._menu.Check(self._axisId[i], False)
            else:
                self._menu.Check(self._axisId[i], True)
        self._toolbar.set_active(self.getActiveAxes())
	
    def _onMenuItemSelected(self, evt):
        """Called whenever one of the specific axis menu items is selected"""
        current = self._menu.IsChecked(evt.GetId())
        if current:
            new = False
        else:
            new = True
        self._menu.Check(evt.GetId(), new)
        self._toolbar.set_active(self.getActiveAxes())
        
    def updateAxes(self, maxAxis):
        """Ensures that there are entries for max_axis axes in the menu
        (selected by default)."""
        if maxAxis > len(self._axisId):
            for i in range(len(self._axisId) + 1, maxAxis + 1, 1):
                menuId = wxNewId()
                self._axisId.append(menuId)
                self._menu.Append(menuId, "Axis %d" % i, "Select axis %d" % i, True)
                self._menu.Check(menuId, True)
                EVT_MENU(self, menuId, self._onMenuItemSelected)
        self._toolbar.set_active(self.getActiveAxes())
        
    def getActiveAxes(self):
        """Return a list of the selected axes."""
        active = []
        for i in range(len(self._axisId)):
            if self._menu.IsChecked(self._axisId[i]):
                active.append(i)
        return active

class NavigationToolbarWx(wxToolBar):
    def __init__(self, figure, parent):
        """
        figure is the Figure instance that the toolboar controls

        win, if not None, is the wxWindow the Figure is embedded in
        """
        wxToolBar.__init__(self, parent, -1)
        DEBUG_MSG("__init__()", 1, self)
        self.figure = figure
        self._lastControl = None
        self._parent = parent
        self._create_menu()
        self._create_controls()
        self.Realize()

        
    def _create_menu(self):
        """
        Creates the 'menu' - implemented as a button which opens a
        pop-up menu since wxPython does not allow a menu as a control
        """
        DEBUG_MSG("_create_menu()", 1, self)
        self._menu = MenuButtonWx(self)
        self.AddControl(self._menu)
        self.AddSeparator()
        
    def _create_controls(self):
        """
        Creates the button controls, and links them to event handlers
        """
        DEBUG_MSG("_create_controls()", 1, self)
        # Need the following line as Windows toolbars default to 15x16
        self.SetToolBitmapSize(wxSize(16,16))
        self.AddSimpleTool(_NTB_X_PAN_LEFT, _load_bitmap('stock_left.xpm'),
                           'Left', 'Scroll left')
        self.AddSimpleTool(_NTB_X_PAN_RIGHT, _load_bitmap('stock_right.xpm'),
                           'Right', 'Scroll right')
        self.AddSimpleTool(_NTB_X_ZOOMIN, _load_bitmap('stock_zoom-in.xpm'),
                           'Zoom in', 'Increase X axis magnification')
        self.AddSimpleTool(_NTB_X_ZOOMOUT, _load_bitmap('stock_zoom-out.xpm'),
                           'Zoom out', 'Decrease X axis magnification')
        self.AddSeparator()
        self.AddSimpleTool(_NTB_Y_PAN_UP,_load_bitmap('stock_up.xpm'),
                           'Up', 'Scroll up')
        self.AddSimpleTool(_NTB_Y_PAN_DOWN, _load_bitmap('stock_down.xpm'),
                           'Down', 'Scroll down')
        self.AddSimpleTool(_NTB_Y_ZOOMIN, _load_bitmap('stock_zoom-in.xpm'),
                           'Zoom in', 'Increase Y axis magnification')
        self.AddSimpleTool(_NTB_Y_ZOOMOUT, _load_bitmap('stock_zoom-out.xpm'),
                           'Zoom out', 'Decrease Y axis magnification')
        self.AddSeparator()
        self.AddSimpleTool(_NTB_REDRWAW, _load_bitmap('stock_refresh.xpm'),
                           'Redraw', 'Redraw plot window')
        self.AddSimpleTool(_NTB_SAVE, _load_bitmap('stock_save_as.xpm'),
                           'Save', 'Save plot contents as images')
        self.AddSeparator()
        self.AddSimpleTool(_NTB_CLOSE, _load_bitmap('stock_close.xpm'),
                           'Close', 'Close plot window')
        self.AddSeparator()
        # TODO: Add traps for mouse wheel events on the tool buttons
        # Events:
        EVT_TOOL(self, _NTB_X_PAN_LEFT, self._onLeftScroll)
        EVT_TOOL(self, _NTB_X_PAN_RIGHT, self._onRightScroll)
        EVT_TOOL(self, _NTB_X_ZOOMIN, self._onXZoomIn)
        EVT_TOOL(self, _NTB_X_ZOOMOUT, self._onXZoomOut)
        EVT_TOOL(self, _NTB_Y_PAN_UP, self._onUpScroll)
        EVT_TOOL(self, _NTB_Y_PAN_DOWN, self._onDownScroll)
        EVT_TOOL(self, _NTB_Y_ZOOMIN, self._onYZoomIn)
        EVT_TOOL(self, _NTB_Y_ZOOMOUT, self._onYZoomOut)
        EVT_TOOL(self, _NTB_REDRWAW, self._onRedraw)
        EVT_TOOL(self, _NTB_SAVE, self._onSave)
        EVT_TOOL(self, _NTB_CLOSE, self._onClose)
        EVT_MOUSEWHEEL(self, self._onMouseWheel)
        
    def set_active(self, ind):
        """
        ind is a list of index numbers for the axes which are to be made active
        """
        DEBUG_MSG("set_active()", 1, self)
        self._ind = ind
        if ind != None:
            self._active = [ self._axes[i] for i in self._ind ]
        else:
            self._active = []
            
    def get_last_control(self):
        """Returns the identity of the last toolbar button pressed."""
        return self._lastControl
        
    def panx(self, direction):
        DEBUG_MSG("panx()", 1, self)
        for a in self._active:
            a.panx(direction)
        self.figure.draw()
        
    def pany(self, direction):
        DEBUG_MSG("pany()", 1, self)
        for a in self._active:
            a.pany(direction)
        self.figure.draw()
        
    def zoomx(self, in_out):
        DEBUG_MSG("zoomx()", 1, self)
        for a in self._active:
            a.zoomx(in_out)
        self.figure.draw()
        
    def zoomy(self, in_out):
        DEBUG_MSG("zoomy()", 1, self)
        for a in self._active:
            a.zoomy(in_out)
        self.figure.draw()
        
    def update(self):
        """
        Update the toolbar menu - called when (e.g.) a new subplot or axes are added
        """
        # TODO: Implement toolbar menu update mechanism
        DEBUG_MSG("update()", 1, self)
        self._axes = self.figure.get_axes()
        self._menu.updateAxes(len(self._axes))
        active = self._menu.updateAxes(len(self._axes))
        self.set_active(active)
        
    # Local event handlers - mainly supply parameters to pan/scroll functions
    def _onLeftScroll(self, evt):
        self.panx(-1)
        self._lastControl = 'panx'
        self._parent.statusbar.set_function(self._lastControl)
        
    def _onRightScroll(self, evt):
        self.panx(1)
        self._lastControl = 'panx'        
        self._parent.statusbar.set_function(self._lastControl)
        
    def _onXZoomIn(self, evt):
        self.zoomx(1)
        self._lastControl = 'zoomx'        
        self._parent.statusbar.set_function(self._lastControl)
        
    def _onXZoomOut(self, evt):
        self.zoomx(-1)
        self._lastControl = 'zoomx'        
        self._parent.statusbar.set_function(self._lastControl)
        
    def _onUpScroll(self, evt):
        self.pany(1)
        self._lastControl = 'pany'        
        self._parent.statusbar.set_function(self._lastControl)

    def _onDownScroll(self, evt):
        self.pany(-1)
        self._lastControl = 'pany'        
        self._parent.statusbar.set_function(self._lastControl)
        
    def _onYZoomIn(self, evt):
        self.zoomy(1)
        self._lastControl = 'zoomy'        
        self._parent.statusbar.set_function(self._lastControl)
        
    def _onYZoomOut(self, evt):
        self.zoomy(-1)
        self._lastControl = 'zoomy'
        self._parent.statusbar.set_function(self._lastControl)

    def _onMouseWheel(self, evt):
        if evt.GetWheelRotation() > 0:
            direction = 1
        else:
            direction = -1
        getattr(self, self._lastControl)(direction)
        
    def _onRedraw(self, evt):
        for a in self.figure.axes:
            a.wash_brushes()
        self.figure.draw()

    def _onSave(self, evt):
        # TODO: add file save action
        pass
    def _onClose(self, evt):
        self.GetParent().Destroy()

class StatusBarWx(wxStatusBar):
    """
    A status bar is added to _FigureFrame to allow measurements and the
    previously selected scroll function to be displayed as a user
    convenience.
    """
    def __init__(self, parent):
        wxStatusBar.__init__(self, parent, -1)
        self.SetFieldsCount(3)
        self.SetStatusText("Function: None", 1)
        self.SetStatusText("Measurement: None", 2)
        #self.Reposition()
        
    def set_function(self, string):
        self.SetStatusText("Function: %s" % string, 1)
        
    def set_measurement(self, string):
        self.SetStatusText("Measurement: %s" % string, 2)

########################################################################
#    
# Now just provide the standard names that backend.__init__ is expecting
# 
########################################################################

Gcf = GcfWx
FigureManager = FigureManagerWx
AxisText = AxisTextWx
Figure = FigureWx
error_msg = error_msg_wx
         
