"""
This is a matlab style functional interface the matplotlib.

The following matlab compatible commands are provided by

   >>> from matplotlib.matlab import *

Plotting commands

  axes     - Create a new axes
  axis     - Set or return the current axis limits
  bar      - make a bar chart
  close    - close a figure window
  errorbar - make an errorbar graph
  figure   - create or change active figure
  gca      - return the current axes
  gcf      - return the current figure
  get      - get a handle graphics property
  hist     - make a histogram
  loglog   - a log log plot
  plot     - make a line plot
  savefig  - save the current figure
  scatter  - make a scatter plot
  semilogx - log x axis
  semilogy - log y axis
  set      - set a handle graphics property
  show     - show the figures
  subplot  - make a subplot (numrows, numcols, axesnum)
  text     - add some text at location x,y to the current axes
  title    - add a title to the current axes
  xlabel   - add an xlabel to the current axes
  ylabel   - add a ylabel to the current axes

Matrix commands

  cumprod   - the cumulative product along a dimension
  cumsum    - the cumulative sum along a dimension
  detrend   - remove the mean or besdt fit line from an array
  diag      - the k-th diagonal of matrix 
  diff      - the n-th differnce of an array
  eig       - the eigenvalues and eigen vectors of v
  eye       - a matrix where the k-th diagonal is ones, else zero 
  find      - return the indices where a condition is nonzero  
  fliplr    - flip the rows of a matrix up/down
  flipud    - flip the columns of a matrix left/right
  linspace  - a linear spaced vector of N values from min to max inclusive
  ones      - an array of ones
  rand      - an array from the uniform distribution [0,1]
  randn     - an array from the normal distribution
  rot90     - rotate matrix k*90 degress counterclockwise
  squeeze   - squeeze an array removing any dimensions of length 1
  tri       - a triangular matrix
  tril      - a lower triangular matrix
  triu      - an upper triangular matrix
  vander    - the Vandermonde matrix of vector x
  svd       - singular value decomposition
  zeros     - a matrix of zeros
  
Probability

  levypdf   - The levy probability density function from the char. func.
  normpdf   - The Gaussian probability density function
  rand      - random numbers from the uniform distribution
  randn     - random numbers from the normal distribution

Statistics

  corrcoef  - correlation coefficient
  cov       - covariance matrix
  max       - the maximum along dimension m
  mean      - the mean along dimension m
  median    - the median along dimension m
  min       - the minimum along dimension m
  norm      - the norm of vector x
  prod      - the product along dimension m
  ptp       - the max-min along dimension m
  std       - the standard deviation along dimension m
  sum       - the sum along dimension m

Time series analysis

  bartlett  - M-point Bartlett window
  blackman  - M-point Blackman window
  cohere    - the coherence using average periodiogram
  csd       - the cross spectral density using average periodiogram
  fft       - the fast Fourier transform of vector x
  hamming   - M-point Hamming window
  hanning   - M-point Hanning window
  hist      - compute the histogram of x
  kaiser    - M length Kaiser window
  psd       - the power spectral density using average periodiogram
  sinc      - the sinc function of array x

Other

  angle     - the angle of a complex array
  polyfit   - fit x, y to an n-th order polynomial
  polyval   - evaluate an n-th order polynomial
  roots     - the roots of the polynomial coefficients in p
  trapz     - trapezoidal integration


Credits: The plotting commands were provided by
John D. Hunter <jdhunter@ace.bsd.uhicago.edu>

Most of the other commands are from the Numeric, MLab and FFT, with
the exception of those in mlab.py provided by matplotlib.
"""

import sys, os
import distutils.sysconfig

# Allow command line access to the backend with -d (matlab compatible
# flag)

for s in sys.argv[1:]:
    if s.startswith('-d'):
        _backend = s[2:].strip()
        break
else:  # default to the old standby
    _backend = 'GTK'

_knownBackends = {'PS':1, 'GTK':1, 'Template':1, 'GD':1, 'WX':1,}

def use(arg):
    """
    Set the matplotlib backend to one of the _knownBackends
    """
    global _backend
    
    if not _knownBackends.has_key(arg):
        print >>sys.stderr, 'unrecognized backend %s.\n' % arg +\
              'Use one of %s' % ', '.join( _knownBackends.keys() )
        sys.exit()
    _backend = arg

def get_backend():
    return _backend

def get_data_path():
    path = os.path.join(distutils.sysconfig.PREFIX, 'share', 'matplotlib')
    if os.path.isdir(path): return path

    path = os.path.join(os.sep.join(__file__.split(os.sep)[:-5]), 
                        'share','matplotlib')
    if os.path.isdir(path): return path

    if os.environ.has_key('MATPLOTLIBDATA'):
        path = os.environ['MATPLOTLIBDATA']
        if os.path.isdir(path): return path
    raise RuntimeError('Could not find the matplotlib data files')
    


major, minor1, minor2, s, tmp = sys.version_info
if major<2 or (major==2 and minor1<3):
    True = 1
    False = 0
else:
    True = True
    False = False

_interactive = False

def interactive(b):
    """
    Set interactive mode to boolean b.

    If b is True, then draw after every plotting command, eg, after xlabel
    """
    global _interactive
    _interactive = b

def is_interactive():
    'Return true if plot mode is interactive'
    return _interactive
    


def switch_backends(fig, FigClass):
    """
    fig is a Figure instance; instantiate an instance of FigClass,
    where FigClass is derived from matplotlib.backend_bases.FigureBase

    This is used for backend switching, eg, to instantiate a FigurePS
    from a FigureGTK.  Note, deep copying is not done, so any changes
    to one of the instances (eg, setting figure size or line props),
    will be reflected in the other
    """
    newfig = FigClass(fig.figsize, fig.dpi.get(), fig.facecolor, fig.edgecolor)
    # override bbox and dpi to preserve refs in axes, etc
    newfig.bbox = fig.bbox
    newfig.dpi = fig.dpi
    newfig.axes = fig.axes
    newfig._text = fig._text

    return newfig

