"""
This is a fully functional do nothing backend to provide a template to
backend writers.  It is fully functional in that you can select it as
a backend with

  import matplotlib
  matplotlib.use('Template')

and your matplotlib scripts will (should!) run without error, though
no output is produced.  This provides a nice starting point for
backend writers because you can selectively implement methods
(draw_rectangle, draw_lines, etc...) and slowly see your figure come
to life w/o having to have a full blown implementation before getting
any results.

Copy this to backend_xxx.py and replace all instances of 'template'
with 'xxx'.  Then implement the class methods and functions below, and
add 'xxx' to the switchyard in matplotlib/backends/__init__.py and
'xxx' to the _knownBackends dict in matplotlib/__init__.py and you're
off.  You can use your backend with

  import matplotlib
  matplotlib.use('xxx')
  from matplotlib.matlab import *
  plot([1,2,3])
  show()

The files that are most relevant to backend_writers are

  matplotlib/backends/backend_your_backend.py
  matplotlib/backend_bases.py
  matplotlib/backends/__init__.py
  matplotlib/__init__.py
  matplotlib/_matlab_helpers.py
  
Naming Conventions

  * classes MixedUpperCase

  * varables lowerUpper

  * functions underscore_separated

REQUIREMENTS

  matplotlib requires python2.2 and Numeric, and I don't yet want to
  make python2.3 a requirement.  I provide the Python Cookbook version
  of enumerate in cbook.py and define the constants True and False if
  version <=2.3.  Of course as a backend writer, you are free to make
  additional requirements, but the less required the better.

"""

import sys
from matplotlib._matlab_helpers import Gcf
from matplotlib.backend_bases import RendererBase, GraphicsContextBase,\
     FigureBase, FigureManagerBase, error_msg

from matplotlib.cbook import True, False
from matplotlib.transforms import Bound1D, Bound2D, Transform
from matplotlib.artist import DPI

def error_msg_template(msg, *args):
    """
    Signal an error condition -- in a GUI, popup a error dialog
    """
    print >>sys.stderr, 'Error:', msg
    sys.exit()
    
class RendererTemplate(RendererBase):
    """
    The renderer handles all the drawing primitives using a graphics
    context instance that controls the colors/styles
    """

    def compute_text_offsets(self, text):
        """
        Return the (x,y) offsets to adjust for the alignment
        specifications.  This method is not required by the API, but
        is a useful to compute the alignment offsets
        """
        return (0,0)  # your actual offsets here
    
    def draw_text(self, gc, x, y, text):
        """
        Render the matplotlib.text.Text instance at x, y in window
        coords using GraphicsContext gc
        """
        pass
         
    def get_text_extent(self, text):
        """
        Return the ink extent of the text as Bound2D instance
        """
        return Bound2D(0,0,1,1)  # your actual extent here
    
    def draw_arc(self, gcEdge, faceColor, x, y, width, height, angle1, angle2):
        """
        Draw an arc centered at x,y with width and height and angles
        from 0.0 to 360.0.

        If faceColor is not None, fill the rectangle with it.  gcEdge
        is a GraphicsContext instance
        """
        pass
    
    def draw_line(self, gc, x1, y1, x2, y2):
        """
        Draw a single line from x1,y1 to x2,y2
        """
        pass
    
    def draw_lines(self, gc, x, y):
        """
        x and y are equal length arrays, draw lines connecting each
        point in x, y
        """
        pass
    
    def draw_polygon(self, gcEdge, faceColor, points):
        """
        Draw a polygon.  points is a len vertices tuple, each element
        giving the x,y coords a vertex.

        If faceColor is not None, fill the rectangle with it.  gcEdge
        is a GraphicsContext instance
        """  
        pass

    def draw_rectangle(self, gcEdge, faceColor, x, y, width, height):
        """
        Draw a rectangle at lower left x,y with width and height.

        If faceColor is not None, fill the rectangle with it.  gcEdge
        is a GraphicsContext instance
        """
        pass

    def draw_point(self, gc, x, y):
        """
        Draw a single point at x,y
        """
        pass


    def new_gc(self):
        """
        Return an instance of a GraphicsContextTemplate
        """
        return GraphicsContextTemplate()


class GraphicsContextTemplate(GraphicsContextBase):
    """
    The graphics context provides the color, line styles, etc...  See
    the gtk and postscript backends for examples of mapping the
    graphics context attributes (cap styles, join styles, line widths,
    colors) to a particular backend.  In GTK this is done by wrapping
    a gtk.gdk.GC object and forwarding the appropriate calls to it
    using a dictionary mapping styles to gdk constants.  In
    Postscript, all the work is done by the renderer, mapping line
    styles to postscript calls.

    The base GraphicsContext stores colors as a RGB tuple on the unit
    interval, eg, (0.5, 0.0, 1.0).  You will probably need to map this
    to colors appropriate for your backend.  Eg, see the ColorManager
    class for the GTK backend.  If it's more appropriate to do the
    mapping at the renderer level (as in the postscript backend), you
    don't need to override any of the GC methods.  If it's more
    approritate to wrap an instance (as in the GTK backend) and do the
    mapping here, you'll need to override several of the setter
    methods.
    """
    pass
              
class FigureTemplate(FigureBase):
    """
    The figure is the main object -- it contains the Axes and Text,
    which in turn contain lines, patches, and more text

    GUI implementers will probably want to look at FigureGTK in
    backend_gtk.py for examples of handling events with respect to
    matplotlib.
    """
    def __init__(self, figsize, dpi):
        FigureBase.__init__(self, figsize, dpi)
        self._isRealized = False
        
    def draw(self, renderer=None, *args, **kwargs):
        """
        Render the figure using RendererTemplate instance renderer
        """
        # at the least, you'll want to do some version of the following

        # the renderer instance is saved so that redraws can be qued
        # and the print_figure instance has access to the last
        # renderer

        if renderer is not None: self.renderer = renderer
        else: renderer=self.renderer
        if renderer is None: return

        # draw the figure bounding box, perhaps none for white figure
        self._figurePatch.draw(renderer)

        # render the axes
        for a in self.axes:
            a.draw(renderer)

        # render the figure text
        for t in self._text:
            t.draw(renderer)
    
    def print_figure(self, filename, dpi=300):
        """
        Render the figure to hardcopy using self.renderer as the
        renderer if neccessary
        """
        pass

    def realize(self, *args):
        """
        This method will be called when the system is ready to draw,
        eg when a GUI window is realized
        """
        self._isRealized = True  
        renderer = RendererTemplate()
        self.draw(renderer)
        
########################################################################
#    
# The following functions and classes are for matlab compatibility
# mode (matplotlib.matlab) and implement window/figure managers,
# etc...
#
########################################################################

def draw_if_interactive():
    """
    This should be overriden in a windowing environment if drawing
    should be done in interactive python mode
    """
    pass

def show():
    """
    This is usually the last line of a matlab script and tells the
    backend that it is time to draw.  In interactive mode, this may be
    a do nothing func.  See the GTK backend for an example of how to
    handle interactive versus batch mode
    """
    for manager in Gcf.get_all_fig_managers():
        manager.figure.realize()


def new_figure_manager(num, figsize, dpi):
    """
    Create a new figure manager instance
    """
    thisFig = FigureTemplate(figsize, dpi)
    manager = FigureManagerTemplate(thisFig, num)
    return manager


class FigureManagerTemplate(FigureManagerBase):
    'GUIs need to override'
    pass


########################################################################
#    
# Now just provide the standard names that backend.__init__ is expecting
# 
########################################################################


FigureManager = FigureManagerTemplate
Figure = FigureTemplate
error_msg = error_msg_template
         
