"""
Place a legend on the axes at location loc.  Labels are a
sequence of strings and loc can be a string or an integer
specifying the legend location

The location codes are

  'best'         : 0,  (currently not supported, defaults to upper right)
  'upper right'  : 1,  (default)
  'upper left'   : 2,
  'lower left'   : 3,
  'lower right'  : 4,
  'right'        : 5,
  'center left'  : 6,
  'center right' : 7,
  'lower center' : 8,
  'upper center' : 9,
  'center'       : 10,

Return value is a sequence of text, line instances that make
up the legend
"""
from __future__ import division
import sys
from Numeric import ones, Float

from artist import Artist
from cbook import enumerate, True, False, is_string_like
from lines import Line2D
from mlab import linspace
from patches import Rectangle, bbox_artist
from text import Text
from transforms import Bound1D, Transform, bound2d_all, \
     transform_bound2d, inverse_transform_bound2d


class Legend(Artist):
    """
    Place a legend on the axes at location loc.  Labels are a
    sequence of strings and loc can be a string or an integer
    specifying the legend location

    The location codes are

      'best'         : 0,  (currently not supported, defaults to upper right)
      'upper right'  : 1,  (default)
      'upper left'   : 2,
      'lower left'   : 3,
      'lower right'  : 4,
      'right'        : 5,
      'center left'  : 6,
      'center right' : 7,
      'lower center' : 8,
      'upper center' : 9,
      'center'       : 10,
 
    Return value is a sequence of text, line instances that make
    up the legend
    """


    codes = {'best'         : 0,
             'upper right'  : 1,  # default
             'upper left'   : 2,
             'lower left'   : 3,
             'lower right'  : 4,
             'right'        : 5,
             'center left'  : 6,
             'center right' : 7,
             'lower center' : 8,
             'upper center' : 9,
             'center'       : 10,
             }


    NUMPOINTS = 4      # the number of points in the legend line
    FONTSIZE = 10
    PAD = 0.2          # the fractional whitespace inside the legend border
    # the following dimensions are in axes coords
    LABELSEP = 0.005   # the vertical space between the legend entries
    HANDLELEN = 0.05     # the length of the legend lines
    HANDLETEXTSEP = 0.02 # the space between the legend line and legend text
    AXESPAD = 0.02     # the border between the axes and legend edge


    def __init__(self, axis, handles, labels, loc):
        Artist.__init__(self, axis.dpi, axis.bbox)
        if is_string_like(loc) and not self.codes.has_key(loc):
            print >>sys.stderr, 'Unrecognized location %s. Falling back on upper right; valid locations are\n%s\t' %(loc, '\n\t'.join(self.codes.keys()))
        if is_string_like(loc): loc = self.codes.get(loc, 1)

        self._axis = axis
        self._loc = loc   

        # use axes coords
        self.transx = Transform( Bound1D(0,1), self.bbox.x )
        self.transy = Transform( Bound1D(0,1), self.bbox.y )

        # make a trial box in the middle of the axes.  relocate it
        # based on it's bbox
        left, upper = 0.5, 0.5  
        self._xdata = linspace(left, left + self.HANDLELEN, self.NUMPOINTS)        
        textleft = left+ self.HANDLELEN+self.HANDLETEXTSEP
        self._texts = self._get_texts(labels, textleft, upper)
        self._handles = self._get_handles(handles, self._texts)
        
        left, top = self._texts[-1].get_position()
        HEIGHT = self._approx_text_height()
        bottom = top-HEIGHT
        left -= self.HANDLELEN + self.HANDLETEXTSEP + self.PAD
        self._patch = Rectangle(
            self.dpi, self.bbox,
            xy=(left, bottom), width=0.5, height=HEIGHT*len(self._texts),
            facecolor=axis.get_axis_bgcolor(), edgecolor='k',
            transx = self.transx,
            transy = self.transy,
            )

    def _approx_text_height(self):
        return self.FONTSIZE/72.0*self.dpi.get()/self.bbox.y.interval()

            
    def _draw(self, renderer):
        self._update_positions(renderer)
        self._patch.draw(renderer)
        for h in self._handles: h.draw(renderer)
        for t in self._texts:
            if 0: bbox_artist(t, renderer)
            t.draw(renderer)

    def _get_handle_text_bbox(self, renderer):
        'Get a bbox for the text and lines in axes coords'
        boxes = [t.get_window_extent(renderer) for t in self._texts]
        boxes.extend([h.get_window_extent() for h in self._handles])
        bbox = bound2d_all(boxes)
        return inverse_transform_bound2d(bbox, self.transx, self.transy)

        
    def _get_handles(self, handles, texts):
        HEIGHT = self._approx_text_height()

        ret = []   # the returned legend lines
        for handle, label in zip(handles, texts):
            x, y = label.get_position()
            x -= self.HANDLELEN + self.HANDLETEXTSEP
            if isinstance(handle, Line2D):
                ydata = (y-HEIGHT/2)*ones(self._xdata.shape, typecode=Float)
                legline = Line2D(self.dpi, self.bbox, self._xdata, ydata,
                                 transx=self.transx, transy=self.transy)
                legline.copy_properties(handle)
                legline.set_markersize(0.6*legline.get_markersize())
                legline.set_data_clipping(False)
                ret.append(legline)
            elif isinstance(handle, Rectangle):

                p = Rectangle(self.dpi, self.bbox,
                              xy=(min(self._xdata), y-3/4*HEIGHT),
                              width = self.HANDLELEN, height=HEIGHT/2,
                              transx=self.transx, transy=self.transy)
                p.copy_properties(handle)
                ret.append(p)
                
                
                    
        return ret


    def _get_texts(self, labels, left, upper):

        # height in axes coords
        HEIGHT = self._approx_text_height()
        pos = upper
        x = left 

        ret = []  # the returned list of text instances
        for l in labels:
            text = Text(
                self.dpi, self.bbox,
                x=x, y=pos,
                text=l, fontsize=self.FONTSIZE,
                verticalalignment='top',
                horizontalalignment='left',
                transx = self.transx,
                transy = self.transy,
                )
            ret.append(text)
            pos -= HEIGHT
            
        return ret

    def get_child_artists(self):
        l =  [self._patch]
        l.extend(self._texts)
        l.extend(self._handles)
        return l
            
    def get_window_extent(self):
        return self._patch.get_window_extent()


    def _offset(self, ox, oy):
        'Move all the artists by ox,oy (axes coords)'
        for t in self._texts:
            x,y = t.get_position()
            t.set_position( (x+ox, y+oy) )

        for h in self._handles:
            if isinstance(h, Line2D):
                x,y = h.get_xdata(), h.get_ydata()
                h.set_data( x+ox, y+oy)
            elif isinstance(h, Rectangle):
                h.xy[0] = h.xy[0] + ox
                h.xy[1] = h.xy[1] + oy
                
        x, y = self._patch.get_x(), self._patch.get_y()
        self._patch.set_x(x+ox)
        self._patch.set_y(y+oy)

    def _update_positions(self, renderer):
        # called from renderer to allow more precise estimates of
        # widths and heights with get_window_extent

        def get_tbounds(text):  #get text bounds in axes coords
            bbox = text.get_window_extent(renderer)
            bboxa = inverse_transform_bound2d(bbox, self.transx, self.transy)
            return bboxa.get_bounds()
            
        hpos = []
        for t, tabove in zip(self._texts[1:], self._texts[:-1]):
            x,y = t.get_position()
            l,b,w,h = get_tbounds(tabove)
            hpos.append( (b,h) )
            t.set_position( (x, b-0.1*h) )

        # now do the same for last line
        l,b,w,h = get_tbounds(self._texts[-1])
        hpos.append( (b,h) )
        
        for handle, tup in zip(self._handles, hpos):
            y,h = tup
            if isinstance(handle, Line2D):
                ydata = y*ones(self._xdata.shape, Float)            
                handle.set_ydata(ydata+h/2)
            elif isinstance(handle, Rectangle):
                handle.set_y(y+1/4*h)
                handle.set_height(h/2)

        # Set the data for the legend patch
        bbox = self._get_handle_text_bbox(renderer)
        bbox.x.scale(1 + self.PAD)
        bbox.y.scale(1 + self.PAD)
        l,b,w,h = bbox.get_bounds()
        self._patch.set_bounds(l,b,w,h)
        
        BEST, UR, UL, LL, LR, R, CL, CR, LC, UC, C = range(11)
        ox, oy = 0, 0                           # center
        if self._loc in (UL, LL, CL):           # left
            ox = self.AXESPAD - l
        if self._loc in (BEST, UR, LR, R, CR):  # right
            ox = 1 - (l + w + self.AXESPAD)
        if self._loc in (BEST, UR, UL, UC):     # upper
            oy = 1 - (b + h + self.AXESPAD)
        if self._loc in (LL, LR, LC):           # lower
            oy = self.AXESPAD - b
        if self._loc in (LC, UC, C):            # center x
            ox = (0.5-w/2)-l
        if self._loc in (CL, CR, C):            # center y
            oy = (0.5-h/2)-b
        self._offset(ox, oy)
