"""
Figure and Axes text
"""
from __future__ import division
from artist import Artist
from transforms import Transform


def _process_text_args(override, fontdict=None, **kwargs):
    "Return an override dict.  See 'text' docstring for info"

    if fontdict is not None:
        override.update(fontdict)

    override.update(kwargs)
    return override

class Text(Artist):
    """
    Handle storing and drawing of text in window or data coordinates

    Publicly accessible attributes
      dpi     : a DPI instance
      bbox    : the display bounding box Bound2D instance
      transx : the transform of the x data
      transy : the transform of the y data

    """
    # fontweights : 'normal', 'bold', 'light'
    # fontangles  : 'italic', 'normal', 'oblique'

    def __init__(self,
                 dpi,
                 bbox,
                 x=0, y=0, text='',
                 color='k',
                 verticalalignment='bottom',
                 horizontalalignment='left',
                 fontname='Times',
                 fontsize=10,
                 fontweight='normal',
                 fontangle='normal',
                 rotation=None,
                 transx=Transform(), 
                 transy=Transform(),
                 ):
        
        Artist.__init__(self, dpi, bbox)
        self.transx = transx
        self.transy = transy
        self._x, self._y = x, y
        
        self._color = color
        self._text = text
        self._verticalalignment = verticalalignment
        self._horizontalalignment = horizontalalignment
        self._rotation = rotation
        self._fontname = fontname
        self._fontsize = fontsize
        self._fontweight = fontweight
        self._fontangle = fontangle

    def copy_properties(self, text):
        text._x = self._x
        text._y = self._y
        text._text = self._text
        text._color = self._color,
        text._verticalalignment = self._verticalalignment
        text._horizontalalignment = self._horizontalalignment
        text._fontname = self._fontname
        text._fontsize = self._fontsize
        text._fontweight = self._fontweight
        text._fontangle = self._fontangle
        text._rotation = self._rotation
        text.transx = self.transx 
        text.transy = self.transy

    def draw(self, renderer):
        x, y  = self.get_xy_display()
        gc = renderer.new_gc()
        gc.set_foreground(self._color)
        if self.get_clip_on():
            gc.set_clip_rectangle(self.bbox.get_bounds())

        renderer.draw_text(gc, x, y, self)

    def get_color(self):
        "Return the color of the text"
        return self._color 
        
    def get_fontname(self):
        "Return the font name as string"
        return self._fontname

    def get_fontsize(self):
        "Return the font size as integer"
        return self._fontsize

    def get_fontweight(self):
        "Get the font weight as string"
        return self._fontweight

    def get_fontangle(self):
        "Get the font angle as string"
        return self._fontangle

    def get_horizontalalignment(self):
        "Return the horizontal alignment as string"
        return self._horizontalalignment

    def get_position(self):
        "Return x, y as tuple"
        return self._x, self._y


    def get_prop_tup(self):
        """
        Return a hashable tuple of properties

        Not intended to be human readable, but useful for backends who
        want to cache derived information about text (eg layouts) and
        need to know if the text has changed
        """
        return (self._x, self._y, self._text, self._color,
        self._verticalalignment, self._horizontalalignment,
        self._fontname, self._fontsize, self._fontweight,
        self._fontangle, self._rotation)

    def get_rotation(self):
        "Return the text rotation arg"
        return self._rotation
    
    def get_text(self):
        "Get the text as string"
        return self._text

    def get_verticalalignment(self):
        "Return the vertical alignment as string"
        return self._verticalalignment

    def get_window_extent(self, renderer):
        """
        Return the bounding box of the text in window coords as a
        Bound2D instance
        """
        return renderer.get_text_extent(self)
        
    def get_xy_display(self):
        try: x = self._x.get()
        except AttributeError: x = self._x
        try: y = self._y.get()
        except AttributeError: y = self._y

        return self.transx.positions(x), self.transy.positions(y)
        
    def set_backgroundcolor(self, color):
        "Set the background color of the text"
        self._backgroundcolor = color

        
    def set_color(self, color):
        "Set the foreground color of the text"
        self._color = color

    def set_horizontalalignment(self, align):
        """
        Set the horizontal alignment to one of
        'center', 'right', or 'left'
        """
        legal = ('center', 'right', 'left')
        if align not in legal:
            raise ValueError('Horizontal alignment must be one of %s' % legal)
        if self._horizontalalignment == align: return     
        self._horizontalalignment = align


    def set_fontname(self, fontname):
        """
        Set the font name, eg, 'Sans', 'Courier', 'Helvetica'
        """
        self._fontname = fontname
        

    def set_fontsize(self, fontsize):
        """
        Set the font size, eg, 8, 10, 12, 14...
        """
        self._fontsize = fontsize
        
        
    def set_fontweight(self, weight):
        """
        Set the font weight, one of:
        'normal', 'bold', 'heavy', 'light', 'ultrabold',  'ultralight'
        """
        self._fontweight = weight
        
        
    def set_fontangle(self, angle):
        """
        Set the font angle, one of 'normal', 'italic', 'oblique'
        """
        self._fontangle = angle
        
        
    def set_position(self, xy):
        self.set_x(xy[0])
        self.set_y(xy[1])

    def set_x(self, x):        
        try: self._x.set(x)
        except AttributeError: self._x = x

    def set_y(self, y):
        try: self._y.set(y)
        except AttributeError: self._y = y
        
    def set_rotation(self, s):
        "Currently only s='vertical', or s='horizontal' are supported"
        self._rotation = s
        
        
    def set_verticalalignment(self, align):
        """
        Set the vertical alignment to one of
        'center', 'top', or 'bottom'
        """

        legal = ('top', 'bottom', 'center')
        if align not in legal:
            raise ValueError('Vertical alignment must be one of %s' % legal)

        self._verticalalignment = align
        
        
    def set_text(self, text):
        "Set the text"
        self._text = text
        

        

    def update_properties(self, d):
        "Update the font attributes with the dictionary in d"
        #check the keys
        
        legal = ('color', 'verticalalignment', 'horizontalalignment',
                 'fontname', 'fontsize', 'fontweight',
                 'fontangle', 'rotation')
        for k,v in d.items():
            if k not in legal:
                raise ValueError('Illegal key %s.  Legal values are %s' % (
                    k, legal))
            self.__dict__['_' + k] = v


