"""
Classes for the ticks and x and y axis
"""
from __future__ import division
import sys, math, re
from numerix import arange, array, asarray, ones, zeros, \
     nonzero, take, Float, log10, logical_and

from artist import Artist
from cbook import enumerate, True, False
from lines import Line2D
from mlab import linspace
from matplotlib import rcParams
from patches import bbox_artist
from ticker import NullFormatter, FixedFormatter, ScalarFormatter, LogFormatter
from ticker import NullLocator, FixedLocator, LinearLocator, LogLocator, AutoLocator
from ticker import AutoTicker, LinearTicker, NullTicker, LogTicker

from transforms import Bound1D, Bound2D, Transform, bound2d_all,\
     identity, logwarn, pow10, Centimeter, Dots, Inches, Millimeter,\
     Points, TransformSize, RRef
from font_manager import FontProperties
from text import Text, _process_text_args

class Tick(Artist):
    """
    Abstract base class for the axis ticks, grid lines and labels

    Publicly accessible attributes

      tick1line  : a Line2D instance
      tick2line  : a Line2D instance
      gridline   : a Line2D instance
      label1      : an Text instance
      label2      : an Text instance
      gridOn     : a boolean which determines whether to draw the tickline
      tick1On    : a boolean which determines whether to draw the 1st tickline
                   (left for xtick and bottom for yticks)
      tick2On    : a boolean which determines whether to draw the 2nd tickline
                   (left for xtick and bottom for yticks)
      label1On    : a boolean which determines whether to draw tick label
      label2On    : a boolean which determines whether to draw tick label
      
      
    """
    def __init__(self, axes, loc, label,
                 size   = None,  # points
                 gridOn = rcParams['axes.grid'],
                 tick1On = True,
                 tick2On = True,
                 label1On = True,
                 label2On = False,                 
                 major = True,                 
                 ):
        """
        bbox is the Bound2D bounding box in display coords of the Axes
        loc is the tick location in data coords
        size is the tick size in relative, axes coords
        """
        Artist.__init__(self, axes.dpi, axes.bbox)
        self.axes = axes

        if size is None:
            if major: size = rcParams['tick.major.size']
            else:     size = rcParams['tick.minor.size']

        self.tick1line = self._get_tick1line(loc, size)
        self.tick2line = self._get_tick2line(loc, size)
        self.gridline = self._get_gridline(loc)        
        self.label1 = self._get_text1(loc, label)
        self.label1.set_clip_on( False )
        self.label = self.label1  # legacy name

        self.label2 = self._get_text2(loc, label)
        self.label2.set_clip_on( False )

        
        self.gridOn = gridOn
        self.tick1On = tick1On
        self.tick2On = tick2On
        self.label1On = label1On
        self.label2On = label2On        
        self._loc = loc
                
        self._size = size
        
    def _get_text1(self, loc, label):
        'Get the default Text 1 instance'
        raise NotImplementedError('Derived must override')

    def _get_text2(self, loc, label):
        'Get the default Text 2 instance'
        raise NotImplementedError('Derived must override')

    def _get_tick1line(self, loc, size):
        'Get the default line2D instance for tick1'
        raise NotImplementedError('Derived must override')

    def _get_tick2line(self, loc, size):
        'Get the default line2D instance for tick2'
        raise NotImplementedError('Derived must override')

    def _get_gridline(self, loc):
        'Get the default grid Line2d instance for this tick'
        raise NotImplementedError('Derived must override')

    def get_child_artists(self):
        return (self.label1, self.label2, self.tick1line, self.tick2line, self.gridline)

    def get_loc(self):
        'Return the tick location (data coords) as a scalar'
        return self._loc
    
    def _draw(self, renderer):
        midPoint = self._tick_is_midpoint(self.get_loc())
        if midPoint and self.tick1On: self.tick1line.draw(renderer)
        if midPoint and self.tick2On: self.tick2line.draw(renderer)
        if midPoint and self.gridOn:  self.gridline.draw(renderer)
        if self.label1On: self.label1.draw(renderer)
        if self.label2On: self.label2.draw(renderer)

    
    def set_loc(self, loc):
        'Set the location of tick in data coords with scalar loc'
        raise NotImplementedError('Derived must override')

    def set_label(self, s):  # legacy name
        'Set the text of ticklabel in with string s'
        self.label1.set_text(s)

    def set_label1(self, s):
        'Set the text of ticklabel in with string s'
        self.label1.set_text(s)

    def set_label2(self, s):
        'Set the text of ticklabel2 in with string s'
        self.label2.set_text(s)

    def _tick_is_midpoint(self, loc):
        'return true if tickloc is not on axes boundary'
        raise NotImplementedError('Derived must override')

    def set_transform(self, trans):
        """
        Set the transform for the dimension the ticks vary along.  Eg,
        for xticks, this is the x dimension and for y ticks this is
        the y dimension.  All the child artists transforms will be ste
        """
        raise NotImplementedError('Derived must override')

class XTick(Tick):
    """
    Contains all the Artists needed to make an x tick - the tick line,
    the label text and the grid line
    """

    def set_transform(self, trans):
        """
        Set the transform for the x dimension the ticks vary along.  e
        """

        self.tick1line.transx = trans
        self.tick2line.transx = trans
        self.gridline.transx = trans
        self.label1.transx = trans
        self.label2.transx = trans                    
    
    def _get_text1(self, loc, label):
        'Get the default Text instance'
        # the y loc is 3 points below the min of y axis
        top = self.bbox.y.get_refmin() - self.dpi*RRef(3/72.0)
        text =  Text(
            dpi=self.dpi, 
            bbox=self.bbox,
            x=loc,
            y=top,
            fontproperties=FontProperties(size=rcParams['tick.labelsize']),
            color=rcParams['tick.color'],
            verticalalignment='top',
            horizontalalignment='center',
            transx = self.axes.xaxis.transData,
            # transy is default, identity transform
            )
        return text

    def _get_text2(self, loc, label):
        'Get the default Text 2 instance'
        # the y loc is 3 points above the max of y axis
        y = self.bbox.y.get_refmax() + self.dpi*RRef(3/72.0)
        text =  Text(
            dpi=self.dpi, 
            bbox=self.bbox,
            x=loc,
            y=y,
            fontproperties=FontProperties(size=rcParams['tick.labelsize']),
            color=rcParams['tick.color'],
            verticalalignment='bottom',
            horizontalalignment='center',
            transx = self.axes.xaxis.transData,
            # transy is default, identity transform
            )
        return text

        
    def _get_tick1line(self, loc, size):
        'Get the default line2D instance'

        line = Line2D( self.dpi, self.bbox,
                       xdata=(loc, loc), ydata=(0, size),
                       color='k',
                       antialiased=False, 
                       transx = self.axes.xaxis.transData,
                       transy = self.axes.yaxis.get_pts_transform(),
                       )
        line.set_data_clipping(False)
        return line

    def _get_tick2line(self, loc, size):
        'Get the default line2D instance'
        offset = self.axes.yaxis.displaylim.get_refmax()
        ytrans = self.axes.yaxis.get_pts_transform(offset)
        line = Line2D( self.dpi, self.bbox,
                       xdata=(loc, loc), ydata=(-size, 0),
                       color='k',
                       antialiased=False, 
                       transx = self.axes.xaxis.transData,
                       transy = ytrans,
                       )
        line.set_data_clipping(False)
        return line

    def _get_gridline(self, loc):
        'Get the default line2D instance'
        line = Line2D( self.dpi, self.bbox,
                       xdata=(loc, loc), ydata=(0, 1),
                       color='k', linestyle=':',
                       antialiased=False, 
                       transx = self.axes.xaxis.transData,
                       transy = self.axes.yaxis.transAxis,
                       )
        line.set_data_clipping(False)
        return line

    def set_loc(self, loc):
        'Set the location of tick in data coords with scalar loc'
        self.tick1line.set_xdata((loc, loc))
        self.tick2line.set_xdata((loc, loc))
        self.gridline.set_xdata((loc, loc))
        self.label1.set_x( loc )
        self.label2.set_x( loc )        
        self._loc = loc

    def _tick_is_midpoint(self, loc):
        'return true if tickloc is not on axes boundary'
        return self.axes.xaxis.viewlim.in_open_interval(loc)

            
        
        
class YTick(Tick):
    """
    Contains all the Artists needed to make a Y tick - the tick line,
    the label text and the grid line
    """
    def set_transform(self, trans):
        """
        Set the transform for the y dimension the ticks vary along.  e
        """

        self.tick1line.transy = trans
        self.tick2line.transy = trans
        self.gridline.transy = trans
        self.label1.transy = trans
        self.label2.transy = trans        

    # how far from the y axis line the right of the ticklabel are
    def _get_text1(self, loc, label):
        'Get the default Text instance'
        right = self.bbox.x.get_refmin() - self.dpi*RRef(3/72.0)
        return Text(
            dpi=self.dpi,
            bbox=self.bbox,
            x=right,
            y=loc,
            fontproperties=FontProperties(size=rcParams['tick.labelsize']),
            color=rcParams['tick.color'],
            verticalalignment='center',
            horizontalalignment='right',
            #transx - default transform is identity
            transy = self.axes.yaxis.transData,
            )

    def _get_text2(self, loc, label):
        'Get the default Text instance'
        x = self.bbox.x.get_refmax() + self.dpi*RRef(3/72.0)
        return Text(
            dpi=self.dpi,
            bbox=self.bbox,
            x=x,
            y=loc,
            fontproperties=FontProperties(size=rcParams['tick.labelsize']),
            color=rcParams['tick.color'],
            verticalalignment='center',
            horizontalalignment='left',
            #transx - default transform is identity
            transy = self.axes.yaxis.transData,
            )


    def _get_tick1line(self, loc, size):
        'Get the default line2D instance'
        line = Line2D( self.dpi, self.bbox,
                       (0, size), (loc, loc), color='k',
                       antialiased=False, 
                       transx = self.axes.xaxis.get_pts_transform(),
                       transy = self.axes.yaxis.transData,
                       )
        line.set_data_clipping(False)
        return line

    def _get_tick2line(self, loc, size):
        'Get the default line2D instance'

        # size points to the left of the xaxis max
        offset = self.axes.xaxis.displaylim.get_refmax() - self.dpi*RRef(size/72.0)
        xtrans = self.axes.xaxis.get_pts_transform(offset)

        line = Line2D( self.dpi, self.bbox,
                       (0, size), (loc, loc), color='k',
                       antialiased=False, 
                       transx = xtrans,
                       transy = self.axes.yaxis.transData,
                       )
        line.set_data_clipping(False)
        return line

    def _get_gridline(self, loc):
        'Get the default line2D instance'
        line = Line2D( self.dpi, self.bbox,
                       xdata=(0,1), ydata=(loc,loc), 
                       color='k', linestyle=':',
                       antialiased=False, 
                       transx = self.axes.xaxis.transAxis,
                       transy = self.axes.yaxis.transData,
                       )
        line.set_data_clipping(False)
        return line


    def set_loc(self, loc):
        'Set the location of tick in data coords with scalar loc'
        self.tick1line.set_ydata((loc, loc))
        self.tick2line.set_ydata((loc, loc))
        self.gridline.set_ydata((loc, loc))
        self.label1.set_y( loc )
        self.label2.set_y( loc )        
        self._loc = loc

    def _tick_is_midpoint(self, loc):
        'return true if tickloc is not on axes boundary'
        return self.axes.yaxis.viewlim.in_open_interval(loc)

    
class Axis(Artist):

    # func / inverse func pairs for transforms
    _scalemap = {'linear' : (identity, identity),
                 'log'    : (logwarn, pow10)}

    LABELPAD = -0.01
    """
    Public attributes
      transData - transform data coords to display coords
      transAxis - transform axis coords to display coords

    """
    def __init__(self, axes):
        """
        Init the axis with the parent Axes instance
        """
        Artist.__init__(self, axes.dpi, axes.bbox)
        self.axes = axes
        self._scale = 'linear'

        self.datalim = Bound1D(None, None) # data interval instance; data coords
        self.viewlim = Bound1D(-1, 1)      # viewport interval instance; data coords
        self.viewlimset = False
        self.displaylim = self._get_display_lim()
        
        self.transData = Transform(self.viewlim, self.displaylim)
        self.transAxis = Transform(Bound1D(0,1), self.displaylim)

        self._majorTicker = AutoTicker()
        self._minorTicker = NullTicker()
        
        self._majorTicker.set_lim(self.viewlim, self.datalim, self.displaylim)
        self._minorTicker.set_lim(self.viewlim, self.datalim, self.displaylim)        
        # whether the grids are on
        self._gridOnMajor = rcParams['axes.grid']  
        self._gridOnMinor = False

        
    def autoscale_view(self):
        'try and set the view lim intelligently based on datalim'
        tup = self._majorTicker.locator.autoscale()
        if tup is None:
            return
        vmin, vmax = tup
        self.viewlim.set_bounds(vmin, vmax)
        
    def build_artists(self):
        """
        Call only after xaxis and yaxis have been initialized in axes
        since the trasnforms require both
        """
        self._label = self._get_label()
        self._label.set_clip_on( False )
        # build a few default ticks; grow as necessary later; only
        # define 1 so properties set on ticks will be copied as they
        # grow
        self._majorTicks = [self._get_tick(major=True)  for i in range(1)]
        self._minorTicks = [self._get_tick(major=False) for i in range(1)]       


    def _draw(self, renderer, *args, **kwargs):
        'Draw the axis lines, grid lines, tick lines and labels'

        ticklabelBoxes = []

        majorTicks = self.get_major_ticks()
        majorLocs = self._majorTicker.locator()
        majorLabels = [self._majorTicker.formatter(val, i) for i, val in enumerate(majorLocs)]

        seen = {}

        for tick, loc, label in zip(majorTicks, majorLocs, majorLabels):
            if not self.viewlim.in_interval(loc): continue
            seen[loc] = 1
            tick.set_loc(loc)
            tick.set_label1(label)
            tick.set_label2(label)            

            tick.draw(renderer)
            extent = tick.label1.get_window_extent(renderer) 
            ticklabelBoxes.append(extent)


        minorTicks = self.get_minor_ticks()
        minorLocs = self._minorTicker.locator()
        minorLabels = [self._minorTicker.formatter(val, i) for i, val in enumerate(minorLocs)]


        for tick, loc, label in zip(minorTicks, minorLocs, minorLabels):
            if seen.has_key(loc): continue
            tick.set_loc(loc)
            tick.set_label(label)

            tick.draw(renderer)  
            extent = tick.label1.get_window_extent(renderer) 
            ticklabelBoxes.append(extent)


        # find the tick labels that are close to the axis labels.  I
        # scale up the axis label box to also find the neighbors, not
        # just the tick labels that actually overlap note we need a
        # *copy* of the axis label box because we don't wan't to scale
        # the actual bbox
        labelBox = self._label.get_window_extent(renderer).copy()
        labelBox.x.scale(1.25)
        labelBox.y.scale(1.25)
        overlaps = [b for b in ticklabelBoxes if b.overlap(labelBox)]


        self._update_label_postion(overlaps)

        self._label.draw(renderer)  # memory leak here, vertical text

        if 0: # draw the bounding boxes around the text for debug
            for tick in majorTicks:
                label = tick.label1
                bbox_artist(label, renderer)
            bbox_artist(self._label, renderer)


    def get_child_artists(self):
        'Return a list of all Artist instances contained by Axis'
        artists = []
        artists.extend(self._majorTicks)
        artists.extend(self._minorTicks)         
        artists.append(self._label)
        return artists

    def get_cm_transform(self, offset=None, transOffset=Transform()):
        """
        Transform val cm to dots

        offset is a RRef instance for TransformSize.  if None, use
        displaylim min.  transOffset is a transform to transform the
        offset to display coords
        """
        if offset is None:
            offset = self.displaylim.get_refmin()

        cm = Centimeter( self.dpi)
        dots =  Dots( self.dpi)
        return TransformSize(cm, dots, offset, transOffset)

    def _get_label(self):
        raise NotImplementedError('Derived must override')

    def _get_display_lim(self):
        raise NotImplementedError('Derived must override')

    def get_gridlines(self):
        'Return the grid lines as a list of Line2D instance'
        return [tick.gridline for tick in self._majorTicks]

    def get_inches_transform(self, offset=None, transOffset=Transform()):
        """
        Transform val inches to dots

        offset is a RRef instance for TransformSize.  if None, use
        displaylim min.  transOffset is a transform to transform the
        offset to display coords
        """
        if offset is None:
            offset = self.displaylim.get_refmin()

        inches = Inches( self.dpi)
        dots =  Dots( self.dpi)
        return TransformSize(inches, dots, offset, transOffset) 

    def get_label(self):
        'Return the axis label as an Text instance'
        return self._label


    def get_mm_transform(self, offset=None, transOffset=Transform()):
        """
        Transform val mm to dots

        offset is a RRef instance for TransformSize.  if None, use
        displaylim min.  transOffset is a transform to transform the
        offset to display coords

        """
        if offset is None:
            offset = self.displaylim.get_refmin()

        mm = Millimeter( self.dpi)
        dots =  Dots( self.dpi)
        return TransformSize(mm, dots, offset, transOffset)


    def get_pts_transform(self, offset=None, transOffset=Transform()):
        """
        Transform val points to dots

        offset is a RRef instance for TransformSize.  if None, use
        displaylim min.  transOffset is a transform to transform the
        offset to display coords
        """

        if offset is None:
            offset = self.displaylim.get_refmin()
        
        pts = Points( self.dpi)
        dots =  Dots( self.dpi)
        return TransformSize(pts, dots, offset, transOffset)

    def get_ticklabels(self):
        'Return a list of Text instances for ticklabels'
        return [tick.label1 for tick in self.get_major_ticks()]

    def get_ticklines(self):
        'Return the ticklines lines as a list of Line2D instance'
        lines = []
        for tick in self._majorTicks:
            lines.append(tick.tick1line)
            lines.append(tick.tick2line)
        return lines

    def get_ticklocs(self):
        "Get the tick locations in data coordinates as a Numeric array"
        return self._majorTicker.locator()

    def _get_tick(self, major):
        'return the default tick intsance'
        raise NotImplementedError('derived must override')

    def _copy_tick_props(self, src, dest):
        'Copy the props from src tick to dest tick'
        dest.label1.copy_properties(src.label1)
        dest.label2.copy_properties(src.label2)

        dest.tick1line.copy_properties(src.tick1line)
        dest.tick1line.copy_properties(src.tick2line)        
        dest.gridline.copy_properties(src.gridline)

        dest.tick1On = src.tick1On
        dest.tick2On = src.tick2On
        dest.label1On = src.label1On
        dest.label2On = src.label2On

        
    def get_major_ticks(self):
        'get the tick instances; grow as necessary'
        numticks = len(self._majorTicker.locator())
        if len(self._majorTicks)<numticks:
            # update the new tick label properties from the old
            protoTick = self._majorTicks[0]
            for i in range(numticks-len(self._majorTicks)):
                tick = self._get_tick(major=True)
                if self._gridOnMajor: tick.gridOn = True
                self._copy_tick_props(protoTick, tick)
                self._majorTicks.append(tick)
        ticks = self._majorTicks[:numticks]

        return ticks


    def get_minor_ticks(self):
        'get the minor tick instances; grow as necessary'
        numticks = len(self._minorTicker.locator())
        if len(self._minorTicks)<numticks:
            protoTick = self._minorTicks[0]
            for i in range(numticks-len(self._minorTicks)):
                tick = self._get_tick(major=False)
                if self._gridOnMinor: tick.gridOn = True
                self._copy_tick_props(protoTick, tick)
                self._minorTicks.append(tick)
        ticks = self._minorTicks[:numticks]

        return ticks

    def grid(self, b, which='major'):
        """
        Set the axis grid on or off; b is a boolean use which =
        'major' | 'minor' to set the grid for major or minor ticks
        """
        if which.lower().find('minor')>=0:
            self._gridOnMinor = b
            for tick in self._minorTicks:  # don't use get_ticks here!
                tick.gridOn = self._gridOnMinor
        else:
            self._gridOnMajor = b
            for tick in self._majorTicks:  # don't use get_ticks here!
                tick.gridOn = self._gridOnMajor
                
        
    def is_log(self):
        'Return true if log scaling is on'
        return self._scale=='log'
    
    def set_major_formatter(self, formatter):
        'Set the formatter of the major ticker'
        formatter.set_lim(self.viewlim, self.datalim, self.displaylim)
        self._majorTicker.formatter = formatter

    def set_minor_formatter(self, formatter):
        'Set the formatter of the minor ticker'        
        formatter.set_lim(self.viewlim, self.datalim, self.displaylim)
        self._minorTicker.formatter = formatter

    def set_major_locator(self, locator):
        'Set the locator of the major ticker'
        locator.set_lim(self.viewlim, self.datalim, self.displaylim)
        self._majorTicker.locator = locator

    def set_minor_locator(self, locator):
        'Set the locator of the minor ticker'
        locator.set_lim(self.viewlim, self.datalim, self.displaylim)
        self._minorTicker.locator = locator
        
        
    def set_scale(self, value):
        "Set the axis scale; either 'linear' or 'log'"
        if value not in self._scalemap.keys():
            raise ValueError('scale must be in %s'%str(self._scalemap.keys()))
        self._scale = value

        if self.is_log():
            self._majorTicker = LogTicker()
            self._majorTicker.set_lim(self.viewlim, self.datalim, self.displaylim)
        else:
            self._majorTicker = LinearTicker()
            self._majorTicker.set_lim(self.viewlim, self.datalim, self.displaylim)
        self.transData.set_funcs(self._scalemap[value])
        self.viewlim.is_positive( self.is_log() )
        self.axes.update_viewlim()

    def set_ticklabels(self, ticklabels, *args, **kwargs):
        """
        Set the text values of the tick labels.  ticklabels is a
        sequence of strings.  Return a list of Text instances
        """
        ticklabels = [str(l) for l in ticklabels]

        formatter = FixedFormatter(ticklabels)
        formatter.set_lim(self.viewlim, self.datalim, self.displaylim)
        self._majorTicker.formatter = formatter
    
        override = {}
        override = _process_text_args(override, *args, **kwargs)

        ret = []
        for i, tick in enumerate(self.get_major_ticks()):
            if i<len(ticklabels): ret.append(tick.label1)
            tick.label1.update_properties(override)
        return ret
    
    def set_ticks(self, ticks):
        'Set the locations of the tick marks from sequence ticks'
        locator = FixedLocator(ticks)
        locator.set_lim(self.viewlim, self.datalim, self.displaylim)
        self._majorTicker.locator = locator

        self.viewlim.update(ticks)
        return self.get_major_ticks()
    
    def _update_label_postion(self, overlaps):
        """
        Update the label position based on the sequence of bounding
        boxes overlaps of all the ticklabels that overlap the current
        ticklabel.  overlaps are the bounding boxes of ticklabels
        """
        raise NotImplementedError('Derived must override')

    def pan(self, numsteps):
        'Pan numticks (can be positive or negative)'
        self._majorTicker.locator.pan(numsteps)

    def zoom(self, direction):
        "Zoom in/out on axis; if direction is >0 zoom in, else zoom out"
        self._majorTicker.locator.zoom(direction)
        
class XAxis(Axis):
    __name__ = 'xaxis'

    def _get_tick(self, major):

        return XTick(self.axes, 0, '', major=major)

    def _get_label(self):
        label = Text(
            self.dpi, self.bbox,
            x = 0.5,               # centered
            y = self.LABELPAD,     # rel coords under axis line
            fontproperties = FontProperties(size=rcParams['axes.labelsize']),
            color = rcParams['axes.labelcolor'],
            verticalalignment='top',
            horizontalalignment='center',
            transx = self.transAxis,
            transy = self.axes.yaxis.transAxis,
            )
        return label

    def _get_display_lim(self):
        return self.bbox.x

    def _update_label_postion(self, overlaps):
        """
        Update the label position based on the sequence of bounding
        boxes overlaps of all the ticklabels that overlap the current
        ticklabel.  overlaps are the bounding boxes of the ticklabels
        """

        x,y = self._label.get_position()
        if not len(overlaps): return

        bbox = bound2d_all(overlaps)
        bottomAxes = (bbox.y.min()-self.bbox.y.min())/self.bbox.y.interval()
        self._label.set_position((x,bottomAxes+self.LABELPAD))
        

    def tick_top(self):
        'use ticks only on top'
        ticks = self.get_major_ticks()
        ticks.extend( self.get_minor_ticks() )
        for t in ticks:
            t.tick1On = False
            t.tick2On = True
            t.label1On = False
            t.label2On = True

    def tick_bottom(self):
        'use ticks only on bottom'
        ticks = self.get_major_ticks()
        ticks.extend( self.get_minor_ticks() )
        for t in ticks:
            t.tick1On = True
            t.tick2On = False
            t.label1On = True
            t.label2On = False


class YAxis(Axis):
    __name__ = 'yaxis'

    def _get_tick(self, major):
        return YTick(self.axes, 0, '', major=major)


    def _get_label(self):
        label = Text(
            dpi=self.dpi,
            bbox = self.bbox, 
            x = self.LABELPAD,    # rel coords to the left of axis line
            y = 0.5,      # centered
            fontproperties=FontProperties(size=rcParams['axes.labelsize']),
            color    = rcParams['axes.labelcolor'],
            verticalalignment='center',
            horizontalalignment='right',
            rotation='vertical', 
            transx = self.axes.xaxis.transAxis,
            transy = self.transAxis,
            )
        return label

    def _get_display_lim(self):
        return self.bbox.y


    def _update_label_postion(self, overlaps):
        """
        Update the label position based on the sequence of bounding
        boxes overlaps of all the ticklabels that overlap the current
        ticklabel.  overlaps are the bounding boxes of the ticklabels
        """
        x,y = self._label.get_position()
        if not len(overlaps):
            #print 'no yticklabel overlaps'
            return
        #print 'found', len(overlaps)
        bbox = bound2d_all(overlaps)
        left = (bbox.x.min()-self.bbox.x.min())/self.bbox.x.interval()
        self._label.set_position((left+self.LABELPAD,y))



    def tick_right(self):
        'use ticks only on right'
        ticks = self.get_major_ticks()
        ticks.extend( self.get_minor_ticks() )
        for t in ticks:
            t.tick1On = False
            t.tick2On = True
            t.label1On = False
            t.label2On = True

    def tick_left(self):
        'use ticks only on left'
        ticks = self.get_major_ticks()
        ticks.extend( self.get_minor_ticks() )
        for t in ticks:
            t.tick1On = True
            t.tick2On = False
            t.label1On = True
            t.label2On = False

