"""
Abstract base classes definine the primitives that renderers and
graphics contexts must implement to serve as a matplotlib backend
"""

from __future__ import division
import sys
from numerix import array

from axes import Axes, Subplot
from cbook import True, False
from patches import Rectangle
from cbook import is_string_like, enumerate, True, False
from transforms import Bound2D
from colors import colorConverter


class RendererBase:

    def flipy(self):
        'return true if y small numbers are top for renderer'
        return True
    
    def points_to_pixels(self, points):
        """
        convert points to display units; unless your backend doesn't
        have dpi, eg, postscript, you need to overrride this function
        """
        return points  

    
    def get_text_extent(self, text):
        """
        Get the text extent in window coords
        """
        return Bound2D(0,0,1,1)  # your values here


    def draw_arc(self, gcEdge, gcFace, x, y, width, height, angle1, angle2):
        """
        Draw an arc centered at x,y with width and height and angles
        from 0.0 to 360.0

        If gcFace is not None, fill the rectangle with it.  gcEdge
        is a GraphicsContext instance
        """
        pass  # derived must override

    def draw_image(self, x, y, im):
        """
        Draw the Image instance into the current axes
        """
        print >>sys.stderr, 'This backend does not yet support images'

    def draw_line(self, gc, x1, y1, x2, y2):
        """
        Draw a single line from x1,y1 to x2,y2
        """
        pass  # derived must override

    def draw_lines(self, gc, x, y):
        """
        x and y are equal length arrays, draw lines connecting each
        point in x, y
        """
        pass  # derived must override

    def draw_point(self, gc, x, y):
        """
        Draw a single point at x,y
        """
        pass  # derived must override

    def draw_polygon(self, gcEdge, gcFace, points):
        """
        Draw a polygon.  points is a len vertices tuple, each element
        giving the x,y coords a vertex

        If gcFace is not None, fill the rectangle with it.  gcEdge
        is a GraphicsContext instance

        """  
        pass # derived must override

    def draw_rectangle(self, gcEdge, gcFace, x, y, width, height):
        """
        Draw a rectangle with lower left at x,y with width and height.

        If gcFace is not None, fill the rectangle with it.  gcEdge
        is a GraphicsContext instance
        """
        pass # derived must override


    def draw_text(self, gc, x, y, t):
        """
        Draw a text instance
        """
        pass

    
    def new_gc(self):
        """
        Return an instance of a GraphicsContextBase
        """
        return GraphicsContextBase()



class GraphicsContextBase:    

    # a mapping from dash styles to suggested offset, dash pairs
    _dashd = {
        'solid'   : (None, None),
        'dashed'  : (0, array([6.0, 6.0])),
        'dashdot' : (0, array([3.0, 5.0, 1.0, 5.0])),
        'dotted'  : (0, array([1.0, 3.0]))
              }

    def __init__(self):
        self._rgb = (0.0, 0.0, 0.0)
        self._linewidth = 1
        self._capstyle = 'butt'
        self._joinstyle = 'miter'
        self._dashes = None, None
        self._cliprect = None
        self._linestyle = 'solid'
        self._antialiased = 1  # use 0,1 not True, False for extension code
        self._alpha = 1.0
    
    def get_antialiased(self):
        "Return true if the object shuold try to do antialiased rendering"
        return self._antialiased

    def get_clip_rectangle(self):
        """
        Return the clip rectangle as (left, bottom, width, height)
        """
        return self._cliprect

    def get_dashes(self):
        """
        Return the dash information as an offset dashlist tuple The
        dash list is a even size list that gives the ink on, ink off
        in pixels.  See p107 of to postscript BLUEBOOK for more info

        Default value is None
        """
        return self._dashes

    def get_alpha(self):
        """
        Return the alpha value used for blending - not supported on
        all backends
        """
        return self._alpha

    def get_capstyle(self):
        """
        Return the capstyle as a string in ('butt', 'round', 'projecting')
        """
        return self._capstyle

    def get_joinstyle(self):
        """
        Return the line join style as one of ('miter', 'round', 'bevel')
        """
        return self._joinstyle

    def get_linestyle(self, style):
        """
        Return the linestyle: one of ('solid', 'dashed', 'dashdot',
        'dotted').  
        """
        return self._linestyle

    def get_rgb(self):
        """
        returns a tuple of three floats from 0-1.  color can be a
        matlab format string, a html hex color string, or a rgb tuple
        """
        return self._rgb

    def set_antialiased(self, b):
        """
        True if object should be drawn with antialiased rendering
        """

        # use 0, 1 to make life easier on extension code trying to read the gc
        if b: self._antialiased = 1
        else: self._antialiased = 0

    def set_clip_rectangle(self, rectangle):
        """
        Set the clip rectangle with sequence (left, bottom, width, height)
        """
        self._cliprect = rectangle



    def set_alpha(self, alpha):
        """
        Set the alpha value used for blending - not supported on
        all backends
        """
        self._alpha = alpha

    def set_linestyle(self, style):
        """
        Set the linestyle to be one of ('solid', 'dashed', 'dashdot',
        'dotted').  
        """
        if style not in ('solid', 'dashed', 'dashdot', 'dotted'):
            error_msg('Unrecognized linestyle.  Found %s' % js)
            return
        self._linestyle = style
        offset, dashes = self._dashd[style]
        self.set_dashes(offset, dashes)

    def set_dashes(self, dash_offset, dash_list):
        """
        Set the dash style for the gc.  dash offset is the offset
        (usually 0).  Dash list specifies the on-off sequence as
        points
        """
        self._dashes = dash_offset, dash_list
    
    def set_foreground(self, fg, isRGB=None):
        """
        Set the foreground color.  fg can be a matlab format string, a
        html hex color string, an rgb unit tuple, or a float between 0
        and 1.  In the latter case, grayscale is used.

        The GraphicsContext converts colors to rgb internally.  If you
        know the color is rgb already, you can set isRGB to True to
        avoid the performace hit of the conversion
        """
        if isRGB:
            self._rgb = fg
        else:
            self._rgb = colorConverter.to_rgb(fg)

    def set_graylevel(self, frac):
        """
        Set the foreground color to be a gray level with frac frac
        """
        self._rgb = (frac, frac, frac)
        

    def set_linewidth(self, w):
        """
        Set the linewidth in points
        """
        self._linewidth = w

    def set_capstyle(self, cs):
        """
        Set the capstyle as a string in ('butt', 'round', 'projecting')
        """
        if cs not in ('butt', 'round', 'projecting'):
            error_msg('Unrecognized cap style.  Found %s' % cs)
        self._capstyle = cs

    def set_joinstyle(self, js):
        """
        Set the join style to be one of ('miter', 'round', 'bevel')
        """
        
        if js not in ('miter', 'round', 'bevel'):
            error_msg('Unrecognized join style.  Found %s' % js)
        self._joinstyle = js


    def get_linewidth(self):
        """
        Return the line width in points as a scalar
        """
        return self._linewidth

    def copy_properties(self, gc):
        'Copy properties from gc to self'
        self._rgb = gc._rgb
        self._linewidth = gc._linewidth
        self._capstyle = gc._capstyle
        self._joinstyle = gc._joinstyle
        self._linestyle = gc._linestyle
        self._dashes = gc._dashes
        self._cliprect = gc._cliprect
        self._antialiased = gc._antialiased
        self._alpha = gc._alpha
        
        
class FigureCanvasBase:
    """
    The canvas the figure renders into.

    Public attribute

      figure - A Figure instance
    """
    def __init__(self, figure):
        self.figure = figure
        
    def draw(self):
        """
        Render the figure
        """
        pass

    def print_figure(self, filename, dpi=300, facecolor='w', edgecolor='w'):
        """
        Render the figure to hardcopy.  Set the figure patch face and
        edge colors.  This is useful because some of the GUIs have a
        gray figure face color background and you'll probably want to
        override this on hardcopy
        """
        pass

    def switch_backends(self, FigureCanvasClass):
        """
        instantiate an instance of FigureCanvasClass

        This is used for backend switching, eg, to instantiate a
        FigureCanvasPS from a FigureCanvasGTK.  Note, deep copying is
        not done, so any changes to one of the instances (eg, setting
        figure size or line props), will be reflected in the other
        """
        newCanvas = FigureCanvasClass(self.figure)
        return newCanvas


class FigureManagerBase:
    """
    Helper class for matlab mode, wraps everything up into a neat bundle

    Public attibutes
    canvas - A FigureCanvas instance
    num    - The figure number
    """
    def __init__(self, canvas, num):
        self.canvas = canvas
        self.num = num
        self.clf()
        
    def clf(self):
        'clear the figure'
        self.axes = {}
        self.currentAxis = None
        self.canvas.figure.clf()
        
    def add_subplot(self, *args, **kwargs):
        """
        Add a subplot to the current figure
        """
        key = (args, tuple(kwargs.items()))
        if self.axes.has_key(key):
            self.currentAxis = self.axes[key]
        else:
            a = Subplot(self.canvas.figure, *args, **kwargs)
            self.canvas.figure.add_axis(a)
            self.axes[key] = a
            self.currentAxis = a
            return a
        
    def add_axes(self, rect, **kwargs):
        """
        Add an axes to the current figure
        """
        rect = tuple(rect)
        key = (rect, tuple(kwargs.items()))
        if self.axes.has_key(key):
            self.currentAxis = self.axes[key]
            return self.currentAxis
        else:
            a = Axes(self.canvas.figure, position=rect, **kwargs)
            self.canvas.figure.add_axis(a)
            self.axes[key] = a
            self.currentAxis = a
            return a

    def get_current_axis(self):
        """
        Return the current axes
        """
        if self.currentAxis is not None:
            return self.currentAxis
        else:
            self.add_subplot(111)
            return self.currentAxis


    def set_current_axes(self, a):
        """
        Set the current axes to be a
        """
        if a is None:
            self.currentAxis = None
            return
        if a not in self.axes.values():
            error_msg('Axes is not in current figure')
        self.currentAxis = a

    def destroy(self):
        pass


def error_msg(msg, *args, **kwargs):
    """
    Alert an error condition with message
    """
    print >>sys.stderr, msg
    sys.exit()
