"""
A collection of modules for collecting, analyzing and plotting
financial data.   User contributions welcome!

"""
from __future__ import division
import sys, os, math
from urllib import urlopen

from artist import Artist
from dates import EpochConverter,\
     MONDAY, TUESDAY, WEDNESDAY, THURSDAY, FRIDAY, SATURDAY, SUNDAY,\
     SEC_PER_DAY
from lines import Line2D
from patches import Rectangle

from matplotlib.matlab import gca

def quotes_historical_yahoo(ticker, date1, date2,
                            converter=EpochConverter()):

    """
    Get historical data for ticker between date1 and date2.  converter
    is a DateConverter class appropriate for converting your dates

    results are a list of

    d, open, close, high, low, volume
    

    where d is an instnace of your datetime supplied by the converter
    """

    y,m,d = converter.ymd(date1)
    d1 = (m, d, y)
    y,m,d = converter.ymd(date2)
    d2 = (m, d, y)

    urlFmt = 'http://table.finance.yahoo.com/table.csv?a=%d&b=%d&c=%d&d=%d&e=%d&f=%d&s=%s&y=0&g=d&ignore=.csv'
    url =  urlFmt % (d1[0], d1[1], d1[2],
                     d2[0], d2[1], d2[2], ticker)

    ticker = ticker.upper()

    results = []
    lines = urlopen(url).readlines()
    for line in lines[1:]:

        vals = line.split(',')

        if len(vals)!=7: continue
        datestr = vals[0]

        d = converter.strptime(datestr, '%d-%b-%y')
        open, high, low, close =  [float(val) for val in vals[1:5]]
        volume = int(vals[5])

        results.append((d, open, close, high, low, volume))
    results.reverse()
    return results


        
def plot_day_summary(ax, quotes, ticksize=0.2,
                     colorup='k', colordown='r',
                     converter = EpochConverter(),
                     ):
    """
    quotes is a list of (time, open, close, high, low, ...) tuples
    
    Represent the time, open, close, high, low as a vertical line
    ranging from low to high.  The left tick is the open and the right
    tick is the close.

    ax          : an Axes instance to plot to
    ticksize    : fraction of a day for the open and close ticks
    colorup     : the color of the lines where close >= open
    colordown   : the color of the lines where close <  open    
    converter   : dates.DateConverter instance used to convert your times.    
                  If your data is already epoch, you can use converter=None
    return value is a list of lines added
    """

    TICKSIZE = ticksize * SEC_PER_DAY


    dpi = ax.dpi
    bbox = ax.bbox

    lines = []
    for q in quotes:

        t, open, close, high, low = q[:5]
        if converter is not None:
            t = converter.epoch(t)

        if close>=open : color = colorup
        else           : color = colordown

        vline = Line2D(
            dpi, bbox,
            xdata=(t, t), ydata=(low, high),
            color=color,
            antialiased=False,   # no need to antialias vert lines
            transx = ax.xaxis.transData,
            transy = ax.yaxis.transData)

        oline = Line2D(
            dpi, bbox,
            xdata=(t - TICKSIZE, t), ydata=(open, open),
            color=color,
            antialiased=False, 
            transx = ax.xaxis.transData,
            transy = ax.yaxis.transData)

        cline = Line2D(
            dpi, bbox,
            xdata=(t, t + TICKSIZE), ydata=(close, close),
            color=color,
            antialiased=False, 
            transx = ax.xaxis.transData,
            transy = ax.yaxis.transData)

        lines.extend((vline, oline, cline))
        ax.add_line(vline)
        ax.add_line(oline)
        ax.add_line(cline)


    ax.xaxis.autoscale_view()
    ax.yaxis.autoscale_view()
    return lines


def candlestick(ax, quotes, width=0.2, colorup='k', colordown='r',
                converter = EpochConverter(),
                alpha=1.0):

    """

    quotes is a list of (time, open, close, high, low, ...)  tuples.
    As long as the first 5 elements of the tuples are these values,
    the tuple can be as long as you want (eg it may store volume).
    
    Plot the time, open, close, high, low as a vertical line ranging
    from low to high.  Use a rectangular bar to represent the
    open-close span.  If close >= open, use colorup to color the bar,
    otherwise use colordown

    ax          : an Axes instance to plot to
    width       : fraction of a day for the rectangle width
    colorup     : the color of the rectangle where close >= open
    colordown   : the color of the rectangle where close <  open    
    converter   : dates.DateConverter instance used to convert your times.
                  If your data is already epoch, you can use converter=None
    alpha       : the rectangle alpha level
    
    return value is lines, patches where lines is a list of lines
    added and patches is a list of the rectangle patches added
    """

    WIDTH = width * SEC_PER_DAY
    OFFSET = WIDTH/2.0
    
    dpi = ax.dpi
    bbox = ax.bbox

    lines = []
    patches = []
    for q in quotes:
        t, open, close, high, low = q[:5]
        if converter is not None:
            t = converter.epoch(t)

        if close>=open :
            color = colorup
            lower = open
            height = close-open
        else           :
            color = colordown
            lower = close
            height = open-close

        vline = Line2D(
            dpi, bbox,
            xdata=(t, t), ydata=(low, high),
            color='k',
            linewidth=0.5,
            antialiased=True,   
            transx = ax.xaxis.transData,
            transy = ax.yaxis.transData)
        

        rect = Rectangle(
            dpi, bbox,
            xy    = (t-OFFSET, lower),
            width = WIDTH,
            height = height, 
            facecolor = color,
            edgecolor = color,            
            transx = ax.xaxis.transData,
            transy = ax.yaxis.transData)
        rect.set_alpha(alpha)

        lines.append(vline)
        patches.append(rect)
        ax.add_line(vline)
        ax.add_patch(rect)        

    ax.xaxis.autoscale_view()
    ax.yaxis.autoscale_view()
    return lines, patches

if __name__=='__main__':

    if 1: # test yahoo historical
        date1 = datetime.date( 2002, 1, 1 )
        date2 = datetime.date( 2002, 3, 1 )
        quotes = quotes_historical_yahoo('INTC', date1, date2)
        
        for t, open, close, high, low, volume in quotes:
            print t, open, close, high, low, volume
