r"""

OVERVIEW 

  mathtext is a module for parsing TeX expressions and drawing them
  into a matplotlib.ft2font image buffer.  You can draw from this
  buffer into your backend.

  A large set of the TeX symbols are provided (see below).
  Subscripting and superscripting are supported, as well as the
  over/under style of subscripting with \sum, \int, etc.

  The module uses pyparsing to parse the TeX expression, an so can
  handle fairly complex TeX expressions Eg, the following renders
  correctly

  s = r'$\cal{R}\prod_{i=\alpha\cal{B}}^\infty a_i\rm{sin}(2 \pi f x_i)$'

  The fonts \cal, \rm, \it, and \tt are allowed.

  If you find TeX expressions that don't parse or render properly,
  please email me, but please check KNOWN ISSUES below first.

REQUIREMENTS

  mathtext requires matplotlib.ft2font.  Set BUILD_FT2FONT=True in
  setup.py.  See BACKENDS below for a summary of availability by
  backend.

LICENSING:

  The computer modern fonts this package uses are part of the BaKoMa
  fonts, which are (in my understanding) free for noncommercial use.
  For commercial use, please consult the licenses in fonts/ttf and the
  author Basil K. Malyshev - see also
  http://www.mozilla.org/projects/mathml/fonts/encoding/license-bakoma.txt

USAGE:

  See http://matplotlib.sourceforge.net/tutorial.html#mathtext for a
  tutorial introduction.
  
  Any text element (xlabel, ylabel, title, text, etc) can use TeX
  markup, as in

    xlabel(r'$\Delta_i$')
           ^
        use raw strings

  The $ symbols must be the first and last symbols in the string.  Eg,
  you cannot do 

    r'My label $x_i$'.  

  But you can change fonts, as in 

    r'\rm{My label} x_i' 

  to achieve the same effect.

  A large set of the TeX symbols are provided.  Subscripting and
  superscripting are supported, as well as the over/under style of
  subscripting with \sum, \int, etc.


  Allowed TeX symbols:

  \Delta \Downarrow \Gamma \Im \LEFTangle \LEFTbrace \LEFTbracket
  \LEFTparen \Lambda \Leftarrow \Leftbrace \Leftbracket \Leftparen
  \Leftrightarrow \Omega \P \Phi \Pi \Psi \RIGHTangle \RIGHTbrace
  \RIGHTbracket \RIGHTparen \Re \Rightarrow \Rightbrace \Rightbracket
  \Rightparen \S \SQRT \Sigma \Sqrt \Theta \Uparrow \Updownarrow
  \Upsilon \Vert \Xi \aleph \alpha \approx \ast \asymp \backslash
  \beta \bigcap \bigcirc \bigcup \bigodot \bigoplus \bigotimes
  \bigtriangledown \bigtriangleup \biguplus \bigvee \bigwedge \bot
  \bullet \cap \cdot \chi \circ \clubsuit \coprod \cup \dag \dashv
  \ddag \delta \diamond \diamondsuit \div \downarrow \ell \emptyset
  \epsilon \equiv \eta \exists \flat \forall \frown \gamma \geq \gg
  \heartsuit \imath \in \infty \int \iota \jmath \kappa \lambda
  \langle \lbrace \lceil \leftangle \leftarrow \leftbrace \leftbracket
  \leftharpoondown \leftharpoonup \leftparen \leftrightarrow \leq
  \lfloor \ll \mid \mp \mu \nabla \natural \nearrow \neg \ni \nu
  \nwarrow \odot \oint \omega \ominus \oplus \oslash \otimes \phi \pi
  \pm \prec \preceq \prime \prod \propto \psi \rangle \rbrace \rceil
  \rfloor \rho \rightangle \rightarrow \rightbrace \rightbracket
  \rightharpoondown \rightharpoonup \rightparen \searrow \sharp \sigma
  \sim \simeq \slash \smile \spadesuit \sqcap \sqcup \sqrt \sqsubseteq
  \sqsupseteq \subset \subseteq \succ \succeq \sum \supset \supseteq
  \swarrow \tau \theta \times \top \triangleleft \triangleright
  \uparrow \updownarrow \uplus \upsilon \varepsilon \varphi \varphi
  \varrho \varsigma \vartheta \vdash \vee \wedge \wp \wr \xi \zeta

  
BACKENDS

  mathtext currently works with GTK, Agg, GTKAgg, TkAgg and WxAgg.  If
  David incorporates ft2font into paint, it will be easy to add to
  Paint.  WX can follow the lead of GTK and use pixels API calls to
  use mathtext.

  PS will require more substantial work, doing the metrics and
  layouts with the AFM versions of the computer modern fonts.
  Backends which don't support mathtext will just render the TeX
  string as a literal.  Stay tuned.


KNOWN ISSUES:

 - some hackish ways I deal with a strange offset in cmex10
 - nested subscripts, eg, x_i_i not working
 - nesting fonts changes in sub/superscript groups not parsing
 - I would also like to add a few more layout commands, like \frac.

Author    : John Hunter <jdhunter@ace.bsd.uchicago.edu>
Copyright : John Hunter (2004)
License   : matplotlib license (PSF compatible)
 
"""
from __future__ import division
import os, sys
from matplotlib.pyparsing import Literal, Word, OneOrMore, ZeroOrMore, \
     Combine, Group, Optional, Forward, NotAny, alphas, nums, alphanums

from matplotlib.artist import DPI
from matplotlib.cbook import True, False, enumerate, iterable
from matplotlib.ft2font import FT2Font
from matplotlib.numerix import absolute
from matplotlib.transforms import Bound2D, bound2d_all
from matplotlib import get_data_path

# the cmex fonts don't align with the other cm* fonts for reasons I
# haven't been able to figure out.  This is a correction factor (512
# is 0.25 size of em; this is a hack that I hope to clean up
OFFSET_EX10 = 256.0/64

# symbolx that have the sub and suprscrips over/under 
overunder = { r'\sum'    : 1,
              r'\int'    : 1,
              r'\prod'   : 1,
              r'\coprod' : 1,
              }
              
fonttable = {

    '.'      : ('cmmi10.ttf', '3A'),
    ','      : ('cmmi10.ttf', '3B'),
    '<'      : ('cmmi10.ttf', '3C'),
    '/'      : ('cmmi10.ttf', '3D'),
    '>'      : ('cmmi10.ttf', '3E'),

    '+'      : ('cmr10.ttf', '2B'),
    '-'      : ('cmr10.ttf', '2D'),
    '='      : ('cmr10.ttf', '3D'),
    ':'      : ('cmr10.ttf', '3A'),
    ';'      : ('cmr10.ttf', '3B'),

    '0'      : ('cmr10.ttf', '30'),
    '1'      : ('cmr10.ttf', '31'),
    '2'      : ('cmr10.ttf', '32'),
    '3'      : ('cmr10.ttf', '33'),
    '4'      : ('cmr10.ttf', '34'),
    '5'      : ('cmr10.ttf', '35'),
    '6'      : ('cmr10.ttf', '26'),
    '7'      : ('cmr10.ttf', '37'),
    '8'      : ('cmr10.ttf', '38'),
    '9'      : ('cmr10.ttf', '39'),


    # Math symbol font
    r'\leftarrow'      : ('cmsy10.ttf', '21'),
    r'\uparrow'        : ('cmsy10.ttf', '22'),
    r'\downarrow'      : ('cmsy10.ttf', '23'),
    r'\leftrightarrow' : ('cmsy10.ttf', '24'),
    r'\nearrow'        : ('cmsy10.ttf', '25'),
    r'\searrow'        : ('cmsy10.ttf', '26'),
    r'\simeq'          : ('cmsy10.ttf', '27'),

    r'\Leftarrow'      : ('cmsy10.ttf', '28'),
    r'\Rightarrow'     : ('cmsy10.ttf', '29'),
    r'\Uparrow'        : ('cmsy10.ttf', '2A'),
    r'\Downarrow'      : ('cmsy10.ttf', '2B'),
    r'\Leftrightarrow' : ('cmsy10.ttf', '2C'),
    r'\nwarrow'        : ('cmsy10.ttf', '2D'),
    r'\swarrow'        : ('cmsy10.ttf', '2E'),
    r'\propto'         : ('cmsy10.ttf', '2F'),

    r'\prime'           : ('cmsy10.ttf', '30'),
    r'\infty'           : ('cmsy10.ttf', '31'),
    r'\in'              : ('cmsy10.ttf', '32'),
    r'\ni'              : ('cmsy10.ttf', '33'),
    r'\bigtriangleup'   : ('cmsy10.ttf', '34'),
    r'\bigtriangledown' : ('cmsy10.ttf', '35'),
    r'\slash'           : ('cmsy10.ttf', '36'),
    #''                 : ('cmsy10.ttf', '37'),
    r'\forall'          : ('cmsy10.ttf', '38'),
    r'\exists'          : ('cmsy10.ttf', '39'),
    r'\neg'             : ('cmsy10.ttf', '3A'),
    r'\emptyset'        : ('cmsy10.ttf', '3B'),
    r'\Re'              : ('cmsy10.ttf', '3C'),
    r'\Im'              : ('cmsy10.ttf', '3D'),
    r'\top'             : ('cmsy10.ttf', '3E'),
    r'\bot'             : ('cmsy10.ttf', '3F'),

    r'\aleph'  : ('cmsy10.ttf', '40'),
    r'\cup'    : ('cmsy10.ttf', '5B'),
    r'\cap'    : ('cmsy10.ttf', '5C'),
    r'\uplus'  : ('cmsy10.ttf', '5D'),
    r'\wedge'  : ('cmsy10.ttf', '5E'),
    r'\vee'    : ('cmsy10.ttf', '5F'),

    r'\vdash'       : ('cmsy10.ttf', '60'),
    r'\dashv'       : ('cmsy10.ttf', '61'),
    r'\lfloor'      : ('cmsy10.ttf', '62'),
    r'\rfloor'      : ('cmsy10.ttf', '63'),
    r'\lceil'       : ('cmsy10.ttf', '64'),
    r'\rceil'       : ('cmsy10.ttf', '65'),
    r'\lbrace'      : ('cmsy10.ttf', '66'),
    r'\rbrace'      : ('cmsy10.ttf', '67'),
    r'\langle'      : ('cmsy10.ttf', '68'),
    r'\rangle'      : ('cmsy10.ttf', '69'),
    r'\mid'         : ('cmsy10.ttf', '6A'),
    r'\Vert'        : ('cmsy10.ttf', '6B'),
    r'\updownarrow' : ('cmsy10.ttf', '6C'),
    r'\Updownarrow' : ('cmsy10.ttf', '6D'),
    r'\backslash'   : ('cmsy10.ttf', '6E'),
    r'\wr'          : ('cmsy10.ttf', '6F'),

    #r'\surd'       : ('cmsy10.ttf', '70'),

    r'\nabla'       : ('cmsy10.ttf', '72'),
    r'\sqcup'       : ('cmsy10.ttf', '74'),
    r'\sqcap'       : ('cmsy10.ttf', '75'),
    r'\sqsubseteq'  : ('cmsy10.ttf', '76'),
    r'\sqsupseteq'  : ('cmsy10.ttf', '77'),
    r'\S'           : ('cmsy10.ttf', '78'),
    r'\dag'         : ('cmsy10.ttf', '79'),
    r'\ddag'        : ('cmsy10.ttf', '7A'),
    r'\P'           : ('cmsy10.ttf', '7B'),
    r'\clubsuit'    : ('cmsy10.ttf', '7C'),
    r'\diamondsuit' : ('cmsy10.ttf', '7D'),
    r'\heartsuit'   : ('cmsy10.ttf', '7E'),

    '-'         : ('cmsy10.ttf', 'A1'),  
    r'\cdot'    : ('cmsy10.ttf', 'A2'),
    r'\times'   : ('cmsy10.ttf', 'A3'),
    r'\ast'     : ('cmsy10.ttf', 'A4'),
    '*'         : ('cmsy10.ttf', 'A4'),
    r'\div'     : ('cmsy10.ttf', 'A5'),
    r'\diamond' : ('cmsy10.ttf', 'A6'),
    r'\pm'      : ('cmsy10.ttf', 'A7'),
    r'\mp'      : ('cmsy10.ttf', 'A8'),
    r'\oplus'    : ('cmsy10.ttf', 'A9'),
    r'\ominus'   : ('cmsy10.ttf', 'AA'),
    r'\otimes'   : ('cmsy10.ttf', 'AD'),
    r'\oslash'   : ('cmsy10.ttf', 'AE'),
    r'\odot'     : ('cmsy10.ttf', 'AF'),

    r'\bigcirc'  : ('cmsy10.ttf', 'B0'),
    r'\circ'     : ('cmsy10.ttf', 'B1'),
    r'\bullet'   : ('cmsy10.ttf', 'B2'),
    r'\asymp'    : ('cmsy10.ttf', 'B3'),
    r'\equiv'    : ('cmsy10.ttf', 'B4'),
    r'\subseteq' : ('cmsy10.ttf', 'B5'),
    r'\supseteq' : ('cmsy10.ttf', 'B6'),
    r'\leq'      : ('cmsy10.ttf', 'B7'),
    r'\geq'      : ('cmsy10.ttf', 'B8'),
    r'\preceq'   : ('cmsy10.ttf', 'B9'),
    r'\succeq'   : ('cmsy10.ttf', 'BA'),
    r'\sim'      : ('cmsy10.ttf', 'BB'),
    r'\approx'   : ('cmsy10.ttf', 'BC'),
    r'\subset'   : ('cmsy10.ttf', 'BD'),
    r'\supset'   : ('cmsy10.ttf', 'BE'),
    r'\ll'       : ('cmsy10.ttf', 'BF'),

    r'\gg'         : ('cmsy10.ttf', 'C0'),
    r'\prec'       : ('cmsy10.ttf', 'C1'),
    r'\succ'       : ('cmsy10.ttf', 'C2'),
    r'\rightarrow' : ('cmsy10.ttf', 'C3'),
    r'\spadesuit'  : ('cmsy10.ttf', 'C4'),


    # Math italic font
    r'\Gamma'   : ('cmmi10.ttf', 'A1'),
    r'\Delta'   : ('cmmi10.ttf', 'A2'),
    r'\Theta'   : ('cmmi10.ttf', 'A3'),
    r'\Lambda'  : ('cmmi10.ttf', 'A4'),
    r'\Xi'      : ('cmmi10.ttf', 'A5'),
    r'\Pi'      : ('cmmi10.ttf', 'A6'),
    r'\Sigma'   : ('cmmi10.ttf', 'A7'),
    r'\Upsilon' : ('cmmi10.ttf', 'A8'),
    r'\Phi'     : ('cmmi10.ttf', 'A9'),
    r'\Psi'     : ('cmmi10.ttf', 'AA'),
    r'\Omega'   : ('cmmi10.ttf', 'AD'),
    r'\alpha'   : ('cmmi10.ttf', 'AE'),
    r'\beta'    : ('cmmi10.ttf', 'AF'),

    r'\gamma'   : ('cmmi10.ttf', 'B0'),
    r'\delta'   : ('cmmi10.ttf', 'B1'),
    r'\epsilon' : ('cmmi10.ttf', 'B2'),
    r'\zeta'    : ('cmmi10.ttf', 'B3'),
    r'\eta'     : ('cmmi10.ttf', 'B4'),
    r'\theta'   : ('cmmi10.ttf', 'B5'),
    r'\iota'    : ('cmmi10.ttf', 'B6'),
    r'\kappa'   : ('cmmi10.ttf', 'B7'),
    r'\lambda'  : ('cmmi10.ttf', 'B8'),
    r'\mu'      : ('cmmi10.ttf', 'B9'),
    r'\nu'      : ('cmmi10.ttf', 'BA'),
    r'\xi'      : ('cmmi10.ttf', 'BB'),
    r'\pi'      : ('cmmi10.ttf', 'BC'),
    r'\rho'     : ('cmmi10.ttf', 'BD'),
    r'\sigma'   : ('cmmi10.ttf', 'BE'),
    r'\tau'     : ('cmmi10.ttf', 'BF'),

    r'\upsilon'    : ('cmmi10.ttf', 'C0'),
    r'\phi'        : ('cmmi10.ttf', 'C1'),
    r'\chi'        : ('cmmi10.ttf', 'C2'),
    r'\psi'        : ('cmmi10.ttf', 'C3'),


    r'\omega'      : ('cmmi10.ttf', '21'),
    r'\varepsilon' : ('cmmi10.ttf', '22'),
    r'\vartheta'   : ('cmmi10.ttf', '23'),
    r'\varphi'     : ('cmmi10.ttf', '24'),
    r'\varrho'     : ('cmmi10.ttf', '25'),
    r'\varsigma'   : ('cmmi10.ttf', '26'),
    r'\varphi'     : ('cmmi10.ttf', '27'),
    r'\leftharpoonup'    : ('cmmi10.ttf', '28'),
    r'\leftharpoondown'  : ('cmmi10.ttf', '29'),
    r'\rightharpoonup'   : ('cmmi10.ttf', '2A'),
    r'\rightharpoondown' : ('cmmi10.ttf', '2B'),
    #''                  : ('cmmi10.ttf', '2C'),
    #''                  : ('cmmi10.ttf', '2D'),
    r'\triangleright'    : ('cmmi10.ttf', '2E'),
    r'\triangleleft'     : ('cmmi10.ttf', '2F'),


    r'\flat'    : ('cmmi10.ttf', '5B'),
    r'\natural' : ('cmmi10.ttf', '5C'),
    r'\sharp'   : ('cmmi10.ttf', '5D'),
    r'\smile'   : ('cmmi10.ttf', '5E'),
    r'\frown'   : ('cmmi10.ttf', '5F'),

    r'\ell'       : ('cmmi10.ttf', '60'),
    r'\imath'     : ('cmmi10.ttf', '7B'),
    r'\jmath'     : ('cmmi10.ttf', '7C'),
    r'\wp'        : ('cmmi10.ttf', '7D'),

    # Math extension font

    # To make parsing easier, I'm going to make up names for left(,
    # left[, etc...  LEFT is bigger than Left is bigger than left.
    # We can add more from this table as needed

    #'('             : ('cmex10.ttf', 'A1'),
    #r'\leftparen'   : ('cmex10.ttf', 'A1'),
    #')'             : ('cmex10.ttf', 'A2'),
    #r'\rightparen'  : ('cmex10.ttf', 'A2'),

    #'['              : ('cmex10.ttf', 'A3'),
    #r'\leftbracket'  : ('cmex10.ttf', 'A3'),
    #']'              : ('cmex10.ttf', 'A4'),
    #r'\rightbracket' : ('cmex10.ttf', 'A4'),    

    '('             : ('cmr10.ttf', '28'),
    r'\leftparen'   : ('cmr10.ttf', '28'),
    ')'             : ('cmr10.ttf', '29'),
    r'\rightparen'  : ('cmr10.ttf', '29'),

    '['              : ('cmr10.ttf', '5B'),
    r'\leftbracket'  : ('cmr10.ttf', '5B'),
    ']'              : ('cmr10.ttf', '5D'),
    r'\rightbracket' : ('cmr10.ttf', '5D'),    

    '{'              : ('cmex10.ttf', 'A9'),
    r'\leftbrace'    : ('cmex10.ttf', 'A9'),
    '}'              : ('cmex10.ttf', 'AA'),    
    r'\rightbrace'   : ('cmex10.ttf', 'AA'),

    r'\leftangle'    : ('cmex10.ttf', 'AD'),
    r'\rightangle'   : ('cmex10.ttf', 'AE'),

    r'\Leftparen'    : ('cmex10.ttf', 'B3'),
    r'\Rightparen'   : ('cmex10.ttf', 'B4'),

    r'\LEFTparen'    : ('cmex10.ttf', 'B5'),
    r'\RIGHTparen'   : ('cmex10.ttf', 'B6'),
    r'\LEFTbracket'  : ('cmex10.ttf', 'B7'),
    r'\RIGHTbracket' : ('cmex10.ttf', 'B8'),
    r'\LEFTbrace'    : ('cmex10.ttf', 'BD'), # 
    r'\RIGHTbrace'   : ('cmex10.ttf', 'BE'),
    r'\LEFTangle'    : ('cmex10.ttf', 'BF'),
    r'\RIGHTangle'   : ('cmex10.ttf', 'C0 '),

    r'\oint'      : ('cmex10.ttf', '49'),
    r'\bigodot'   : ('cmex10.ttf', '4B'),
    r'\bigoplus'  : ('cmex10.ttf', '4D'),
    r'\bigotimes' : ('cmex10.ttf', '4F'),

    #todo , multiple sigms
    r'\sum'      : ('cmex10.ttf', '58'),
    r'\prod'     : ('cmex10.ttf', '59'),
    r'\int'      : ('cmex10.ttf', '5A'),
    r'\bigcup'   : ('cmex10.ttf', '5B'),
    r'\bigcap'   : ('cmex10.ttf', '5C'),
    r'\biguplus' : ('cmex10.ttf', '5D'),
    r'\bigwedge' : ('cmex10.ttf', '5E'),
    r'\bigvee'   : ('cmex10.ttf', '5F'),

    r'\coprod'       : ('cmex10.ttf', '61'),
    r'\Leftbracket'  : ('cmex10.ttf', '68'),
    r'\Rightbracket' : ('cmex10.ttf', '69'),    
    r'\Leftbrace'    : ('cmex10.ttf', '6E'),    
    r'\Rightbrace'   : ('cmex10.ttf', '6F'),

    # making up some names for big sqrt symbols
    r'\sqrt'  : ('cmex10.ttf', '70'),
    r'\Sqrt'  : ('cmex10.ttf', '71'),
    r'\SQRT'  : ('cmex10.ttf', '72'),
    }



class MathText:

    fontfile = 'cmmi10.ttf'
    size = 12
    ox = 0
    oy = 0
    nx = 0
    ny = 0
    dpi = 200

            
        
    def set_origin(self, ox, oy):
        'Set the pen location where drawing should begin'
        self.ox = ox
        self.oy = oy

    def set_next(self, nx, ny):
        """
        Set the pen location where drawing should begin for the
        following MathText
        """
        self.nx = nx
        self.ny = ny


    def get_bbox(self):
        bboxes = [c.get_bbox() for c in self.get_children()]
        return bound2d_all(bboxes)
            
    def get_children(self):
        raise NotImplementedError('Derived must override')

    def set_offsets(self, previous):
        """
        Set the offsets assuming previous is the symbol before
        you. return a new previous
        """
        nx, ny = previous.nx, previous.ny
        self.ox = nx
        self.oy = ny

        for child in self.get_children():
            previous = child.set_offsets(previous)
            
        left, top, width, height = self.get_bbox().get_bounds()
        self.nx = self.ox+width
        self.ny = self.oy
        return previous

    def draw(self, baseline):
        for child in self.get_children():
            child.draw(baseline)

    def load_glyphs(self):
        for child in self.get_children():
            child.load_glyphs()

    def set_size(self, size):
        for child in self.get_children():
            child.set_size(size)

    def set_fontfile(self, fontfile):
        for child in self.get_children():
            child.set_fontfile(fontfile)
        
class Expr(MathText):
    def __init__(self, objs):
        self.objs = objs
    def __repr__(self):
        return "Expr\n\t%s"%'\n\t'.join(['%s'%obj for obj in self.objs])

    def get_children(self):
        return self.objs

    
class Single(MathText):
    def __init__(self, c):
        self.c = c
        if fonttable.has_key(self.c):    
            self.fontfile, num  = fonttable[self.c]
            num = int(num, 16)
        elif len(self.c)==1:
            num = ord(self.c)
        else:
            raise ValueError('unrecognized symbol "%s"' % self.c)

        self.num = num
        self.glyph = None

    
    def get_alignment(self):
        'return the descent below baseline'

        if self.fontfile=='cmex10.ttf':
            d =   self.glyph.height/64.0/2 + OFFSET_EX10
        else:
            xmin, ymin, xmax, ymax = [val/64.0 for val in self.glyph.bbox]
            d =  ymax
        return abs(d)
    
    def load_glyphs(self):
        self.fonts[self.fontfile].set_size(self.size, self.dpi) 
        self.glyph = self.fonts[self.fontfile].load_char(self.num) 


    def get_bbox(self):
        xmin, ymin, xmax, ymax = [val/64.0 for val in self.glyph.bbox]
        width = xmax-xmin
        height = ymax-ymin
        return Bound2D(self.ox, self.oy, width, height)

    def get_children(self):
        return [self]
    
    def __repr__(self):
        return "Single '%s'"%self.c

    def set_offsets(self, previous):
        xmin, ymin, xmax, ymax = [val/64.0 for val in self.glyph.bbox]


        self.ox = previous.nx + xmin
        self.oy = previous.ny - self.get_alignment()

        self.nx = self.ox + self.glyph.horiAdvance/64.0
        self.ny = previous.ny
        return self
        
    def draw(self, baseline):
        self.fonts[self.fontfile].draw_glyph_to_bitmap(
            int(self.ox), int(baseline + self.oy), self.glyph)

    def set_size(self, size):
        self.size = size

    def set_fontfile(self, fontfile):
        if self.glyph is not None:
            raise RuntimeError("You cannot set font after glyph loaded")
        self.fontfile = fontfile
        
    
class Grp(MathText):
    def __init__(self, members):
        self.members = members

    def __repr__(self):
        return "Group [%s]"%', '.join(['%s'%x for x in self.members])

    def get_children(self):
        return self.members


class Font(MathText):
    symmap = { r'\cal' : 'cmsy10.ttf',
               r'\rm'  : 'cmr10.ttf',
               r'\tt'  : 'cmtt10.ttf',
               r'\it'  : 'cmmi10.ttf',
               }
                          
    def __init__(self, name, grp):
        self.name = name
        self.grp = grp
        self.fontfile = self.symmap[name]
        for child in self.get_children():
            child.set_fontfile(self.fontfile)

    def __repr__(self):
        return "Font %s %s" % (self.name, self.grp)

    def get_children(self):
        return self.grp.get_children()


class SubSup(MathText):
    SUBSCALE = 0.7
    SUBOFFSET = 0.5
    SUPOFFSET = 0.75
    
    def __init__(self, type, obj):
        self.obj = obj   
        self.type = type
        self.set_size(self.size)
        
    #def __repr__(self):
    #    return "SubSup %s "% ' '.join(['%s'%obj for obj in self.obj])

    def __repr__(self):
        return "SubSup %s %s" % (self.type, self.obj)

    def get_children(self):
        #return self.obj.get_children()
        #print 'gc', self.obj, type(self.obj)
        #return [o for o in self.obj]
        return [self.obj]


    def set_size(self, size):

        self.size = size
        for child in self.get_children():
            child.set_size(size*self.SUBSCALE)


    def set_offsets(self, previous):

        # set offset of children with dummy instance to get width
        # height of children
        class Previous: nx, ny = 0, 0
        tmp = Previous()
        for child in self.get_children():
            tmpprev = child.set_offsets(tmp)

        # ou is True for Over/Under layout 
        ou = (isinstance(previous, Single) and
              overunder.has_key(previous.c))


        if ou:
            # estimate self width, height for centering
            cumw = 0
            height = 0
            left, top, width, height = self.get_bbox().get_bounds()

            l,t,w,h = previous.get_bbox().get_bounds()
            bottomPrev = t + h

            
            
            midx = l+w/2.0
            pad = 0.05*h

            self.ox = midx-width/2.0  # centered on x

            if self.type == '_':
                self.oy = bottomPrev + pad + (top-t) # this is wrong; implement ascender
            else:
                self.oy = t-pad

            self.nx = self.ox
            self.ny = self.oy
        else:
            l,t,w,h = previous.get_bbox().get_bounds()
            if self.type == '_': deltay = self.SUBOFFSET*h
            else: deltay = -self.SUPOFFSET*h

            # move the sub/supscript back a little from a normal
            # offset
            deltax = -0.1*(previous.nx-previous.ox)


            self.ox = previous.nx
            self.oy = previous.ny

            # set the temporarily for setting child offsets in tmpprev
            # below and then restore
            self.nx = previous.nx + deltax
            self.ny = previous.ny + deltay


        tmp = self
        for child in self.get_children():
            tmp = child.set_offsets(tmp)
            
        if ou: return previous
        else:
            # now restore the proper offsets
            self.nx = child.nx
            self.ny = child.ny - deltay
            return self

class Handler:
    
    def expr(self, s, loc, toks):
        self.expr = Expr(toks)
        return loc, [self.expr]

    def singlesym(self, s, loc, toks):
        assert(len(toks)==1)
        #print 'singlesym', toks
        single = Single(toks[0])
        return loc, [single]
    
    def grp(self, s, loc, toks):
        assert(len(toks)==1)
        #print 'grp', toks
        grp = Grp(toks[0])
        return loc, [grp]

    def fontgrp(self, s, loc, toks):
        assert(len(toks)==1)
        name, grp = toks[0]
        #print 'fontgrp', toks
        font = Font(name, grp)
        return loc, [font]

    def subsupgrp(self, s, loc, toks):
        assert(len(toks)==1)
        #print 'subsup', toks
        updown = toks[0][0]
        objs = toks[0][1]
        subsup = SubSup(updown, objs)
        return loc, [SubSup(updown, objs)]


handler = Handler()

lbrace = Literal('{').suppress()
rbrace = Literal('}').suppress()
lbrack = Literal('[')
rbrack = Literal(']')
lparen = Literal('(')
rparen = Literal(')')
grouping = lbrack | rbrack | lparen | rparen

bslash = Literal('\\')


langle = Literal('<')
rangle = Literal('>')
equals = Literal('=')
relation = langle | rangle | equals

colon =  Literal(':')
comma =  Literal(',')
period =  Literal('.')
semicolon =  Literal(';')
exclamation =  Literal('!')

punctuation = colon | comma | period | semicolon 

at =  Literal('@')
percent =  Literal('%')
ampersand =  Literal('&')

misc = exclamation | at | percent | ampersand


plus = Literal('+')
minus = Literal('-')
times = Literal('*')
div = Literal('/')
binop = plus | minus | times | div

underscore = Literal('_')
superscript = Literal('^')


roman = Literal('rm')
cal   = Literal('cal')
italics = Literal('it')
typewriter = Literal('tt')
fontname = roman | cal | italics | typewriter

texsym = Combine(bslash + Word(alphanums) + NotAny(lbrace))

char = Word(alphanums + ' ', exact=1)

singlesym = (texsym ^ char ^ binop ^ relation ^ punctuation ^ misc ^ grouping ).setParseAction(handler.singlesym).leaveWhitespace()

subsupgrp = Forward().setParseAction(handler.subsupgrp)
symbol    = singlesym ^ subsupgrp 
grp       = Group( lbrace +  OneOrMore(symbol) + rbrace).setParseAction(handler.grp)
fontgrp   = Group( Combine(bslash + fontname) + grp).setParseAction(handler.fontgrp)
#subsupgrp << Group( (underscore ^ superscript) + (symbol ^ grp ^ fontgrp ) )
subsupgrp << Group( (underscore ^ superscript) + (symbol ^ grp ^ fontgrp ) + Optional(subsupgrp) )

group  = grp ^ fontgrp.leaveWhitespace() ^ subsupgrp
#group  = grp ^ fontgrp.leaveWhitespace() ^ OneOrMore(subsupgrp)
expr = OneOrMore(symbol ^ group).setParseAction(handler.expr)

def math_parse_s(s, dpi, fontsize):
    """
    Parse the math expression s, return the (bbox, fonts) tuple needed
    to render it.

    fontsize must be in points

    bbox: left, bottom, width, heigt
    fonts: a seq of FT2Fonts
    """
    cacheKey = (s, dpi)
    s = s[1:-1]  # strip the $ from front and back
    if math_parse_s.cache.has_key(cacheKey):
        return math_parse_s.cache[cacheKey]
    fnames = ('cmmi10.ttf', 'cmsy10.ttf', 'cmex10.ttf',
              'cmtt10.ttf', 'cmr10.ttf')
    # allocate a new set of fonts
    basepath = get_data_path()
    MathText.fonts = dict([ (name, FT2Font(os.path.join(basepath, name)))
                            for name in fnames])

    MathText.dpi = dpi
    tokens = expr.parseString( s )
    handler.expr.set_size(fontsize)
    handler.expr.load_glyphs()
    #for  o in handler.expr.objs:
    #    print o
    #print expr

    class Previous:
        nx = 0
        ny = 0

    previous = Previous()
    handler.expr.set_offsets(previous)


    l,t,w,h  = handler.expr.get_bbox().get_bounds()
    #h += OFFSET_EX10
    oys = [o.oy for o in handler.expr.get_children()]
    maxoy = abs(max(oys))

    padx = 5
    pady = int(max(absolute(oys)))
    pady = 60
    #pady = 5

    for name, font in MathText.fonts.items():
        font.set_bitmap_size(int(w)+padx, int(h)+pady) # todo: hack alert

    bounds = l, t+h, w, h
    handler.expr.draw(h)


    ret = bounds, MathText.fonts.values()
    math_parse_s.cache[cacheKey] = ret
    return ret
math_parse_s.cache = {}
