"""
This is a matlab style functional interface the matplotlib.

The following matlab compatible commands are provided

Plotting commands

  axes     - Create a new axes
  axis     - Set or return the current axis limits
  bar      - make a bar chart
  cla      - clear current axes
  clf      - clear a figure window
  close    - close a figure window
  cohere   - make a plot of coherence
  csd      - make a plot of cross spectral density
  errorbar - make an errorbar graph
  figure   - create or change active figure
  fill     - make filled polygons
  gca      - return the current axes
  gcf      - return the current figure
  get      - get a handle graphics property
  hist     - make a histogram
  loglog   - a log log plot
  imshow   - plot image data
  pcolor   - make a pseudocolor plot
  plot     - make a line plot
  psd      - make a plot of power spectral density
  savefig  - save the current figure
  scatter  - make a scatter plot
  set      - set a handle graphics property
  semilogx - log x axis
  semilogy - log y axis
  show     - show the figures
  specgram - a spectrogram plot
  subplot  - make a subplot (numrows, numcols, axesnum)
  text     - add some text at location x,y to the current axes
  title    - add a title to the current axes
  xlabel   - add an xlabel to the current axes
  ylabel   - add a ylabel to the current axes

Matrix commands

  cumprod   - the cumulative product along a dimension
  cumsum    - the cumulative sum along a dimension
  detrend   - remove the mean or besdt fit line from an array
  diag      - the k-th diagonal of matrix 
  diff      - the n-th differnce of an array
  eig       - the eigenvalues and eigen vectors of v
  eye       - a matrix where the k-th diagonal is ones, else zero 
  find      - return the indices where a condition is nonzero  
  fliplr    - flip the rows of a matrix up/down
  flipud    - flip the columns of a matrix left/right
  linspace  - a linear spaced vector of N values from min to max inclusive
  meshgrid  - repeat x and y to make regular matrices
  ones      - an array of ones
  rand      - an array from the uniform distribution [0,1]
  randn     - an array from the normal distribution
  rot90     - rotate matrix k*90 degress counterclockwise
  squeeze   - squeeze an array removing any dimensions of length 1
  tri       - a triangular matrix
  tril      - a lower triangular matrix
  triu      - an upper triangular matrix
  vander    - the Vandermonde matrix of vector x
  svd       - singular value decomposition
  zeros     - a matrix of zeros
  
Probability

  levypdf   - The levy probability density function from the char. func.
  normpdf   - The Gaussian probability density function
  rand      - random numbers from the uniform distribution
  randn     - random numbers from the normal distribution

Statistics

  corrcoef  - correlation coefficient
  cov       - covariance matrix
  max       - the maximum along dimension m
  mean      - the mean along dimension m
  median    - the median along dimension m
  min       - the minimum along dimension m
  norm      - the norm of vector x
  prod      - the product along dimension m
  ptp       - the max-min along dimension m
  std       - the standard deviation along dimension m
  sum       - the sum along dimension m

Time series analysis

  bartlett  - M-point Bartlett window
  blackman  - M-point Blackman window
  cohere    - the coherence using average periodiogram
  csd       - the cross spectral density using average periodiogram
  fft       - the fast Fourier transform of vector x
  hamming   - M-point Hamming window
  hanning   - M-point Hanning window
  hist      - compute the histogram of x
  kaiser    - M length Kaiser window
  psd       - the power spectral density using average periodiogram
  sinc      - the sinc function of array x

Other

  angle     - the angle of a complex array
  load     - load ASCII data into array
  polyfit   - fit x, y to an n-th order polynomial
  polyval   - evaluate an n-th order polynomial
  roots     - the roots of the polynomial coefficients in p
  save      - save an array to an ASCII file
  trapz     - trapezoidal integration


Credits: The plotting commands were provided by
John D. Hunter <jdhunter@ace.bsd.uhicago.edu>

Most of the other commands are from the Numeric, MLab and FFT, with
the exception of those in mlab.py provided by matplotlib.
"""

# bring all the MLab and mlab symbols in so folks can import them from
# matplotlib.matlab in one fell swoop
from numerix import *
from mlab import *

import _matlab_helpers
import mlab  #so I can override hist, psd, etc...

from axes import Axes
from backends import new_figure_manager, error_msg, \
     draw_if_interactive, show
from cbook import flatten, is_string_like, exception_to_str
from colors import ColormapJet
from figure import Figure
from matplotlib import rcParams
def colors():
    """
    This is a do nothing function to provide you with help on how
    matplotlib handles colors.

    Commands which take color arguments can use several formats to
    specify the colors.  For the basic builtin colors, you can use a
    single letter

      b  : blue
      g  : green
      r  : red
      c  : cyan
      m  : magenta
      y  : yellow
      k  : black 
      w  : white


    For a greater range of colors, you have two options.  You can
    specify the color using an html hex string, as in

      color = '#eeefff'

    or you can pass an R,G,B tuple, where each of R,G,B are in the
    range [0,1].  The example below creates a subplot with a dark
    slate gray background

       subplot(111, axisbg=(0.1843, 0.3098, 0.3098))

    Here is an example that creates a pale turqoise title

      title('Is this the best color?', color='#afeeee')

    """
    pass

def plotting():
    """
    Plotting commands

    axes     - Create a new axes
    axis     - Set or return the current axis limits
    bar      - make a bar chart
    cla      - clear the current axes    
    clf      - clear the current figure window
    close    - close the current figure window
    cohere   - make a plot of coherence
    csd      - make a plot of cross spectral density
    errorbar - make an errorbar graph
    figure   - create or change active figure
    gca      - return the current axes
    gcf      - return the current figure
    get      - get a handle graphics property
    hist     - make a histogram
    loglog   - a log log plot
    pcolor   - make a pseudocolor plot
    plot     - make a line plot
    psd      - make a plot of power spectral density
    savefig  - save the current figure
    scatter  - make a scatter plot
    semilogx - log x axis
    semilogy - log y axis
    set      - set a handle graphics property
    show     - show the figures
    subplot  - make a subplot (numrows, numcols, axesnum)
    text     - add some text at location x,y to the current axes
    title    - add a title to the current axes
    xlabel   - add an xlabel to the current axes
    ylabel   - add a ylabel to the current axes
    """
    pass




def get_current_fig_manager():
    figManager = _matlab_helpers.Gcf.get_active()
    if figManager is None:
        gcf()  # creates an active figure as a side effect
        figManager = _matlab_helpers.Gcf.get_active()
    return figManager

def get_plot_commands(): return ('axes', 'axis', 'bar', 'cla', 'clf', 'close', 'cohere', 'csd', 'errorbar', 'figure', 'fill', 'gca', 'gcf', 'get', 'hist', 'imshow', 'loglog', 'pcolor', 'plot', 'psd', 'savefig', 'scatter', 'set', 'semilogx', 'semilogy', 'show', 'specgram', 'subplot', 'text', 'title', 'xlabel', 'ylabel')

def raise_msg_to_str(msg):
    """msg is a return arg from a raise.  Join with new lines"""
    if not is_string_like(msg):
        msg = '\n'.join(map(str, msg))
    return msg

#----- Now we get started with the matlab commands  ----#

def axis(*v):
    """
    axis() returns the current axis as a length a length 4 vector

    axis(v) where v= [xmin xmax ymin ymax] sets the min and max of the
    x and y axis limits

    axis('off') turns off the axis lines and labels
    """
    
    if len(v)==1 and is_string_like(v[0]):
        s = v[0]
        if s.lower()=='on': gca().set_axis_on()
        elif s.lower()=='off': gca().set_axis_off()
        else:
            error_msg('Unrecognized string %s to axis; try on or off' % s)
        return
    
    try: v[0]
    except IndexError:
        xlim = gca().get_xlim()
        ylim = gca().get_ylim()
        return [xlim[0], xlim[1], ylim[0], ylim[1]]
    
    v = v[0]
    if len(v) != 4:
        error_msg('v must contain [xmin xmax ymin ymax]')
        return 
    gca().set_xlim([v[0], v[1]])
    gca().set_ylim([v[2], v[3]])
    draw_if_interactive()
    
def axes(*args, **kwargs):
    """
    Add an axis at positon rect specified by

      axes() by itself creates a default full subplot(111) window axis

      axes(rect, axisbg='w') where rect=[left, bottom, width, height] in
        normalized (0,1) units background is the background color for
        the axis, default white

      axes(h) where h is an axes instance makes h the
        current axis An Axes instance is returned

    axisbg is a color format string which sets the background color of
    the axes

    If axisbg is a length 1 string, assume it's a color format string
    (see plot for legal color strings).  If it is a length 7 string,
    assume it's a hex color string, as used in html, eg, '#eeefff'.
    If it is a len(3) tuple, assume it's an rgb value where r,g,b in
    [0,1].
    
    """

    nargs = len(args)
    if len(args)==0: return subplot(111, **kwargs)
    if nargs>1:
        error_msg('Only one non keyword arg to axes allowed')
        return
    
    arg = args[0]

    if isinstance(arg, Axes):
        get_current_fig_manager().set_current_axes(arg)
        return arg
    else:
        rect = arg
        ret = get_current_fig_manager().add_axes(rect, **kwargs)
        return ret


def bar(*args, **kwargs):
    """
    BAR(left, height)

    Make a bar plot with rectangles at
      left, left+width, 0, height
    left and height are Numeric arrays

    Return value is a list of Rectangle patch instances

    BAR(left, height, width, bottom,
        color, yerr, xerr, capsize, yoff)

    xerr and yerr, if not None, will be used to generate errorbars
    on the bar chart

    color specifies the color of the bar

    capsize determines the length in points of the error bar caps


    The optional arguments color, width and bottom can be either
    scalars or len(x) sequences

    This enables you to use bar as the basis for stacked bar
    charts, or candlestick plots
    """
    
    try: patches =  gca().bar(*args, **kwargs)
    except Exception, msg:
        s = exception_to_str(msg)
        error_msg(s)
        raise RuntimeError(msg)
    draw_if_interactive()
    return patches

def close(*args):
    """
    Close a figure window

    close() by itself closes the current figure

    close(num) closes figure number num

    close(h) where h is a figure handle(instance) closes that figure

    close('all') closes all the figure windows
    """

    if len(args)==0:
        figManager = _matlab_helpers.Gcf.get_active()
        if figManager is None: return
        else: _matlab_helpers.Gcf.destroy(figManager.num)
    elif len(args)==1:
        arg = args[0]
        if arg=='all':
            for manager in _matlab_helpers.Gcf.get_all_fig_managers():
                _matlab_helpers.Gcf.destroy(manager.num)
        elif isinstance(arg, int):
            _matlab_helpers.Gcf.destroy(arg)
        elif isinstance(arg, Figure):
            for manager in _matlab_helpers.Gcf.get_all_fig_managers():
                if manager.figure==arg:
                    _matlab_helpers.Gcf.destroy(manager.num)
        else:
            msg = 'Unrecognized argument type to close'
            popup_err(msg)
    else:
        msg = 'close takes 0 or 1 arguments'
        popup_err(msg)


def cla():
    """
    Clear the current axes
    """
    gca().cla()
    draw_if_interactive()
    
def clf():
    """
    Clear the current figure
    """
    manager = get_current_fig_manager()
    manager.clf()
    draw_if_interactive()


def cohere(x, y, NFFT=256, Fs=2, detrend=mlab.detrend_none,
           window=mlab.window_hanning, noverlap=0):
    """
    Compute the coherence between x and y.  Coherence is the
    normalized cross spectral density

    Cxy = |Pxy|^2/(Pxx*Pyy)

    The return value is (Cxy, f), where f are the frequencies of the
    coherence vector.  See the docs for psd and csd for information
    about the function arguments NFFT, detrend, windowm noverlap, as
    well as the methods used to compute Pxy, Pxx and Pyy.

    Returns the tuple Cxy, freqs

    Refs:
      Bendat & Piersol -- Random Data: Analysis and Measurement
        Procedures, John Wiley & Sons (1986)

    """

    try: ret =  gca().cohere(x, y, NFFT, Fs, detrend, window, noverlap)
    except ValueError, msg:
        msg = raise_msg_to_str(msg)
        error_msg(msg)
    else:
        draw_if_interactive()
        return ret


def csd(x, y, NFFT=256, Fs=2, detrend=mlab.detrend_none,
        window=mlab.window_hanning, noverlap=0):
    """
    The cross spectral density Pxy by Welches average periodogram
    method.  The vectors x and y are divided into NFFT length
    segments.  Each segment is detrended by function detrend and
    windowed by function window.  noverlap gives the length of the
    overlap between segments.  The product of the direct FFTs of x and
    y are averaged over each segment to compute Pxy, with a scaling to
    correct for power loss due to windowing.  Fs is the sampling
    frequency.

    NFFT must be a power of 2

    detrend and window are functions, unlike in matlab where they are
    vectors.  For detrending you can use detrend_none, detrend_mean,
    detrend_linear or a custom function.  For windowing, you can use
    window_none, window_hanning, or a custom function

    Returns the tuple Pxy, freqs.  Pxy is the cross spectrum (complex
    valued), and 10*log10(|Pxy|) is plotted

    Refs:
      Bendat & Piersol -- Random Data: Analysis and Measurement
        Procedures, John Wiley & Sons (1986)

    """
    try: ret =  gca().csd(x, y, NFFT, Fs, detrend, window, noverlap)
    except ValueError, msg:
        msg = raise_msg_to_str(msg)
        error_msg(msg)
    else:
        draw_if_interactive()
        return ret

def errorbar(x, y, yerr=None, xerr=None, fmt='b-', capsize=3):
    """
    Plot x versus y with error deltas in yerr and xerr.
    Vertical errorbars are plotted if yerr is not None
    Horizontal errorbars are plotted if xerr is not None

    xerr and yerr may be any of:
        a rank-0, Nx1 Numpy array  - symmetric errorbars +/- value
        an N-element list or tuple - symmetric errorbars +/- value
        a rank-1, Nx2 Numpy array  - asymmetric errorbars -column1/+column2
    
    fmt is the plot format symbol for y

    Return value is a length 2 tuple.  The first element is a list of
    y symbol lines.  The second element is a list of error bar lines.

    capsize is the size of the error bar caps in points
    """
    try: ret =  gca().errorbar(x, y, yerr, xerr, fmt, capsize)
    except ValueError, msg:
        msg = raise_msg_to_str(msg)
        error_msg(msg)
    else:
        draw_if_interactive()
        return ret

def figlegend(handles, labels, loc):
    """
    Place a legend in the figure.  Labels are a sequence of
    strings, handles is a sequence of line or patch instances, and
    loc can be a string or an integer specifying the legend
    location

    USAGE: 
      legend( (line1, line2, line3),
              ('label1', 'label2', 'label3'),
              'upper right')

    See help(legend) for information about the location codes
    """
    l=  gcf().legend(handles, labels, loc)
    draw_if_interactive()
    return l
    

def figure(num=1,
           figsize   = rcParams['figure.figsize'],
           dpi       = rcParams['figure.dpi'],
           facecolor = rcParams['figure.facecolor'],
           edgecolor = rcParams['figure.edgecolor'],
           ):
    """
    figure(num = 1, figsize=(8, 6), dpi=80, facecolor='w', edgecolor='k')


    Create a new figure and return a handle to it

    If figure(num) already exists, make it active and return the
    handle to it.

      figure(1)

    figsize - width in height x inches
    dpi     - resolution
    facecolor - the background color    
    edgecolor - the border color

    rcParams gives the default values from the .matplotlibrc file

    """
    if num==0:
        error_msg('Figure number can not be 0.\n' + \
                  'Hey, give me a break, this is matlab compatability')

    figManager = _matlab_helpers.Gcf.get_fig_manager(num)
    if figManager is None:
        figManager = new_figure_manager(num, figsize, dpi, facecolor, edgecolor)
        _matlab_helpers.Gcf.set_active(figManager)
    
    return figManager.canvas.figure

def fill(*args, **kwargs):
    """
    plot filled polygons.  *args is a variable length argument,
    allowing for multiple x,y pairs with an optional color format
    string.  For example, all of the following are legal, assuming a
    is the Axis instance:

      fill(x,y)            # plot polygon with vertices at x,y
      fill(x,y, 'b' )      # plot polygon with vertices at x,y in blue

    An arbitrary number of x, y, color groups can be specified, as in 
      fill(x1, y1, 'g', x2, y2, 'r')  

    Return value is a list of patches that were added

    The following color strings are supported

      b  : blue
      g  : green
      r  : red
      c  : cyan
      m  : magenta
      y  : yellow
      k  : black 
      w  : white

    The kwargs that are can be used to set line properties (any
    property that has a set_* method).  You can use this to set edge
    color, face color, etc.

    Example code:

    from matplotlib.matlab import *
    t = arange(0.0, 1.01, 0.01)
    s = sin(2*2*pi*t)

    fill(t, s, 'r')
    grid(True)
    show()

    """
    try: patch =  gca().fill(*args, **kwargs)
    except Exception, msg:
        s = exception_to_str(msg)
        error_msg(s)
        raise RuntimeError('Could not exectute fill')
    draw_if_interactive()
    return patch

def gca():
    """
    Return the current axis instance.  This can be used to control
    axis properties either using set or the Axes methods.

    Example:

      plot(t,s)
      set(gca(), 'xlim', [0,10])  # set the x axis limits

    or

      plot(t,s)
      a = gca()
      a.set_xlim([0,10])          # does the same
    """

    return get_current_fig_manager().get_current_axis()
        
def gcf():
    "Return a handle to the current figure"
    
    figManager = _matlab_helpers.Gcf.get_active()
    if figManager is not None:
        return figManager.canvas.figure
    else:
        return figure()

def get(o, s):
    """
    Return the value of handle property s

    h is an instance of a class, eg a Line2D or an Axes or Text.
    if s is 'somename', this function returns

      o.get_somename()
    
    """
    func = 'o.get_%s()' % s
    return eval(func, {}, {'o': o})
    

def grid(b):
    """
    Set the figure grid to be on or off (b is a boolean)
    """
    gca().grid(b)
    draw_if_interactive()
    
def hist(x, bins=10, noplot=0, normed=0):
    """
    Compute the histogram of x.  bins is either an integer number of
    bins or a sequence giving the bins.  x are the data to be binned.

    if noplot is True, just compute the histogram and return the
    number of observations and the bins as an (n, bins) tuple.

    If noplot is False, compute the histogram and plot it, returning
    n, bins, patches

    If normed is true, the first element of the return tuple will be the
    counts normalized to form a probability distribtion, ie,
    n/(len(x)*dbin)
    
    """
    if noplot: return mlab.hist(x, bins, normed)
    else:
        try:
            ret = gca().hist(x, bins, normed)
        except ValueError, msg:
            msg = raise_msg_to_str(msg)
            error_msg(msg)
            raise RuntimeError, msg
    draw_if_interactive()
    return ret

    

def hlines(*args, **kwargs):    
    """
    lines = hlines(self, y, xmin, xmax, fmt='k-')

    plot horizontal lines at each y from xmin to xmax.  xmin or
    xmax can be scalars or len(x) numpy arrays.  If they are
    scalars, then the respective values are constant, else the
    widths of the lines are determined by xmin and xmax

    Returns a list of line instances that were added

    """
    try: lines =  gca().hlines(*args, **kwargs)
    except ValueError, msg:
        msg = raise_msg_to_str(msg)
        error_msg(msg)
        raise RuntimeError, msg
    draw_if_interactive()
    return lines


def imshow(*args, **kwargs):
    """
    Display the image in array X to current axes.  X must be a
    float array

    Usage: 

     IMSHOW(X)
    
     If X is MxN, assume luminance (grayscale)
     If X is MxNx3, assume RGB
     If X is MxNx4, assume RGBA

     IMSHOW(X, cmap)
     cmap is a colors.Colormap instance used to make a pseudo-color plot
    
    An Image instance is returned
    """
    try: im =  gca().imshow(*args, **kwargs)
    except ValueError, msg:
        msg = raise_msg_to_str(msg)
        error_msg(msg)
        raise RuntimeError, msg
    except RuntimeError, msg:
        msg = raise_msg_to_str(msg)
        error_msg(msg)
        raise RuntimeError, msg
    draw_if_interactive()
    return im


def legend(*args, **kwargs):
    """
    Place a legend on the current axes at location loc.  Labels are a
    sequence of strings and loc can be a string or an integer
    specifying the legend location

    USAGE: 

      Make a legend with existing lines
      legend( LABELS )
      >>> legend( ('label1', 'label2', 'label3') ) 

      Make a legend for Line2D instances lines1, line2, line3
      legend( LINES, LABELS )
      >>> legend( (line1, line2, line3), ('label1', 'label2', 'label3') )

      Make a legend at LOC
      legend( LABELS, LOC )  or
      legend( LINES, LABELS, LOC )
      >>> legend( ('label1', 'label2', 'label3'), loc='upper left')
      >>> legend( (line1, line2, line3),
                  ('label1', 'label2', 'label3'),
                  loc=2)

    The LOC location codes are

    The LOC location codes are

      'best' : 0,          (currently not supported, defaults to upper right)
      'upper right'  : 1,  (default)
      'upper left'   : 2,
      'lower left'   : 3,
      'lower right'  : 4,
      'right'        : 5,
      'center left'  : 6,
      'center right' : 7,
      'lower center' : 8,
      'upper center' : 9,
      'center'       : 10,

    If none of these are suitable, loc can be a 2-tuple giving x,y
    in axes coords, ie,

      loc = 0, 1 is left top
      loc = 0.5, 0.5 is center, center

      and so on


    The legend instance is returned
    """

    return gca().legend(*args, **kwargs)

def load(fname):
    """
    Load ASCII data from fname into an array and return the array.

    The data must be regular, same number of values in every row

    fname can be a filename or a file handle

    matfile data is not currently supported, but see
    Nigel Wade's matfile ftp://ion.le.ac.uk/matfile/matfile.tar.gz

    Example usage:

    x,y = load('test.dat')  # data in two columns

    X = load('test.dat')    # a matrix of data

    x = load('test.dat')    # a single column of data

    """

    if is_string_like(fname):
        fh = file(fname)
    elif hasattr(fname, seek):
        fh = fname
    else:
        raise ValueError('fname must be a string or file handle')
    
    X = []
    numCols = None
    for line in fh:
        line = line.strip()
        if not len(line): continue
        row = [float(val) for val in line.split()]
        thisLen = len(row)
        if numCols is not None and thisLen != numCols:
            raise ValueError('All rows must have the same number of columns')
        X.append(row)

    X = array(X)
    r,c = X.shape
    if r==1 or c==1:
        X.shape = max([r,c]),
    return X


    
            

def loglog(*args, **kwargs):
    """
    Make a loglog plot with log scaling on the a and y axis.  The args
    to semilog x are the same as the args to plot.  See help plot for
    more info
    """
    try: ret =  gca().loglog(*args, **kwargs)
    except ValueError, msg:
        msg = raise_msg_to_str(msg)
        error_msg(msg)
    else:
        draw_if_interactive()
        return ret


def pcolor(*args, **kwargs):
    """
    
    pcolor(C) - make a pseudocolor plot of matrix C using
    rectangle patches

    pcolor(C, cmap=colormapInstance) - make a pseudocolor plot of
    matrix C using rectangle patches using a custom colormap derived
    from matplotlib.colors.Colormap.  You must pass this as a kwarg.

    This function is currently very slow.  Consider using imshow with
    a colormap in it's place
    
    pcolor(X, Y, C) - a pseudo color plot of C on the matrices X and Y  

    Shading:

      The optional keyword arg shading ('flat' or 'faceted') will
      determine whether the black grid is drawn around each pcolor
      square.  Defaul 'faceteted'
         e.g.,   
         pcolor(C, shading='flat')  
         pcolor(X, Y, C, shading='faceted')

    returns a list of patch objects.

    Note, the behavior of meshgrid in matlab is a bit
    counterintuitive for x and y arrays.  For example,

      x = arange(7)
      y = arange(5)
      X, Y = meshgrid(x,y)

      Z = rand( len(x), len(y))
      pcolor(X, Y, Z)

    will fail in matlab and matplotlib.  You will probably be
    happy with

      pcolor(X, Y, transpose(Z))

    Likewise, for nonsquare Z,

      pcolor(transpose(Z))

    will make the x and y axes in the plot agree with the numrows
    and numcols of Z

    """
    return gca().pcolor(*args, **kwargs)

def pcolor(*args, **kwargs):
    """
    pcolor_patch(C) - make a pseudocolor plot of matrix C

    pcolor(X, Y, C) - a pseudo color plot of C on the matrices X and Y  

    Shading:

      The optional keyword arg shading ('flat' or 'faceted') will
      determine whether the black grid is drawn around each pcolor
      square.  Defaul 'faceteted'
         e.g.,   
         pcolor(C, shading='flat')  
         pcolor(X, Y, C, shading='faceted')

    returns a list of patch objects.

    Note, the behavior of meshgrid in matlab is a bit
    counterintuitive for x and y arrays.  For example,

      x = arange(7)
      y = arange(5)
      X, Y = meshgrid(x,y)

      Z = rand( len(x), len(y))
      pcolor(X, Y, Z)

    will fail in matlab and matplotlib.  You will probably be
    happy with

      pcolor(X, Y, transpose(Z))

    Likewise, for nonsquare Z,

      pcolor(transpose(Z))

    will make the x and y axes in the plot agree with the numrows
    and numcols of Z

    """
    return gca().pcolor(*args, **kwargs)


def plot(*args, **kwargs):
    """
    plot lines.  *args is a variable length argument, allowing for
    multiple x, y pairs with an optional format string.  For
    example, all of the following are legal
        
      plot(x,y)            # plot Numeric arrays y vs x
      plot(x,y, 'bo')      # plot Numeric arrays y vs x with blue circles
      plot(y)              # plot y using x = arange(len(y))
      plot(y, 'r+')        # ditto with red plusses

    An arbitrary number of x, y, fmt groups can be specified, as in 

      plot(x1, y1, 'g^', x2, y2, 'l-')  

    Return value is a list of lines that were added

    The following line styles are supported:

      -  : solid line
      -- : dashed line
      -. : dash-dot line
      :  : dotted line
      .  : points
      ,  : pixels
      o  : circle symbols
      ^  : triangle up symbols
      v  : triangle down symbols
      <  : triangle left symbols
      >  : triangle right symbols
      s  : square symbols
      +  : plus symbols

    The following color strings are supported

      b  : blue
      g  : green
      r  : red
      c  : cyan
      m  : magenta
      y  : yellow
      k  : black 
      w  : white

    Line styles and colors are combined in a single format string

    The kwargs that are can be used to set line properties (any
    property that has a set_* method).  You can use this to set a line
    label (for auto legends), linewidth, anitialising, marker face
    color, etc.  Here is an example:

      plot([1,2,3], [1,2,3], 'go-', label='line 1', linewidth=2)
      plot([1,2,3], [1,4,9], 'rs',  label='line 2')
      axis([0, 4, 0, 10])
      legend()

    If you make multiple lines with one plot command, the kwargs apply
    to all those lines, eg

      plot(x1, y1, x2, y2, antialising=False)

    Neither line will be antialiased.
    """
    
    try: lines =  gca().plot(*args, **kwargs)
    except ValueError, msg:
        msg = raise_msg_to_str(msg)
        error_msg(msg)
    else:
        draw_if_interactive()
        return lines



def plot_date(*args, **kwargs):
    """
    PLOT_DATE(d, y, converter, fmt='bo', **kwargs)

    d is a sequence of dates; converter is a dates.DateConverter
    instance that converts your dates to seconds since the epoch for
    plotting.  y are the y values at those dates.  fmt is a plot
    format string.  kwargs are passed on to plot.  See plot for more
    information.
    """
    
    try: lines =  gca().plot_date(*args, **kwargs)
    except ValueError, msg:
        msg = raise_msg_to_str(msg)
        error_msg(msg)
    else:
        draw_if_interactive()
        return lines


def psd(x, NFFT=256, Fs=2, detrend=mlab.detrend_none,
        window=mlab.window_hanning, noverlap=0):
    """
    The power spectral density by Welches average periodogram method.
    The vector x is divided into NFFT length segments.  Each segment
    is detrended by function detrend and windowed by function window.
    noperlap gives the length of the overlap between segments.  The
    absolute(fft(segment))**2 of each segment are averaged to compute Pxx,
    with a scaling to correct for power loss due to windowing.  Fs is
    the sampling frequency.

    -- NFFT must be a power of 2

    -- detrend and window are functions, unlike in matlab where they
       are vectors.  For detrending you can use detrend_none,
       detrend_mean, detrend_linear or a custom function.  For
       windowing, you can use window_none, window_hanning, or a custom
       function

    -- if length x < NFFT, it will be zero padded to NFFT
    

    Returns the tuple Pxx, freqs

    For plotting, the power is plotted as 10*log10(pxx)) for decibels,
    though pxx itself is returned
    
    Refs:
      Bendat & Piersol -- Random Data: Analysis and Measurement
        Procedures, John Wiley & Sons (1986)

    """
    try: ret =  gca().psd(x, NFFT, Fs, detrend, window, noverlap)
    except ValueError, msg:
        msg = raise_msg_to_str(msg)
        error_msg(msg)
    else:
        draw_if_interactive()
        return ret
    
        
def save(fname, X, fmt='%1.4f'):
    """
    Save the data in X to file fname using fmt string to convert the
    data to strings

    fname can be a filename or a file handle    

    Example usage:

    save('test.out', X)         # X is an array
    save('test1.out', (x,y,z))  # x,y,z equal sized 1D arrays
    save('test2.out', x)        # x is 1D
    save('test3.out', x, fmt='%1.4e')  # use exponential notation

    """

    if is_string_like(fname):
        fh = file(fname, 'w')
    elif hasattr(fname, seek):
        fh = fname
    else:
        raise ValueError('fname must be a string or file handle')

    
    X = asarray(X)
    origShape = None
    if len(X.shape)==1:
        origShape = X.shape
        X.shape = len(X), 1
    for row in X:
        fh.write(' '.join([fmt%val for val in row]) + '\n')

    if origShape is not None:
        X.shape = origShape
def savefig(*args, **kwargs):
    """
    def savefig(fname, dpi=150, facecolor='w', edgecolor='w',
                orientation='portrait'):
    Save the current figure to filename fname.  dpi is the resolution
    in dots per inch.

    Output file types currently supported are jpeg and png and will be
    deduced by the extension to fname

    facecolor and edgecolor are the colors os the figure rectangle

    orientation is either 'landscape' or 'portrait' - not supported on
    all backends; currently only on postscript output."""

    for key in ('dpi', 'facecolor', 'edgecolor'):
        if not kwargs.has_key(key):
            kwargs[key] = rcParams['savefig.%s'%key]

    manager = get_current_fig_manager()
    manager.canvas.print_figure(*args, **kwargs)

def scatter(*args, **kwargs):
    """

    scatter(self, x, y, s=None, c='b'):

    Make a scatter plot of x versus y.  s is a size (in data
    coords) and can be either a scalar or an array of the same
    length as x or y.  c is a color and can be a single color
    format string or an length(x) array of intensities which will
    be mapped by the colormap jet.        

    If size is None a default size will be used
    """

    try: patches =  gca().scatter(*args, **kwargs)
    except ValueError, msg:
        msg = raise_msg_to_str(msg)
        error_msg(msg)
        raise RuntimeError, msg
    draw_if_interactive()
    return patches



def semilogx(*args, **kwargs):
    """
    Make a semilog plot with log scaling on the x axis.  The args to
    semilog x are the same as the args to plot.  See help plot for
    more info    
    """
    try: ret =  gca().semilogx(*args, **kwargs)
    except ValueError, msg:
        msg = raise_msg_to_str(msg)
        error_msg(msg)
    else:
        draw_if_interactive()
        return ret


def semilogy(*args, **kwargs):
    """
    Make a semilog plot with log scaling on the y axis.  The args to
    semilog x are the same as the args to plot.  See help plot for
    more info    
    """
    try: ret =  gca().semilogy(*args, **kwargs)
    except ValueError, msg:
        msg = raise_msg_to_str(msg)
        error_msg(msg)
    else:
        draw_if_interactive()
        return ret
        
    
def set(h, *args, **kwargs):
    """
    Set handle h property in string s to value val

    h can be a handle or vector of handles.

    h is an instance (or vector of instances) of a class, eg a Line2D
    or an Axes or Text.

    args is a list of string, value pairs.  if the string
    is 'somename', set function calls

      o.set_somename(value)

    for every instance in h.

    
    """

    if not iterable(h): h = [h]
    else: h = flatten(h)

    if len(args)%2:
        error_msg('The set args must be string, value pairs')

    funcvals = []
    for i in range(0, len(args)-1, 2):
        funcvals.append((args[i], args[i+1]))
    funcvals.extend(kwargs.items())

    ret = []
    for o in h:
        for s, val in funcvals:
            s = s.lower()
            funcName = "set_%s"%s
            func = getattr(o,funcName)        
            try: ret.extend( [func(val)] )
            except ValueError, msg:
                msg = raise_msg_to_str(msg)
                error_msg(msg)
                raise RuntimeError, msg
        
    draw_if_interactive()
    return [x for x in flatten(ret)]


def specgram(x, NFFT=256, Fs=2, detrend=mlab.detrend_none,
             window=mlab.window_hanning, noverlap=128,
             cmap = ColormapJet(256)):
    """
    Compute a spectrogram of data in x.  Data are split into NFFT
    length segements and the PSD of each section is computed.  The
    windowing function window is applied to each segment, and the
    amount of overlap of each segment is specified with noverlap

    cmap is a a matplotlib.colors.ColorMap
    
    See help(psd) for information on the other arguments

    return value is Pxx, freqs, bins, im

    bins are the time points the spectrogram is calculated over
    freqs is an array of frequencies
    Pxx is a len(times) x len(freqs) array of power
    im is a matplotlib image
    """
    try: ret =  gca().specgram(x, NFFT, Fs, detrend, window, noverlap, cmap)
    except ValueError, msg:
        msg = raise_msg_to_str(msg)
        error_msg(msg)
    else:
        draw_if_interactive()
        return ret


def subplot(*args, **kwargs):
    """
    Create a subplot command, creating axes with

      subplot(numRows, numCols, plotNum)

    where plotNum=1 is the first plot number and increasing plotNums
    fill rows first.  max(plotNum)==numRows*numCols

    You can leave out the commas if numRows<=numCols<=plotNum<10, as
    in

      subplot(211)    # 2 rows, 1 column, first (upper) plot

    subplot(111) is the default axis

    The background color of the subplot can be specified via keyword
    argument 'axisbg', which takes a color string or gdk.Color as value, as in

    subplot(211, axisbg='y')
    """
    try:
        get_current_fig_manager().add_subplot(*args, **kwargs)
        a =  gca()
    except ValueError, msg:
        msg = raise_msg_to_str(msg)
        error_msg(msg)
        raise RuntimeError, msg
    draw_if_interactive()
    return a

def text(x, y, label, fontdict=None, **kwargs):
    """
    Add text to axis at location x,y

    fontdict is a dictionary to override the default text properties.
    If fontdict is None, the default is

      'fontsize'            : 9,
      'verticalalignment'   : 'bottom',
      'horizontalalignment' : 'left'

    **kwargs can in turn be used to override the fontdict, as in

      a.text(x,y,label, fontsize=12)

    This command supplies no override dict, and so will have
    'verticalalignment'='bottom' and 'horizontalalignment'='left' but
    the keyword arg 'fontsize' will create a fontsize of 12

    The purpose these options is to make it easy for you to create a
    default font theme for your plots by creating a single dictionary,
    and then being able to selective change individual attributes for
    the varous text creation commands, as in

        fonts = {
          'color'               : 'k',
          'fontname'            : 'Courier',
          'fontweight'          : 'bold'
          }

        title('My title', fonts, fontsize=12)
        xlabel('My xlabel', fonts, fontsize=10)
        ylabel('My ylabel', fonts, fontsize=10)
        text(12, 20, 'some text', fonts, fontsize=8)

    The Text defaults are

        'color'               : 'k',
        'fontname'            : 'Sans',
        'fontsize'            : 10,
        'fontweight'          : 'bold',
        'fontangle'           : 'normal',
        'horizontalalignment' : 'left'
        'rotation'            : 'horizontal',
        'verticalalignment'   : 'bottom',
        'transx'              : gca().xaxis.transData,
        'transy'              : gca().yaxis.transData,            

        transx and transy specify that text is in data coords,
        alternatively, you can specify text in axis coords (0,0 lower
        left and 1,1 upper right).  The example below places text in
        the center of the axes

        ax = subplot(111)
        text(0.5, 0.5,'matplotlib', 
             horizontalalignment='center',
             verticalalignment='center',
             transx = ax.xaxis.transAxis,
             transy = ax.yaxis.transAxis,
        )

    """
    t =  gca().text(x, y, label, fontdict, **kwargs)
    draw_if_interactive()
    return t

def title(s, *args, **kwargs):
    """
    Set the title of the current axis to s

    Default font override is:
      override = {
        'fontsize'            : 11,
        'verticalalignment'   : 'bottom',
        'horizontalalignment' : 'center'
      }

    See the text docstring for information of how override and the
    optional args work

    """
    l =  gca().set_title(s, *args, **kwargs)
    draw_if_interactive()
    return l



def vlines(*args, **kwargs):    
    """
    lines =  vlines(x, ymin, ymax, color='k'):

    Plot vertical lines at each x from ymin to ymax.  ymin or ymax
    can be scalars or len(x) numpy arrays.  If they are scalars,
    then the respective values are constant, else the heights of
    the lines are determined by ymin and ymax

    Returns a list of lines that were added
    """
    try: lines =  gca().vlines(*args, **kwargs)
    except ValueError, msg:
        msg = raise_msg_to_str(msg)
        error_msg(msg)
        raise RuntimeError, msg
    draw_if_interactive()
    return lines

def xlabel(s, *args, **kwargs):
    """
    Set the x axis label of the current axis to s

    Default override is

      override = {
          'fontsize'            : 10,
          'verticalalignment'   : 'top',
          'horizontalalignment' : 'center'
          }

    See the text docstring for information of how override and
    the optional args work

    """
    l =  gca().set_xlabel(s, *args, **kwargs)
    draw_if_interactive()
    return l

def ylabel(s, *args, **kwargs):
    """
    Set the y axis label of the current axis to s

    Defaults override is

        override = {
           'fontsize'            : 10,
           'verticalalignment'   : 'center',
           'horizontalalignment' : 'right',
           'rotation'='vertical' : }

    See the text docstring for information of how override and the
    optional args work
    
    """
    l = gca().set_ylabel(s, *args, **kwargs)
    draw_if_interactive()
    return l


def table(*args, **kwargs):
    """
    table(cellText=None, cellColours=None,
          cellLoc='right', colWidths=None,
          rowLabels=None, rowColours=None, rowLoc='left',
          colLabels=None, colColours=None, colLoc='center',
          loc='bottom', bbox=None):

    Add a table to the current axes.  Returns a table instance.  For
    finer grained control over tables, use the Table class and add it
    to the axes with add_table.

    Thanks to John Gill for providing the class and table.

    """
    t = gca().table(*args, **kwargs)
    draw_if_interactive()
    return t
    
