"""
Figure and Axes text
"""
from __future__ import division
from artist import Artist
from cbook import True, False, enumerate
from font_manager import FontProperties
from transforms import Transform
from matplotlib import rcParams

def _process_text_args(override, fontdict=None, **kwargs):
    "Return an override dict.  See 'text' docstring for info"

    if fontdict is not None:
        override.update(fontdict)

    override.update(kwargs)
    return override


class Text(Artist):
    """
    Handle storing and drawing of text in window or data coordinates

    Publicly accessible attributes
      dpi     : a DPI instance
      bbox    : the display bounding box Bound2D instance
      transx : the transform of the x data
      transy : the transform of the y data

    """
    def __init__(self, 
                 dpi, bbox,                 
                 x=0, y=0, text='',
                 color=rcParams['text.color'],
                 verticalalignment='bottom',
                 horizontalalignment='left',
                 fontproperties=FontProperties(),
                 rotation=None,
                 transx=Transform(), 
                 transy=Transform()):
        #...snip...#
        Artist.__init__(self, dpi, bbox)
        self.transx = transx
        self.transy = transy
        self._x, self._y = x, y
        
        self._color = color
        self._text = text
        self._verticalalignment = verticalalignment
        self._horizontalalignment = horizontalalignment
        self._rotation = rotation

        self._fontproperties = fontproperties
    def copy_properties(self, text):
        text._x = self._x
        text._y = self._y
        text._text = self._text
        text._color = self._color,
        text._verticalalignment = self._verticalalignment
        text._horizontalalignment = self._horizontalalignment
        text._fontproperties = self._fontproperties.copy()
        text._rotation = self._rotation
        text.transx = self.transx 
        text.transy = self.transy

    def _draw(self, renderer):
        x, y  = self.get_xy_display()
        gc = renderer.new_gc()
        gc.set_foreground(self._color)
        gc.set_alpha(self._alpha)
        if self.get_clip_on():
            gc.set_clip_rectangle(self.bbox.get_bounds())

        renderer.draw_text(gc, x, y, self)

    def get_color(self):
        "Return the color of the text"
        return self._color 

    def get_font_properties(self):
        "Return the font object"
        return self._fontproperties

    def get_fontname(self):
        "Return the font name as string"
        return self._fontproperties.get_family()[-1]  #  temporary hack.

    def get_fontstyle(self):
        "Return the font style as string"
        return self._fontproperties.get_style()

    def get_fontsize(self):
        "Return the font size as integer"
        return self._fontproperties.get_size_in_points()

    def get_fontweight(self):
        "Get the font weight as string"
        return self._fontproperties.get_weight()

    def get_fontangle(self):
        "Get the font angle as string"
        return self._fontproperties.get_style()

    def get_horizontalalignment(self):
        "Return the horizontal alignment as string"
        return self._horizontalalignment

    def get_position(self):
        "Return x, y as tuple"
        return self._x, self._y


    def get_prop_tup(self):
        """
        Return a hashable tuple of properties

        Not intended to be human readable, but useful for backends who
        want to cache derived information about text (eg layouts) and
        need to know if the text has changed
        """

        x, y = self.get_xy_display()
        return (self._x, self._y, self._text, self._color,
                self._verticalalignment, self._horizontalalignment,
                hash(self._fontproperties), self._rotation, self.dpi.get())

    def get_rotation(self):
        "Return the text rotation arg"
        return self._rotation
    
    def get_text(self):
        "Get the text as string"
        return self._text

    def get_verticalalignment(self):
        "Return the vertical alignment as string"
        return self._verticalalignment

    def get_window_extent(self, renderer):
        """
        Return the bounding box of the text in window coords as a
        Bound2D instance
        """
        return renderer.get_text_extent(self)
        
    def get_xy_display(self):
        try: x = self._x.get()
        except AttributeError: x = self._x
        try: y = self._y.get()
        except AttributeError: y = self._y

        return self.transx.positions(x), self.transy.positions(y)
        
    def set_backgroundcolor(self, color):
        "Set the background color of the text"
        self._backgroundcolor = color

        
    def set_color(self, color):
        "Set the foreground color of the text"
        self._color = color

    def set_horizontalalignment(self, align):
        """
        Set the horizontal alignment to one of
        'center', 'right', or 'left'
        """
        legal = ('center', 'right', 'left')
        if align not in legal:
            raise ValueError('Horizontal alignment must be one of %s' % legal)
        if self._horizontalalignment == align: return     
        self._horizontalalignment = align

    def set_family(self, fontname):
        """
        Set the font name, eg, 'Sans', 'Courier', 'Helvetica'
        """
        self._fontproperties.set_family(fontname)

    def set_fontname(self, fontname):
        """
        Set the font name, eg, 'Sans', 'Courier', 'Helvetica'
        """
        self._fontproperties.set_name(fontname)
        
    def set_fontstyle(self, fontstyle):
        """
        Set the font style, one of 'normal', 'italic', 'oblique'
        """
        self._fontproperties.set_style(fontstyle)

    def set_fontsize(self, fontsize):
        """
        Set the font size, eg, 8, 10, 12, 14...
        """
        self._fontproperties.set_size(fontsize)
        
    def set_fontweight(self, weight):
        """
        Set the font weight, one of:
        'normal', 'bold', 'heavy', 'light', 'ultrabold',  'ultralight'
        """
        self._fontproperties.set_weight(weight)
        
    def set_fontangle(self, style):
        """
        Set the font angle, one of 'normal', 'italic', 'oblique'
        """
        self._fontproperties.set_style(style)
        
    def set_position(self, xy):
        self.set_x(xy[0])
        self.set_y(xy[1])

    def set_x(self, x):        
        try: self._x.set(x)
        except AttributeError: self._x = x

    def set_y(self, y):
        try: self._y.set(y)
        except AttributeError: self._y = y
        
    def set_rotation(self, s):
        "Currently only s='vertical', or s='horizontal' are supported"
        self._rotation = s
        
    def set_verticalalignment(self, align):
        """
        Set the vertical alignment to one of
        'center', 'top', or 'bottom'
        """

        legal = ('top', 'bottom', 'center')
        if align not in legal:
            raise ValueError('Vertical alignment must be one of %s' % legal)

        self._verticalalignment = align
        
    def set_text(self, text):
        "Set the text"
        self._text = text

    def set_transx(self, trans):
        "Set the x transform"
        self.transx = trans

    def set_transy(self, trans):
        "Set the y transform"
        self.transy = trans

    def update_properties(self, d):
        "Update the font attributes with the dictionary in d"
        #check the keys
        
        legal = ('family', 'color', 'verticalalignment', 'horizontalalignment',
                 'fontname', 'fontstyle', 'fontangle', 'fontvariant',
                 'fontweight', 'fontsize', 'rotation', 'transx', 'transy',
                 'fontproperties')

        
        for k,v in d.items():
            if k not in legal:
                raise ValueError('Illegal key %s.  Legal values are %s' % (
                    k, legal))
            funcname = 'set_' + k
            assert(hasattr(self, funcname))
            func = getattr(self, funcname)
            func(v)


    def is_math_text(self):
        return ( self._text.startswith('$') and
                 self._text.endswith('$') )

    def set_fontproperties(self, fp):
        self._fontproperties = fp

    def set_transx(self, tx):
        self.transx = tx

    def set_transy(self, ty):
        self.transy = ty
        
