"""
A class for converting color arguments to RGB

This class instantiates a single instance colorConverter that is used
to convert matlab color strings to RGB.  RGB is a tuple of float RGB
values in the range 0-1.

Commands which take color arguments can use several formats to specify
the colors.  For the basic builtin colors, you can use a single letter

      b  : blue
      g  : green
      r  : red
      c  : cyan
      m  : magenta
      y  : yellow
      k  : black 
      w  : white


For a greater range of colors, you have two options.  You can specify
the color using an html hex string, as in

      color = '#eeefff'

or you can pass an R,G,B tuple, where each of R,G,B are in the range
[0,1].
"""
from __future__ import division
from numerix import MLab, array, arange, Float, take, Float, Int, where, \
     zeros, asarray

from cbook import True, False, enumerate, is_string_like, iterable

def looks_like_color(c):
    if is_string_like(c):
        if len(c)==1: return True
        elif len(s)==7 and c.startswith('#') and len(s)==7: return True
        else: return False
    elif iterable(c) and len(c)==3:
        try:
            rgb = [float(val) for val in c]
            return True
        except:
            return False
    else:
        return False
    
def hex2color(s):
    "Convert hex string (like html uses, eg, #efefef) to a r,g,b tuple"
    if s.find('#')!=0 or len(s)!=7:
        raise ValueError('s must be a hex string like "#efefef#')
    r,g,b = map(lambda x: int('0x' + x, 16)/256.0, (s[1:3], s[3:5], s[5:7]))
    return r,g,b

class ColorConverter:
    colors = {
        'b' : (0.0, 0.0, 1.0),
        'g' : (0.0, 0.5, 0.0),
        'r' : (1.0, 0.0, 0.0),
        'c' : (0.0, 0.75, 0.75),
        'm' : (0.75, 0, 0.75),
        'y' : (0.75, 0.75, 0),
        'k' : (0.0, 0.0, 0.0),
        'w' : (1.0, 1.0, 1.0),
        }

    cache = {}
    def to_rgb(self, arg):
        """
        returns a tuple of three floats from 0-1.  arg can be a matlab
        format string, a html hex color string, an rgb tuple, or a
        float between 0 and 1.  In the latter case, grayscale is used
        """
        try: return self.cache[arg]
        except KeyError: pass
        except TypeError: # could be unhashable rgb seq
            arg = tuple(arg)
            try: self.cache[arg]
            except KeyError: pass

        
        try: float(arg)
        except: 
            if is_string_like(arg) and len(arg)==7 and arg[0]=='#':
                color =   hex2color(arg)
            else:
                # see if it looks like rgb.  If so, just return arg
                try: float(arg[2])
                except: color = self.colors.get(arg, (0.0, 0.0, 0.0))
                else: color =  tuple(arg)
        else:
            if arg>=0 and arg<=1:
                color =  (arg,arg,arg)
            else:
                msg = 'Floating point color arg must be between 0 and 1\n' +\
                      'Found %1.2f' % arg
                raise RuntimeError(msg)

        self.cache[arg] = color
        return color

    def to_rgba(self, arg, alpha=1.0):
        """
        returns a tuple of four floats from 0-1.  arg can be a matlab
        format string, a html hex color string, an rgb tuple, or a
        float between 0 and 1.  In the latter case, grayscale is used
        """
        r,g,b = self.to_rgb(arg)
        return r,g,b,alpha

colorConverter = ColorConverter()

class Colormap:
   def __init__(self, N=256, color='jet'):
      self.N = N
      self.color=color
      self.indmax = self.N-1

      self._make_red()
      self._make_blue()
      self._make_green()
      self.clim = None
      self.rgbs =  zip(self.red, self.green, self.blue)
      
   def _make_red(self):
      if self.color=='jet':
         fracon = 0.35
         fracmax = 0.66
         fracdown = 0.89
         mup = 1/(fracmax - fracon)
         mdown = -0.5/(1-fracdown)
         self.red = []
         for i in range(self.N):
            frac = i/self.N
            if frac < fracon: thisval = 0
            elif frac >=fracon and frac<=fracmax: thisval = mup * (frac-fracon)
            elif frac > fracmax and frac<fracdown: thisval = 1.0
            else: thisval = 1+mdown*(frac-fracdown)
            self.red.append( thisval )
      if self.color=='gray':
         self.red = 1.0/self.N*arange(0,self.N,1,Float)

   def _make_blue(self):
      if self.color in ['jet']:
         #blue is just red flipped left/right
         if not self.__dict__.has_key('red'):
            s = 'You must call _jet_make_red before _jet_make_blue'
            raise RuntimeError, s
                   
         self.blue = [self.red[i] for i in range(self.indmax,-1,-1)]
      if self.color=='gray':
         self.blue = 1.0/self.N*arange(0,self.N,1,Float)


   def _make_green(self):
      if self.color=='jet':
         fracon = 0.1250
         fracmax = 0.375
         fracdown = 0.64
         fracoff = 0.91
         mup = 1/(fracmax - fracon)
         mdown = -1/(fracoff-fracdown)
         self.green = []
         for i in range(self.N):
            frac = i/self.N
            if frac < fracon: thisval = 0
            elif frac >=fracon and frac<=fracmax: thisval = mup * (frac-fracon)
            elif frac > fracmax and frac<fracdown: thisval = 1.0
            elif frac >= fracdown and frac<fracoff:
                thisval = 1+mdown*(frac-fracdown)
            else: thisval = 0.0
            self.green.append( thisval )
      if self.color=='gray':
         self.green = 1.0/self.N*arange(0,self.N,1,Float)

   def get_color(self, val, valmin, valmax):
       # map val to a range from 0 to 1
       if iterable(val):
          s = "val must be a scalar.  Perhaps you meant to call get_colors?"
          raise ValueError, s
       #print valmin, valmax
       ind = self.indmax*(val-valmin)/(valmax-valmin)
       return self.rgbs[self._bound_ind(ind)]

   def get_color_alpha(self, val, valmin, valmax, alpha=1.0):
       # map val to a range from 0 to 1
       if iterable(val):
          s = "val must be a scalar.  Perhaps you meant to call get_colors?"
          raise ValueError, s
       #print valmin, valmax
       ind = self.indmax*(val-valmin)/(valmax-valmin)
       r,g,b = self.rgbs[self._bound_ind(ind)]
       return r,g,b,alpha



   def _bound_ind(self, ind):
      if ind < 0: return 0
      if ind > self.indmax: return self.indmax
      return int(ind)
      

   def get_colors(self, vals, valmin=None, valmax=None):
      # map Numeric array vals to colors
      vals = asarray(vals)
      if valmin is None: valmin = min(vals)
      if valmax is None: valmax = max(vals)
      if valmin==valmax: return self.rgbs[0]
      ind = (self.N-1.0)/(valmax-valmin)*(vals-valmin)
      return [self.rgbs[self._bound_ind(i)] for i in ind]

   def get_colors_alpha(self, vals, valmin=None, valmax=None, alpha=1.0):
      # map Numeric array vals to colors
      vals = asarray(vals)
      if valmin is None: valmin = min(vals)
      if valmax is None: valmax = max(vals)
      if valmin==valmax: return self.rgbs[0]
      ind = (self.N-1.0)/(valmax-valmin)*(vals-valmin)
      return [self.rgbs[self._bound_ind(i)] for i in ind]

   def array_as_rgb(self, X):
       """
       X is an MxN array of floats.  Set the clim if you want to
       control the color range (default min t max)      
       """
       if self.clim is None:
        vmin = MLab.min(MLab.min(X))
        vmax = MLab.max(MLab.max(X))

       else:
           vmin, vmax = self.clim

       M,N = X.shape      
       ind = (float(self.N)/(vmax-vmin))*(X-vmin)
       ind = ind.astype(Int)
       ind = where(ind<0,0, ind)
       ind = where(ind>=self.N,self.N-1, ind)

       r = take(self.red, ind)
       g = take(self.green, ind)
       b = take(self.blue, ind)
       A =  zeros((M,N,3), Float)
       A[:,:,0] = r
       A[:,:,1] = g
       A[:,:,2] = b
       return A
   def set_clim(self, cmin, cmax):
       'Set the color limits for mapping scalar data to colormap'
       self.clim = cmin, cmax

class ColormapJet(Colormap):
   pass


class Grayscale(Colormap):
    def __init__(self, N=256):
        Colormap.__init__(self, N, 'gray')
        
   
