"""
The image module supports basic image loading, rescaling and display
operations.

"""
from __future__ import division
from artist import Artist
from numerix import arange
import _image

class Image(Artist):


    def __init__(self, viewlim, datalim, displaylim):
        """
        The viewlim instance that give the x and y viewlimits so that
        the image can respond to changes in pan/zoom etc...

        The x and y datalim give the data range of the image.  Eg if
        datalim.interalx() = (xmin, xmax), then the 0th column is at
        point x = xmin and the last image column is at point x = xmax;
        likewise for y.  The default data lim are 0, width-1 and 0,
        height-1

        A linear scaling between datalim and viewlim is assumed
        """
        Artist.__init__(self)

        # map interpolation strings to module constants
        self._interpd = {
            'bicubic'     : _image.BICUBIC,
            'bilinear'    : _image.BILINEAR,
            'blackman100' : _image.BLACKMAN100,
            'blackman256' : _image.BLACKMAN256,
            'blackman64'  : _image.BLACKMAN64,
            'nearest'     : _image.NEAREST,
            'sinc144'     : _image.SINC144,
            'sinc256'     : _image.SINC256,
            'sinc64'      : _image.SINC64,
            'spline16'    : _image.SPLINE16,
            'spline36'    : _image.SPLINE36,       
        }

        # map aspect ratio strings to module constants
        self._aspectd = {
            'free'     : _image.ASPECT_FREE,
            'preserve' : _image.ASPECT_PRESERVE,
        }

        self.displaylim = displaylim
        # reverse interp dict
        self._interpdr = dict([ (v,k) for k,v in self._interpd.items()])

        # reverse aspect dict
        self._aspectdr = dict([ (v,k) for k,v in self._aspectd.items()])

        self._im = None
        self._viewlim = viewlim
        self._datalim = datalim

    def draw(self, renderer, *args, **kwargs):
        if self._im is None:
            raise ValueError('You must first set the image')

        # image input dimensions
        numrows, numcols = self._im.get_size()
        self._im.reset_matrix()

        dxintv = self._datalim.width()
        dyintv = self._datalim.height()

        # the viewport scale factor
        sx = dxintv/self._viewlim.width()
        sy = dyintv/self._viewlim.height()



        # the viewport translation
        tx = (self._datalim.xmin()-self._viewlim.xmin())/dxintv * numcols
        if renderer.flipy():
            ty = (self._datalim.ymax()-self._viewlim.ymax())/dyintv * numrows
        else:
            ty = (self._datalim.ymin()-self._viewlim.ymin())/dyintv * numrows

        if renderer.flipy(): ty = -ty
        l, b, widthDisplay, heightDisplay = self.displaylim.get_bounds()

        self._im.apply_translation(tx, ty)
        self._im.apply_scaling(sx, sy)

        # resize viewport to display
        rx = widthDisplay / numcols
        ry = heightDisplay  / numrows

        if self._im.get_aspect()==_image.ASPECT_PRESERVE:
            if ry < rx: rx = ry
            # todo: center the image in viewport
            self._im.apply_scaling(rx, rx)
        else:
            self._im.apply_scaling(rx, ry)
            
        self._im.resize(int(heightDisplay), int(widthDisplay))

        renderer.draw_image(l, b+heightDisplay, self._im)


    def fromarray(self, A):
        'Load the image from numeric/numarray A'
        self._im = _image.fromarray(A)
        self._image_init()

        
    def fromfile(self, fname):
        'Load the image from file fname'
        raise NotImplementedError('Not there yet')
        self._im = _image.fromfile(fname)
        self._image_init()

    def get_aspect(self):
        """
        Return the method used to constrain the aspoect ratio of the
        One of
        
        'free'     : aspect ratio not constrained
        'preserve' : preserve aspect ration when resizing
        """
        if self._im is None:
            raise ValueError('You must first set the image')
        a = self._im.get_aspect()
        return self._aspectdr[a]


    def get_size(self):
        'Get the width, height of the input image'
        if self._im is None:
            raise ValueError('You must first set the image')
        
        return self._im.get_size()

    def get_interpolation(self):
        """
        Return the interpolation method the image uses when resizing.

        One of
        
        'bicubic', 'bilinear', 'blackman100', 'blackman256', 'blackman64',
        'nearest', 'sinc144', 'sinc256', 'sinc64', 'spline16', 'spline36'
        """
        if self._im is None:
            raise ValueError('You must first set the image')
        
        i = self._im.get_interpolation()
        return self._interpdr[i]

    def _image_init(self):
        """
        Call this after loading an image from any src
        """

        width, height = self._im.get_size()
        self.set_data_extent(0, width-1, 0, height-1)
        

    def set_aspect(self, s):
        """
        Set the method used to constrain the aspoect ratio of the
        image ehen resizing,

        One of
        
        'free'     : aspect ratio not constrained
        'preserve' : preserve aspect ration when resizing
        """
        if self._im is None:
            raise ValueError('You must first set the image')

        s = s.lower()
        if not self._aspectd.has_key(s):
            raise ValueError('Illegal aspect string')
        self._im.set_aspect(self._aspectd[s])

    def set_interpolation(self, s):
        """
        Set the interpolation method the image uses when resizing.

        One of
        
        'bicubic', 'bilinear', 'blackman100', 'blackman256', 'blackman64',
        'nearest', 'sinc144', 'sinc256', 'sinc64', 'spline16', 'spline36'
        """
        if self._im is None:
            raise ValueError('You must first set the image')
        
        s = s.lower()
        if not self._interpd.has_key(s):
            raise ValueError('Illegal interpolation string')
        self._im.set_interpolation(self._interpd[s])


    def set_data_extent(self, xmin, xmax, ymin, ymax):
        """
        Set the data extent that the image pixels represent.  If you
        are using the matlab interface, use Axes.set_image_extent
        instead
        """
        self._datalim.intervalx().set_bounds(xmin, xmax)
        self._datalim.intervaly().set_bounds(ymin, ymax)


        
