"""
This module contains all the 2D line class which can draw with a
variety of line styles, markers and colors
"""

# TODO: expose cap and join style attrs
from __future__ import division

import sys

from numerix import Float, alltrue, arange, array, logical_and,\
     nonzero, searchsorted, take, asarray, ones, where, less

from artist import Artist
from cbook import True, False
from colors import colorConverter
from patches import bbox_artist
from transforms import lbwh_to_bbox
from matplotlib import rcParams

TICKLEFT, TICKRIGHT, TICKUP, TICKDOWN = range(4)
lineStyles  = {'-':1, '--':1, '-.':1, ':':1}
lineMarkers =    {'.':1, ',':1, 'o':1, '^':1, 'v':1, '<':1, '>':1, 's':1,
                  '+':1, 'x':1, 'd':1, 'D':1, '|':1, '_':1, 'h':1, 'H':1,
                  'p':1, '1':1, '2':1, '3':1, '4':1,
                  TICKLEFT:1,
                  TICKRIGHT:1,
                  TICKUP:1,
                  TICKDOWN:1,
                  }

class Line2D(Artist):

    def __init__(self, xdata, ydata,
                 linewidth=rcParams['lines.linewidth'],
                 linestyle=rcParams['lines.linestyle'],
                 color=rcParams['lines.color'], 
                 marker=None,
                 markersize=rcParams['lines.markersize'],
                 markeredgecolor=None, markerfacecolor=None,
                 antialiased=rcParams['lines.antialiased'],
                 ):
        """
        xdata is a sequence of x data
        ydata is a sequence of y data
        linewidth is the width of the line in points
        linestyle is one of lineStyles
        marker is one of lineMarkers or None
        markersize is the size of the marker in points
        markeredgecolor  and markerfacecolor can be any color arg 

        """
        Artist.__init__(self)
        #convert sequences to numeric arrays

        self._linestyle = linestyle
        self._linewidth = linewidth
        self._color = color
        self._antialiased = antialiased
        self._markersize = markersize
        self._label = ''
        self._dashSeq = None
        
        if markeredgecolor is None: markeredgecolor=color
        
        self._markerfacecolor = markerfacecolor
        self._markeredgecolor = markeredgecolor

        self.verticalOffset = None        
        self.set_data(xdata, ydata)
        self._lineStyles =  {
            '-'  : self._draw_solid,
            '--' : self._draw_dashed, 
            '-.' : self._draw_dash_dot, 
            ':'  : self._draw_dotted,
            None : self._draw_nothing}

        self._markers =  {
            '.'  : self._draw_point,
            ','  : self._draw_pixel, 
            'o'  : self._draw_circle, 
            'v'  : self._draw_triangle_down, 
            '^'  : self._draw_triangle_up, 
            '<'  : self._draw_triangle_left, 
            '>'  : self._draw_triangle_right, 
            '1'  : self._draw_tri_down, 
            '2'  : self._draw_tri_up, 
            '3'  : self._draw_tri_left, 
            '4'  : self._draw_tri_right, 
            's'  : self._draw_square,
            'p'  : self._draw_pentagon,
            'h'  : self._draw_hexagon1,
            'H'  : self._draw_hexagon2,
            '+'  : self._draw_plus,
            'x'  : self._draw_x,
            'D'  : self._draw_diamond,
            'd'  : self._draw_thin_diamond,
            '|'  : self._draw_vline,
            '_'  : self._draw_hline,
            TICKLEFT    : self._draw_tickleft, 
            TICKRIGHT   : self._draw_tickright,
            TICKUP      : self._draw_tickup,
            TICKDOWN    : self._draw_tickdown,
            None : self._draw_nothing
            }

        if not self._lineStyles.has_key(linestyle):
            print >>sys.stderr, 'Unrecognized line style', linestyle
        if not self._markers.has_key(marker):
            print >>sys.stderr, 'Unrecognized marker style', marker
 
        self.set_marker(marker)
        self._lineFunc = self._lineStyles.get(linestyle, self._draw_nothing)
        self._markerFunc = self._markers.get(marker, self._draw_nothing)
        self._useDataClipping = rcParams['lines.data_clipping']
        
    def get_window_extent(self, renderer):
        x, y = self._get_numeric_clipped_data_in_range()

        x, y = self._transform.numerix_x_y(x, y)
        #x, y = self._transform.seq_x_y(x, y)

        left = min(x)
        bottom = min(y)
        width = max(x) - left
        height = max(y) - bottom

        # correct for marker size, if any
        if self._marker is not None:
            ms = self._markersize/72.0*self.figure.dpi.get()
            left -= ms/2
            bottom -= ms/2
            width += ms
            height += ms
        return lbwh_to_bbox( left, bottom, width, height)
        
    def set_data(self, x, y):
        try: del self._xc, self._yc
        except AttributeError: pass

        self._x = asarray(x, Float)
        self._y = asarray(y, Float)
        if len(self._y)==1 and len(self._x)>1:
            self._y = self._y*ones(self._x.shape, Float)

        self._xsorted = self._is_sorted(self._x)

    def set_data_clipping(self, b):
        'b is a boolean that sets whether data clipping is on'
        self._useDataClipping = b

    def set_label(self, s):
        'Set the line label to s for auto legend'
        self._label = s
        
    def set_vertical_offset(self, voff, transOffset=None):
        # JDH: todo; handle the offset
        raise NotImplementedError('jdh fix me!')
        
    def _is_sorted(self, x):
        "return true if x is sorted"
        if len(x)<2: return 1
        return alltrue(x[1:]-x[0:-1]>=0)
    
    def _get_numeric_clipped_data_in_range(self):
        # if the x or y clip is set, only plot the points in the
        # clipping region
        try: self._xc, self._yc
        except AttributeError: x, y = self._x, self._y
        else: x, y = self._xc, self._yc

            
        return x, y

    def draw(self, renderer):

        x, y = self._get_numeric_clipped_data_in_range()
        if len(x)==0: return 

        xt, yt = self._transform.numerix_x_y(x, y)
        #xt, yt = self._transform.seq_x_y(x, y)


        gc = renderer.new_gc()
        gc.set_foreground(self._color)
        gc.set_antialiased(self._antialiased)
        gc.set_linewidth(self._linewidth)
        gc.set_alpha(self._alpha)
        if self.get_clip_on():
            gc.set_clip_rectangle(self.clipbox.get_bounds())


        self._lineFunc(renderer, gc, xt, yt)


        if self._marker is not None:
            gc = renderer.new_gc()
            gc.set_foreground(self._markeredgecolor)
            gc.set_linewidth(self._linewidth)
            if self.get_clip_on():
                gc.set_clip_rectangle(self.clipbox.get_bounds())
            self._markerFunc(renderer, gc, xt, yt)

        #if 1: bbox_artist(self, renderer)
        
    def get_antialiased(self): return self._antialiased
    def get_color(self): return self._color
    def get_linestyle(self): return self._linestyle
    def get_label(self): return self._label
    def get_linewidth(self): return self._linewidth
    def get_marker(self): return self._marker
    def get_markeredgecolor(self): return self._markeredgecolor
    def get_markerfacecolor(self): return self._markerfacecolor
    def get_markersize(self): return self._markersize
    def get_xdata(self): return self._x
    def get_ydata(self):  return self._y

        
    def _set_clip(self):

        if not self._useDataClipping: return
        try: self._xmin, self._xmax
        except AttributeError: indx = arange(len(self._x))
        else:
            if len(self._x)==1:
                indx = 0
            elif self._xsorted:
                # for really long signals, if we know they are sorted
                # on x we can save a lot of time using search sorted
                # since the alternative approach requires 3 O(len(x) ) ops
                indMin, indMax = searchsorted(
                    self._x, array([self._xmin, self._xmax]))
                indMin = max(0, indMin-1)
                indMax = min(indMax+1, len(self._x))
                skip = 0
                if self._lod:
                    # if level of detail is on, decimate the data
                    # based on pixel width
                    l, b, w, h = self.get_window_extent().get_bounds()
                    skip = int((indMax-indMin)/w)                    
                if skip>0:  indx = arange(indMin, indMax, skip)
                else: indx = arange(indMin, indMax)
            else:
                indx = nonzero(
                    logical_and( self._x>=self._xmin,
                                 self._x<=self._xmax ))
        

        self._xc = take(self._x, indx)
        self._yc = take(self._y, indx)

        # y data clipping for connected lines can introduce horizontal
        # line artifacts near the clip region.  If you really need y
        # clipping for efficiency, consider using plot(y,x) instead.
        if ( self._yc.shape==self._xc.shape and 
             self._linestyle is None):
            try: self._ymin, self._ymax
            except AttributeError: indy = arange(len(self._yc))
            else: indy = nonzero(
                logical_and(self._yc>=self._ymin,
                                  self._yc<=self._ymax ))
        else:
            indy = arange(len(self._yc))
            
        self._xc = take(self._xc, indy)
        self._yc = take(self._yc, indy)

    def set_antialiased(self, b):
        """
        True if line should be drawin with antialiased rendering
        """            
        self._antialiased = b

    def set_color(self, color):
        self._color = color

    def set_linewidth(self, w):
        self._linewidth = w

    def set_linestyle(self, s):
        self._linestyle = s
        self._lineFunc = self._lineStyles.get(self._linestyle, '-')

    def set_marker(self, marker):
        self._marker = marker
        self._markerFunc = self._markers.get(marker, self._draw_nothing)

    def set_markeredgecolor(self, ec):
        self._markeredgecolor = ec

    def set_markerfacecolor(self, fc):
        self._markerfacecolor = fc

    def set_markersize(self, sz):
        self._markersize = sz

    def set_xdata(self, x):
        self.set_data(x, self._y)

    def set_ydata(self, y):

        self.set_data(self._x, y)
        
    def set_xclip(self, xmin, xmax):
        if xmax<xmin: xmax, xmin = xmin, xmax
        self._xmin, self._xmax = xmin, xmax
        self._set_clip()

    def set_yclip(self, ymin, ymax):
        if ymax<ymin: ymax, ymin = ymin, ymax
        self._ymin, self._ymax = ymin, ymax
        self._set_clip()

    def set_dashes(self, seq):
        'seq is a sequence of dashes with on off ink in points'
        self._dashSeq = seq
        
    def _draw_nothing(self, renderer, gc, xt, yt):
        pass
    
    def _draw_solid(self, renderer, gc, xt, yt):
        if len(xt)<2: return
        gc.set_linestyle('solid')
        gc.set_capstyle('projecting')   
        #print gc._rgb, type(gc._rgb), len(gc._rgb)
        renderer.draw_lines(gc, xt,yt)

    def _draw_dashed(self, renderer, gc, xt, yt):
        if len(xt)<2: return
        gc.set_linestyle('dashed')
        if self._dashSeq is not None:
            gc.set_dashes(0, self._dashSeq)
        gc.set_capstyle('butt')   
        gc.set_joinstyle('miter') 
        renderer.draw_lines(gc, xt, yt)

    def _draw_dash_dot(self, renderer, gc, xt, yt):
        if len(xt)<2: return
        gc.set_linestyle('dashdot')
        gc.set_capstyle('butt')   
        gc.set_joinstyle('miter') 
        renderer.draw_lines(gc, xt, yt)


    def _draw_dotted(self, renderer, gc, xt, yt):

        if len(xt)<2: return
        gc.set_linestyle('dotted')
        gc.set_capstyle('butt')   
        gc.set_joinstyle('miter')
        renderer.draw_lines(gc, xt, yt)

        
    def _draw_point(self, renderer, gc, xt, yt):
        for (x,y) in zip(xt, yt):
            renderer.draw_arc(gc, None, x, y, 1, 1, 0.0, 360.0)

    def _draw_pixel(self, renderer, gc, xt, yt):
        for (x,y) in zip(xt, yt):
            renderer.draw_point(gc, x, y)


    def _draw_circle(self, renderer, gc, xt, yt):
        w = h = renderer.points_to_pixels(self._markersize)

        
        rgbFace = self._get_rgb_face()
        for (x,y) in zip(xt, yt):
            renderer.draw_arc(gc, rgbFace,
                              x, y, w, h, 0.0, 360.0)

    def _draw_triangle_up(self, renderer, gc, xt, yt):
        
        offset = 0.5*renderer.points_to_pixels(self._markersize)
        rgbFace = self._get_rgb_face()
        for (x,y) in zip(xt, yt):
            verts = ( (x, y+offset),
                      (x-offset, y-offset),
                      (x+offset, y-offset) )
            renderer.draw_polygon(gc, rgbFace, verts)


    def _draw_triangle_down(self, renderer, gc, xt, yt):
        offset = 0.5*renderer.points_to_pixels(self._markersize)
        rgbFace = self._get_rgb_face()
        for (x,y) in zip(xt, yt):            
            verts = ( (x-offset, y+offset),
                      (x+offset, y+offset),
                      (x, y-offset))
            renderer.draw_polygon(gc, rgbFace, verts)

    def _draw_triangle_left(self, renderer, gc, xt, yt):
        offset = 0.5*renderer.points_to_pixels(self._markersize)
        rgbFace = self._get_rgb_face()
        for (x,y) in zip(xt, yt):            
            verts = ( (x-offset, y),
                      (x+offset, y-offset),
                      (x+offset, y+offset))
            renderer.draw_polygon(gc, rgbFace, verts)


    def _draw_triangle_right(self, renderer, gc, xt, yt):
        offset = 0.5*renderer.points_to_pixels(self._markersize)
        rgbFace = self._get_rgb_face()
        for (x,y) in zip(xt, yt):            
            verts = ( (x+offset, y),
                      (x-offset, y-offset),
                      (x-offset, y+offset))
            renderer.draw_polygon(gc, rgbFace, verts)

            

    def _draw_square(self, renderer, gc, xt, yt):
        side = renderer.points_to_pixels(self._markersize)
        offset = side*0.5
        rgbFace = self._get_rgb_face()
        for (x,y) in zip(xt, yt):            
            renderer.draw_rectangle(
                gc, rgbFace,
                x-offset, y-offset, side, side) 

    def _draw_diamond(self, renderer, gc, xt, yt):
        offset = 0.6*renderer.points_to_pixels(self._markersize)
        rgbFace = self._get_rgb_face()
        for (x,y) in zip(xt, yt):            
            verts = ( (x+offset, y),
                      (x, y-offset),
                      (x-offset, y),
                      (x, y+offset))
            renderer.draw_polygon(gc, rgbFace, verts)

    def _draw_thin_diamond(self, renderer, gc, xt, yt):
        offset = 0.7*renderer.points_to_pixels(self._markersize)
        xoffset = 0.6*offset
        rgbFace = self._get_rgb_face()
        for (x,y) in zip(xt, yt):            
            verts = ( (x+xoffset, y),
                      (x, y-offset),
                      (x-xoffset, y),
                      (x, y+offset))
            renderer.draw_polygon(gc, rgbFace, verts)

    def _draw_pentagon(self, renderer, gc, xt, yt):
        offset = 0.6*renderer.points_to_pixels(self._markersize)
        offsetX1 = offset*0.95
        offsetY1 = offset*0.31
        offsetX2 = offset*0.59
        offsetY2 = offset*0.81
        rgbFace = self._get_rgb_face()
        for (x,y) in zip(xt, yt):            
            verts = ( (x, y+offset),
                      (x-offsetX1, y+offsetY1),
                      (x-offsetX2, y-offsetY2),
                      (x+offsetX2, y-offsetY2),
                      (x+offsetX1, y+offsetY1))
            renderer.draw_polygon(gc, rgbFace, verts)

    def _draw_hexagon1(self, renderer, gc, xt, yt):
        offset = 0.6*renderer.points_to_pixels(self._markersize)
        offsetX1 = offset*0.87
        offsetY1 = offset*0.5
        rgbFace = self._get_rgb_face()
        for (x,y) in zip(xt, yt):            
            verts = ( (x, y+offset),
                      (x-offsetX1, y+offsetY1),
                      (x-offsetX1, y-offsetY1),
                      (x, y-offset),
                      (x+offsetX1, y-offsetY1),
                      (x+offsetX1, y+offsetY1))
            renderer.draw_polygon(gc, rgbFace, verts)

    def _draw_hexagon2(self, renderer, gc, xt, yt):
        offset = 0.6*renderer.points_to_pixels(self._markersize)
        offsetX1 = offset*0.5
        offsetY1 = offset*0.87
        rgbFace = self._get_rgb_face()
        for (x,y) in zip(xt, yt):            
            verts = ( (x+offset, y),
                      (x+offsetX1, y+offsetY1),
                      (x-offsetX1, y+offsetY1),
                      (x-offset, y),
                      (x-offsetX1, y-offsetY1),
                      (x+offsetX1, y-offsetY1))
            renderer.draw_polygon(gc, rgbFace, verts)

    def _draw_vline(self, renderer, gc, xt, yt):
        offset = 0.5*renderer.points_to_pixels(self._markersize)
        for (x,y) in zip(xt, yt):            
            renderer.draw_line(gc, x, y-offset, x, y+offset)

    def _draw_hline(self, renderer, gc, xt, yt):
        offset = 0.5*renderer.points_to_pixels(self._markersize)
        for (x,y) in zip(xt, yt):
            renderer.draw_line(gc, x-offset, y, x+offset, y)

    def _draw_tickleft(self, renderer, gc, xt, yt):
        offset = renderer.points_to_pixels(self._markersize)
        for (x,y) in zip(xt, yt):
            renderer.draw_line(gc, x-offset, y, x, y)

    def _draw_tickright(self, renderer, gc, xt, yt):

        offset = renderer.points_to_pixels(self._markersize)
        for (x,y) in zip(xt, yt):
            renderer.draw_line(gc, x, y, x+offset, y)

    def _draw_tickup(self, renderer, gc, xt, yt):
        offset = renderer.points_to_pixels(self._markersize)

        for (x,y) in zip(xt, yt):
            renderer.draw_line(gc, x, y, x, y+offset)

    def _draw_tickdown(self, renderer, gc, xt, yt):
        offset = renderer.points_to_pixels(self._markersize)

        rgbFace = self._get_rgb_face()
        for (x,y) in zip(xt, yt):
            renderer.draw_line(gc, x, y-offset, x, y)

    def _draw_plus(self, renderer, gc, xt, yt):
        offset = 0.5*renderer.points_to_pixels(self._markersize)
        for (x,y) in zip(xt, yt):
            renderer.draw_line(gc, x-offset, y, x+offset, y)
            renderer.draw_line(gc, x, y-offset, x, y+offset)

    def _draw_tri_down(self, renderer, gc, xt, yt):
        offset = 0.5*renderer.points_to_pixels(self._markersize)
        offset1 = offset*0.8
        offset2 = offset*0.5
        for (x,y) in zip(xt, yt):
            renderer.draw_line(gc, x, y, x, y-offset)
            renderer.draw_line(gc, x, y, x+offset1, y+offset2)
            renderer.draw_line(gc, x, y, x-offset1, y+offset2)

    def _draw_tri_up(self, renderer, gc, xt, yt):
        offset = 0.5*renderer.points_to_pixels(self._markersize)
        offset1 = offset*0.8
        offset2 = offset*0.5
        for (x,y) in zip(xt, yt):
            renderer.draw_line(gc, x, y, x, y+offset)
            renderer.draw_line(gc, x, y, x+offset1, y-offset2)
            renderer.draw_line(gc, x, y, x-offset1, y-offset2)

    def _draw_tri_left(self, renderer, gc, xt, yt):
        offset = 0.5*renderer.points_to_pixels(self._markersize)
        offset1 = offset*0.8
        offset2 = offset*0.5
        for (x,y) in zip(xt, yt):
            renderer.draw_line(gc, x, y, x-offset, y)
            renderer.draw_line(gc, x, y, x+offset2, y+offset1)
            renderer.draw_line(gc, x, y, x+offset2, y-offset1)

    def _draw_tri_right(self, renderer, gc, xt, yt):
        offset = 0.5*renderer.points_to_pixels(self._markersize)
        offset1 = offset*0.8
        offset2 = offset*0.5
        for (x,y) in zip(xt, yt):
            renderer.draw_line(gc, x, y, x+offset, y)
            renderer.draw_line(gc, x, y, x-offset2, y+offset1)
            renderer.draw_line(gc, x, y, x-offset2, y-offset1)

    def _draw_x(self, renderer, gc, xt, yt):
        offset = 0.5*renderer.points_to_pixels(self._markersize)
        for (x,y) in zip(xt, yt):
            renderer.draw_line(gc, x-offset, y-offset, x+offset, y+offset)
            renderer.draw_line(gc, x-offset, y+offset, x+offset, y-offset)

    def copy_properties(self, line):
        'copy properties from line to self'
        self._linestyle = line._linestyle
        self._linewidth = line._linewidth
        self._color = line._color
        self._markersize = line._markersize        
        self._markerfacecolor = line._markerfacecolor
        self._markeredgecolor = line._markeredgecolor
        self._dashSeq = line._dashSeq

        self._linestyle = line._linestyle
        self._marker = line._marker
        self._lineFunc = line._lineStyles[line._linestyle]
        self._markerFunc = line._markers[line._marker]
        self._useDataClipping = line._useDataClipping

    def _get_rgb_face(self):
        if self._markerfacecolor is None: rgbFace = None
        else: rgbFace = colorConverter.to_rgb(self._markerfacecolor)
        return rgbFace
