/*
 * ====================================================================
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Tomcat", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 *
 */


package php.java.servlet;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.Method;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLEncoder;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.Vector;

import javax.servlet.ServletConfig;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.UnavailableException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import php.java.bridge.Util;


/**
 *  CGI-invoking servlet for web applications, used to execute scripts which
 *  comply to the Common Gateway Interface (CGI) specification and are named
 *  in the path-info used to invoke this servlet.
 *
 * <p>
 *
 * <b>Example</b>:<br>
 * If an instance of this servlet was mapped (using
 *       <code>&lt;web-app&gt;/WEB-INF/web.xml</code>) to:
 * </p>
 * <p>
 * <code>
 * &lt;web-app&gt;/cgi-bin/*
 * </code>
 * </p>
 * <p>
 * then the following request:
 * </p>
 * <p>
 * <code>
 * http://localhost:8080/&lt;web-app&gt;/cgi-bin/dir1/script/pathinfo1
 * </code>
 * </p>
 * <p>
 * would result in the execution of the script
 * </p>
 * <p>
 * <code>
 * &lt;web-app-root&gt;/WEB-INF/cgi/dir1/script
 * </code>
 * </p>
 * <p>
 * with the script's <code>PATH_INFO</code> set to <code>/pathinfo1</code>.
 * </p>
 * <p>
 * Recommendation:  House all your CGI scripts under
 * <code>&lt;webapp&gt;/WEB-INF/cgi</code>.  This will ensure that you do not
 * accidentally expose your cgi scripts' code to the outside world and that
 * your cgis will be cleanly ensconced underneath the WEB-INF (i.e.,
 * non-content) area.
 * </p>
 * <p>
 * The default CGI location is mentioned above.  You have the flexibility to
 * put CGIs wherever you want, however:
 * </p>
 * <p>
 *   The CGI search path will start at
 *   webAppRootDir + File.separator + cgiPathPrefix
 *   (or webAppRootDir alone if cgiPathPrefix is
 *   null).
 * </p>
 * <p>
 *   cgiPathPrefix is defined by setting
 *   this servlet's cgiPathPrefix init parameter
 * </p>
 *
 * <p>
 *
 * <B>CGI Specification</B>:<br> derived from
 * <a href="http://cgi-spec.golux.com">http://cgi-spec.golux.com</a>.
 * A work-in-progress & expired Internet Draft.  Note no actual RFC describing
 * the CGI specification exists.  Where the behavior of this servlet differs
 * from the specification cited above, it is either documented here, a bug,
 * or an instance where the specification cited differs from Best
 * Community Practice (BCP).
 * Such instances should be well-documented here.  Please email the
 * <a href="mailto:tomcat-dev@jakarta.apache.org">Jakarta Tomcat group [tomcat-dev@jakarta.apache.org]</a>
 * with amendments.
 *
 * </p>
 * <p>
 *
 * <b>Canonical metavariables</b>:<br>
 * The CGI specification defines the following canonical metavariables:
 * <br>
 * [excerpt from CGI specification]
 * <PRE>
 *  AUTH_TYPE
 *  CONTENT_LENGTH
 *  CONTENT_TYPE
 *  GATEWAY_INTERFACE
 *  PATH_INFO
 *  PATH_TRANSLATED
 *  QUERY_STRING
 *  REMOTE_ADDR
 *  REMOTE_HOST
 *  REMOTE_IDENT
 *  REMOTE_USER
 *  REQUEST_METHOD
 *  SCRIPT_NAME
 *  SERVER_NAME
 *  SERVER_PORT
 *  SERVER_PROTOCOL
 *  SERVER_SOFTWARE
 * </PRE>
 * <p>
 * Metavariables with names beginning with the protocol name (<EM>e.g.</EM>,
 * "HTTP_ACCEPT") are also canonical in their description of request header
 * fields.  The number and meaning of these fields may change independently
 * of this specification.  (See also section 6.1.5 [of the CGI specification].)
 * </p>
 * [end excerpt]
 *
 * </p>
 * <h2> Implementation notes</h2>
 * <p>
 *
 * <b>standard input handling</b>: If your script accepts standard input,
 * then the client must start sending input within a certain timeout period,
 * otherwise the servlet will assume no input is coming and carry on running
 * the script.  The script's the standard input will be closed and handling of
 * any further input from the client is undefined.  Most likely it will be
 * ignored.  If this behavior becomes undesirable, then this servlet needs
 * to be enhanced to handle threading of the spawned process' stdin, stdout,
 * and stderr (which should not be too hard).
 * <br>
 * If you find your cgi scripts are timing out receiving input, you can set
 * the init parameter <code></code> of your webapps' cgi-handling servlet
 * to be
 * </p>
 * <p>
 *
 * <b>Metavariable Values</b>: According to the CGI specificion,
 * implementations may choose to represent both null or missing values in an
 * implementation-specific manner, but must define that manner.  This
 * implementation chooses to always define all required metavariables, but
 * set the value to "" for all metavariables whose value is either null or
 * undefined.  PATH_TRANSLATED is the sole exception to this rule, as per the
 * CGI Specification.
 *
 * </p>
 * <p>
 *
 * <b>NPH --  Non-parsed-header implementation</b>:  This implementation does
 * not support the CGI NPH concept, whereby server ensures that the data
 * supplied to the script are preceisely as supplied by the client and
 * unaltered by the server.
 * </p>
 * <p>
 * The function of a servlet container (including Tomcat) is specifically
 * designed to parse and possible alter CGI-specific variables, and as
 * such makes NPH functionality difficult to support.
 * </p>
 * <p>
 * The CGI specification states that compliant servers MAY support NPH output.
 * It does not state servers MUST support NPH output to be unconditionally
 * compliant.  Thus, this implementation maintains unconditional compliance
 * with the specification though NPH support is not present.
 * </p>
 * <p>
 *
 * The CGI specification is located at
 * <a href="http://cgi-spec.golux.com">http://cgi-spec.golux.com</a>.
 *
 * </p>
 * <p>
 *
 * @author Martin T Dengler (original author)
 * @author Amy Roh (for Tomcat)
 * @author Jost Boekemeier (for the PHP/JavaBridge)
 * @since Tomcat 4
 * @since PHP/JavaBridge 2.0.8
 *
 */
public class CGIServlet extends HttpServlet {

    // IO buffer size
    public static final int BUF_SIZE = 8192;

    /* some vars below copied from Craig R. McClanahan's InvokerServlet */

    /** the string manager for this package. */
    /* YAGNI
       private static StringManager sm =
       StringManager.getManager(Constants.Package);
    */

    private static final long serialVersionUID = 3258132448955937337L;

    /** the Context container associated with our web context. */
    public ServletContext context = null;

    /** the debugging detail level for this servlet. */
    private int debug = 0;

    /** the time in ms to wait for the client to send us CGI input data */
    protected int iClientInputTimeout = 100;

    /**
     *  The CGI search path will start at
     *    webAppRootDir + File.separator + cgiPathPrefix
     *    (or webAppRootDir alone if cgiPathPrefix is
     *    null)
     */
    public String cgiPathPrefix = "/WEB-INF/cgi";

    /**
     * Header encoding
     */
    private static final String ASCII = "ASCII";
    private static final String UTF = "UTF-8";

    public HashMap processEnvironment = null;
    private static final File winnt = new File("c:/winnt");
    private static final File windows = new File("c:/windows");
    private static final Class[] EMPTY_PARAM = new Class[0];
    private static final Object[] EMPTY_ARG = new Object[0];
    /**
     * Get the current process environment which will be passed to the sub-process.
     * Requires jdk1.5 or higher. In jdk1.4, where System.getenv() is not available,
     * we allocate an empty map.<p>
     * To add custom environment variables (such as PATH=... or LD_ASSUME_KERNEL=2.4.21, ...),
     * use a custom PhpCgiServlet, for example:<br>
     * <code>
     * public class MyPhpCgiServlet extends PhpCgiServlet {<br>
     * &nbsp;&nbsp;protected HashMap getProcessEnvironment() {<br>
     * &nbsp;&nbsp;&nbsp;&nbsp;HashMap map = new HashMap();<br>
     * &nbsp;&nbsp;&nbsp;&nbsp;map.put("PATH", "/usr/local/bin");<br>
     * &nbsp;&nbsp;&nbsp;&nbsp;return map; <br>
     * &nbsp;&nbsp;}<br>
     * }<br>
     * </code>
     * @return The current process environment.
     */    
    protected HashMap getProcessEnvironment() {
	HashMap defaultEnv = new HashMap();
        String val = null;
	// Bug in WINNT and WINXP.
	// If SystemRoot is missing, php cannot access winsock.
	if(winnt.isDirectory()) val="c:\\winnt";
	else if(windows.isDirectory()) val = "c:\\windows";
	try {
	    String s = getServletConfig().getInitParameter("Windows.SystemRoot");
	    if(s!=null) val=s;
	} catch (Throwable t) {/*ignore*/}
	try {
	    String s = System.getenv("SystemRoot"); 
	    if(s!=null) val=s;
        } catch (Throwable t) {/*ignore*/}
        try {
	    String s = System.getProperty("Windows.SystemRoot");
	    if(s!=null) val=s;
        } catch (Throwable t) {/*ignore*/}
	if(val!=null) defaultEnv.put("SystemRoot", val);
        try {
          Method m = System.class.getMethod("getenv", EMPTY_PARAM);
          Map map = (Map) m.invoke(System.class, EMPTY_ARG);
          defaultEnv.putAll(map);
      } catch (Exception e) {
	  defaultEnv.putAll(php.java.bridge.Util.COMMON_ENVIRONMENT);
      }
      return defaultEnv;
    }

    /**
     * UTF-8 encode a URL parameter.
     * @param parameter
     * @return the URLEncoded parameter
     */
    protected static String encode(String parameter, HttpServlet servlet) {
	try {
	    return URLEncoder.encode(parameter, UTF);
	} catch (UnsupportedEncodingException e) {servlet.log("Error: " + e); }
	return parameter;
    }




    /**
     * Sets instance variables.
     * <P>
     * Modified from Craig R. McClanahan's InvokerServlet
     * </P>
     *
     * @param config                    a <code>ServletConfig</code> object
     *                                  containing the servlet's
     *                                  configuration and initialization
     *                                  parameters
     *
     * @exception ServletException      if an exception has occurred that
     *                                  interferes with the servlet's normal
     *                                  operation
     */
    public void init(ServletConfig config) throws ServletException {

        super.init(config);
        
        processEnvironment = getProcessEnvironment();
        
        // Verify that we were not accessed using the invoker servlet
        String servletName = getServletConfig().getServletName();
        if (servletName == null)
            servletName = "";
        if (servletName.startsWith("org.apache.catalina.INVOKER."))
            throw new UnavailableException
                ("Cannot invoke CGIServlet through the invoker");

        // Set our properties from the initialization parameters
        String value = null;
        try {
            value = getServletConfig().getInitParameter("debug");
            if(value!=null) debug = Integer.parseInt(value);
            value =
                getServletConfig().getInitParameter("cgiPathPrefix");
            if(value!=null) cgiPathPrefix = value;
            value =
                getServletConfig().getInitParameter("iClientInputTimeout");
            if(value!=null) iClientInputTimeout = Integer.parseInt(value);
        } catch (Throwable t) {
            //NOOP
        }
        //log("init: loglevel set to " + debug);

        // Identify the internal container resources we need
        //Wrapper wrapper = (Wrapper) getServletConfig();
        //context = (Context) wrapper.getParent();

	context = config.getServletContext();
        if (debug >= 1) {
            //log("init: Associated with Context '" + context.getPath() + "'");
        }

    }



    /**
     * Provides CGI Gateway service -- delegates to <code>doGet</code>
     *
     * @param  req   HttpServletRequest passed in by servlet container
     * @param  res   HttpServletResponse passed in by servlet container
     *
     * @exception  ServletException  if a servlet-specific exception occurs
     * @exception  IOException  if a read/write exception occurs
     *
     * @see javax.servlet.http.HttpServlet
     *
     */
    protected void doPost(HttpServletRequest req, HttpServletResponse res)
        throws IOException, ServletException {
        doGet(req, res);
    }



    /**
     * Provides CGI Gateway service
     *
     * @param  req   HttpServletRequest passed in by servlet container
     * @param  res   HttpServletResponse passed in by servlet container
     *
     * @exception  ServletException  if a servlet-specific exception occurs
     * @exception  IOException  if a read/write exception occurs
     *
     * @see javax.servlet.http.HttpServlet
     *
     */
    protected void doGet(HttpServletRequest req, HttpServletResponse res)
        throws ServletException, IOException {

        CGIEnvironment cgiEnv = createCGIEnvironment(req, res, getServletContext());

        if (cgiEnv.isValid()) {
            CGIRunner cgi = cgiRunnerFactory.createCGIRunner(cgiEnv);
            if ("POST".equals(req.getMethod())) {
                cgi.setInput(req.getInputStream());
            } 
            String query = req.getQueryString();
            if(query==null) query="";
            cgi.setParams(query);
            cgi.setResponse(res);
            cgi.run();
        }

    } //doGet

    /**
     * Only for internal use.
     * 
     * Returns the port# of the local port
     * @param req The servlet request
     * @return The local port or the value from the server port variable.
     */
    public static int getLocalPort(ServletRequest req) {
	int port = -1;
	try {
	    port = req.getLocalPort(); 
	} catch (Throwable t) {/*ignore*/}
	if(port<=0) port = req.getServerPort();
	return port;
    }

    protected class CGIRunnerFactory {
        protected CGIRunner createCGIRunner(CGIEnvironment cgiEnv) {
            return new CGIRunner(cgiEnv);
	}
    };
    protected CGIRunnerFactory cgiRunnerFactory = new CGIRunnerFactory();

    /**
     * @param req HttpServletRequest
     * @param res HttpServletResponse
     * @param servletContext ServletContext
     * @return cgi environment
     */
    protected CGIEnvironment createCGIEnvironment(HttpServletRequest req, HttpServletResponse res, ServletContext servletContext) {
	CGIEnvironment env = new CGIEnvironment(req, res, servletContext);
	env.init(req, res);
	return env;
    }


    /**
     * Encapsulates the CGI environment and rules to derive
     * that environment from the servlet container and request information.
     *
     * <p>
     * </p>
     *
     * @author   Martin Dengler [root@martindengler.com]
     * @since    Tomcat 4.0
     *
     */
    protected class CGIEnvironment {


        /** context of the enclosing servlet */
        protected ServletContext context = null;

        /** context path of enclosing servlet */
        protected String contextPath = null;

        /** servlet URI of the enclosing servlet */
        protected String servletPath = null;

        /** pathInfo for the current request */
        protected String pathInfo = null;

        /** real file system directory of the enclosing servlet's web app */
        private String webAppRootDir = null;

        /** derived cgi environment */
        protected HashMap env = null;

        /** cgi command to be invoked */
        private String command = null;

        /** cgi command's desired working directory */
        private File workingDirectory = null;

        /** whether or not this object is valid or not */
        protected boolean valid = false;


        /**
         * Creates a CGIEnvironment and derives the necessary environment,
         * query parameters, working directory, cgi command, etc.
         * @param  req       HttpServletRequest for information provided by
         *                   the Servlet API
         * @param res HttpServletResponse
         * @param  context   ServletContext for information provided by the
         *                   Servlet API
         *
         */
        protected CGIEnvironment(HttpServletRequest req,
                                 HttpServletResponse res, ServletContext context) {
            setupFromContext(context);
            setupFromRequest(req);
        }

        public void init(HttpServletRequest req, HttpServletResponse res) {
            this.valid = setCGIEnvironment(req, res);

            if (this.valid) {
		int idx = command.lastIndexOf(File.separator);
		if(idx==-1) idx = command.lastIndexOf("/");
		if(idx==-1) throw new IllegalArgumentException("command is not a file path");
                workingDirectory = new File(command.substring(0,idx));
            }
        }

	/**
         * Uses the ServletContext to set some CGI variables
         *
         * @param  context   ServletContext for information provided by the
         *                   Servlet API
         */
        protected void setupFromContext(ServletContext context) {
            this.context = context;
            this.webAppRootDir = CGIServlet.getRealPath(context,"/");
        }



        /**
         * Uses the HttpServletRequest to set most CGI variables
         *
         * @param  req   HttpServletRequest for information provided by
         *               the Servlet API
         */
        protected void setupFromRequest(HttpServletRequest req) {
            this.contextPath = req.getContextPath();
            this.pathInfo = req.getPathInfo();
            this.servletPath = req.getServletPath();
        }



        /**
         * Resolves core information about the cgi script.
         *
         * <p>
         * Example URI:
         * <PRE> /servlet/cgigateway/dir1/realCGIscript/pathinfo1 </PRE>
         * <ul>
         * <LI><b>path</b> = $CATALINA_HOME/mywebapp/dir1/realCGIscript
         * <LI><b>scriptName</b> = /servlet/cgigateway/dir1/realCGIscript
         * <LI><b>cgiName</b> = /dir1/realCGIscript
         * <LI><b>name</b> = realCGIscript
         * </ul>
         * </p>
         * <p>
         * CGI search algorithm: search the real path below
         *    &lt;my-webapp-root&gt; and find the first non-directory in
         *    the getPathTranslated("/"), reading/searching from left-to-right.
         *</p>
         *<p>
         *   The CGI search path will start at
         *   webAppRootDir + File.separator + cgiPathPrefix
         *   (or webAppRootDir alone if cgiPathPrefix is
         *   null).
         *</p>
         *<p>
         *   cgiPathPrefix is defined by setting
         *   this servlet's cgiPathPrefix init parameter
         *
         *</p>
         *
         * @param pathInfo       String from HttpServletRequest.getPathInfo()
         * @param webAppRootDir  String from context.getRealPath("/")
         * @param contextPath    String as from
         *                       HttpServletRequest.getContextPath()
         * @param servletPath    String as from
         *                       HttpServletRequest.getServletPath()
         * @param cgiPathPrefix  subdirectory of webAppRootDir below which
         *                       the web app's CGIs may be stored; can be null.
         *                       The CGI search path will start at
         *                       webAppRootDir + File.separator + cgiPathPrefix
         *                       (or webAppRootDir alone if cgiPathPrefix is
         *                       null).  cgiPathPrefix is defined by setting
         *                       the servlet's cgiPathPrefix init parameter.
         *
         *
         * @return
         * <ul>
         * <li>
         * <code>path</code> -    full file-system path to valid cgi script,
         *                        or null if no cgi was found
         * <li>
         * <code>scriptName</code> -
         *                        CGI variable SCRIPT_NAME; the full URL path
         *                        to valid cgi script or null if no cgi was
         *                        found
         * <li>
         * <code>cgiName</code> - servlet pathInfo fragment corresponding to
         *                        the cgi script itself, or null if not found
         * <li>
         * <code>name</code> -    simple name (no directories) of the
         *                        cgi script, or null if no cgi was found
         * </ul>
         *
         * @author Martin Dengler [root@martindengler.com]
         * @since Tomcat 4.0
         */
        protected String[] findCGI(String pathInfo, String webAppRootDir,
                                   String contextPath, String servletPath,
                                   String cgiPathPrefix) {
            String path = null;
            String name = null;
            String scriptname = null;
            String cginame = null;

            if ((webAppRootDir != null)
                && (webAppRootDir.lastIndexOf(File.separator) ==
                    (webAppRootDir.length() - 1))) {
		//strip the trailing "/" from the webAppRootDir
		webAppRootDir =
                    webAppRootDir.substring(0, (webAppRootDir.length() - 1));
            }

            if (cgiPathPrefix != null) {
                webAppRootDir = webAppRootDir + File.separator
                    + cgiPathPrefix;
            }

            if (debug >= 2) {
                log("findCGI: path=" + pathInfo + ", " + webAppRootDir);
            }

            File currentLocation = new File(webAppRootDir);
            StringTokenizer dirWalker =
		new StringTokenizer(pathInfo, File.separator);
            if (debug >= 3) {
                log("findCGI: currentLoc=" + currentLocation);
            }
            while (!currentLocation.isFile() && dirWalker.hasMoreElements()) {
                if (debug >= 3) {
                    log("findCGI: currentLoc=" + currentLocation);
                }
                currentLocation = new File(currentLocation,
                                           (String) dirWalker.nextElement());
            }
            if (!currentLocation.isFile()) {
                return new String[] { null, null, null, null };
            } else {
                if (debug >= 2) {
                    log("findCGI: FOUND cgi at " + currentLocation);
                }
                path = currentLocation.getAbsolutePath();
                name = currentLocation.getName();
                cginame =
		    currentLocation.getParent().substring(webAppRootDir.length())
		    + File.separator
		    + name;

                if (".".equals(contextPath)) {
                    scriptname = servletPath + cginame;
                } else {
                    scriptname = contextPath + servletPath + cginame;
                }
            }

            if (debug >= 1) {
                log("findCGI calc: name=" + name + ", path=" + path
                    + ", scriptname=" + scriptname + ", cginame=" + cginame);
            }
            return new String[] { path, scriptname, cginame, name };

        }

        /**
         * Constructs the CGI environment to be supplied to the invoked CGI
         * script; relies heavliy on Servlet API methods and findCGI
         * @param res HttpServletResponse
         * @param    HttpServletRequest request associated with the CGI
         *           invokation
         *
         * @return   true if environment was set OK, false if there
         *           was a problem and no environment was set
         */
        protected boolean setCGIEnvironment(HttpServletRequest req, HttpServletResponse res) {
	    HashMap envp = (HashMap)processEnvironment.clone();
            
            String sPathInfoOrig = null;
            String sPathTranslatedOrig = null;
            String sCGIFullPath = null;
            String sCGIScriptName = null;
            String sCGIFullName = null;
            String sCGIName = null;
            String[] sCGINames;


            sPathInfoOrig = this.pathInfo;
            sPathInfoOrig = sPathInfoOrig == null ? "" : sPathInfoOrig;

            sPathTranslatedOrig = req.getPathTranslated();
            sPathTranslatedOrig =
                sPathTranslatedOrig == null ? "" : sPathTranslatedOrig;

            sCGINames = findCGI(sPathInfoOrig,
                                webAppRootDir,
                                contextPath,
                                servletPath,
                                cgiPathPrefix);

            sCGIFullPath = sCGINames[0];
            sCGIScriptName = sCGINames[1];
            sCGIFullName = sCGINames[2];
            sCGIName = sCGINames[3];

            if (sCGIFullPath == null
                || sCGIScriptName == null
                || sCGIFullName == null
                || sCGIName == null) {
                return false;
            }

            envp.put("SERVER_SOFTWARE", "TOMCAT");
            envp.put("SERVER_NAME", nullsToBlanks(req.getServerName()));
            envp.put("GATEWAY_INTERFACE", "CGI/1.1");
            envp.put("SERVER_PROTOCOL", nullsToBlanks(req.getProtocol()));
            int port = getLocalPort(req);
            Integer iPort = (port == 0 ? new Integer(-1) : new Integer(port));
            envp.put("SERVER_PORT", iPort.toString());
            envp.put("REQUEST_METHOD", nullsToBlanks(req.getMethod()));
            envp.put("SCRIPT_NAME", nullsToBlanks(sCGIScriptName));
            envp.put("QUERY_STRING", nullsToBlanks(req.getQueryString()));
            envp.put("REMOTE_HOST", nullsToBlanks(req.getRemoteHost()));
            envp.put("REMOTE_ADDR", nullsToBlanks(req.getRemoteAddr()));
            envp.put("AUTH_TYPE", nullsToBlanks(req.getAuthType()));
            envp.put("REMOTE_USER", nullsToBlanks(req.getRemoteUser()));
            envp.put("REMOTE_IDENT", ""); //not necessary for full compliance
            envp.put("CONTENT_TYPE", nullsToBlanks(req.getContentType()));
            setPathInfo(req, envp, sCGIFullName);

            /* Note CGI spec says CONTENT_LENGTH must be NULL ("") or undefined
             * if there is no content, so we cannot put 0 or -1 in as per the
             * Servlet API spec.
             */
            int contentLength = req.getContentLength();
            String sContentLength = (contentLength <= 0 ? "" :
                                     (new Integer(contentLength)).toString());
            envp.put("CONTENT_LENGTH", sContentLength);


            Enumeration headers = req.getHeaderNames();
            String header = null;
            while (headers.hasMoreElements()) {
                header = null;
                header = ((String) headers.nextElement()).toUpperCase();
                //REMIND: rewrite multiple headers as if received as single
                //REMIND: change character set
                //REMIND: I forgot what the previous REMIND means
                if ("AUTHORIZATION".equalsIgnoreCase(header) ||
                    "PROXY_AUTHORIZATION".equalsIgnoreCase(header)) {
                    //NOOP per CGI specification section 11.2
                } else if("HOST".equalsIgnoreCase(header)) {
                    String host = req.getHeader(header);
		    int idx =  host.indexOf(":");
		    if(idx < 0) idx = host.length();
                    envp.put("HTTP_" + header.replace('-', '_'),
                             host.substring(0, idx));
                } else {
                    envp.put("HTTP_" + header.replace('-', '_'),
                             req.getHeader(header));
                }
            }

            command = sCGIFullPath;
            envp.put("X_TOMCAT_SCRIPT_PATH", command);  //for kicks

            this.env = envp;

            return true;

        }


        protected void setPathInfo(HttpServletRequest req, HashMap envp, String sCGIFullName) {
          String sPathInfoCGI = null;
          String sPathTranslatedCGI = null;
          /*-
           * PATH_INFO should be determined by using sCGIFullName:
           * 1) Let sCGIFullName not end in a "/" (see method findCGI)
           * 2) Let sCGIFullName equal the pathInfo fragment which
           *    corresponds to the actual cgi script.
           * 3) Thus, PATH_INFO = request.getPathInfo().substring(
           *                      sCGIFullName.length())
           *
           * (see method findCGI, where the real work is done)
           *
           */
          if (pathInfo == null
              || (pathInfo.substring(sCGIFullName.length()).length() <= 0)) {
              sPathInfoCGI = "";
          } else {
              sPathInfoCGI = pathInfo.substring(sCGIFullName.length());
          }
          envp.put("PATH_INFO", sPathInfoCGI);


          /*-
           * PATH_TRANSLATED must be determined after PATH_INFO (and the
           * implied real cgi-script) has been taken into account.
           *
           * The following example demonstrates:
           *
           * servlet info   = /servlet/cgigw/dir1/dir2/cgi1/trans1/trans2
           * cgifullpath    = /servlet/cgigw/dir1/dir2/cgi1
           * path_info      = /trans1/trans2
           * webAppRootDir  = servletContext.getRealPath("/")
           *
           * path_translated = servletContext.getRealPath("/trans1/trans2")
           *
           * That is, PATH_TRANSLATED = webAppRootDir + sPathInfoCGI
           * (unless sPathInfoCGI is null or blank, then the CGI
           * specification dictates that the PATH_TRANSLATED metavariable
           * SHOULD NOT be defined.
           *
           */
          if (sPathInfoCGI != null && !("".equals(sPathInfoCGI))) {
              sPathTranslatedCGI = CGIServlet.getRealPath(context, sPathInfoCGI);
          } else {
              sPathTranslatedCGI = null;
          }
          if (sPathTranslatedCGI == null || "".equals(sPathTranslatedCGI)) {
              //NOOP
          } else {
              envp.put("PATH_TRANSLATED", nullsToBlanks(sPathTranslatedCGI));
          }
	}

	/**
         * Gets derived command string
         *
         * @return  command string
         *
         */
        protected String getCommand() {
            return command;
        }



        /**
         * Gets derived CGI working directory
         *
         * @return  working directory
         *
         */
        protected File getWorkingDirectory() {
            return workingDirectory;
        }



        /**
         * Gets derived CGI environment
         *
         * @return   CGI environment
         *
         */
        protected HashMap getEnvironment() {
            return env;
        }


        /**
         * Gets validity status
         *
         * @return   true if this environment is valid, false
         *           otherwise
         *
         */
        protected boolean isValid() {
            return valid;
        }



        /**
         * Converts null strings to blank strings ("")
         *
         * @param    string to be converted if necessary
         * @return   a non-null string, either the original or the empty string
         *           ("") if the original was <code>null</code>
         */
        protected String nullsToBlanks(String s) {
            return nullsToString(s, "");
        }



        /**
         * Converts null strings to another string
         *
         * @param    string to be converted if necessary
         * @param    string to return instead of a null string
         * @return   a non-null string, either the original or the substitute
         *           string if the original was <code>null</code>
         */
        protected String nullsToString(String couldBeNull,
                                       String subForNulls) {
            return (couldBeNull == null ? subForNulls : couldBeNull);
        }



        /**
         * Converts blank strings to another string
         *
         * @param    string to be converted if necessary
         * @param    string to return instead of a blank string
         * @return   a non-null string, either the original or the substitute
         *           string if the original was <code>null</code> or empty ("")
         */
        protected String blanksToString(String couldBeBlank,
					String subForBlanks) {
            return (("".equals(couldBeBlank) || couldBeBlank == null)
                    ? subForBlanks
                    : couldBeBlank);
        }



    } //class CGIEnvironment



    /**
     * Converts a Hashtable to a String array by converting each
     * key/value pair in the Hashtable to a String in the form
     * "key=value" (hashkey + "=" + hash.get(hashkey).toString())
     *
     * @param  h   Hashtable to convert
     *
     * @return     converted string array
     *
     * @exception  NullPointerException   if a hash key has a null value
     *
     */
    protected static String[] hashToStringArray(Map h)
        throws NullPointerException {
        Vector v = new Vector();
        Iterator e = h.keySet().iterator();
        while (e.hasNext()) {
            String k = e.next().toString();
            v.add(k + "=" + h.get(k));
        }
        String[] strArr = new String[v.size()];
        v.copyInto(strArr);
        return strArr;
    }


    /**
     * Identical to context2.getRealPath(pathInfoCGI). On BEA
     * WebLogic, which has a broken getRealPath() implementation, we
     * use context2.getResource(pathInfoCGI)) instead.
     * @param context2 The servlet context.
     * @param pathInfoCGI  may be "" or "/" for example.
     * @return a valid path or null
     * @throws MalformedURLException 
     */
    public static String getRealPath(ServletContext context2, String pathInfoCGI) {
	String ret = context2.getRealPath(pathInfoCGI);
	if(ret!=null) return ret;

	// The following is the workaround for BEA WebLogic
	boolean stripSlash=false; 
	if(!pathInfoCGI.endsWith("/")) {
	    stripSlash=true;
	    if("".equals(pathInfoCGI)) pathInfoCGI = "/";
	}
	URL url = null;
	try { 
	    url = context2.getResource(pathInfoCGI);
	} catch (MalformedURLException e) {
	    Util.printStackTrace(e);
	    return null;
	}
	if(!"file".equals(url.getProtocol())) return null;
	ret = url.getPath();
	if(stripSlash&&ret.endsWith("/")) ret = ret.substring(0, ret.length()-1);
	return ret.replace('/', File.separatorChar);
    }

    /**
     * Encapsulates the knowledge of how to run a CGI script, given the
     * script's desired environment and (optionally) input/output streams
     *
     * <p>
     *
     * Exposes a <code>run</code> method used to actually invoke the
     * CGI.
     *
     * </p>
     * <p>
     *
     * The CGI environment and settings are derived from the information
     * passed to the constuctor.
     *
     * </p>
     * <p>
     *
     * The input and output streams can be set by the <code>setInput</code>
     * and <code>setResponse</code> methods, respectively.
     * </p>
     *
     * @author    Martin Dengler [root@martindengler.com]
     */

    protected class CGIRunner {

        /** script/command to be executed */
        protected String command = null;

        /** environment used when invoking the cgi script */
        protected HashMap env = null;

        /** working directory used when invoking the cgi script */
        protected File wd = null;

        /** query parameters to be passed to the invoked script */
        protected String params = null;

        /** stdin to be passed to cgi script */
        protected InputStream stdin = null;

        /** response object used to set headers & get output stream */
        protected HttpServletResponse response = null;



        /**
         *  Creates a CGIUtil and initializes its environment, working
         *  directory, and query parameters.
         *  <BR>
         *  Input/output streams (optional) are set using the
         *  <code>setInput</code> and <code>setResponse</code> methods,
         *  respectively.
         *
         * @param  command  string full path to command to be executed
         * @param  env      Hashtable with the desired script environment
         * @param  wd       File with the script's desired working directory
         * @param  params   Hashtable with the script's query parameters
         *
         * @param  response       HttpServletResponse object for setting headers
         *                   based on CGI script output
         *
         */
        protected CGIRunner(CGIEnvironment env) {
            this.command = env.getCommand();
            this.env = env.getEnvironment();
            this.wd = env.getWorkingDirectory();
        }



        /**
	 * @param queryParameters
	 */
	public void setParams(String queryParameters) {
	    this.params=queryParameters;
	}


        /**
         * Sets HttpServletResponse object used to set headers and send
         * output to
         *
         * @param  response   HttpServletResponse to be used
         *
         */
        protected void setResponse(HttpServletResponse response) {
            this.response = response;
        }



        /**
         * Sets standard input to be passed on to the invoked cgi script
         *
         * @param  stdin   InputStream to be used
         *
         */
        protected void setInput(InputStream stdin) {
            this.stdin = stdin;
        }


        protected void addHeader(String line) {
	    try {
		if (debug >= 2) {
		    log("runCGI: addHeader(\"" + line + "\")");
		}
		if (line.startsWith("Status")) {
		    line = line.substring(line.indexOf(":") + 1).trim();
		    int i = line.indexOf(' ');
		    if (i>0)
			line = line.substring(0,i);

		    response.setStatus(Integer.parseInt(line));
		} else {
                    response.addHeader
                        (line.substring(0, line.indexOf(":")).trim(),
                         line.substring(line.indexOf(":") + 1).trim());
		}
	    } catch (ArrayIndexOutOfBoundsException e) {/*not a valid header*/}
	    catch (StringIndexOutOfBoundsException e){/*not a valid header*/}
        }

        /**
         * Executes a CGI script with the desired environment, current working
         * directory, and input/output streams
         *
         * <p>
         * This implements the following CGI specification recommedations:
         * <UL>
         * <LI> Servers SHOULD provide the "<code>query</code>" component of
         *      the script-URI as command-line arguments to scripts if it
         *      does not contain any unencoded "=" characters and the
         *      command-line arguments can be generated in an unambiguous
         *      manner.
         * <LI> Servers SHOULD set the AUTH_TYPE metavariable to the value
         *      of the "<code>auth-scheme</code>" token of the
         *      "<code>Authorization</code>" if it was supplied as part of the
         *      request header.  See <code>getCGIEnvironment</code> method.
         * <LI> Where applicable, servers SHOULD set the current working
         *      directory to the directory in which the script is located
         *      before invoking it.
         * <LI> Server implementations SHOULD define their behavior for the
         *      following cases:
         *     <ul>
         *     <LI> <u>Allowed characters in pathInfo</u>:  This implementation
         *             does not allow ASCII NUL nor any character which cannot
         *             be URL-encoded according to internet standards;
         *     <LI> <u>Allowed characters in path segments</u>: This
         *             implementation does not allow non-terminal NULL
         *             segments in the the path -- IOExceptions may be thrown;
         *     <LI> <u>"<code>.</code>" and "<code>..</code>" path
         *             segments</u>:
         *             This implementation does not allow "<code>.</code>" and
         *             "<code>..</code>" in the the path, and such characters
         *             will result in an IOException being thrown;
         *     <LI> <u>Implementation limitations</u>: This implementation
         *             does not impose any limitations except as documented
         *             above.  This implementation may be limited by the
         *             servlet container used to house this implementation.
         *             In particular, all the primary CGI variable values
         *             are derived either directly or indirectly from the
         *             container's implementation of the Servlet API methods.
         *     </ul>
         * </UL>
         * </p>
         *
         * @exception IOException if problems during reading/writing occur
         * @throws ServletException
         *
         * @see    java.lang.Runtime#exec(String command, String[] envp,
         *                                File dir)
         */
        protected void run() throws IOException, ServletException {
            if (debug >= 1 ) {
                log("runCGI(envp=[" + env + "], command=" + command + ")");
            }

            if ((command.indexOf(File.separator + "." + File.separator) >= 0)
                || (command.indexOf(File.separator + "..") >= 0)
                || (command.indexOf(".." + File.separator) >= 0)) {
                throw new IOException(this.getClass().getName()
                                      + "Illegal Character in CGI command "
                                      + "path ('.' or '..') detected.  Not "
                                      + "running CGI [" + command + "].");
            }
	    //create query arguments
            StringBuffer cmdAndArgs = new StringBuffer(command);
            if(params!=null && params.trim().length()>0){
		cmdAndArgs.append(" ");
		cmdAndArgs.append(params.replace('&', ' '));
            }

            Runtime rt = Runtime.getRuntime();
            Process proc = null;
            InputStream natIn = null;
            OutputStream natOut = null;
            InputStream in = null;
            OutputStream out = null;
	    try {
		proc = rt.exec(cmdAndArgs.toString(), hashToStringArray(env), wd);
		(new Util.Thread("CGIErrorReader") {
			private Process proc;
			public Thread init(Process proc) {
			    this.proc = proc;
			    return this;
			}
			public void run() {
			    InputStream in = proc.getErrorStream();
			    int c;
			    try { while((c=in.read())!=-1) System.err.write(c); } catch (IOException e) {/*ignore*/}
			    try { in.close();} catch (IOException e1) {/*ignore*/}
			}
		    }).init(proc).start();

		natIn = proc.getInputStream();
		natOut = proc.getOutputStream();
		in = stdin;
		out = response.getOutputStream();
        
		String line = null;
		byte[] buf = new byte[BUF_SIZE];// headers cannot be larger than this value!
		int i=0, n, s=0;
		boolean eoh=false;

		// the post variables
		if(in!=null) {
		    while((n=in.read(buf))!=-1) {
			natOut.write(buf, 0, n);
		    }
		}
		natOut.flush();
		
		// the header and content
		while((n = natIn.read(buf, i, buf.length-i)) !=-1 ) {
		    int N = i + n;
		    // header
		    while(!eoh && i<N) {
			switch(buf[i++]) {
			
			case '\n':
			    if(s+2==i && buf[s]=='\r') {
				eoh=true;
			    } else {
				line = new String(buf, s, i-s-2, ASCII);
				addHeader(line);
				s=i;
			    }
			}
		    }
		    // body
		    if(eoh) {
			if(out!=null && i<N) out.write(buf, i, N-i);
			i=0;
		    }
		}
		proc=null;
	    }  finally {
		if(in!=null) try {in.close();} catch (IOException e) {}
		if(natIn!=null) try {natIn.close();} catch (IOException e) {}
		if(natOut!=null) try {natOut.close();} catch (IOException e) {}

		if(proc!=null) try {proc.destroy(); } catch (Exception e) {}
	    }
	}
    } //class CGIRunner

} //class CGIServlet
