<?php
/**
* @package phpDocumentor
*/
//
// PhpDoc, a program for creating javadoc style documentation from php code
// Copyright (C) 2000-2001 Joshua Eichorn
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//

//
// Copyright 2000-2001 Joshua Eichorn
// Email jeichorn@phpdoc.org
// Web 		http://phpdoc.org/
// Mirror 	http://phpdocu.sourceforge.net/
// Project    	http://sourceforge.net/projects/phpdocu/
//

/**
* @see Linker::getFormattedInheritedMethods(), Linker::getFormattedInheritedVars()
*/
function sortInherited($a, $b)
{
	fancy_debug('a',$a);
	fancy_debug('b',$b);
	list($aparent,$at) = each ($a);
	list($bparent,$bt) = each ($b);
	list($aval,) = each($at);
	list($bval,) = each($bt);
	if (strnatcasecmp($aparent,$bparent)==0)
	{
		return strnatcasecmp($aval,$bval);
	}
	return strnatcasecmp($aparent,$bparent);
}

/** The Linker Class
 *
 *  The Class tree/see link/inheritance engine for Php Documentor
 *
 *  @author Gregory Beaver <cellog@users.sourceforge.net>
 *	@version	$Id: Linker.inc,v 1.20 2002/04/24 03:29:16 CelloG Exp $
 *  @copyright 2002 Gregory beaver
 *  @package phpDocumentor
 */
class Linker extends Publisher
{
	/**
	*@access private
	*/
	var $last	= "";
	
	/**
	* alphabetical index of elements
	* @see Linker::generateElementIndex()
	*/
	var $elements = array();
	
	/**
	* alphabetical index of elements sorted by package
	* @see Linker::generateElementIndex()
	*/
	var $pkg_elements = array();
	
	var $package = 'default';
	
	var $subpackage = '';
	
	/**
	* suppress display of private elements marked with @access private in output docs
	*/
	var $parsePrivate = false;
	
	/**
	* @access private
	*/
	var $private_class = false;

	/**
	* Render object, used to find the default object to look for a see reference
	*
	* @see Linker::getLink()
	*/
	var $render;
	
	/**
	* current procedural directory, used to set up package directory links (all files are in a directory with the same name
	* as their package, or 'default' if no package is specified
	*
	* @see Linker::HandleEvent()
	*/
	var $dir = "";
	
	/**
	* current page, used to reset the parent variable when the parser leaves a class definition
	*
	* @see $parent
	* @see Linker::HandleEvent()
	*/
	var $cur_page = "";

	/**
	* current class directory, used to set up package directory links (all files are in a directory with the same name
	* as their package, or 'default' if no package is specified
	*
	* @see Linker::HandleEvent()
	*/
	var $classdir = "";

	/**
	* array of class inheritance.
	*
	* Format: Classname => parent classname
	* @see Linker::generateClassTreeFromClass(), Linker::getInheritedVars(), Linker::getInheritedMethods(), Linker::getFormattedInheritedVars(), Linker::getFormattedInheritedMethods(), Linker::getLink()
	*/
	var $class_parents_by_name = array();

	/**
	* array of class inheritance indexed by parent class
	*
	* Format: ParentClassname => array(Child1name,Child2name)
	* @see Linker::getRootTree()
	*/
	
	var $class_children = array();
	
	/**
	* array of class methods.
	*
	* Format: Classname => method
	* @see Linker::getLink()
	*/
	var $methods = array();

	/**
	* array of class variables.
	*
	* Format: Classname => var
	* @see Linker::getLink()
	*/
	var $vars = array();

	/**
	* array of class links.
	*
	* Format: Classname => html <a> link to show this class's documentation
	* @see Linker::getClassLink()
	*/
	var $class_links = array();

	/**
	* array of class packages.
	*
	* Format: Classname => array(package, subpackage)
	*/
	var $class_packages = array();

	/**
	* array of function links.
	*
	* Format: Functionname => html <a> link to show this function's documentation
	* @see Linker::getFunctionLink()
	*/
	var $function_links = array();

	/**
	* array of method links.
	*
	* Format: Methodname => html <a> link to show this function's documentation
	* @see Linker::getMethodLink()
	*/
	var $method_links = array();

	/**
	* array of pages by function
	*
	* Format: function name => parent page
	* @see Linker::getLink()
	*/
	var $func_page = array();

	/**
	* array of classes by method
	*
	* Format: method name => parent class
	* @see Linker::getLink()
	*/
	var $method_class = array();

	/**
	* array of var links.
	*
	* Format: $ClassVariableName => html <a> link to show this variable's documentation
	* @see Linker::getVarLink()
	*/
	var $var_links = array();

	/**
	* array of constant links.
	*
	* Format: $ConstantName => html <a> link to show this variable's documentation
	* @see Linker::getConstLink()
	*/
	var $define_links = array();

	// bunches of vars used as we build files up from templates
	var $page	= array();
	var $constants	= array();

	/**
	* The extenstion that the files will be output with
	*/
	var $outputExt	=	".html";

	/**
	* The parent object
	*/
	var $parent	=	"";

	/**
	* The last type of element handled
	*/
	var $lasttype = '';
	
	/**
	* The number of times a local link must add ../ to the path to get the correct link
	*/
	var $dir_level = 1;

	/**
	* The number of times a local link must add ../ to the path to get the correct link
	*/
	var $classdir_level = 1;

	/**
	* The current type of element handled
	*/
	var $type = '';

	function Linker()
	{
	}

	/**
	* Parse the entire project for class hierarchy and file location in order to create clickable links in the documentation.
	*
	* {@link Parser::parse()} passes the current event, which is either PHPDOC_EVENT_NEWSTATE or an object 
	* @param integer $event current event
	* @param mixed $data either the new state constant or an object containing data parsed from the file
	* @see Linker::getLink(), Linker::getFormattedInheritedMethods(), Linker::getFormattedInheritedVars()
	* @see Data, DataFile, DataDefine, DataDocBlock, DataVar, DataFunction, DataClass, DataPage
	*/
	// $data is always a class from DataTypes
	function HandleEvent($event,$data)
	{
		if ($event == PHPDOC_EVENT_NEWSTATE)
		{
			if ($data==STATE_END_CLASS)
			{
				$this->parent = $this->cur_page;
				$this->private_class = false;
			} elseif ($data == PHPDOC_EVENT_END_PAGE)
			{
				$this->package = $this->dir = 'default';
				$this->subpackage = '';
			}
//			echo $data . "\n";
		} 
		 else 
		{
			$this->lasttype = $this->type;
			$type = $data->getType();
			$this->type = $type;
			//echo $type . "\n";
			
			switch($type)
			{
				case "include":
					unset($this->last);
					unset($data);
				break;
				case "docblock":
					if ($this->lasttype == "docblock" && isset($this->page_template_would_be_set))
					{
						$this->page_template_would_be_set = 0;
						// add this procedural page to the index
						if ($this->last->getKeyword('package'))
						{
							$this->dir = $this->package = $this->last->getKeyword('package');

							if ($this->last->getKeyword('subpackage'))
							{
								$this->dir_level = 2;
								$this->dir .= PATH_DELIMITER . $this->subpackage = $this->last->getKeyword('subpackage');
							} else {
								$this->subpackage = '';
							}
						} else
						{
							$this->dir = $this->package = 'default';
							$this->subpackage = '';
						}
						$this->dir = $this->last->getDirFromPackage();
						$this->page = array();
					} 
					$this->last = $data;
					unset($data);
				break;
				case "page":

					if (count($this->page) >0)
					{
						$this->page = array();
					}
					// Create a new template, this is where non class stuff goes
					$this->page_template_would_be_set = 1;
					$this->package = $this->dir = 'default';
					// Setting stuff about the page
					$this->cur_page = $this->page['name'] = "_" . $data->getName();
					$this->parent = "_" . $data->getName();

				break;
				case "class":
					if (empty($this->last))
					{
						// we don't have a docblock, create an empty one to get rid of errors
						$this->last = new DataDocblock();
						$this->classdir = 'default';
						$this->classdir_level = 1;
						$this->package = 'default';
						$this->subpackage = '';
					}
					if ($access = $this->last->getKeyword('access'))
					{
						if (($access == 'private') && (!$this->parsePrivate))
						{
							$access = false;
							$this->parent = $data->getName();
							$this->private_class = true;
							unset($this->last);
							break;
						}
					}
					$this->classdir = $this->last->getDirFromPackage();
					if ($this->last->getKeyword('package'))
					{
						$this->classdir_level = 1;
						$this->package = $this->last->getKeyword('package');

						if ($this->last->getKeyword('subpackage'))
						{
							$this->classdir_level = 2;
							$this->subpackage = $this->last->getKeyword('subpackage');
						} else {
							$this->subpackage = '';
						}
					} else
					{
						$this->classdir_level = 1;
						$this->package = 'default';
						$this->subpackage = '';
					}
					$func['type'] = 'class';
					$func['name'] = $data->getName();
					$this->elements[substr(strtolower($data->getName()),0,1)][] = $func;
					$this->pkg_elements[$this->package][$this->subpackage][substr(strtolower($data->getName()),0,1)][] = $func;
					$this->class_packages[$data->getName()] = array($this->package,$this->subpackage);
					$this->class_links[$data->getName()] = $this->classdir . SMART_PATH_DELIMITER . $data->getName() . $this->outputExt;
					if ($parent = $data->getExtends())
					{
						$this->class_parents_by_name[$data->getName()] = $data->getExtends();
						$this->class_children[$data->getExtends()][] = $data->getName();
					} else
					{
						$this->class_parents_by_name[$data->getName()] = '';
					}
					$this->parent = $data->getName();

					unset($this->last);
				break;
				case "define":
					if (empty($this->last))
					{
						// we don't have a docblock, create an empty one to get rid of errors
						$this->last = new DataDocblock();
						$this->last->dir = $this->dir;
					}
					if ($access = $this->last->getKeyword('access'))
					{
						if (($access == 'private') && (!$this->parsePrivate))
						{
							unset($this->last);
							break;
						}
					}
					$data->setDir($this->last);
					$func['type'] = 'define';
					$func['name'] = $data->getName();
					$func['page'] = $this->parent;
					$func['package'] = $this->package;
					$this->define_links[$this->package][$data->getName()] = $this->dir . SMART_PATH_DELIMITER . $this->parent.'.html#'.$data->getName();
					$this->elements[substr(strtolower($data->getName()),0,1)][] = $func;
					$this->pkg_elements[$this->package][$this->subpackage][substr(strtolower($data->getName()),0,1)][] = $func;
					$data->setParent($this->parent);	

					$this->defines[$data->getName()] = array(
								"package"		=> $this->package,
								"constant_value" 	=> $data->getValue(), 
								"sdesc" 		=> $this->last->getShortDesc(),
								"desc" 			=> $this->last->getDesc(),
								"inner_loop"		=> array("docblock" => $this->last->listKeywords())
							);
					unset($this->last);
				break;
				case "function":

					if (empty($this->last))
					{
						// we don't have a docblock, create an empty one to get rid of errors
						$this->last = new DataDocblock();
					}
					if ($this->parent != $this->cur_page)
					{
						if (($this->private_class) && (!$this->parsePrivate))
						{
							unset($this->last);
							break;
						}
					}
					if ($access = $this->last->getKeyword('access'))
					{
						if (($access == 'private') && (!$this->parsePrivate))
						{
							unset($this->last);
							break;
						}
					}
					$data->setParent($this->parent);	
					if ($this->parent==$this->cur_page)
					{
						$func['type'] = 'function';
						$func['name'] = $data->getName();
						$func['page'] = $this->parent;
						$func['package'] = $this->package;
						$this->elements[substr(strtolower($data->getName()),0,1)][] = $func;
						$this->pkg_elements[$this->package][$this->subpackage][substr(strtolower($data->getName()),0,1)][] = $func;
						$this->func_page[$data->getName()] = $this->parent;
						$this->function_links[$this->package][$data->getName()] = $this->dir . SMART_PATH_DELIMITER . $this->parent.'.html#'.$data->getName();
					} else
					{
						$func['type'] = 'method';
						$func['name'] = $data->getName();
						$func['class'] = $this->parent;
						$this->elements[substr(strtolower($data->getName()),0,1)][] = $func;
						$this->pkg_elements[$this->package][$this->subpackage][substr(strtolower($data->getName()),0,1)][] = $func;
						$this->method_class[$data->getName()] = $this->parent;
						$this->method_links[$this->parent][$data->getName()] = $this->classdir . SMART_PATH_DELIMITER . $this->parent.'.html#'.$data->getName();
					}

					$function_call = $data->getName() . " ( ";
					$tmp = 0;
					foreach($data->listParams() as $param)
					{
						if ($tmp == 0)
						{
							$tmp = 1;
						} else {
							$function_call .= ", ";
						}
						if (!empty($param[1]))
						{
							$function_call .= "[$param[0] = $param[1]]";
						} else {
							$function_call .= $param[0];
						}
					}
					$function_call .= " )";

					if ($this->parent != $this->cur_page)
					{
						$this->methods[$this->parent][$data->getName()] = 
							array(
								"ifunction_name" 	=> $data->getName(),
								"ifunction_sdesc" 	=> trim($this->last->getShortDesc()),
								"ifunction_call" 	=> $function_call,
								"ipage" => $this->parent . $this->outputExt
								);
					} else
					{
						$this->functions[$this->parent][$data->getName()] = array(
								"ifunction_name" 	=> $data->getName(),
								"ifunction_sdesc" 	=> trim($this->last->getShortDesc()),
								"ifunction_call" 	=> $function_call,
								"ipage" => $this->parent . $this->outputExt
								);
					}

					unset($this->last);
				break;
				case "var":
					if (empty($this->last))
					{
						// we don't have a docblock, create an empty one to get rid of errors
						$this->last = new DataDocblock();
						$this->last->dir = $this->classdir;
					}
					if (($this->private_class) && (!$this->parsePrivate))
					{
						unset($this->last);
						break;
					}
					if ($access = $this->last->getKeyword('access'))
					{
						if (($access == 'private') && (!$this->parsePrivate))
						{
							unset($this->last);
							break;
						}
					}
					$data->setDir($this->last);
					$func['type'] = 'variable';
					$func['name'] = substr($data->getName(),1);
					$func['class'] = $this->parent;
					$this->elements[substr(strtolower($data->getName()),1,1)][] = $func;
					$this->pkg_elements[$this->package][$this->subpackage][substr(strtolower($data->getName()),1,1)][] = $func;
					$this->var_links[$this->parent][$data->getName()] = $this->classdir . SMART_PATH_DELIMITER . $this->parent.'.html#'.$data->getName();
					$this->vars[$this->parent][$data->getName()] = array(
							"ivar_name" => $data->getName(),
							"ivar_default" => $data->getValue(), 
							"ivar_sdesc" => trim($this->last->getShortDesc()),
							"ipage" => $this->parent . $this->outputExt
							);

					unset($this->last);
				break;
			}

		}
	}
	
	/**
	* Generate alphabetical index of all elements
	*
	* @see Linker::$elements, Linker::HandleEvent()
	*/
	function generateElementIndex()
	{
		$elementindex = array();
		unset($this->elements[0]);
		uksort($this->elements,'strnatcasecmp');
		while(list($letter,) = each($this->elements))
		{
			uasort($this->elements[$letter],array($this,"elementCmp"));
			$elindex['letter'] = $letter;
			for($i=0;$i<count($this->elements[$letter]);$i++)
			{
				switch($this->elements[$letter][$i]['type'])
				{
					case 'class':
						$elindex['inner_loop']['index'][]['listing'] = $this->getClassLink($this->elements[$letter][$i]['name'],0,'class '.$this->elements[$letter][$i]['name']).'<br>';
					break;
					case 'define':
						$elindex['inner_loop']['index'][]['listing'] = $this->getConstLink($this->elements[$letter][$i]['package'],$this->elements[$letter][$i]['name'],'constant '.$this->elements[$letter][$i]['name']).'<br>';
					break;
					case 'function':
						$elindex['inner_loop']['index'][]['listing'] = $this->getFunctionLink($this->elements[$letter][$i]['package'],$this->elements[$letter][$i]['name'],'function '.$this->elements[$letter][$i]['name'].'()').'<br>';
					break;
					case 'method':
						$elindex['inner_loop']['index'][]['listing'] = $this->getMethodLink($this->elements[$letter][$i]['class'],$this->elements[$letter][$i]['name'],'method '.$this->elements[$letter][$i]['class'].'::'.$this->elements[$letter][$i]['name'].'()').'<br>';
					break;
					case 'variable':
						$elindex['inner_loop']['index'][]['listing'] = $this->getVarLink($this->elements[$letter][$i]['class'],$this->elements[$letter][$i]['name'],'variable '.$this->elements[$letter][$i]['class'].'::$'.$this->elements[$letter][$i]['name']).'<br>';
					break;
				}
			}
			$elementindex[] = $elindex;
			$elindex = array();
		}
		return $elementindex;
	}
	
	/**
	* Generate alphabetical index of all elements by package and subpackage
	*
	* @param string $package name of a package
	* @see Linker::$pkg_elements, Linker::HandleEvent(), Linker::generatePkgElementIndexes()
	*/
	function generatePkgElementIndex($package)
	{
		$elementindex = array();
//		unset($this->elements[0]);
		uksort($this->pkg_elements[$package],'strnatcasecmp');
		$subp = '';
		foreach($this->pkg_elements[$package] as $subpackage => $els)
		{
			if (!empty($subpackage)) $subp = " (<b>subpackage:</b> $subpackage)"; else $subp = '';
			uksort($els,'strnatcasecmp');
			while(list($letter,) = each($els))
			{
				uasort($els[$letter],array($this,"elementCmp"));
				$elindex[$letter]['letter'] = $letter;
				for($i=0;$i<count($els[$letter]);$i++)
				{
					switch($els[$letter][$i]['type'])
					{
						case 'class':
							$elindex[$letter]['inner_loop']['index'][]['listing'] = $this->getClassLink($els[$letter][$i]['name'],0,'class '.$els[$letter][$i]['name'])."$subp<br>";
						break;
						case 'define':
							$elindex[$letter]['inner_loop']['index'][]['listing'] = $this->getConstLink($els[$letter][$i]['package'],$els[$letter][$i]['name'],'constant '.$els[$letter][$i]['name'])."$subp<br>";
						break;
						case 'function':
							$elindex[$letter]['inner_loop']['index'][]['listing'] = $this->getFunctionLink($els[$letter][$i]['package'],$els[$letter][$i]['name'],'function '.$els[$letter][$i]['name'].'()')."$subp<br>";
						break;
						case 'method':
							$elindex[$letter]['inner_loop']['index'][]['listing'] = $this->getMethodLink($els[$letter][$i]['class'],$els[$letter][$i]['name'],'method '.$els[$letter][$i]['class'].'::'.$els[$letter][$i]['name'].'()')."$subp<br>";
						break;
						case 'variable':
							$elindex[$letter]['inner_loop']['index'][]['listing'] = $this->getVarLink($els[$letter][$i]['class'],$els[$letter][$i]['name'],'variable '.$els[$letter][$i]['class'].'::$'.$els[$letter][$i]['name'])."$subp<br>";
						break;
					}
				}
			}
		}
		while(list(,$tempel) = each($elindex))
		$elementindex[] = $tempel;
		return $elementindex;
	}
	
	/**
	*
	* @see Linker::generatePkgElementIndex()
	*/
	function generatePkgElementIndexes()
	{
		$packages = array();
		$package_names = array();
		$pkg = array();
		foreach($this->pkg_elements as $package => $trash)
		{
			$pkgs['package'] = $package;
			$pkg['package'] = $package;
			$pkg['inner_loop']['pindex'] = $this->generatePkgElementIndex($package);
			$packages[] = $pkg;
			$package_names[] = $pkgs;
			unset($pkgs);
			unset($pkg);
		}
		return array($packages,$package_names);
	}
	
	/**
	* does a nat case sort on the specified second level value of the array
	*
	* @param	mixed	$a
	* @param	mixed	$b
	* @return	int
	* @see		Linker::generateElementIndex()
	*/
	function elementCmp ($a, $b)
	{
		return strnatcasecmp($a['name'], $b['name']);
	}

	/**
	* Generate all simple class tree arrays
	*
	* @return	array	array(root classes),array([parent] => [child])
	*/
	function generateClassTree()
	{
		$roots = array();
		foreach($this->class_parents_by_name as $child => $parent)
		{
			if (!$parent) $roots[] = $child;
		}
		$tree = array_flip($this->class_parents_by_name);
		foreach($roots as $root)
		{
			while ($tree[$root])
			{
				$branches[$root] = $tree[$root];
				$root = $tree[$root];
			}
		}
		return array($roots,$branches);
	}
	
	/**
	* returns a template-enabled array of class trees
	* 
	* @param	string	$package	package to generate a class tree for
	* @see Linker::$class_children, Linker::getRootTree()
	*/
	function generateFormattedClassTrees($package)
	{
		$roots = $trees = array();
		foreach($this->class_parents_by_name as $child => $parent)
		{
			if (!$parent)
			{
				if ($this->class_packages[$child][0] == $package) $roots[] = $child;
			}
		}
		for($i=0;$i<count($roots);$i++)
		{
			$trees[] = array('class' => $roots[$i],'class_tree' => "<ul>\n".$this->getRootTree($roots[$i])."</ul>\n");
		}
		return $trees;
	}
	
	/**
	* walk through the $class_children array, return formatted class list
	*
	* @param string $class name of a class with children
	* @see Linker::$class_children, Linker::generateFormattedClassTrees()
	*/
	function getRootTree($class)
	{
		$my_tree = "<li>";
		if (!isset($this->class_children[$class])) return '<li>'.$this->getClassLink($class)."</li>\n";
		$my_tree .= $this->getClassLink($class);
		if ($this->class_children[$class])
		$my_tree .= '<ul>';
		for($i=0;$i<count($this->class_children[$class]);$i++)
		{
			$my_tree .= $this->getRootTree($this->class_children[$class][$i]);
		}
		if ($this->class_children[$class])
		$my_tree .= "</ul>\n";
		$my_tree .= "</li>\n";
		return $my_tree;
	}
	
	/**
	* Generate a simple class tree array from $branch
	*
	* @param	string	$branch name of the child class
	* @return	array	[parent] => [child]
	* @see Linker::generateFormattedClassTree()
	*/
	function generateClassTreeFromClass($branch)
	{
		$tree = $this->class_parents_by_name;
		if (!$tree) return false;
		while(isset($tree[$branch]) && $tree[$branch])
		{
			$branches[$branch] = $tree[$branch];
			$branch = $tree[$branch];
		}
		$branches[$branch] = 0;
		return $branches;
	}
	
	/**
	* returns a string containing the class inheritance tree from the root object to the class
	*
	* @param string	classname
	* @see Render::HandleEvent()
	*/
	
	function generateFormattedClassTree($class)
	{
		$tree = $this->generateClassTreeFromClass($class);
		$out = '';
		if (count($tree) - 1)
		{
			$result = array($class);
			$parent = $tree[$class];
			while ($parent)
			{
				$result[] = 
					$this->getClassLink($parent,1)."\n" .
					"%s|\n" .
					"%s--";
				$parent = $tree[$parent];
			}
			$nbsp = '   ';
			for($i=count($result) - 1;$i>=0;$i--)
			{
				$my_nbsp = '';
				for($j=0;$j<count($result) - $i;$j++) $my_nbsp .= $nbsp;
				$out .= sprintf($result[$i],$my_nbsp,$my_nbsp);
			}
			return "<pre>$out</pre>";
		} else
		{
			return $class;
		}
	}
	
	/**
	* Return simple list of inherited methods
	*
	* @see Linker::getFormattedInheritedMethods()
	*/
	
	function getInheritedMethods($child)
	{
		$tree = $this->generateClassTreeFromClass($child);
		$parent = $child;
		$meths = array();
		$vars_ = array();
		while(isset($tree[$parent]))
		{
			if (isset($this->methods[$tree[$parent]]))
			{
				$meths[][$tree[$parent]] = $this->methods[$tree[$parent]];
				if(is_array($this->methods[$tree[$parent]]))
				while(list($var,) = each($this->methods[$tree[$parent]]))
				{
					if (!isset($vars_[$var]))
					$vars_[$var] = $tree[$parent];
				}
			}
			$parent = $tree[$parent];
		}
		return array($meths,$vars_);
	}
	
	/**
	* Return simple list of inherited variables
	*
	* @see Linker::getFormattedInheritedVars()
	*/
	
	function getInheritedVars($child)
	{
		$tree = $this->generateClassTreeFromClass($child);
		$parent = $child;
		$vars = array();
		$vars_ = array();
		while(isset($tree[$parent]) && $tree[$parent])
		{
			if (isset($this->vars[$tree[$parent]]))
			{
				$vars[][$tree[$parent]] = $this->vars[$tree[$parent]];
				while(list($var,) = each($this->vars[$tree[$parent]]))
				{
					if (!isset($vars_[$var]))
					$vars_[$var] = $tree[$parent];
				}
			}
			$parent = $tree[$parent];
		}
		return array($vars,$vars_);
	}
	
	/**
	* Return template-enabled list of inherited variables
	*
	* This function splits inherited variables into groupings by parent class, and it also understands over-riding.
	* If class foo defines variable $bar = 2 and class foo1 overrides the variable by defining $bar = 4, this function
	* will correctly say that the variable $bar is inherited from class foo1 and has default value 4
	* @see Render::HandleEvent(), Linker::HandleEvent()
	*/
	
	function getFormattedInheritedVars($child)
	{
		list($ivs,$cut) = $this->getInheritedVars($child);
		if (!isset($ivs[0])) return false;
		for($i = 0;$i < count($ivs);$i++)
		{
			list($a,$b) = each($ivs[$i]);
			$vars[$a] = $b;
		}
		$results = array();
		foreach($vars as $class => $var)
		{
			ksort($var,"strnatcasecmp");
			$a = '';
			for($i=0;$i<$this->classdir_level;$i++) $a .= '..'.PATH_DELIMITER;
			$result['parent_class'] = $this->getClassLink($class,1);
			foreach($var as $name => $info)
			{
				if ($cut[$info['ivar_name']])
				{
					if ($class==$cut[$info['ivar_name']])
					{
						if (isset($this->class_links[$class]))
						{
							$info['ipath'] = $a . $this->class_links[$class];
						}
						$result['inner_loop']["ivars"][] = $info;
					}
				} else
				{
					if (isset($this->class_links[$class]))
					{
						$info['ipath'] = $a . $this->class_links[$class];
					}
					$result['inner_loop']["ivars"][] = $info;
				}
			}
			if (count($result))
			$results[] = $result;
			$result = array();
		}
		return $results;
	}
	
	/**
	* Return template-enabled list of inherited methods
	*
	* This function splits inherited methods into groupings by parent class, and it also understands over-riding.
	* If class foo defines method bar() and class foo1 overrides the method by defining bar() again, this function
	* will correctly say that the method bar() is inherited from class foo1 and not from class foo
	* @see Render::HandleEvent(), Linker::HandleEvent()
	*/

	function getFormattedInheritedMethods($child)
	{
		list($ivs,$cut) = $this->getInheritedMethods($child);
		if (!isset($ivs[0])) return false;
		for($i = 0;$i < count($ivs);$i++)
		{
			list($a,$b) = each($ivs[$i]);
			$vars[$a] = $b;
		}
		$result = array();
		foreach($vars as $class => $var)
		{
			ksort($var,"strnatcasecmp");
			$result['parent_class'] = $this->getClassLink($class,1);
			$a = '';
			for($i=0;$i<$this->classdir_level;$i++) $a .= '..'.PATH_DELIMITER;
			foreach($var as $name => $info)
			{
				if ($cut[$info['ifunction_name']])
				{
					if ($class==$cut[$info['ifunction_name']])
					{
						if (isset($this->class_links[$class]))
						{
							$info['ipath'] = $a . $this->class_links[$class];
						}
						$result['inner_loop']["ifunctions"][] = $info;
					}
				} else
				{
					if (isset($this->class_links[$class]))
					{
						$info['ipath'] = $a . $this->class_links[$class];
					}
					$result['inner_loop']["ifunctions"][] = $info;
				}
			}
			if (count($result))
			$results[] = $result;
			$result = array();
		}
		return $results;
	}
	
	function getClassPackage($class)
	{
		while(isset($this->class_parents_by_name[$class]) && !empty($this->class_parents_by_name[$class]))
		{
			$class = $this->class_parents_by_name[$class];
		}
		if (isset($this->class_packages[$class]))
		return $this->class_packages[$class];
		else return false;
	}
	
	/**
	* generates an html <a> tag for viewing documentation of $class
	*
	* @see Render::HandleEvent()
	*/
	
	function getClassLink($class,$local = false,$expr = '')
	{
		if (empty($expr)) $expr = $class;
		$a = '';
		if ($local && isset($this->class_links[$class]))
		{
			for($i=0;$i<$this->classdir_level;$i++) $a .= '..'.PATH_DELIMITER;
		}
		if (isset($this->class_links[$class])) return '<a href="'.$a.$this->class_links[$class].'">'.$expr.'</a>';
		else return $class;
	}

	/**
	* generates an html <a> tag for viewing documentation of function $function.
	*
	* if $function is a method, $parent is a classname, otherwise $parent is the file that the function is in (phpdocumentor format)
	* $expr is the text to display in the link
	*
	* @see Render::HandleEvent()
	*/
	
	function getFunctionLink($package,$function,$expr,$local = false)
	{
		$a = '';
		if ($local && isset($this->function_links[$package][$function]))
		{
			for($i=0;$i<$this->dir_level;$i++) $a .= '..'.PATH_DELIMITER;
		}
		if (isset($this->function_links[$package][$function])) return '<a href="'.$a.$this->function_links[$package][$function].'">'.$expr.'</a>';
		else return $expr;
	}

	/**
	* generates an html <a> tag for viewing documentation of method $function.
	*
	* $expr is the text to display in the link
	*
	* @see Render::HandleEvent()
	*/
	
	function getMethodLink($class,$function,$expr,$local = false)
	{
		$a = '';
		if ($local && isset($this->method_links[$class][$function]))
		{
			for($i=0;$i<$this->classdir_level;$i++) $a .= '..'.PATH_DELIMITER;
		}
		if (isset($this->method_links[$class][$function])) return '<a href="'.$a.$this->method_links[$class][$function].'">'.$expr.'</a>';
		else return $expr;
	}

	/**
	* generates an html <a> tag for viewing documentation of $varname.
	*
	* $class is a classname, $expr is the text to display in the link
	*
	* @see Render::HandleEvent()
	*/
	
	function getVarLink($class,$varname,$expr='',$local = false)
	{
		if (empty($expr)) $expr = '$'.$varname;
		$a = '';
		if ($local && isset($this->var_links[$class]['$'.$varname]))
		{
			for($i=0;$i<$this->classdir_level;$i++) $a .= '..'.PATH_DELIMITER;
		}
		if (isset($this->var_links[$class]['$'.$varname])) return '<a href="'.$a.$this->var_links[$class]['$'.$varname].'">'.$expr.'</a>';
		else return $expr;
	}

	/**
	* generates an html <a> tag for viewing documentation of $const
	*
	* @see Render::HandleEvent()
	*/
	
	function getConstLink($package,$const,$expr='',$local = false)
	{
		$a = '';
		if ($local && isset($this->define_links[$package][$const]))
		{
			for($i=0;$i<$this->dir_level;$i++) $a .= '..'.PATH_DELIMITER;
		}
		if (empty($expr)) $expr = $const;
		if (isset($this->define_links[$package][$const])) return '<a href="'.$a.$this->define_links[$package][$const].'">'.$expr.'</a>';
		else return $expr;
	}
	
	/**
	* The meat of the see tag
	*
	* $expr is a string with many allowable formats:
	* <ol><li>constant_name</li><li>classname::function()</li><li>classname::$variablename</li><li>classname</li></ol>
	* If classname:: is not present, and the see tag is in a documentation block within a class, then the function uses the classname to
	* search for $expr as a function or variable within classname, or any of its parent classes.
	* given an $expr without '$', '::' or '()' getLink first searches for methods and variables within the default class, then for any function,
	* then for constants and then for classes.
	*
	* @see DataDocBlock, Render::HandleEvent(), getConstLink(), getVarLink(), getFunctionLink(), getClassLink()
	*/
	function getLink($expr,$classtree = 66,$par = 66,$get = '',$local = false,$orgget = '')
	{
		if (isset($this->class_links[$expr])) return $this->getClassLink($expr,$local);
		if (isset($this->defines[$expr])) return $this->getConstLink($this->defines[$expr]['package'],$expr,$local);
		// is $expr a comma-delimited list?
		if (strpos($expr,','))
		{
			$a = explode(',',$expr);
			$b = '';
			for($i=0;$i<count($a);$i++)
			{
				if (!empty($b)) $b .= ',';
				// if so return each component with a link
				$b .= ' '.$this->getLink(trim($a[$i]),66,66,'',$local,$orgget);
			}
			return $b;
		}
		// are we in a class?
		if ($this->render->parent != $this->render->cur_page)
		{
			// recursion test
			if ($classtree == 66)
			{
				// top-level, generate the class tree for the class currently being parsed
				$classtree = $this->generateClassTreeFromClass($this->render->parent);
				$par = $this->render->parent;
			}
			// is $expr simply a word? if so, check to see if it is a method or variable of this class tree
			if (!strpos($expr,'::'))
			{
				$orgget = $expr;
				// strip any commentary "function xxx", leave "xxx" to do "classname::xxx"
				$get = str_replace('function ','',$expr);
				// if get is neither get() nor $get, assume get is a function, add () to make get()
				if (!is_numeric(strpos($get,'$'))&&!strpos($get,'()')) $get = $get.'()';
				// recur
				$a = $this->getLink($par.'::'.$get,$classtree,$par,$get,$local,$orgget);
				// test to see whether getLink returned a link, or returned the value unmodified
				if ($a != $par.'::'.$get) return $a;
				else
				{
					// recursion test: $classtree is empty at the root object
					if (count($classtree))
					{
						$tpar = $par;
						// set $par to the parent of the object we just examined, and set $classtree to the parent's class tree
						$par = $classtree[$par];
						unset($classtree[$tpar]);
						// recur
						$a  = $this->getLink($par.'::'.$get,$classtree,$par,$get,$local,$orgget);
						// test to see whether getLink returned a link, or returned the value unmodified
						if ($a != $par.'::'.$get) return $a;
					}
				}
			} elseif (!empty($get))
			{
				// get is set if we are testing a classname::get (which is $expr)
				$a = $this->getLink($expr,66,66,'',$local,$orgget);
				// test to see whether getLink returned a link, or returned the value unmodified
				if ($a != $expr) return $a;
				else
				{
					// go deeper!
					if (count($classtree))
					{
						$tpar = $par;
						$par = $classtree[$par];
						unset($classtree[$tpar]);
						if (!$par) $expr = $orgget;
						else
						return $this->getLink($par.'::'.$get,$classtree,$par,$get,$local,$orgget);
					}
				}
				
			}
		}
//		echo $this->render->parent.' '.$expr."\n";
//		flush();

		// is $expr in class::method() or class::$variable format?
		if (strpos($expr,'::'))
		{
			$class_method = explode('::',$expr);
			if (strpos($class_method[1],'()'))
			{
				// strip everything but the function name, return a link
				return $this->getMethodLink($class_method[0],str_replace('function ','',str_replace('()','',$class_method[1])),$expr,$local);
			}
			// strip the leading $ from the variable name
			return $this->getVarLink($class_method[0],str_replace('$','',$class_method[1]),$expr,$local);
		}
		// $expr does not have ::
		if (is_numeric(strpos('$',$expr)))
		{
			// default to current class, whose name is contained in $this->render->parent
			return $this->getVarLink($this->render->parent,str_replace('$','',$expr),$expr,$local);
		}
		// $expr is a function? (non-method)
		if (strpos($expr,'()'))
		{
			// extract the function name, use it to retrieve the file that the function is in
//			$page = $this->func_page[str_replace('function ','',str_replace('()','',$expr))];
			// return the link
			return $this->getFunctionLink($this->render->package, str_replace('function ','',str_replace('()','',$expr)),$expr,$local);
		}
		/*
		
		BUG: should look for a function in the current package, which means function links should be indexed by package and have
		a parent field.  The parser should also halt with a fatal exception when a name collision occurs between 2 functions of the
		same package
		
		*/
		// $expr is just a word.  First, test to see if it is a function of the current page
		if (isset($page) && isset($this->func_page[str_replace('function ','',str_replace('()','',$expr))]))
		{
			if ($page == $this->func_page[str_replace('function ','',str_replace('()','',$expr))])
			return 'function '.$this->getFunctionLink($this->render->package,str_replace('function ','',str_replace('()','',$expr)),$expr,$local);
		}
		// otherwise, see if it is a method
		if (isset($this->method_class[str_replace('function ','',str_replace('()','',$expr))]))
		{
			if ($this->render->parent == $this->method_class[str_replace('function ','',str_replace('()','',$expr))])
			return 'method '.$this->getFunctionLink($this->render->parent,str_replace('function ','',str_replace('()','',$expr)),$expr,$local);
		}
		// no links found
		return $expr;
	}

	/**
	* set parsing (on or off)
	*
	* @param	bool $parse
	*/
	function setParsePrivate($parse)
	{
		$this->parsePrivate = $parse;
	}

	/**
	*
	* @see phpdoc.inc
	*/
	function setRenderer(&$render)
	{
		$this->render = &$render;
	}
}
?>
