<?php
//
// PhpDoc, a program for creating javadoc style documentation from php code
// Copyright (C) 2000-2001 Joshua Eichorn
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//

//
// Copyright 2000-2001 Joshua Eichorn
// Email jeichorn@phpdoc.org
// Web 		http://phpdoc.org/
// Mirror 	http://phpdocu.sourceforge.net/
// Project    	http://sourceforge.net/projects/phpdocu/
//

/**
* @package 	phpDocumentor
*/
/** used when a backslash is encountered in parsing a string or other escapable entity */ 
define("PARSER_EVENT_ESCAPE"		,	900);
define("STATE_ESCAPE"			,	1000);

/** Class published to Render with this event */
define("PHPDOC_EVENT_CLASS"		,	800);
/** DocBlock published to Render with this event */
define("PHPDOC_EVENT_DOCBLOCK"		,	801);
/** Function published to Render with this event */
define("PHPDOC_EVENT_FUNCTION"		,	802);
/** Class Variable published to Render with this event */
define("PHPDOC_EVENT_VAR"		,	803);
/** New File (page) published to Render with this event */
define("PHPDOC_EVENT_PAGE"		,	804);
/** Constant (define) published to Render with this event */
define("PHPDOC_EVENT_DEFINE"		,	805);
/** @deprecated */
define("PHPDOC_EVENT_MESSAGE"		,	806);
/** use to inform Render of a new element being parsed */
define("PHPDOC_EVENT_NEWSTATE"		,	807);
/**
* used to inform Render that the current file has been completely parsed.
* Render then flushes all buffers for functions/classes/defines/includes on the current page
* @see Render::HandleEvent()
*/
define("PHPDOC_EVENT_END_PAGE"		,	808);
/** Package-level page published to Render with this event */
define("PHPDOC_EVENT_PACKAGEPAGE"		,	809);
/** Include (include/require/include_once/require_once) published to Render with this event */
define("PHPDOC_EVENT_INCLUDE"		,	810);

/** used when double quotation mark (") encountered in parsing */
define("PARSER_EVENT_QUOTE"		,	101);
/** currently parsing a quote */
define("STATE_QUOTE"			,	201);

/** { encountered in parsing a function or php code */
define("PARSER_EVENT_LOGICBLOCK"	,	102);
/** currently parsing a { } block */
define("STATE_LOGICBLOCK"		,	202);

/** used for the beginning of parsing, before first < ? php encountered */
define("PARSER_EVENT_NOEVENTS"		,	103);
/** out of < ? php tag */
define("STATE_NOEVENTS"			,	203);

/** used when long comment /x x/ where x is an asterisk is encountered in parsing */
define("PARSER_EVENT_COMMENTBLOCK"	,	104);
/** currently parsing a long comment /x x/ where x is an asterisk */
define("STATE_COMMENTBLOCK"		,	204);

/** used when short comment // is encountered in parsing */
define("PARSER_EVENT_COMMENT"		,	105);
/** currently parsing a short comment // */
define("STATE_COMMENT"			,	205);

/** used when php code processor instruction (< ? php) is encountered in parsing */
define("PARSER_EVENT_PHPCODE"		,	106);
/** currently parsing php code */
define("STATE_PHPCODE"			,	206);

/** used when a define statement is encountered in parsing */
define("PARSER_EVENT_DEFINE"		,	107);
/** currently parsing a define statement */
define("STATE_DEFINE"			,	207);

/** used when a define statement opening parenthesis is encountered in parsing */
define("PARSER_EVENT_DEFINE_PARAMS"	,	108);
/** currently parsing the stuff in ( ) of a define statement */
define("STATE_DEFINE_PARAMS"		,	208);

/** used when a function statement opening parenthesis is encountered in parsing */
define("PARSER_EVENT_FUNCTION_PARAMS"	,	109);
/** currently parsing the stuff in ( ) of a function definition */
define("STATE_FUNCTION_PARAMS"		,	209);

/** used when a single quote (') is encountered in parsing */
define("PARSER_EVENT_SINGLEQUOTE"	,	110);
/** currently parsing a string enclosed in single quotes (') */
define("STATE_SINGLEQUOTE"		,	210);

/** used when a class definition is encountered in parsing */
define("PARSER_EVENT_CLASS"		,	111);
/** currently parsing a class definition */
define("STATE_CLASS"			,	211);
/** used to tell Render that a class has been completely parsed, and to flush buffers */
define("STATE_END_CLASS"		,	311);

/** used when a DocBlock is encountered in parsing */
define("PARSER_EVENT_DOCBLOCK"		,	112);
/** currently parsing a DocBlock */
define("STATE_DOCBLOCK"			,	212);

/** used when a @tag is encountered in DocBlock parsing */
define("PARSER_EVENT_DOCKEYWORD"	,	113);
/** currently parsing a @tag in a DocBlock */
define("STATE_DOCKEYWORD"		,	213);

/** used when a <email@address> is encountered in parsing an @author tag*/
define("PARSER_EVENT_DOCKEYWORD_EMAIL"	,	114);
/** currently parsing an email in brackets in an @author tag of a DocBlock */
define("STATE_DOCKEYWORD_EMAIL"		,	214);

/** used when an array definition is encountered in parsing */
define("PARSER_EVENT_ARRAY"		,	115);
/** currently parsing an array */
define("STATE_ARRAY"			,	215);

/** used when a var statement is encountered in parsing a class definition */
define("PARSER_EVENT_VAR"		,	116);
/** currently parsing a Class variable */
define("STATE_VAR"			,	216);

/** used when a function definition is encountered in parsing */
define("PARSER_EVENT_FUNCTION"		,	117);
/** currently parsing a Function or Method */
define("STATE_FUNCTION"			,	217);

/** used when a ? > (with no space) is encountered in parsing */
define("PARSER_EVENT_OUTPHP"		,	118);
/** currently out of php code */
define("STATE_OUTPHP"			,	218);

/** used when an inline {@tag} is encountered in parsing a DocBlock */
define("PARSER_EVENT_INLINE_DOCKEYWORD"	,	119);
/** currently parsing an inline tag like { @link} in a DocBlock */
define("STATE_INLINE_DOCKEYWORD"		,	219);

/** used when a define statement's opening parenthesis is encountered in parsing */
define("PARSER_EVENT_DEFINE_PARAMS_PARENTHESIS"	,	120);
/** currently parsing an inner parenthetical statement of a define( ) */
define("STATE_DEFINE_PARAMS_PARENTHESIS"		,	220);

define("PARSER_EVENT_END_STATEMENT",	121);

/** used when a <<< is encountered in parsing */
define("PARSER_EVENT_EOFQUOTE"	,	122);
/** currently parsing a string defined using Perl <<< */
define("STATE_EOFQUOTE"		,	222);

/** used when an include/require/include_once/require_once statement is encountered in parsing */
define("PARSER_EVENT_INCLUDE"	,	123);
/** currently parsing an include/require/include_once/require_once */
define("STATE_INCLUDE"	,	223);

/** used when an opening parenthesis of an include/require/include_once/require_once statement is encountered in parsing */
define("PARSER_EVENT_INCLUDE_PARAMS"	,	124);
/** currently parsing the stuff in ( ) of a define statement */
define("STATE_INCLUDE_PARAMS"	,	224);

/** used when an inner ( ) is encountered while parsing an include/require/include_once/require_once statement */
define("PARSER_EVENT_INCLUDE_PARAMS_PARENTHESIS"	,	125);
/** currently parsing an inner parenthetical statement of an include/includeonce/require/requireonce( ) */
define("STATE_INCLUDE_PARAMS_PARENTHESIS"	,	225);

/**
* @author	Joshua Eichorn <jeichorn@phpdoc.org>, Gregory Beaver <cellog@users.sourceforge.net>
* @version	$Id: Parser.inc,v 1.65 2002/04/24 04:35:07 CelloG Exp $
* @package 	phpDocumentor
*/
class Parser extends Publisher
{
	/**
	* Word parser
	* @see WordParser
	*/
	var $wp;
	
	/**
	* temporary parser variables
	*/
	var $p_vars = array('func' => false, 'function_data' => false, 'quote_data' => '', 'event_stack' => false, 'last_pevent' => 0,
						'two_words_ago' => '', 'temp_word' => '', 'docblock' => false, 'line' => '', 'linecount' => 0, 'startword' => '',
						'periodline' => 0, 'shortdesc' => '', 'docblock_desc' => '', 'class' => false, 'source_location' => '',
						'define_params_data' => '', 'define' => false, 'define_name' => '', 'define_value' => '', 'var' => false,
						'oldtoken' => false, 'comment_data' => '', 'function_param' => NULL, 'inline_dockeyword_type' => false,
						'inline_dockeyword_data' => false, 'dockeyword_type' => false, 'dockeyword_data' =>false, 'param_var' => false,
						'include_name' => '', 'include_value' => '','include' => false);
	
	/**
	* parser flags, for states that don't warrant a new event (like new line in a docblock)
	*/
	var $p_flags = array('docblocknewline' => false, 'docblockintags' => false, 'useperiod' => false,
						'definename_isset' => false, 'define_parens' => false, 'reset_quote_data' => false,
						'in_desc' => false, 'in_tag' => false, 'newline' => true, 'tempnewline' => false,
						'start_docblock' => false, 'includename_isset' => false);

	/**
	* lookup table for event handler methods
	* @see Parser::parse()
	*/
	var $eventHandlers = array(
								'handleArray' => PARSER_EVENT_ARRAY,
								'handleClass' => PARSER_EVENT_CLASS,
								'handleComment' => PARSER_EVENT_COMMENT,
								'handleEscape' => PARSER_EVENT_ESCAPE,
								'defaultHandler' => PARSER_EVENT_LOGICBLOCK,
//								'defaultHandler' => PARSER_EVENT_NOEVENTS, (set in constructor below)
//								'defaultHandler' => PARSER_EVENT_COMMENTBLOCK,
//								'defaultHandler' => PARSER_EVENT_OUTPHP,
								'handleDefine' => PARSER_EVENT_DEFINE,
								'handleDefineParams' => PARSER_EVENT_DEFINE_PARAMS,
								'handleDefineParamsParenthesis' => PARSER_EVENT_DEFINE_PARAMS_PARENTHESIS,
//								'handleDefineParamsParenthesis' => PARSER_EVENT_INCLUDE_PARAMS_PARENTHESIS,
								'handleDocBlock' => PARSER_EVENT_DOCBLOCK,
								'handleDockeyword' => PARSER_EVENT_DOCKEYWORD,
								'handleDockeywordEmail' => PARSER_EVENT_DOCKEYWORD_EMAIL,
								'handleEOFQuote' => PARSER_EVENT_EOFQUOTE,
								'handleFunction' => PARSER_EVENT_FUNCTION,
								'handleFunctionParams' => PARSER_EVENT_FUNCTION_PARAMS,
								'handleInlineDockeyword' => PARSER_EVENT_INLINE_DOCKEYWORD,
								'handleInclude' => PARSER_EVENT_INCLUDE,
								'handleIncludeParams' => PARSER_EVENT_INCLUDE_PARAMS,
								'handleQuote' => PARSER_EVENT_QUOTE,
								'handlePhpCode' => PARSER_EVENT_PHPCODE,
								'handleSingleQuote' => PARSER_EVENT_SINGLEQUOTE,
								'handleVar' => PARSER_EVENT_VAR,
	);

	/**
	* An array of allowable @tags
	*/
	var $allowableTags = array("param","return","access","author","copyright","version","see","since","deprecated","exception","throws",
					"package","subpackage","link","todo","name","global","TODO","todo","deprec","var","magic","abstract"); // fc@fc.clever-soft.com 11/30/2001


	/**
	* An array of allowed inline @tags
	*/
	var $allowableInlineTags = array("link");
	
	/**
	* Sets the states up, and creates a new WordParser
	*/
	function Parser()
	{
		$this->wp = new WordParser;
		$this->setupStates();
		// strange PHP behavior: it converts constants to strings without warning if it's an array index
		$this->eventHandlers = array_flip($this->eventHandlers);
		$this->eventHandlers[PARSER_EVENT_NOEVENTS] = 'defaultHandler';
		$this->eventHandlers[PARSER_EVENT_COMMENTBLOCK] = 'defaultHandler';
		$this->eventHandlers[PARSER_EVENT_OUTPHP] = 'defaultHandler';
		$this->eventHandlers[PARSER_EVENT_INCLUDE_PARAMS_PARENTHESIS] = 'handleDefineParamsParenthesis';
	}

	/**
	* Parse a new file
	*
	* @param	string	$parse_data
	* @param	string	$path
	* @param	int	$base	number of directories to drop off the bottom when creating names using path
	* @return	bool
	*/
	function parse (&$parse_data, $path, $base = 0)
	{
		$this->p_vars = array('func' => false, 'function_data' => false, 'quote_data' => '', 'event_stack' => false, 'last_pevent' => 0,
						'two_words_ago' => '', 'temp_word' => '', 'docblock' => false, 'line' => '', 'linecount' => 0, 'startword' => '',
						'periodline' => 0, 'shortdesc' => '', 'docblock_desc' => '', 'class' => false, 'source_location' => '',
						'define_params_data' => '', 'define' => false, 'define_name' => '', 'define_value' => '', 'var' => false,
						'oldtoken' => false, 'comment_data' => '', 'function_param' => NULL, 'inline_dockeyword_type' => false,
						'inline_dockeyword_data' => false, 'dockeyword_type' => false, 'dockeyword_data' =>false, 'param_var' => false,
						'include_name' => '', 'include_value' => '','include' => false);
	
		$thsi->p_flags = array('docblocknewline' => false, 'docblockintags' => false, 'useperiod' => false,
						'definename_isset' => false, 'define_parens' => false, 'reset_quote_data' => false,
						'in_desc' => false, 'in_tag' => false, 'newline' => true, 'tempnewline' => false,
						'start_docblock' => false, 'includename_isset' => false);
		if (strlen($parse_data) == 0)
		{
			return false;
		}

		global $event_stack;
		// initialize variables so E_ALL error_reporting doesn't complain
		$pevent = 0;
		$word = 0;
		$this->p_vars['event_stack'] = new EventStack;

		$this->wp->setup($parse_data);
		

		$page = new DataPage;
		$page->setPath($path);
		$page->setFile(basename($path));
		//$name = str_replace("/","_",dirname($path)) . "_" . array_shift(explode(".",$page->getFile()));
		// fc@fc.clever-soft.com 11/29/2001
		$name = str_replace(PATH_DELIMITER,"_",dirname($path)) . "_" .  str_replace(".","_",$page->getFile());
		$tmp = explode("_",$name);
		$name = implode("_",array_slice($tmp,$base));

		$page->setName($name);
		$this->p_vars['source_location'] = $source_location = "Program_Root" .PATH_DELIMITER. implode(PATH_DELIMITER,
			array_slice(explode(PATH_DELIMITER,$path),$base));
		
		$page->setSourceLocation($source_location);

		$this->publishEvent(PHPDOC_EVENT_PAGE,$page);
		unset($page);
		$this->p_flags['reset_quote_data'] = true;

		do
		{
			$lpevent = $pevent;
			$pevent = $this->p_vars['event_stack']->getEvent();
			if ($lpevent != $pevent)
			{
				$this->p_vars['last_pevent'] = $lpevent;
			}

			if ($this->p_vars['last_pevent'] != $pevent)
			{
				// its a new event so the word parser needs to be reconfigured 
				$this->configWordParser($pevent);
			}

		
			$this->publishEvent(PHPDOC_EVENT_NEWSTATE,($pevent + 100));


			$this->p_vars['last_word'] = $word;
			$word = $this->wp->getWord();

			if (DEBUG == true)
			{
				echo "LAST: |" . $this->p_vars['last_word'] . "|\n";
				echo "PEVENT: " . $this->getParserEventName($pevent) . "\n";
				echo $this->wp->getPos() . ": |$word|\n";
			}
			if (isset($this->eventHandlers[$pevent]))
			{
				$handle = $this->eventHandlers[$pevent];
				$this->$handle($word, $pevent);
			} else
			{
				debug('WARNING: possible error, no handler for event number '.$pevent);
			}
		} while (!($word === false));
		$this->publishEvent(PHPDOC_EVENT_NEWSTATE,PHPDOC_EVENT_END_PAGE);
	}
	
	/**
	* handler for NOEVENTS, OUTPHP, COMMENTBLOCK, LOGICBLOCK
	*/
	
	function defaultHandler($word, $pevent)
	{
		$this->checkEventPush( $word, $pevent);
		$this->checkEventPop($word,$pevent);
	}
	
	/**
	* handler for ESCAPE.
	* this event handler parses "this string \"with its escape backslashes\"" and returns:
	* <code>this string "with its escape backslashes"</code>
	* to make it human-readable
	*/
	
	function handleEscape($word, $pevent)
	{
		$this->p_vars['event_stack']->popEvent();
	}
	
	/**
	* handler for COMMENT.
	* this event handler parses single-line comments like:
	* // this one
	*/
	
	function handleComment($word, $pevent)
	{
		$this->checkEventPush( $word, $pevent);
	
		if (!isset($this->p_vars['comment_data'])) $this->p_vars['comment_data'] = '';
		$this->p_vars['comment_data'] .= $word;
	
		$this->checkEventPop($word,$pevent);
	}

	/**
	* handler for ARRAY.
	* this event handler parses arrays in default values of function and var definitions
	*/
	
	function handleArray($word, $pevent)
	{
		$this->checkEventPush( $word, $pevent); 

		if (!isset($this->p_vars['function_data']))
		{
			$this->p_vars['function_data'] = "array";
		}

		if ( ($this->p_vars['last_word'] == "'"))
		{
			$this->p_vars['function_data'] .= $this->p_vars['quote_data']."'";
		}
		if ( ($this->p_vars['last_word'] == "\""))
		{
			$this->p_vars['function_data'] .= $this->p_vars['quote_data']."\"";
		}

		$this->p_vars['function_data'] .= $word;
		//echo "function_data = |$this->p_vars['function_data']|\n";

		if ($this->checkEventPop($word,$pevent))
		{
		}
	}

	/**
	* handler for DEFINE.
	* handles define(constant, value); statements
	*/
	
	function handleDefine($word, $pevent)
	{
		$this->checkEventPush( $word, $pevent);

		$this->p_flags['definename_isset'] = false;
		$this->p_vars['define_params_data'] = '';
		unset($this->p_vars['quote_data']);
		if ($this->checkEventPop($word,$pevent))
		{
			$this->p_vars['define'] = new DataDefine;
			$this->p_vars['define']->setName($this->p_vars['define_name']);
			$this->p_vars['define']->setValue($this->p_vars['define_value']);
			$this->publishEvent(PHPDOC_EVENT_DEFINE,$this->p_vars['define']);
			$this->p_flags['definename_isset'] = false;
			unset($this->p_vars['define']);
			unset($this->p_vars['define_name']);
			unset($this->p_vars['define_value']);
			unset($this->p_vars['define_params_data']);
		}
	}
	
	/**
	* handler for DEFINE_PARAMS.
	* handles the parsing of constant and value in define(constant, value);
	*/
	
	function handleDefineParams($word, $pevent)
	{
		$this->checkEventPush( $word, $pevent);
		
		$this->p_flags['define_parens'] = true;
		if(!isset($this->p_vars['define_params_data'])) $this->p_vars['define_params_data'] = '';
		
		if ($this->checkEventPop($word,$pevent))
		{
			if (!empty($this->p_vars['quote_data']))
			{
				$this->p_vars['define_value'] = $this->p_vars['quote_data'];
			} else {
				if (!empty($this->p_vars['define_params_data']))
				$this->p_vars['define_value'] = $this->p_vars['define_params_data'];
				else
				$this->p_vars['define_value'] = trim($this->p_vars['last_word']);
			}
		}
		if ($this->p_flags['definename_isset'])
		{
			if (isset($this->p_vars['quote_data']))
			{
				$this->p_vars['define_params_data'] .= '&quot;'.$this->p_vars['quote_data'].'&quot;';
				unset($this->p_vars['quote_data']);
			}
			if (($word != "'") && ($word != '"'))
			$this->p_vars['define_params_data'] .= $word;
		} else
		{
			if ($word != ",")
			{
				if (isset($this->p_vars['quote_data']))
				{
					$this->p_vars['define_params_data'] .= $this->p_vars['quote_data'];
					unset($this->p_vars['quote_data']);
				}
				if (($word != "'") && ($word != '"'))
				$this->p_vars['define_params_data'] .= $word;
			} else
			{
				$this->p_flags['definename_isset'] = true;
				$this->p_vars['define_name'] = $this->p_vars['define_params_data'];
				unset($this->p_vars['quote_data']);
				unset($this->p_vars['define_params_data']);
			}
		}
	}
	
	/**
	* handler for DEFINE_PARAMS_PARENTHESIS.
	* this handler takes all parenthetical statements within constant or value in:
	* define(constant, value) of a define statement, and handles them properly
	*/
	
	function handleDefineParamsParenthesis($word, $pevent)
	{
		$this->p_vars['define_params_data'] .= $word;
		$this->checkEventPush( $word, $pevent);
		$this->checkEventPop( $word, $pevent);
	}

	/**
	* handler for CLASS.
	* this handler parses a class statement
	*/
	
	function handleClass($word, $pevent)
	{
		if ($this->checkEventPush( $word, $pevent) == PARSER_EVENT_DOCBLOCK)
		{
			$this->wp->setWhitespace(true);
		}

		if (!isset($this->p_vars['class'])) $this->p_vars['class'] = false;
		if (!is_subclass_of($this->p_vars['class'],"data"))
		{
			$this->p_vars['class'] = new DataClass;
			$this->p_vars['class']->setname($word);
			$this->p_vars['class']->setSourceLocation($this->p_vars['source_location']);
		}

		if (strtolower($this->p_vars['last_word']) == "extends")
		{
			$this->p_vars['class']->setExtends($word);
		}

		if ($word == "{")
		{
			$this->publishEvent(PHPDOC_EVENT_CLASS,$this->p_vars['class']);
		}
		//echo $this->wp->getPos() . ": |$word|\n";
		if ($this->checkEventPop($word,$pevent))
		{
			// throw an event when class is done
			$this->publishEvent(PHPDOC_EVENT_NEWSTATE,STATE_END_CLASS);
			$this->p_vars['class'] = false;
		}
	}

	/**
	* handler for VAR.
	* handle a var $varname = default_value; or var $varname; statement in a class definition
	*/
	
	function handleVar($word, $pevent)
	{
		$this->checkEventPush( $word, $pevent);

		if (!isset($this->p_vars['var'])) $this->p_vars['var'] = false;
		if (!is_subclass_of($this->p_vars['var'],"data"))
		{
			$this->p_vars['var'] = new DataVar;
			$this->p_vars['var']->setName($word);
		}
		if ($this->p_vars['last_word'] == "=")
		{
			$this->p_vars['var']->setValue($word);
		}
		if ($this->p_vars['last_pevent'] == PARSER_EVENT_QUOTE || $this->p_vars['last_pevent'] == PARSER_EVENT_SINGLEQUOTE)
		{
			$this->p_vars['var']->setValue($this->p_vars['quote_data']);
			$this->p_vars['quote_data'] = '';
		}
		if ($this->p_vars['last_pevent'] == PARSER_EVENT_ARRAY)
		{
			$this->p_vars['var']->setValue($this->p_vars['function_data']);
			$this->p_vars['function_data'] = false;
		}
			
		if ($this->checkEventPop($word,$pevent))
		{
			$this->publishEvent(PHPDOC_EVENT_VAR,$this->p_vars['var']);
			unset($this->p_vars['var']);
		}
	}

	/**
	* handler for QUOTE.
	* this handler recognizes strings defined with double quotation marks (") and handles them correctly
	* in any place that they legally appear in php code
	*/
	
	function handleQuote($word, $pevent)
	{
		if ($this->p_flags['reset_quote_data'] === true)
		{
			$this->p_flags['reset_quote_data'] = false;
			$this->p_vars['quote_data'] = "";
		}
		$this->checkEventPush( $word, $pevent);
		if ($word != "\"")
		{
			$this->p_vars['quote_data'] .= $word;
		}
		if ($this->checkEventPop($word,$pevent))
		{
			$this->p_flags['reset_quote_data'] = true;
		}
	}
	
	/**
	* handler for SINGLEQUOTE.
	* this handler recognizes strings defined with single quotation marks (') and handles them correctly
	* in any place that they legally appear in php code
	*/
	
	function handleSingleQuote($word, $pevent)
	{
		$this->checkEventPush( $word, $pevent);
		if ($this->checkEventPop($word,$pevent))
		{
			if ($this->p_vars['last_word'] != "'")
			{
				$this->p_vars['quote_data'] = $this->p_vars['last_word'];
			} else {
				$this->p_vars['quote_data'] = "";
			}
		}
	}

	/**
	* handler for EOFQUOTE.
	* this handler recognizes strings defined with perl-style <<< EOF quotes, and handles them correctly
	* in any place that they legally appear in php code
	*
	* an example:
	* <code>$var <<< EOF
	* blah blah blah
	* EOF;</code>
	*/
	
	function handleEOFQuote($word, $pevent)
	{
		//	echo $this->wp->getPos() . ": word=|$word|\t\t\tlastword=|$this->p_vars['last_word']|\n";
		if (trim($this->p_vars['last_word']) == "<<<")
		{
			// ok we found the keyword
			//echo "Keyword == $word\n";
			$this->p_vars['oldtoken'] = $this->tokens[STATE_EOFQUOTE];
			$this->tokens[STATE_EOFQUOTE] = array($word);
		} 
		else if ($this->p_vars['last_pevent'] || PARSER_EVENT_EOFQUOTE) 
		{
			// i don't think anything will ever use this so were not going to set it
			//$this->p_vars['quote_data'] = $this->p_vars['last_word']; 
			$this->p_vars['event_stack']->popEvent();
			$this->tokens[STATE_EOFQUOTE] = $this->p_vars['oldtoken'];
		}
	}

	/**
	* handler for PHPCODE.
	* this handler recognizes the < ? php processor directive (no spaces, of course), and begins parsing php code
	*/
	
	function handlePhpCode($word, $pevent)
	{
		if ($this->checkEventPush( $word, $pevent) == PARSER_EVENT_DOCBLOCK)
		{
			$this->wp->setWhitespace(true);
		}
	}
	
	/**
	* handler for FUNCTION.
	* this handler recognizes function declarations, and parses them
	*/
	
	function handleFunction($word, $pevent)
	{
		$this->checkEventPush( $word, $pevent); 
	
		if (!isset($this->p_vars['func'])) $this->p_vars['func'] = false;
		if (! is_subclass_of($this->p_vars['func'],"data")) 
		{ 
			$this->p_vars['func'] = new DataFunction; 
			$this->p_vars['func']->setName($word); 
		} 
	
		if ($this->p_vars['last_pevent'] == PARSER_EVENT_LOGICBLOCK) 
		{ 
			$this->publishEvent(PHPDOC_EVENT_FUNCTION,$this->p_vars['func']); 
			$this->p_vars['func'] = false; 
		} 
		else if ($this->checkEventPop($word,$pevent)) 
		{ 
			$this->publishEvent(PHPDOC_EVENT_FUNCTION,$this->p_vars['func']); 
			$this->p_vars['func'] = false; 
		} 
		if ($this->p_vars['last_pevent'] == PARSER_EVENT_LOGICBLOCK) 
		{ 
			$this->p_vars['event_stack']->popEvent(); 
			// We need to backup a word because of this wiredness 
			$this->wp->setPos($this->wp->getPos() - strlen($word)); 
		} 
	}

	/**
	* handler for FUNCTION_PARAMS.
	* this handler recognizes the parameters of a function within parentheses like function(param, param = default_value)
	* and parses them
	*/
	
	function handleFunctionParams($word, $pevent)
	{
		//echo $this->wp->getPos() . ": word=|$word|\t\t\tlastword=|$this->p_vars['last_word']|\n";
		//echo "function_param = '$this->p_vars['function_param']'\n";
		//echo "function_data = '$this->p_vars['function_data']'\n";
		$e1 = $this->checkEventPush( $word, $pevent); 

		if ($e1 != PARSER_EVENT_ARRAY)
		{

			if (empty($this->p_vars['function_param']) && $word != "," && $word != "\n" && $word != ")")
			{
				 $this->p_vars['function_param'] = $word;
			}
			if ($this->p_vars['last_word'] == "=" && $word != "'" && $word != "\"")
			{
				$this->p_vars['func']->addParam($this->p_vars['function_param'],$word);
				unset($this->p_vars['function_param']);
			}
			if ( ($this->p_vars['last_word'] == "'") && isset($this->p_vars['function_param']) )
			{
				$this->p_vars['func']->addParam($this->p_vars['function_param'],"'".$this->p_vars['quote_data']."'");
				unset($this->p_vars['function_param']);
			}
			if ( ($this->p_vars['last_word'] == "\"") && isset($this->p_vars['function_param']) )
			{
				$this->p_vars['func']->addParam($this->p_vars['function_param'],"\"".$this->p_vars['quote_data']."\"");
				unset($this->p_vars['function_param']);
			}
			if ($word == "," && !empty($this->p_vars['function_param']))
			{
				if (isset($this->p_vars['function_data']))
				{
					$this->p_vars['func']->addParam($this->p_vars['function_param'],$this->p_vars['function_data']);
					unset($this->p_vars['function_param']);
					unset($this->p_vars['function_data']);
				} else {
					$this->p_vars['func']->addParam($this->p_vars['function_param'],"");
					unset($this->p_vars['function_param']);
				}
			}
			
			if ($this->checkEventPop($word,$pevent))
			{
				if (isset($this->p_vars['function_param']))
				{
					if (isset($this->p_vars['function_data']))
					{
						$this->p_vars['func']->addParam($this->p_vars['function_param'],$this->p_vars['function_data']);
						unset($this->p_vars['function_param']);
						unset($this->p_vars['function_data']);
					} else {
						$this->p_vars['func']->addParam($this->p_vars['function_param'],"");
						unset($this->p_vars['function_param']);
					}
				}
			}
		}
	}

	/**
	* handler for DOCBLOCK.
	* this handler recognizes DocBlocks, and parses out the short description, long description, and publishes the
	* docblock data to Render
	*/
	
	function handleDocBlock($word, $pevent)
	{
		$e1 = $this->checkEventPush( $word, $pevent);

		if (!isset($this->p_vars['docblock'])) $this->p_vars['docblock'] = false;
		if (!is_subclass_of($this->p_vars['docblock'],"data"))
		{
			$this->p_flags['in_desc'] = true;
			$this->p_flags['in_tag'] = false;

			$this->p_vars['docblock'] = new DataDocBlock();
			$this->p_flags['useperiod'] = false;
			$this->p_vars['line'] = array();
			$this->p_vars['linecount'] = 0;
			$this->p_flags['start_docblock'] = true;
		}

		//echo $this->wp->getPos() . ": |$word|\n";

		if (!isset($this->p_vars['linecount'])) $this->p_vars['linecount'] = 0;
		if (!$e1 && $word != "*" && $word != "*/" && $this->p_vars['last_word'] != "\n")
		{
			if ($this->p_flags['in_tag'])
			{
				$this->p_vars['dockeyword_data'] .= $word;
			} else
			{
				if (!isset($this->p_vars['line'][$this->p_vars['linecount']])) $this->p_vars['line'][$this->p_vars['linecount']] = '';
				$this->p_vars['line'][$this->p_vars['linecount']] .= $word;
			}
		}

		if ($this->p_vars['last_word'] == "." && strlen(trim($word)) == 0 && $this->p_flags['useperiod'] == false)
		{
			$this->p_flags['useperiod'] = true;
			$this->p_vars['periodline'] = $this->p_vars['linecount'];
		}

		if ($word == "\n")
		{
			$this->p_flags['newline'] = true;
			$this->p_flags['start_docblock'] = false;
			if ($this->p_flags['in_desc'])
			$this->p_vars['linecount']++;
		} else
		{
			if ($word == '/**' || $this->p_flags['start_docblock'])
			{
				$this->p_flags['newline'] = true;
				$this->p_flags['start_docblock'] = false;
			}
			if ($this->p_flags['newline'])
			{
				preg_match('/^([\r\n\t ]+)$/',$word,$match);
				if (!isset($match[0]) && $word != '*' && $word != '@' && $word != '/**')
				{
					$this->p_flags['newline'] = false;
				}
			}
		}
//		fancy_debug('newline',$this->p_flags['newline'],$word);

		if ($this->checkEventPop($word,$pevent))
		{
			//print_r($this->p_vars['line']);
			
			$this->p_flags['in_tag'] = false;
			$this->p_flags['new_line'] = true;
			
			// we aren't going to use period line if its over line 3, or 4 lines of short desc
			if (!isset($this->p_vars['periodline'])) $this->p_vars['periodline'] = 0;
			if ($this->p_vars['periodline'] > 3)
			{
				$this->p_flags['useperiod'] = false;
			}

			// figure out the shortdesc
			if ($this->p_flags['useperiod'] === false)
			{
				// use the first non blank line for short desc
				for($i = 0; $i < count($this->p_vars['line']); $i++)
				{
					if (!isset($this->p_vars['line'][$i])) $this->p_vars['line'][$i] = '';
					if (strlen(trim($this->p_vars['line'][$i])) > 0)
					{
						$this->p_vars['periodline'] = $i;
						$i = count($this->p_vars['line']);
					}
				}
						
				// check to see if we are going to use a blank line to end the shortdesc
				// this can only be in the first 4 lines
				if (count($this->p_vars['line']) > 4)
				{
					$max = 4;
				} else {
					$max = count($this->p_vars['line']);
				}

				for($i = $this->p_vars['periodline']; $i < $max; $i++)
				{
					if (strlen(trim($this->p_vars['line'][$i])) == 0)
					{
						$this->p_vars['periodline'] = $i;
						$i = $max;
					}
				}
			}

			$this->p_vars['shortdesc'] = "";
			for($i = 0; ($i <= $this->p_vars['periodline']) && ($i < count($this->p_vars['line'])); $i++)
			{
				$this->p_vars['shortdesc'] .= $this->p_vars['line'][$i];
			}
			$this->p_vars['periodline']++;

			$this->p_vars['docblock_desc'] = "";
			//echo "i = $this->p_vars['periodline']; i < " . count($this->p_vars['line']) . "\n";
			for($i = $this->p_vars['periodline']; $i < count($this->p_vars['line']); $i++)
			{
				$this->p_vars['docblock_desc'] .= $this->p_vars['line'][$i];
			}


			$this->p_vars['docblock']->setShortDesc($this->p_vars['shortdesc']);
			$this->p_vars['docblock']->setDesc($this->p_vars['docblock_desc']);
			unset($this->p_vars['docblock_desc']);
			$this->publishEvent(PHPDOC_EVENT_DOCBLOCK,$this->p_vars['docblock']);
			unset($this->p_vars['docblock']);
			$this->wp->setWhitespace(false);
		}
	}
	
	/**
	* handler for DOCKEYWORD.
	* this handler recognizes @tags in DocBlocks and parses them for display
	*/
	
	function handleDockeyword($word, $pevent)
	{
		//		echo $this->wp->getPos() . ": |$word|$this->p_vars['startword']\n";

		//		echo "docktype: $this->p_vars['dockeyword_type']\n";

		// if the @tag is not at the start of a line, ignore it
		$justset = false;
		if ($word == "\n")
		{
			$this->p_flags['tempnewline'] = true;
		}
		if ($this->p_flags['tempnewline'])
		{
			if ($this->p_vars['last_word'] == '@')
			{
				$this->p_flags['newline'] = true;
				$this->p_flags['tempnewline'] = false;
			}
		}
//		fancy_debug('testing',$this->p_flags['newline'], $word);
		if ($this->p_flags['newline'])
		{
			// we're in tags now
			$this->p_flags['newline'] = false;
			$this->p_flags['in_desc'] = false;

			if (!in_array($word,$this->allowableTags))
			{
				if ($word != "*" && $word != "*/" && $this->p_vars['last_word'] != "\n")
				{
					if ($this->p_flags['in_tag'])
					{
						$this->p_vars['dockeyword_data'] .= "@";
					} else
					{
						$this->p_flags['in_desc'] = true;
						$this->p_vars['line'][$this->p_vars['linecount']] .= '@'.$word;
						$this->p_vars['event_stack']->popEvent();
						$this->p_vars['dockeyword_data'] = '';
						$this->p_vars['dockeyword_type'] = false;
						return;
					}
				}
			} else
			{
				if ($this->p_vars['dockeyword_type'])
				{
					if (!empty($this->p_vars['param_var']))
					{
						$this->p_vars['docblock']->addParam($this->p_vars['param_var'],trim($this->p_vars['dockeyword_data']));
					} else {
						if ($this->p_vars['dockeyword_type']=='param')
						{
							$this->p_vars['docblock']->addParam(NULL,trim($this->p_vars['dockeyword_data']));
						}
						if ($this->p_vars['dockeyword_type']=='see')
						{
							if (isset($this->subscriber['*']->linker))
							{
								$this->p_vars['dockeyword_data'] = $this->subscriber['*']->linker->getLink(trim($this->p_vars['dockeyword_data']),66,66,'',1);
							}
						}
						$this->p_vars['docblock']->addKeyword($this->p_vars['dockeyword_type'],trim($this->p_vars['dockeyword_data']));
//						fancy_debug('other adding',$this->p_vars['dockeyword_type'],trim($this->p_vars['dockeyword_data']));
					}
				}
//				debug('found '.$word);
				$this->p_flags['in_tag'] = true;
				$justset = true;
				$this->p_vars['param_var'] = false;
				$this->p_vars['dockeyword_type'] = $word;
				$this->p_vars['dockeyword_data'] = '';
				$this->p_flags['newline'] = false;
			}
		} else
		{
			if ($this->p_vars['last_word'] == '@')
			{
				if ($this->p_flags['in_tag'])
				{
					$this->p_vars['dockeyword_data'] .= '@';
				} else
				{
					$this->p_vars['line'][$this->p_vars['linecount']] .= '@'.$word;
					$this->p_vars['event_stack']->popEvent();
					$this->p_flags['tempnewline'] = false;
					$this->p_vars['dockeyword_data'] = '';
					$this->p_vars['dockeyword_type'] = false;
					return;
				}
			}
		}
		if (!isset($this->p_vars['dockeyword_type'])) $this->p_vars['dockeyword_type'] = false;
		if (!isset($this->p_vars['dockeyword_data'])) $this->p_vars['dockeyword_data'] = '';
		$this->wp->setWhitespace(true);
		$e1 = $this->checkEventPush( $word, $pevent); 
		if (!$e1) {
			if ($this->p_vars['dockeyword_type'] == "param" && empty($this->p_vars['param_var']))
			{
				if (substr($word,0,1) == "$")
				{
					if ($word != "*" && $word != "*/" && $word != '@' && $this->p_vars['last_word'] != "\n")
					$this->p_vars['param_var'] = $word;
				} else {
					if (!$justset && ($word != "*" && $word != "*/" && $word != '@' && $this->p_vars['last_word'] != "\n"))
					$this->p_vars['dockeyword_data'] .= $word;
				}
			} else if ($word != "*/" && $word != "*"/* && $word != "\n"*/) {//$word != "@" &&
				if (!$justset && ($word != "*" && $word != "*/" && $word != '@' && $this->p_vars['last_word'] != "\n"))
				$this->p_vars['dockeyword_data'] .= $word;
			}
		}

		if ($this->p_flags['newline'] || (!$this->p_flags['newline'] && $word != '@'))
		{
			if ($this->checkEventPop($word,$pevent))
			{
				if (!empty($this->p_vars['param_var']))
				{
					$this->p_vars['docblock']->addParam($this->p_vars['param_var'],trim($this->p_vars['dockeyword_data']));
				} else {
					if ($this->p_vars['dockeyword_type']=='see')
					{
						if (isset($this->subscriber['*']->linker))
						{
							$this->p_vars['dockeyword_data'] = $this->subscriber['*']->linker->getLink(trim($this->p_vars['dockeyword_data']),66,66,'',1);
						}
					}
//								fancy_debug('adding',$this->p_vars['dockeyword_type'],trim($this->p_vars['dockeyword_data']));
					$this->p_vars['docblock']->addKeyword($this->p_vars['dockeyword_type'],trim($this->p_vars['dockeyword_data']));
				}
				$this->p_flags['tempnewline'] = false;
				$this->p_vars['param_var'] = false;
				$this->p_vars['dockeyword_type'] = false;
				$this->p_vars['dockeyword_data'] = '';
				$this->wp->setWhitespace(false);
				// walk back a word
				$this->wp->setPos($this->wp->getPos() - strlen($word));
			}
		}
	}
	
	/**
	* handler for DOCKEYWORD_EMAIL.
	* this handler recognizes angle brackets < and > surrounding an email address in an @author tag,
	* and returns a mailto: hyperlink
	*/
	
	function handleDockeywordEmail($word, $pevent)
	{
		//echo $this->wp->getPos() . ": |$word|\n";
		if (!$this->checkEventPop($word,$pevent) && $word != "<")
		{
			if (strstr($word,"@"))
			{
				$this->p_vars['dockeyword_data'] .= "&lt;<a href='mailto:$word'>$word</a>&gt;";
			} else {
				$this->p_vars['dockeyword_data'] .= "<$word>";
			}
		}
	}

	/**
	* handler for INLINE_DOCKEYWORD.
	* this handler recognizes {@inline tags} like link, and parses them, replacing them directly
	* in the text flow with their output.
	*/
	
	function handleInlineDockeyword($word, $pevent)
	{
		//		echo $this->wp->getPos() . ": |$word|\n";

		//		echo "docktype: $this->p_vars['inline_dockeyword_type']\n";
		if (!isset($this->p_vars['inline_dockeyword_type'])) $this->p_vars['inline_dockeyword_type'] = false;
		if (!isset($this->p_vars['inline_dockeyword_data'])) $this->p_vars['inline_dockeyword_data'] = '';
		if (!$this->p_vars['inline_dockeyword_type'])
		{
			if (in_array($word,$this->allowableInlineTags))
			{
				$this->p_vars['inline_dockeyword_type'] = strtolower($word);
			} else {
				$this->p_vars['line'][$this->p_vars['linecount']] .= '{@'.$word;
				$this->p_vars['event_stack']->popEvent();
			}
		} else
		{
			if ($word != "}")
			{
				$this->p_vars['inline_dockeyword_data'] .= $word;
			}
		}
		if ($this->checkEventPop($word,$pevent))
		{
			if ($this->p_vars['inline_dockeyword_type']=='link')
			{
				// support hyperlinks of any protocol
				if (is_numeric(strpos($this->p_vars['inline_dockeyword_data'],'://')))
				{
					// if there is more than 1 parameter, the stuff after the space is the hyperlink text
					if (strpos(trim($this->p_vars['inline_dockeyword_data']),' '))
					{
						$i1 = strpos(trim($this->p_vars['inline_dockeyword_data']),' ') + 1;
						$link = substr(trim($this->p_vars['inline_dockeyword_data']),0,$i1 - 1);
						$text = substr(trim($this->p_vars['inline_dockeyword_data']),$i1);
						$this->p_vars['inline_dockeyword_data'] = '<a href="'.$link.'">'.$text.'</a>';
					}
					else
					$this->p_vars['inline_dockeyword_data'] = '<a href="'.$this->p_vars['inline_dockeyword_data'].'">'.$this->p_vars['inline_dockeyword_data'].'</a>';
				} elseif (isset($this->subscriber['*']->linker))
				{
//								debug('parsing '.$this->p_vars['inline_dockeyword_data']);
					$this->p_vars['inline_dockeyword_data'] = $this->subscriber['*']->linker->getLink(trim($this->p_vars['inline_dockeyword_data']),66,66,'',1);
				}
			}
			if ($this->p_flags['in_desc'])
			{
				$this->p_vars['line'][$this->p_vars['linecount']] .= $this->p_vars['inline_dockeyword_data'];
				$this->p_vars['dockeyword_type'] = false;
				$this->p_vars['dockeyword_data'] = '';
			}
			elseif ($this->p_flags['in_tag'])
			$this->p_vars['dockeyword_data'] .= $this->p_vars['inline_dockeyword_data'];
		}
	}

	/**
	* handler for INCLUDE.
	* this handler recognizes include/require/include_once/require_once statements, and publishes the
	* data to Render
	*/
	
	function handleInclude($word, $pevent)
	{
		$this->checkEventPush( $word, $pevent);
		if (!$this->p_flags['includename_isset'])
		{
			$this->p_flags['includename_isset'] = true;
			$this->p_vars['include_name'] = $this->p_vars['last_word'];
		}

		$this->p_vars['include_params_data'] = '';
		unset($this->p_vars['quote_data']);
		if ($this->checkEventPop($word,$pevent))
		{
			$this->p_vars['include'] = new DataInclude;
			$this->p_vars['include']->setName($this->p_vars['include_name']);
			$this->p_vars['include']->setValue($this->p_vars['include_value']);
			$this->publishEvent(PHPDOC_EVENT_INCLUDE,$this->p_vars['include']);
			$this->p_flags['includename_isset'] = false;
			unset($this->p_vars['include']);
			unset($this->p_vars['include_name']);
			unset($this->p_vars['include_value']);
			unset($this->p_vars['include_params_data']);
		}
	}
	
	/**
	* handler for INCLUDE_PARAMS.
	* this handler parses the contents of ( ) in include/require/include_once/require_once statements
	*/
	
	function handleIncludeParams($word, $pevent)
	{
		$this->checkEventPush( $word, $pevent);
		
		$this->p_flags['include_parens'] = true;
		if(!isset($this->p_vars['include_params_data'])) $this->p_vars['include_params_data'] = '';
		
		if ($this->checkEventPop($word,$pevent))
		{
			if (!empty($this->p_vars['quote_data']))
			{
				$this->p_vars['include_value'] = $this->p_vars['quote_data'];
			} else {
				if (!empty($this->p_vars['include_params_data']))
				$this->p_vars['include_value'] = $this->p_vars['include_params_data'];
				else
				$this->p_vars['include_value'] = trim($this->p_vars['last_word']);
			}
		}
		if (isset($this->p_vars['quote_data']))
		{
			$this->p_vars['include_params_data'] .= '&quot;'.$this->p_vars['quote_data'].'&quot;';
			unset($this->p_vars['quote_data']);
		}
		if (($word != "'") && ($word != '"'))
		$this->p_vars['include_params_data'] .= $word;
	}
	
	/**
	* this function checks whether parameter $word is a token for pushing a new event onto the Event Stack.
	* @return mixed	returns false, or the event number
	*/
	
	function checkEventPush($word,$pevent)
	{
		$e = false;
		if (isset($this->pushEvent[$pevent]))
		{
			if (isset($this->pushEvent[$pevent][strtolower($word)]))
			$e = $this->pushEvent[$pevent][strtolower($word)];
		}
		if ($e)
		{
			$this->p_vars['event_stack']->pushEvent($e);
			return $e;
		} else {
			return false;
		}
	}

	/**
	* this function checks whether parameter $word is a token for popping the current event off of the Event Stack.
	* @return mixed	returns false, or the event number popped off of the stack
	*/
	
	function checkEventPop($word,$pevent)
	{
		if (!isset($this->popEvent[$pevent])) return false;
		if (in_array(strtolower($word),$this->popEvent[$pevent]))
		{
			return $this->p_vars['event_stack']->popEvent();
		} else {
			return false;
		}
	}

	/**
	* setup the parser tokens, and the pushEvent/popEvent arrays
	* @see $tokens, $pushEvent, $popEvent
	*/
	
	function setupStates()
	{
		$this->tokens[STATE_PHPCODE]		= array(" ", "\t",";","?>","/**", "//","/*","#","\r\n","\n","\r","(",'<<<');
		$this->tokens[STATE_QUOTE]		= array("\\\"","\\\\","\"");
		$this->tokens[STATE_LOGICBLOCK]		= array("{","}","\"","'","/*","//","#","?>",'<<<');
		$this->tokens[STATE_NOEVENTS]		= array("<?php","<?");
		$this->tokens[STATE_COMMENTBLOCK]	= array("*/");
		$this->tokens[STATE_COMMENT]		= array("\r\n","\r","\n");
		$this->tokens[STATE_DEFINE]		= array(" ","(",";");
		$this->tokens[STATE_DEFINE_PARAMS]	= array("/*",",",")"," ","'","\"","(");
		$this->tokens[STATE_DEFINE_PARAMS_PARENTHESIS]	= array("(",")");
		$this->tokens[STATE_FUNCTION_PARAMS]	= array("/*","\"",",",")","="," ","'","(");
		$this->tokens[STATE_SINGLEQUOTE]	= array("'","\\'","\\\\");
		$this->tokens[STATE_CLASS]		= array(" ","\t","?>",";","}","{","/**","//","/*","#","\r\n","\n","\r","(");
		$this->tokens[STATE_DOCBLOCK]		= array("*/","*","@","\r\n","\n","\r",".",'{@');
		$this->tokens[STATE_DOCKEYWORD]		= array("@","*/","*","\n","\r\n","\r","\t"," ","<",">");
		$this->tokens[STATE_INLINE_DOCKEYWORD]		= array("{@","}","\t"," ");
		$this->tokens[STATE_DOCKEYWORD_EMAIL]	= array(">","\n","\r\n","\r");
		$this->tokens[STATE_VAR]		= array(" ","\t",";","=",",","\"","'","array");
		$this->tokens[STATE_ARRAY]		= array("(",")","\"","'","array");
		$this->tokens[STATE_FUNCTION]		= array("(","{","}"," ","\t");
		$this->tokens[STATE_OUTPHP]		= array("<?php","<?");
		$this->tokens[STATE_EOFQUOTE]		= array(" ","\t","\n");
		$this->tokens[STATE_ESCAPE]		= false;// this tells the word parser to just cycle
		$this->tokens[STATE_INCLUDE]	= array(" ","(",";");
		$this->tokens[STATE_INCLUDE_PARAMS]	= array("/*",")"," ","'","\"","(");
		$this->tokens[STATE_INCLUDE_PARAMS_PARENTHESIS]	= array("(",")");

		// For each event word to event mapings
		$this->pushEvent[PARSER_EVENT_QUOTE] = 
			array(
				"\\"	=> PARSER_EVENT_ESCAPE
			);
		$this->popEvent[PARSER_EVENT_QUOTE] = array("\"");
##########################
		 
		$this->pushEvent[PARSER_EVENT_LOGICBLOCK] = 
			array(
				"\""	=> PARSER_EVENT_QUOTE,
				"'"	=> PARSER_EVENT_SINGLEQUOTE,
				"//" 	=> PARSER_EVENT_COMMENT,
				"#" 	=> PARSER_EVENT_COMMENT,
				"/*" 	=> PARSER_EVENT_COMMENTBLOCK,
				"{"	=> PARSER_EVENT_LOGICBLOCK,
				"?>"	=> PARSER_EVENT_OUTPHP,
				"<<<"	=> PARSER_EVENT_EOFQUOTE
			);
		$this->popEvent[PARSER_EVENT_LOGICBLOCK] = array("}");
##########################

		$this->pushEvent[PARSER_EVENT_NOEVENTS] = 
			array(
				"<?php" => PARSER_EVENT_PHPCODE,
				"<?" => PARSER_EVENT_PHPCODE
			);
##########################

		$this->pushEvent[PARSER_EVENT_PHPCODE] = 
			array(
				"function" 	=> PARSER_EVENT_FUNCTION,
				"class" 	=> PARSER_EVENT_CLASS,
				"define" 	=> PARSER_EVENT_DEFINE,
				"include_once" => PARSER_EVENT_INCLUDE,
				"require_once" => PARSER_EVENT_INCLUDE,
				"include"	=> PARSER_EVENT_INCLUDE,
				"require"	=> PARSER_EVENT_INCLUDE,
				"//" 		=> PARSER_EVENT_COMMENT,
				"#" 		=> PARSER_EVENT_COMMENT,
				"/*" 		=> PARSER_EVENT_COMMENTBLOCK,
				"/**" 		=> PARSER_EVENT_DOCBLOCK,
				"\""		=> PARSER_EVENT_QUOTE,
				"'"		=> PARSER_EVENT_SINGLEQUOTE,
				"<<<"		=> PARSER_EVENT_EOFQUOTE,
				"?>" 		=> PARSER_EVENT_OUTPHP
			);
##########################

		$this->pushEvent[PARSER_EVENT_FUNCTION] = 
			array(
				"(" 	=> PARSER_EVENT_FUNCTION_PARAMS,
				"{" 	=> PARSER_EVENT_LOGICBLOCK
			);
		$this->popEvent[PARSER_EVENT_FUNCTION] = array("}");
##########################

		$this->pushEvent[PARSER_EVENT_DOCBLOCK] = 
			array(
				"@" 	=> PARSER_EVENT_DOCKEYWORD,
				"{@"	=> PARSER_EVENT_INLINE_DOCKEYWORD
			);
		$this->popEvent[PARSER_EVENT_DOCBLOCK] = array("*/");
##########################

		$this->pushEvent[PARSER_EVENT_CLASS] = 
			array(
				"function" 	=> PARSER_EVENT_FUNCTION,
				"var" 		=> PARSER_EVENT_VAR,
				"/**" 		=> PARSER_EVENT_DOCBLOCK,
				"//" 		=> PARSER_EVENT_COMMENT,
				"#" 		=> PARSER_EVENT_COMMENT,
				"?>"		=> PARSER_EVENT_OUTPHP
			);
		$this->popEvent[PARSER_EVENT_CLASS] = array("}");
##########################

		$this->pushEvent[PARSER_EVENT_DEFINE] = 
			array(
				"/*" 	=> PARSER_EVENT_COMMENTBLOCK,
				"(" 	=> PARSER_EVENT_DEFINE_PARAMS
			);
		$this->popEvent[PARSER_EVENT_DEFINE] = array(";");
##########################

		$this->pushEvent[PARSER_EVENT_INCLUDE] = 
			array(
				"/*" 	=> PARSER_EVENT_COMMENTBLOCK,
				"(" 	=> PARSER_EVENT_INCLUDE_PARAMS
			);
		$this->popEvent[PARSER_EVENT_INCLUDE] = array(";");
##########################

		$this->pushEvent[PARSER_EVENT_DEFINE_PARAMS] = 
			array(
				"("	=>	PARSER_EVENT_DEFINE_PARAMS_PARENTHESIS
			);
		$this->popEvent[PARSER_EVENT_DEFINE_PARAMS] = array(")");
##########################

		$this->pushEvent[PARSER_EVENT_INCLUDE_PARAMS] = 
			array(
				"("	=>	PARSER_EVENT_INCLUDE_PARAMS_PARENTHESIS
			);
		$this->popEvent[PARSER_EVENT_INCLUDE_PARAMS] = array(")");
##########################

		$this->pushEvent[PARSER_EVENT_DEFINE_PARAMS_PARENTHESIS] =
			array(
				"("	=>	PARSER_EVENT_DEFINE_PARAMS_PARENTHESIS
			);
		$this->popEvent[PARSER_EVENT_DEFINE_PARAMS_PARENTHESIS] = array(")");
##########################

		$this->pushEvent[PARSER_EVENT_INCLUDE_PARAMS_PARENTHESIS] =
			array(
				"("	=>	PARSER_EVENT_INCLUDE_PARAMS_PARENTHESIS
			);
		$this->popEvent[PARSER_EVENT_INCLUDE_PARAMS_PARENTHESIS] = array(")");
##########################

		$this->pushEvent[PARSER_EVENT_VAR] = 
			array(
				"\""	=> PARSER_EVENT_QUOTE,
				"'"	=> PARSER_EVENT_SINGLEQUOTE,
				"array" => PARSER_EVENT_ARRAY
			);
		$this->popEvent[PARSER_EVENT_VAR] = array(";");
##########################

		$this->pushEvent[PARSER_EVENT_COMMENT] = 
			array(
				"\\"	=> PARSER_EVENT_ESCAPE
			);
		$this->popEvent[PARSER_EVENT_COMMENT] = array("\n");
##########################

		$this->popEvent[PARSER_EVENT_COMMENTBLOCK] = array("*/");
##########################
		$this->pushEvent[PARSER_EVENT_SINGLEQUOTE] = 
			array(
				"\\"	=> PARSER_EVENT_ESCAPE
			);

		$this->popEvent[PARSER_EVENT_SINGLEQUOTE] = array("'");
##########################
		$this->pushEvent[PARSER_EVENT_FUNCTION_PARAMS] = 
			array(
				"\""	=> PARSER_EVENT_QUOTE,
				"'"	=> PARSER_EVENT_SINGLEQUOTE,
				"array" => PARSER_EVENT_ARRAY
			);
		$this->popEvent[PARSER_EVENT_FUNCTION_PARAMS] = array(")");
##########################
		$this->pushEvent[PARSER_EVENT_DOCKEYWORD] = 
			array(
				"<"	=> PARSER_EVENT_DOCKEYWORD_EMAIL,
				"{@" => PARSER_EVENT_INLINE_DOCKEYWORD,
			);

		$this->popEvent[PARSER_EVENT_DOCKEYWORD] = array("@","*/");
##########################

		$this->popEvent[PARSER_EVENT_INLINE_DOCKEYWORD] = array("}");
##########################

		$this->popEvent[PARSER_EVENT_OUTPHP] = array("<?php","<?");
##########################

		$this->popEvent[PARSER_EVENT_DOCKEYWORD_EMAIL] = array(">","\n");

##########################
		$this->pushEvent[PARSER_EVENT_ARRAY] = 
			array(
				"\""	=> PARSER_EVENT_QUOTE,
				"'"	=> PARSER_EVENT_SINGLEQUOTE,
				"array" => PARSER_EVENT_ARRAY
			);
		$this->popEvent[PARSER_EVENT_ARRAY] = array(")");
##########################
	}

	/**
	* tell the parser's WordParser {@link $wp} to set up tokens to parse words by.
	* tokens are word separators.  In English, a space or punctuation are examples of tokens.
	* In PHP, a token can be a ;, a parenthesis, or even the word "function"
	* @param	$value integer an event number
	* @see WordParser
	*/
	
	function configWordParser($e)
	{
		$this->wp->setSeperator($this->tokens[($e + 100)]);
	}

	/**
	* Debugging function, takes an event number and attempts to return its name
	* @param	$value integer an event number
	*/
	

	function getParserEventName ($value)
	{	
		$lookup = array(
			PARSER_EVENT_NOEVENTS 		=> "PARSER_EVENT_NOEVENTS",
			PARSER_EVENT_PHPCODE		=> "PARSER_EVENT_PHPCODE",
			PARSER_EVENT_DOCBLOCK		=> "PARSER_EVENT_DOCBLOCK",
			PARSER_EVENT_FUNCTION		=> "PARSER_EVENT_FUNCTION",
			PARSER_EVENT_CLASS		=> "PARSER_EVENT_CLASS",
			PARSER_EVENT_DEFINE		=> "PARSER_EVENT_DEFINE",
			PARSER_EVENT_DEFINE_PARAMS	=> "PARSER_EVENT_DEFINE_PARAMS",
			PARSER_EVENT_COMMENT		=> "PARSER_EVENT_COMMENT",
			PARSER_EVENT_COMMENTBLOCK	=> "PARSER_EVENT_COMMENTBLOCK",
			PARSER_EVENT_ESCAPE		=> "PARSER_EVENT_ESCAPE",
			PARSER_EVENT_QUOTE		=> "PARSER_EVENT_QUOTE",
			PARSER_EVENT_FUNCTION_PARAMS	=> "PARSER_EVENT_FUNCTION_PARAMS",
			PARSER_EVENT_SINGLEQUOTE	=> "PARSER_EVENT_SINGLEQUOTE",
			PARSER_EVENT_VAR		=> "PARSER_EVENT_VAR",
			PARSER_EVENT_LOGICBLOCK		=> "PARSER_EVENT_LOGICBLOCK",
			PARSER_EVENT_OUTPHP		=> "PARSER_EVENT_OUTPHP",
			PARSER_EVENT_DOCKEYWORD		=> "PARSER_EVENT_DOCKEYWORD",
			PARSER_EVENT_DOCKEYWORD_EMAIL	=> "PARSER_EVENT_DOCKEYWORD_EMAIL",
			PARSER_EVENT_ARRAY		=> "PARSER_EVENT_ARRAY",
			PARSER_EVENT_INLINE_DOCKEYWORD	=>	"PARSER_EVENT_INLINE_DOCKEYWORD",
			PARSER_EVENT_EOFQUOTE	=>	"PARSER_EVENT_EOFQUOTE",
			PARSER_EVENT_INCLUDE	=>	"PARSER_EVENT_INCLUDE",
			PARSER_EVENT_INCLUDE_PARAMS	=>	"PARSER_EVENT_INCLUDE_PARAMS",
			PARSER_EVENT_INCLUDE_PARAMS_PARENTHESIS	=> "PARSER_EVENT_INCLUDE_PARAMS_PARENTHESIS",
			PARSER_EVENT_DEFINE_PARAMS_PARENTHESIS => "PARSER_EVENT_DEFINE_PARAMS_PARENTHESIS",
		);
		if (isset($lookup[$value]))
		return $lookup[$value];
		else return $value;
	}
}

/**
* Global package page parser
*
* @package phpDocumentor
*/
class ppageParser extends Parser
{
	
	function setupStates()
	{
		$this->tokens[STATE_NOEVENTS]		= array("{@","}");
		$this->tokens[STATE_INLINE_DOCKEYWORD]		= array("{@","}","\t"," ");

##########################

		$this->pushEvent[PARSER_EVENT_NOEVENTS] = 
			array(
				"{@" => PARSER_EVENT_INLINE_DOCKEYWORD
			);
##########################

		$this->popEvent[PARSER_EVENT_INLINE_DOCKEYWORD] = array("}");
	}
	
	/**
	* Parse a new file
	*
	* @param	string	$parse_data
	* @param	string	$package
	* @param	int	$subpackage
	* @return	mixed	false or parsed data
	*/
	function parse (&$parse_data,$package = 'default',$subpackage = '')
	{
		if (!isset($subpackage) || !$subpackage) $subpackage = '';
		if (strlen($parse_data) == 0)
		{
			return false;
		}

		// initialize variables so E_ALL error_reporting doesn't complain
		$pevent = 0;
		$word = 0;
		$this->p_vars['event_stack'] = new EventStack;
		$total = '';

		$this->wp->setup($parse_data);

		$this->p_flags['reset_quote_data'] = true;

		do
		{
			$lpevent = $pevent;
			$pevent = $this->p_vars['event_stack']->getEvent();
			if ($lpevent != $pevent)
			{
				$this->p_vars['last_pevent'] = $lpevent;
			}

			if ($this->p_vars['last_pevent'] != $pevent)
			{
				// its a new event so the word parser needs to be reconfigured 
				$this->configWordParser($pevent);
			}

		
			$this->publishEvent(PHPDOC_EVENT_NEWSTATE,($pevent + 100));


			$this->p_vars['last_word'] = $word;
			$word = $this->wp->getWord();

			if (DEBUG == true)
			{
				echo "LAST: |" . $this->p_vars['last_word'] . "|\n";
				echo "PEVENT: " . $this->getParserEventName($pevent) . "\n";
				echo $this->wp->getPos() . ": |$word|\n";
			}
			switch ($pevent)
			{
				case PARSER_EVENT_NOEVENTS:
					if (!$this->checkEventPush( $word, $pevent)) $total .= $word;
				break;
				case PARSER_EVENT_INLINE_DOCKEYWORD:
//					echo $this->wp->getPos() . ": |$word|\n";

					if (!isset($this->p_vars['inline_dockeyword_type'])) $this->p_vars['inline_dockeyword_type'] = false;
//					echo "docktype: $this->p_vars['inline_dockeyword_type']\n";
					if (!isset($this->p_vars['inline_dockeyword_data'])) $this->p_vars['inline_dockeyword_data'] = '';
					if (!$this->p_vars['inline_dockeyword_type'])
					{
						if (in_array($word,$this->allowableInlineTags))
						{
							$this->p_vars['inline_dockeyword_type'] = strtolower($word);
						} else {
							$total .= '{@'.$word;
						}
					} else
					{
						if ($word != "}")
						{
							$this->p_vars['inline_dockeyword_data'] .= $word;
						}
					}
					if ($this->checkEventPop($word,$pevent))
					{
						if ($this->p_vars['inline_dockeyword_type']=='link')
						{
							// support hyperlinks of any protocol
							if (is_numeric(strpos($this->p_vars['inline_dockeyword_data'],'://')))
							{
								// if there is more than 1 parameter, the stuff after the space is the hyperlink text
								if (strpos(trim($this->p_vars['inline_dockeyword_data']),' '))
								{
									$i1 = strpos(trim($this->p_vars['inline_dockeyword_data']),' ') + 1;
									$link = substr(trim($this->p_vars['inline_dockeyword_data']),0,$i1 - 1);
									$text = substr(trim($this->p_vars['inline_dockeyword_data']),$i1);
									$this->p_vars['inline_dockeyword_data'] = '<a href="'.$link.'">'.$text.'</a>';
								}
								else
								$this->p_vars['inline_dockeyword_data'] = '<a href="'.$this->p_vars['inline_dockeyword_data'].'">'.$this->p_vars['inline_dockeyword_data'].'</a>';
							} elseif (isset($this->subscriber['*']->linker))
							{
//								debug('parsing '.$this->p_vars['inline_dockeyword_data']);
								$this->p_vars['inline_dockeyword_data'] = $this->subscriber['*']->linker->getLink(trim($this->p_vars['inline_dockeyword_data']),66,66,'',1);
							}
						}
						$total .= $this->p_vars['inline_dockeyword_data'];
						$this->p_vars['dockeyword_type'] = false;
						$this->p_vars['dockeyword_data'] = '';
					}
				break;
			}
		} while (!($word === false));
		$data = new DataPackagePage($total,$package,$subpackage);
		$this->PublishEvent(PHPDOC_EVENT_PACKAGEPAGE,$data);
	}
}
?>