<?php
/**
* Intermediate class parsing structure.
* This structure parses classes, vars and methods by file, and then iterates over the class tree to set up inheritance.  The {@link Inherit()}
* method is the meat of the class, and processes the class trees from root to branch, ensuring that parsing order is unimportant
* @package phpDocumentor
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: Classes.inc,v 1.22 2002/05/23 21:12:40 CelloG Exp $
*/
/**
* Intermediate class parsing structure.
* This structure parses classes, vars and methods by file, and then iterates over the class tree to set up inheritance.  The {@link Inherit()}
* method is the meat of the class, and processes the class trees from root to branch, ensuring that parsing order is unimportant
* @package phpDocumentor
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: Classes.inc,v 1.22 2002/05/23 21:12:40 CelloG Exp $
*/
class Classes
{
	/**
	* file being parsed, used in every add function to match up elements with the file that contains them
	* @see addClass(), addMethod(), addVar(), nextFile()
	* @var string
	*/
	var $curfile;
	/**
	* class being parsed, used to match up methods and vars with their parent class
	* @see addMethod(), addVar()
	* @var string
	*/
	var $curclass;
	
	/**
	* Used when a definite match is made between a parent class and a child class
	* Format: array(parent => array(parentfile => array(child => childfile)))
	* @var array
	*/
	var $definitechild;
	/**
	* array of parsed classes organized by the name of the file that contains the class.
	* Format:
	* array(filename => array(classname => {@link parserClass}))
	* @var array
	*/
	var $classesbyfile = array();
	/**
	* array of file names organized by classes that are in the file.
	* This structure is designed to handle name conflicts.  Two files can contain classes with the same name, and this array will
	* record both filenames to help control linking and inheritance errors
	* Format:
	* array(classname => array(name of file containing classname, name of file 2 containing classname...)
	* @var array
	*/
	var $classesbynamefile = array();
	/**
	* array of parsed methods organized by the file that contains them.
	* Format:
	* array(filename => array(classname => array({@link parserMethod} 1, {@link parserMethod} 2,...))
	* @var array
	*/
	var $methodsbyfile = array();
	/**
	* array of parsed vars organized by the file that contains them.
	* Format:
	* array(filename => array(classname => array({@link parserVar} 1, {@link parserVar} 2,...))
	* @var array
	*/
	var $varsbyfile = array();
	/**
	* keeps track of extend declarations by file, used to find inheritance
	* Format:
	* array(filename => array(classname => parentclassname))
	* @var array
	*/
	var $extendsbyfile = array();
	/**
	* Keeps track of child classes by file.
	* Since PHPDocumentor can document collections of files that contain name conflicts (PHP would give a fatal error), it
	* is impossible to assume a class that declares "extends foo" necessarily extends the class foo in file X.  It could be an
	* extended class of class foo in file Y.  Because of this, PHPDocumentor relies on packaging to resolve the name conflict
	* This array keeps track of the packages of a child class
	* Format:
	* array(parentclassname => array(filename => array(childclassname => array(packagename, packagename)))
	* @var array
	*/
	var $classchildrenbyfile = array();
	/**
	* Keeps track of class packages found in a file.
	* This is used in {@link getParentClass()} to determine the number of packages in a file, in order to resolve inheritance
	* issues
	* Format: array(filename => array(packagename1, packagename2,...))
	* @var array
	*/
	var $classpackagebyfile = array();
	/**
	* post-inheritance data structure passed to IntermediateParser.  Format is same as $classtree
	* @see IntermediateParser::$classtree
	* @var array
	*/
	var $classparents = array();
	/**
	* post-inheritance data structure passed to IntermediateParser.  Format is same as $classpackages
	* @see IntermediateParser::$classpackages
	* @var array
	*/
	var $classpackages = array();
	/**
	* post-inheritance data structure passed to IntermediateParser.  Format is same as $methods
	* @see IntermediateParser::$methods
	* @var array
	*/
	var $methods = array();
	/**
	* post-inheritance data structure passed to IntermediateParser.  Format is same as $methods
	* @see IntermediateParser::$vars
	* @var array
	*/
	var $vars = array();
	/** @access private */
	var $revcpbf = array();
	/**
	* All classes with no parents (no extends clause) are tracked in this array by the file that contains them.
	* Format: array(classname => array(name of file1 that contains root classname, name of file2 that contains root classname,...))
	* @var array
	*/
	var $roots = array();
	
	/**
	* @access private
	*/
	var $potentialclassconflicts = array();
	
	var $classconflicts = array();
	
	/**
	* sets up the {@link $classesbyfile, $classesbynamefile, $extendsbyfile, $classchildrenbyfile, $roots} arrays, and sets {@link $curclass}
	* @param parserClass &$element element is a {@link parserClass}
	*/
	function addClass(&$element)
	{
		$this->curclass = $element->getName();
		$element->curfile = $this->curfile;
		$this->classesbyfile[$this->curfile][$this->curclass] = $element;
		$this->classesbynamefile[$this->curclass][] = $this->curfile;
		$this->extendsbyfile[$this->curfile][$this->curclass] = $element->getExtends();
		$this->classchildrenbyfile[$element->getExtends()][$this->curfile][$this->curclass][] = $element->docblock->package;
		if (!$element->getExtends())
		{
			$this->roots[$this->curclass][] = $this->curfile;
		}
		$this->addPackageToFile($element->docblock->package);
	}
	
	/**
	* sets up the {@link $methodsbyfile} array using {@link $curfile} and {@link $curclass}
	* @param parserMethod &$element element is a {@link parserMethod}
	*/
	function addMethod(&$element)
	{
		$this->methodsbyfile[$this->curfile][$this->curclass][] = $element;
	}
	
	/**
	* sets up the {@link $varsbyfile} array using {@link $curfile} and {@link $curclass}
	* @param parserVar &$element element is a {@link parserVar}
	*/
	function addVar(&$element)
	{
		$this->varsbyfile[$this->curfile][$this->curclass][] = $element;
	}
	
	/**
	* sets {@link $curfile} to $file and {@link $curclass} to false (no class being parsed)
	* @param string $file file currently being parsed
	*/
	function nextFile($file)
	{
		$this->curfile = $file;
		$this->curclass = false;
	}
	
	/**
	* sets up the {@link $methodsbyfile} array using {@link $curfile} and {@link $curclass}
	* @param string $package package name
	*/
	function addPackageToFile($package)
	{
		// don't care about default
		if ($package == $GLOBALS['PHPDocumentor_DefaultPackageName'])
		$this->revcpbf[$this->curfile][$package] = 1;
		if (!isset($this->revcpbf[$this->curfile][$package]))
		$this->classpackagebyfile[$this->curfile][] = $package;
		$this->revcpbf[$this->curfile][$package] = 1;
	}
	
	/**
	* uses {@link getParentClass()} to find the parent class, and then modifies the {@link parserClass} element in
	* {@link $classesbyfile} to use the parent's package, and inherit methods/vars
	* @param string $class child class to find parent class
	* @param string $file file child class is located in
	*/
	function setClassParent($class,$file)
	{
		if (is_array($par = $this->getParentClass($class,$file)))
		{
//			phpdoc_out("$file class $class extends ".$par[1]." file ".$par[0]."\n");
			$this->classesbyfile[$file][$class]->setParent($par[1],$par[0],$this);
			$this->definitechild[$par[1]][$par[0]][$class] = $file;
		} else
		{
			$this->classesbyfile[$file][$class]->setParentNoClass($par);
		}
	}
	
	/**
	* Main processing engine for setting up class inheritance.
	* This function uses {@link $roots} to traverse the inheritance tree via {@link processChild()} and returns
	* the data structures IntermediateParser needs to convert parsed data to output using {@link IntermediateParser::Convert()}
	* @param IntermediateParser &$render {@link IntermediateParser} that needs the updated elements
	*/
	function Inherit(&$render)
	{
		phpdoc_out("\nProcessing Class Inheritance\n\n");
		phpdoc_out("\nProcessing Root Trees\n\n");
		foreach($this->roots as $class => $files)
		{
			for ($i=0; $i<count($files); $i++)
			{
				$this->processChild($render,$class,$files[$i]);
			}
		}
		phpdoc_out("\nProcessing leftover classes (classes that extend root classes not found in this package)\n");
		foreach($this->classesbyfile as $i => $j)
		{
			foreach($j as $k => $m)
			{
				$this->processChild($render,$k,$i,true);
			}
		}
		phpdoc_out("done processing leftover classes\n");
		$this->setupClassConflicts();
	}
	
	/**
	* @access private
	*/
	function setupClassConflicts()
	{
		foreach($this->potentialclassconflicts as $class => $files)
		{
			if (count($files) - 1)
			{
				$packages = array();
				$fileclass = array();
				for($i=0;$i<count($files);$i++)
				{
					if (in_array($this->classesbyfile[$files[$i]][$class]->docblock->package,$packages))
					{
						$fur = $files[$i];
						for($i=0;$i<count($files);$i++)
						{
							if (isset($filepackage[$files[$i]]))
							{
								if ($filepackage[$files[$i]] == $this->classesbyfile[$fur][$class]->docblock->package) $fer = $i;
							}
						}
						$i = count($files) + 1;
						// error
						addError(PDERROR_CLASS_CONFLICT,$class,$this->classesbyfile[$fur][$class]->docblock->package,$fur,$files[$fer]);
						$this->classconflicts[$class] = $files;
					} else
					{
						$filepackage[$files[$i]] = $this->classesbyfile[$files[$i]][$class]->docblock->package;
						$packages[] = $this->classesbyfile[$files[$i]][$class]->docblock->package;
					}
				}
			}
		}
	}
	
	/**
	* This function recursively climbs up the class tree, setting inherited information like package and adds the elements
	* to the IntermediateParser that needs them.
	* Using structures defined in {@link Classes}, the function first sets package information, and then seeks out child classes.
	* It uses 3 tests to determine whether a class is a child class.
	* <ol>
	*	<li>child class is in the same file as the parent class and extends parent class</li>
	*	<li>child class is in a different file and specifies the parent's @package in its docblock</li>
	*	<li>child class is in a different file and is in a different @package, with one possible parent class</li>
	* </ol>
	* @param IntermediateParser &$render {@link IntermediateParser} class that needs the processed child information
	* @param string $class class to process
	* @param string $file name of file $class is located in
	* @param boolean $furb flag used privately to control informational output while parsing
	*							(used when processing leftover classes in {@link Inherit()}
	*/
	function processChild(&$render,$class,$file,$furb = false)
	{
		if (isset($this->classesbyfile[$file][$class]->processed)) return;
		$this->potentialclassconflicts[$class][] = $file;
		if ($furb) phpdoc_out("Processing $class in file $file\n");
		$this->classesbyfile[$file][$class]->processed = true;
		if ($furb) $this->roots[$class][] = $file;
		$this->setClassParent($class,$file);
		$db = $this->classesbyfile[$file][$class];
		$render->addElement($db->docblock->package,$db->docblock->subpackage,$db,$file);
		$render->pages[$file]->addElement($db);
		if (isset($this->varsbyfile[$file]) && isset($this->varsbyfile[$file][$class]))
		foreach($this->varsbyfile[$file][$class] as $vr)
		{
			$vr->docblock->package = $db->docblock->package;
			$vr->docblock->subpackage = $db->docblock->subpackage;
			$render->addElement($db->docblock->package,$db->docblock->subpackage,$vr,$file);
			$render->pages[$file]->addElement($vr);
			$this->vars[$db->docblock->package][$class][$vr->getName()] = $vr;
		}
		if (isset($this->methodsbyfile[$file]) && isset($this->methodsbyfile[$file][$class]))
		foreach($this->methodsbyfile[$file][$class] as $vr)
		{
			$vr->docblock->package = $db->docblock->package;
			$vr->docblock->subpackage = $db->docblock->subpackage;
			$render->addElement($db->docblock->package,$db->docblock->subpackage,$vr,$file);
			$render->pages[$file]->addElement($vr);
			$this->methods[$db->docblock->package][$class][$vr->getName()] = $vr;
		}
		$this->classpackages[$class][] = array($db->docblock->package,$db->docblock->subpackage);
		if (is_array($db->parent))
		$this->classparents[$db->docblock->package][$class] = $db->parent[1];
		else
		$this->classparents[$db->docblock->package][$class] = $db->getExtends();
		if (is_array($db->parent))
		{
			$z = $this->getClass($db->parent[1],$db->parent[0]);
			$this->classchildren[$z->docblock->package][$db->parent[1]][] = $db;
		}
		if (isset($this->classchildrenbyfile[$class]))
		{
			foreach($this->classchildrenbyfile[$class] as $childfile => $other)
			{
				// test 1, inherits in same file (must be same package)
				if ($childfile == $file)
				{
					foreach($other as $child => $packages)
					{
//						debug("parent $class same file $child");
						$this->processChild($render,$child,$childfile);
						$x = $this->getClass($child,$childfile);
						if (($x->docblock->package != $GLOBALS['PHPDocumentor_DefaultPackageName']) && ($x->docblock->package != $db->docblock->package))
						{
							// child package need root for class trees
							if ($x->docblock->package != $db->docblock->package)
							$this->roots[$child][] = $childfile;
						}
					}
				} else
				{
				// test 2, different file, same package
					foreach($other as $child => $packages)
					{
						for($j=0; $j<count($packages); $j++)
						{
							if ($this->classesbyfile[$file][$class]->docblock->package == $packages[$j])
							{
								$this->processChild($render,$child,$childfile);
//								debug("$childfile diff file $child, parent $class, same package ".$packages[$j]);
							} else
							{
								// test 3, different file, different package, only 1 parent is possible
								if (isset($this->classesbynamefile[$child]))
								{
									// 1 possible
									if (count($this->classesbynamefile[$child]) == 1)
									{
//										debug("$childfile diff file $child, diff package, 1 possible parent root $class");
										$this->processChild($render,$child,$childfile);
										if ($packages[$j] != $GLOBALS['PHPDocumentor_DefaultPackageName'])
										$this->roots[$child][] = $childfile;
									}
								}
							}
						}
					}
				}
			}
		}
	}
	
	/**
	* @return parserClass
	* @param string $class classname
	* @param string $file file classname is located in
	*/
	function &getClass($class,$file)
	{
		return $this->classesbyfile[$file][$class];
	}
	
	/**
	* @return parserClass
	* @param string $class classname
	* @param string $package package classname is in
	*/
	function &getClassByPackage($class,$package)
	{
		for($i=0; $i < count($this->classesbynamefile[$class]); $i++)
		{
			if ($this->classesbyfile[$this->classesbynamefile[$class][$i]][$class]->getPackage() == $package)
			return $this->classesbyfile[$this->classesbynamefile[$class][$i]][$class];
		}
	}
	
	/**
	* uses 3 tests to find the parent classname:
	* <ol>
	*	<li>only one class with the parent classname</li>
	*	<li>more than one class, but only one in the same file as the child</li>
	*	<li>only one parent class in the same package as the child</li>
	* </ol>
	* @return mixed false if no parent class, a string if no parent class found by that name,
	*				and an array(file parentclass is in,parentclassname)
	*/
	function getParentClass($class,$file)
	{
		if (!isset($this->classesbyfile[$file][$class]))
		{
			// possibly raise error
			return false;
		}
		$element = $this->classesbyfile[$file][$class];
		if (!($ex = $element->getExtends())) return false;
		// first check to see if there is one and only one class with the parent class's name
		if (isset($this->classesbynamefile[$ex]))
		{
			if (count($this->classesbynamefile[$ex]) == 1)
			{
				return array($this->classesbynamefile[$ex][0],$ex);
			} else
			{
				// next check to see if there is a parent class in the same file
				if (isset($this->classesbyfile[$file][$ex])) return array($file,$ex);
				// next check to see if there is only one package used in the file, try to resolve it that way
				if (isset($this->classpackagebyfile[$file]))
				{
					if (count($this->classpackagebyfile[$file]) == 1)
					{
						for($i=0;$i<count($this->classesbynamefile[$ex]);$i++)
						{
							if ($this->classesbyfile[$this->classesbynamefile[$element->getExtends()][$i]][$ex]->getPackage() == $this->classpackagebyfile[$file][0])
								return array($this->classesbynamefile[$ex][$i],$ex);
						}
					}
				}
				// name conflict
				addWarning(PDERROR_INHERITANCE_CONFLICT, $class, $file, $ex);
				return $ex;
			}
		} else
		{
			addWarning(PDERROR_PARENT_NOT_FOUND, $class, $ex);
			return $ex;
		}
	}
	
	/**
	* returns a tree of parent classes for the class located in file $file
	* @return array format array(classname => parentclass,classname2 =>parentclass2)
	* @param string $class classname
	* @param string $file file class is found in
	*/
	function getParClassTree($class,$file)
	{
		$partree = array();
		$par = $this->getParentClass($class,$file);
		if ($par)
		{
			if (is_array($par))
			$partree[$par[1]] = $this->getParClassTreeBranch($par);
			else
			{
				$partree[$par] = array($par,false);
				return $partree;
			}
		} else return array($class => false);
	}
	
	/**
	* @return array array(package => array(rootclassname, rootclassname,...),...)
	*/
	function getRoots()
	{
		$roots = array();
		foreach($this->roots as $class => $files)
		{
			for ($i=0; $i<count($files); $i++)
			{
				$x = $this->getClass($class,$files[$i]);
				$roots[$x->getPackage()][] = $class;
			}
		}
		foreach($roots as $package => $root)
		{
			usort($roots[$package],"strnatcasecmp");
		}
		return $roots;
	}
	
	/**
	* Get all classes confirmed in parsing to be descended class $parclass in file $file
	* @return mixed either false if no children, or array of format array(childname => childfile,childname2 => childfile2,...)
	* @param string $parclass name of parent class
	* @param string $file file parent class is found in
	* @see parserClass::getChildClassList()
	*/
	function getDefiniteChildren($parclass,$file)
	{
		if (isset($this->definitechild[$parclass][$file])) return $this->definitechild[$parclass][$file];
		return false;
	}
	
	function sortRoots()
	{
	}
}
?>
