<?php
/**
* Base class for all output converters.
* A Converter takes output from the {@link Parser} and converts it to template-friendly output.  A converter for the standard PHPDocumentor
* template, {@link HTMLConverter}, is provided with this release.  Future releases will have support for other formats and templates, including
* DocBook, XML, and possibly other HTML templates.  The converter takes output directly from {@link IntermediateParser} and using {@link walk()} it
* "walks" over an array of PHPDocumentor elements, as represented by descendants of {@link parserElement}
*
* a converter must define the abstract function Convert (an example is {@link HTMLConverter::Convert()}), a function that takes a passed object
* and uses it to generate structures for an output template, or directly generates output.  Since all elements have their DocBlocks linked
* directly, this allows for more logical parsing than earlier versions of PHPDocumentor.
*
* A Converter is passed several data structures in its constructor, all of which are optional in output and may be handled in any way
* the converter would like.  These structures are arrays:
* <ul>
*	<li>array of methods by class (see {@link IntermediateParser::$methods})</li>
*	<li>array of variables by class (see {@link IntermediateParser::$vars})</li>
*	<li>array of links to documented elements (see {@link IntermediateParser::$links})</li>
*	<li>array of class parents by name (see {@link IntermediateParser::$classtree})</li>
*	<li>array of class packages by classname (see {@link IntermediateParser::$classpackages})</li>
*	<li>array of packages to document (see {@link IntermediateParser::$packageoutput})</li>
*	<li>array of all documented elements by name (see {@link IntermediateParser::$elements})</li>
*	<li>array of all documented elements by name, split by package (see {@link IntermediateParser::$pkg_elements})</li>
*	<li>boolean option, set to true to parse all elements marked @access private (see {@link IntermediateParser::$parsePrivate})</li>
*	<li>boolean option, set to true to stop informative output while parsing (good for cron jobs) (see {@link IntermediateParser::$quietMode})</li>
*	<li>a {@link Classes} object that contains all class information adjusted for proper inheritance</li>
*	<li>string template directory</li>
*	<li>string output directory</li>
* </ul>
* @package phpDocumentor
* @subpackage Converters
* @abstract
* @see parserDocBlock, parserInclude, parserPage, parserClass, parserDefine, parserFunction, parserMethod, parserVar
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: Converter.inc,v 1.42 2002/05/23 19:12:18 CelloG Exp $
*/
/**
* Base class for all output converters.
* A Converter takes output from the {@link Parser} and converts it to template-friendly output.  A converter for the standard PHPDocumentor
* template, {@link HTMLConverter}, is provided with this release.  Future releases will have support for other formats and templates, including
* DocBook, XML, and possibly other HTML templates.  The converter takes output directly from {@link IntermediateParser} and using {@link walk()} it
* "walks" over an array of PHPDocumentor elements, as represented by descendants of {@link parserElement}
*
* a converter must define the abstract function Convert (an example is {@link HTMLConverter::Convert()}), a function that takes a passed object
* and uses it to generate structures for an output template, or directly generates output.  Since all elements have their DocBlocks linked
* directly, this allows for more logical parsing than earlier versions of PHPDocumentor.
*
* A Converter is passed several data structures in its constructor, all of which are optional in output and may be handled in any way
* the converter would like.  These structures are arrays:
* <ul>
*	<li>array of methods by class (see {@link IntermediateParser::$methods})</li>
*	<li>array of variables by class (see {@link IntermediateParser::$vars})</li>
*	<li>array of links to documented elements (see {@link IntermediateParser::$links})</li>
*	<li>array of class parents by name (see {@link IntermediateParser::$classtree})</li>
*	<li>array of class packages by classname (see {@link IntermediateParser::$classpackages})</li>
*	<li>array of packages to document (see {@link IntermediateParser::$packageoutput})</li>
*	<li>array of all documented elements by name (see {@link IntermediateParser::$elements})</li>
*	<li>array of all documented elements by name, split by package (see {@link IntermediateParser::$pkg_elements})</li>
*	<li>boolean option, set to true to parse all elements marked @access private (see {@link IntermediateParser::$parsePrivate})</li>
*	<li>boolean option, set to true to stop informative output while parsing (good for cron jobs) (see {@link IntermediateParser::$quietMode})</li>
*	<li>a {@link Classes} object that contains all class information adjusted for proper inheritance</li>
*	<li>string template directory</li>
*	<li>string output directory</li>
* </ul>
* @package phpDocumentor
* @subpackage Converters
* @abstract
* @see parserDocBlock, parserInclude, parserPage, parserClass, parserDefine, parserFunction, parserMethod, parserVar
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: Converter.inc,v 1.42 2002/05/23 19:12:18 CelloG Exp $
*/
class Converter
{
	/**
	* @var string
	*/
	var $package = 'default';
	/**
	* @var string
	*/
	var $subpackage = '';
	/**
	* set to a classname if currently parsing a class, false if not
	* @var mixed
	*/
	var $class = false;
	/**
	* array of classes descended from {@link abstractLink}
	* @see functionLink, defineLink, classLink, pageLink, methodLink, varLink
	* @var array
	*/
	var $links;
	/**
	* array of classes descended from {@link abstractLink} organized by file
	* @see functionLink, defineLink, classLink, methodLink, varLink
	* @var array
	*/
	var $linkswithfile;
	/**
	* @var array
	*/
	var $methods;
	/**
	* @var array
	*/
	var $vars;
	/**
	* @var array
	*/
	var $class_parents_by_name;
	/**
	* @var mixed
	*/
	var $package_output;
	/**
	* @var array
	*/
	var $class_packages;
	/**
	* @var array
	*/
	var $elements;
	/**
	* @var array
	*/
	var $pkg_elements;
	/**
	* @var bool
	*/
	var $parseprivate;
	/**
	* @var bool
	*/
	var $quietmode;
	
	var $curfile;
	
	var $classes;
	
	function Converter(&$methods, &$vars, &$links, &$linksw, &$c, $po, &$cp, &$els, &$pkg_els, $pp, $qm, &$classes, $templateDir, $targetDir)
	{
		$this->package = $GLOBALS['PHPDocumentor_DefaultPackageName'];
		$this->links = $links;
		$this->linkswithfile = $linksw;
		$this->methods = $methods;
		$this->vars = $vars;
		$this->class_parents_by_name = $c;
		$this->class_packages = $cp;
		$this->package_output = $po;
		$this->elements = $els;
		$this->pkg_elements = $pkg_els;
		$this->parseprivate = $pp;
		$this->quietmode = $qm;
		$this->classes = $classes;
		$this->classes->sortroots();
		$this->roots = $classes->getRoots();
	}
	
	/**
	* @abstract
	*/
	function endClass()
	{
	}
	
	/**
	* @abstract
	*/
	function endPage()
	{
	}
	
	/**
	* called by {@link IntermediateParser::Convert()} to traverse the array of pages and their elements, converting them to the output format
	*/
	function walk(&$pages,&$package_pages)
	{
		foreach($pages as $j => $flub)
		{
			phpdoc_out('Converting '.$pages[$j]->parent->getFile()."\n");
			flush();
			$this->package = $pages[$j]->parent->package;
			$this->subpackage = $pages[$j]->parent->subpackage;
			$this->class = false;
			$this->curfile = $pages[$j]->parent->getFile();
			if ($this->package_output)
			{
				if (!in_array($this->package,$this->package_output))
				$this->Convert($pages[$j]);
			} else $this->Convert($pages[$j]);
			for($i=0; $i<count($pages[$j]->elements); $i++)
			{
				$this->package = $pages[$j]->elements[$i]->docblock->package;
				if ($this->package_output) if (!in_array($this->package,$this->package_output)) continue;
				$this->subpackage = $pages[$j]->elements[$i]->docblock->subpackage;
				if ($this->class)
				{
					if ($pages[$j]->elements[$i]->type != 'var' && $pages[$j]->elements[$i]->type != 'method')
					{
						$this->class = false;
						$this->endClass();
					}
					if ($pages[$j]->elements[$i]->type == 'class') $this->class = $pages[$j]->elements[$i]->name;
				} else
				{
					if ($pages[$j]->elements[$i]->type == 'class')
					{
						$this->class = $pages[$j]->elements[$i]->name;
					}
				}
				$this->Convert($pages[$j]->elements[$i]);
			}
			$this->endPage();
		}
		foreach($package_pages as $i => $perp)
		{
			phpdoc_out('Converting package page for package '.$package_pages[$i]->package."\n");
			$this->package = $package_pages[$i]->package;
			$this->subpackage = '';
			$this->class = false;
			$this->Convert($package_pages[$i]);
		}
	}
	
	/**
	* Generate a simple class tree array from $branch
	*
	* @param	string	$branch name of the child class
	* @param string $package package name this class is in
	* @return	array	[parent] => [child]
	* @see generateFormattedClassTree()
	*/
	function generateClassTreeFromClass($branch,$package)
	{
		if (!isset($this->class_parents_by_name[$package])) return false;
		$tree = $this->class_parents_by_name[$package];
		if (!$tree) return array();
		while(isset($tree[$branch]) && $tree[$branch])
		{
			$branches[$branch] = $tree[$branch];
			$branch = $tree[$branch];
		}
		$branches[$branch] = 0;
		return $branches;
	}
	
	/**
	* @param string $class name of a class
	* @return mixed returns the name of $class's package, or false if not found or a name conflict between packages exists
	*/
	function getClassPackage($class)
	{
		foreach($this->class_parents_by_name as $package => $pbn)
		{
			$class2 = $class1 = $class;
			while(isset($pbn[$class1]))
			{
				if ($pbn[$class1])
				{
					$class2 = $pbn[$class1];
				}
				$class1 = $pbn[$class1];
			}
			if (isset($this->class_packages[$class2]))
			{
				if (count($this->class_packages[$class2]) != 1) return false;
				return $this->class_packages[$class2][0];
			}
		}
		return false;
	}
	
	/**
	* @return bool true if a link to this class exists in package $package and subpackage $subpackage
	* @param string $expr class name
	* @param string $package package to search in
	* @param string $subpackage subpackage to search in
	*/
	function isLinkedClass($expr,$package,$subpackage,$file=false)
	{
		return isset($this->links[$package][$subpackage]['class'][$expr]);
	}
	
	/**
	* @return bool true if a link to this function exists in package $package and subpackage $subpackage
	* @param string $expr function name
	* @param string $package package to search in
	* @param string $subpackage subpackage to search in
	*/
	function isLinkedFunction($expr,$package,$subpackage,$file=false)
	{
		return isset($this->links[$package][$subpackage]['function'][$expr]);
	}
	
	/**
	* @return bool true if a link to this define exists in package $package and subpackage $subpackage
	* @param string $expr define name
	* @param string $package package to search in
	* @param string $subpackage subpackage to search in
	*/
	function isLinkedDefine($expr,$package,$subpackage,$file=false)
	{
		return isset($this->links[$package][$subpackage]['define'][$expr]);
	}
	
	/**
	* @return bool true if a link to this procedural page exists in package $package and subpackage $subpackage
	* @param string $expr procedural page name
	* @param string $package package to search in
	* @param string $subpackage subpackage to search in
	*/
	function isLinkedPage($expr,$package,$subpackage)
	{
		return isset($this->links[$package][$subpackage]['page'][$expr]);
	}
	
	/**
	* @return bool true if a link to this method exists in package $package, subpackage $subpackage and class $class
	* @param string $expr method name
	* @param string $class class name
	* @param string $package package to search in
	* @param string $subpackage subpackage to search in
	*/
	function isLinkedMethod($expr,$package,$subpackage,$class,$file=false)
	{
		return isset($this->links[$package][$subpackage]['method'][$class][$expr]);
	}
	
	/**
	* @return bool true if a link to this method exists in package $package, subpackage $subpackage and class $class
	* @param string $expr method name
	* @param string $class class name
	* @param string $package package to search in
	* @param string $subpackage subpackage to search in
	*/
	function isLinkedVar($expr,$package,$subpackage,$class,$file=false)
	{
		return isset($this->links[$package][$subpackage]['var'][$class][$expr]);
	}
	
	/**
	* @param string $expr class name
	* @param string $package package name
	* @return mixed returns a {@link classLink} or false if the element is not found in package $package
	* @see classLink
	*/
	function getClassLink($expr,$package,$file=false)
	{
		if (!isset($this->links[$package])) return false;
		foreach($this->links[$package] as $subpackage => $notused)
		{
			if ($this->isLinkedClass($expr,$package,$subpackage,$file))
			{
				if ($file)
				return $this->linkswithfile[$package][$subpackage]['class'][$file][$expr];
				return $this->links[$package][$subpackage]['class'][$expr];
			}
		}
		return false;
	}

	/**
	* @param string $expr function name
	* @param string $package package name
	* @return mixed returns a {@link functionLink} or false if the element is not found in package $package
	* @see functionLink
	*/
	function getFunctionLink($expr,$package,$file=false)
	{
		if (!isset($this->links[$package])) return false;
		foreach($this->links[$package] as $subpackage => $notused)
		{
			if ($this->isLinkedFunction($expr,$package,$subpackage,$file))
			{
				if ($file)
				return $this->linkswithfile[$package][$subpackage]['function'][$file][$expr];
				return $this->links[$package][$subpackage]['function'][$expr];
			}
		}
		return false;
	}

	/**
	* @param string $expr constant name
	* @param string $package package name
	* @return mixed returns a {@link defineLink} or false if the element is not found in package $package
	* @see defineLink
	*/
	function getDefineLink($expr,$package,$file=false)
	{
		if (!isset($this->links[$package])) return false;
		foreach($this->links[$package] as $subpackage => $notused)
		{
			if ($this->isLinkedDefine($expr,$package,$subpackage,$file))
			{
				if ($file)
				return $this->linkswithfile[$package][$subpackage]['define'][$file][$expr];
				return $this->links[$package][$subpackage]['define'][$expr];
			}
		}
		return false;
	}

	/**
	* @param string $expr procedural page name
	* @param string $package package name
	* @return mixed returns a {@link pageLink} or false if the element is not found in package $package
	* @see pageLink
	*/
	function getPageLink($expr,$package)
	{
		if (!isset($this->links[$package])) return false;
		foreach($this->links[$package] as $subpackage => $notused)
		{
			if ($this->isLinkedPage($expr,$package,$subpackage)) return $this->links[$package][$subpackage]['page'][$expr];
		}
		return false;
	}

	/**
	* @param string $expr method name
	* @param string $class class name
	* @param string $package package name
	* @return mixed returns a {@link methodLink} or false if the element is not found in package $package, class $class
	* @see methodLink
	*/
	function getMethodLink($expr,$class,$package,$file=false)
	{
		if (!isset($this->links[$package])) return false;
		foreach($this->links[$package] as $subpackage => $notused)
		{
			if ($this->isLinkedMethod($expr,$package,$subpackage,$class,$file))
			{
				if (0)//($file && !isset($this->linkswithfile[$package][$subpackage]['method'][$file]))
				{
					var_dump($package,$subpackage,'method',$file,$class,$expr,$this->linkswithfile[$package][$subpackage]['method']);
				}
				if ($file)
				return $this->linkswithfile[$package][$subpackage]['method'][$file][$class][$expr];
				return $this->links[$package][$subpackage]['method'][$class][$expr];
			}
		}
		return false;
	}

	/**
	* @param string $expr var name
	* @param string $class class name
	* @param string $package package name
	* @return mixed returns a {@link varLink} or false if the element is not found in package $package, class $class
	* @see varLink
	*/
	function getVarLink($expr,$class,$package,$file=false)
	{
		if (!isset($this->links[$package])) return false;
		foreach($this->links[$package] as $subpackage => $notused)
		{
			if ($this->isLinkedVar($expr,$package,$subpackage,$class,$file))
			{
				if ($file)
				return $this->linkswithfile[$package][$subpackage]['var'][$file][$class][$expr];
				return $this->links[$package][$subpackage]['var'][$class][$expr];
			}
		}
		return false;
	}

	/**
	* The meat of the see tag
	*
	* $expr is a string with many allowable formats:
	* <ol>
	*	<li>proceduralpagename.ext</li>
	*	<li>constant_name</li>
	*	<li>classname::function()</li>
	*	<li>classname::$variablename</li>
	*	<li>classname</li>
	* </ol>
	* If classname:: is not present, and the see tag is in a documentation block within a class, then the function uses the classname to
	* search for $expr as a function or variable within classname, or any of its parent classes.
	* given an $expr without '$', '::' or '()' getLink first searches for classes, procedural pages, constants, and then searches for
	* methods and variables within the default class, and finally for any function
	*
	* @param string $expr expression to search for a link
	* @param mixed $classtree internal recursion variable is set to 66 when not used, is set to an array returned from {@link generateClassTreeFromClass()} otherwise
	* @param mixed $par internal recursion variable is set to 66 when not used, is set to the class to be prepended to $expr in attempts to find the link ($par::$expr)
	* @param string $get internal recursion variable is '' when not used, is set to $expr after formatting for link searching,
	*                    by removing () from function names, and extraneous spaces.
	* @param string $orgget internal recursion variable set to '' when not used, is set to the original value of $expr
	* @return mixed getLink returns a descendant of {@link abstractLink} if it finds a link, otherwise it returns a string
	* @see parserDocBlock, Render::HandleEvent(), getPageLink(), getDefineLink(), getVarLink(), getFunctionLink(), getClassLink()
	* @see pageLink, functionLink, defineLink, classLink, methodLink, varLink
	*/
	function &getLink($expr,$classtree = 66,$par = 66,$get = '',$orgget = '')
	{
		$links = &$this->links;
		$package = $this->package;
		$class = $this->class;
		if ($test = $this->getClassLink($expr,$package)) return $test;
		if ($test = $this->getPageLink($expr,$package)) return $test;
		if ($test = $this->getDefineLink($expr,$package)) return $test;
		// is $expr a comma-delimited list?
		if (strpos($expr,','))
		{
			$a = explode(',',$expr);
			$b = array();
			for($i=0;$i<count($a);$i++)
			{
				// if so return each component with a link
				$b[] = $this->getLink(trim($a[$i]),66,66,'',$orgget);
			}
			return $b;
		}
		// are we in a class?
		if ($class)
		{
			// recursion test
			if ($classtree == 66)
			{
				// top-level, generate the class tree for the class currently being parsed
				$classtree = $this->generateClassTreeFromClass($class, $package);
				$par = $class;
			}
			// is $expr simply a word? if so, check to see if it is a method or variable of this class tree
			if (!strpos($expr,'::'))
			{
				$orgget = $expr;
				// strip any commentary "function xxx", leave "xxx" to do "classname::xxx"
				$get = str_replace('function ','',$expr);
				// if get is neither get() nor $get, assume get is a function, add () to make get()
				if (!is_numeric(strpos($get,'$'))&&!strpos($get,'()')) $get = $get.'()';
				// recur
				$a = $this->getLink($par.'::'.$get,$classtree,$par,$get,$orgget);
				// test to see whether getLink returned a link, or returned the value unmodified
				if (is_object($a)) return $a;
				else
				{
					// recursion test: $classtree is empty at the root object
					if (count($classtree))
					{
						$tpar = $par;
						// set $par to the parent of the object we just examined, and set $classtree to the parent's class tree
						$par = $classtree[$par];
						unset($classtree[$tpar]);
						// recur
						$a  = $this->getLink($par.'::'.$get,$classtree,$par,$get,$orgget);
						// test to see whether getLink returned a link, or returned the value unmodified
						if (is_object($a)) return $a;
					}
				}
			} elseif (!empty($get))
			{
				// get is set if we are testing a classname::get (which is $expr)
				$a = $this->getLink($expr,66,66,'',$orgget);
				// test to see whether getLink returned a link, or returned the value unmodified
				if (is_object($a)) return $a;
				else
				{
					// go deeper!
					if (count($classtree))
					{
						$tpar = $par;
						$par = $classtree[$par];
						unset($classtree[$tpar]);
						if (!$par) $expr = $orgget;
						else
						return $this->getLink($par.'::'.$get,$classtree,$par,$get,$orgget);
					}
				}
				
			}
		}
//		echo $this->render->parent.' '.$expr."\n";
//		flush();

		// is $expr in class::method() or class::$variable format?
		if (@strpos($expr,'::'))
		{
			$class_method = explode('::',$expr);
			if (strpos($class_method[1],'()'))
			{
				// strip everything but the function name, return a link
				if ($test = Converter::getMethodLink(str_replace('function ','',str_replace('()','',$class_method[1])), $class_method[0], $package)) return $test;
			}
			if ($test = Converter::getVarLink($class_method[1], $class_method[0], $package)) return $test;
		}
		// $expr does not have ::
		if (is_numeric(@strpos('$',$expr)))
		{
			// default to current class, whose name is contained in $this->render->parent
			if ($test = Converter::getVarLink($expr, $class, $package)) return $test;
		}
		// $expr is a function? (non-method)
		if (@strpos($expr,'()'))
		{
			// extract the function name, use it to retrieve the file that the function is in
//			$page = $this->func_page[str_replace('function ','',str_replace('()','',$expr))];
			// return the link
			if ($test = Converter::getFunctionLink(str_replace('function ','',str_replace('()','',$expr)), $package)) return $test;
		}
		// $expr is just a word.  First, test to see if it is a function of the current package
		if ($test = $this->getFunctionLink(str_replace('function ','',str_replace('()','',$expr)), $package)) return $test;
		// otherwise, see if it is a method
		if ($class)
		{
			if ($test = Converter::getMethodLink(str_replace('function ','',str_replace('()','',$expr)), $class, $package)) return $test;
		}
		// no links found
		return $expr;
	}
	
	/**
	* @abstract
	*/
	function returnLink($link,$text)
	{
	}

	/**
	* @abstract
	*/
	function returnSee(&$element, $eltext = false)
	{
	}
	
	/**
	* @abstract
	*/
	function Convert(&$element)
	{
	}
	
	/**
	* @abstract
	*/
	function Output($title)
	{
		phpdoc_out("WARNING: Generic Converter was used, no output will be generated");
	}
}

?>