<?php
/**
* Parser Data Structures
* @package phpDocumentor
* @subpackage ParserData
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: ParserData.inc,v 1.35 2002/05/25 03:38:53 CelloG Exp $
*/
/**
* @package phpDocumentor
* @subpackage ParserData
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: ParserData.inc,v 1.35 2002/05/25 03:38:53 CelloG Exp $
*/
class parserPage
{
	/**
	* @var string
	*/
	var $type = 'page';
	/**
	* not implemented in this version, will be used to link xml output pages
	* @var string
	*/
	var $id = '';
	/**
	* filename.ext (no path)
	* @var string
	*/
	var $file = '';
	/**
	* relative source location
	* @var string
	*/
	var $sourceLocation = '';
	/**
	* phpdoc-safe name (only letters, numbers and _)
	* @var string
	*/
	var $name = '';
	/**
	* @var string
	*/
	var $package = 'default';
	/**
	* @var string
	*/
	var $subpackage = '';
	/**
	* not implemented yet
	* @var string
	*/
	var $parserVersion = '';
	/**
	* not implemented yet
	* file modification date, will be used for makefiles
	* @var string
	*/
	var $modDate = '';
	/**
	* false or in-memory representation of parsed elements found in this file
	* @var mixed
	*/
	var $parsedData = false;
	/**
	* @var string
	*/
	var $path = '';
	/**
	* either false or an array of packages
	* @see IntermediateParser::$packageoutput, Converter::$package_output
	* @var mixed
	*/
	var $packageOutput = false;
	
	function parserPage()
	{
		$this->package = $GLOBALS['PHPDocumentor_DefaultPackageName'];
	}
	
	/**
	* @return string always "page"
	*/
	function getType()
	{
		return 'page';
	}
	
	/**
	* @param string $file
	*/
	function setFile($file)
	{
		$this->file = $file;
	}
	
	/**
	* @return string filename.ext
	*/
	function getFile()
	{
		if (!isset($this->file)) return false;
		return $this->file;
	}
	
	/**
	* @param string $path
	*/
	function setPath($path)
	{
		$this->path = $path;
	}
	
	/**
	* @return string fully delimited path (OS-dependent format)
	*/
	function getPath()
	{
		if (!isset($this->path)) return false;
		return $this->path;
	}
	
	/**
	* @param array $packages array of packages to display in documentation (package1,package2,...)
	* @see IntermediateParser::$packageoutput
	*/
	function setPackageOutput($packages)
	{
		$this->packageOutput = $packages;
	}
	
	/**
	* @return array array of packages (package1,package2,...)
	* @see IntermediateParser::$packageoutput
	*/
	function getPackageOutput()
	{
		return $this->packageOutput;
	}
	
	/**
	* @param string $name phpdoc-safe name (only _, numbers and letters)
	*/
	function setName($name)
	{
		$this->name = $name;
	}
	
	/**
	* @return string phpdoc-safe name (only _, numbers and letters)
	*/
	function getName()
	{
		if (!isset($this->name)) return false;
		return $this->name;
	}
	
	/**
	* @param string $source source location relative to program root
	*/
	function setSourceLocation($source)
	{
		$this->sourceLocation = $source;
	}
	
	/**
	* @return string location relative to program root
	*/
	function getSourceLocation ()
	{
		if (!isset($this->sourceLocation)) return false;
		return $this->sourceLocation;
	}
	/**
	*
	* @return boolean tell the parser whether to parse the file, otherwise
	* 				  this function will retrieve the parsed data from external file
	*/
	function getParseData()
	{
		return true;
	}
}

/**
* Contains an in-memory representation of all documentable elements ({@link parserPage}, {@link parserFunction}, {@link parserDefine},
* {@link parserInclude}, {@link parserClass}, {@link parserMethod}, {@link parserVar}) and their DocBlocks ({@link parserDocBlock}).
*
* This class works in coordination with {@link IntermediateParser} to take output from {@link Parser::handleEvent()} and create indexes,
* links, and other assorted things (all documented in IntermediateParser and {@link Converter})
* @package phpDocumentor
* @subpackage ParserData
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: ParserData.inc,v 1.35 2002/05/25 03:38:53 CelloG Exp $
*/
class parserData
{
	/**
	* {@link parserPage} element that is this parserData's parent, or false if not set
	* @var mixed
	*/
	var $parent = false;
	/**
	* array of parsed elements
	* @var array
	*/
	var $elements = array();
	/**
	* array of links descended from {@link abstractLink}
	* @var array
	* @see pageLink, defineLink, classLink, functionLink, methodLink, varLink
	*/
	var $links = array();
	/**
	* used by {@link IntermediateParser::handleDocBlock()} to determine whether a docblock is a page-level docblock or not.
	* $clean is true as long as only 0 or 1 docblock has been parsed, and no element other than parserPage has been parsed
	* @var bool
	*/
	var $clean = true;
	/**
	* DocBlock ({@link parserDocBlock}) for this page, or false if not set
	* @var mixed
	*/
	var $docblock = false;
	/**
	* always 'page', used in element indexing and conversion functions found in {@link Converter}
	* @var string
	*/
	var $type = 'page';
	
	/**
	* @param parserElement add a parsed element to the {@link $elements} array, also sets {@link $clean} to false
	*/
	function addElement(&$element)
	{
		$this->elements[] = $element;
		$this->clean = false;
	}
	
	/**
	* @param parserElement element to add a new link (descended from {@link abstractLink})to the {@link $links} array
	* @param string classname for elements that are class-based (this may be deprecated in the future, as the classname
	*               should be contained within the element.  if $element is a page, this parameter is a package name
	* @param string subpackage name for page elements
	*/
	function addLink(&$element,$classorpackage = '', $subpackage = '')
	{
		switch($element->type)
		{
			case 'function':
				$x = new functionLink;
				$x->addLink($this->parent->file, $this->parent->name, $element->name, $element->docblock->package, $element->docblock->subpackage);
				return $x;
			break;
			case 'define':
				$x = new defineLink;
				$x->addLink($this->parent->file, $this->parent->name, $element->name, $element->docblock->package, $element->docblock->subpackage);
				return $x;
			break;
			case 'class':
				$x = new classLink;
				$x->addLink($this->parent->file, $this->parent->name, $element->name, $element->docblock->package, $element->docblock->subpackage);
				return $x;
			break;
			case 'method':
				$x = new methodLink;
				$x->addLink($classorpackage, $this->parent->file, $this->parent->name, $element->name, $element->docblock->package, $element->docblock->subpackage);
				return $x;
			break;
			case 'var':
				$x = new varLink;
				$x->addLink($classorpackage, $this->parent->file, $this->parent->name, $element->name, $element->docblock->package, $element->docblock->subpackage);
				return $x;
			break;
			case 'page':
				if (empty($classorpackage)) $classorpackage = $GLOBALS['PHPDocumentor_DefaultPackageName'];
				$x = new pageLink;
				$x->addLink($element->file,$element->name,$element->file,$classorpackage, $subpackage);
				return $x;
			break;
		}
	}
	
	/**
	* returns a list of all classes declared in a file
	* @param Converter &$c
	* @return array Format: array(packagename => parserClass,packagename => parserClass,...)
	*/
	function getClasses(&$c)
	{
		$r = array();
		if (!isset($c->classes->classesbyfile[$this->parent->file])) return array();
		foreach($c->classes->classesbyfile[$this->parent->file] as $class => $obj)
		{
			$r[$obj->docblock->package][] = $obj;
		}
		return $r;
	}
	
	/**
	* @param parserPage parent element of this parsed data
	*/
	function setParent(&$parent)
	{
		$this->parent = $parent;
	}
	
	/**
	* @return bool returns the value of {@link $clean}
	*/
	function isClean()
	{
		return $this->clean;
	}
	
	/**
	* @param parserDocBlock
	* @see parserDocBlock
	*/
	function setDocBlock(&$docblock)
	{
		$this->docblock = $docblock;
	}
}

/**
* base class for all elements
* @package phpDocumentor
* @subpackage ParserData
* @abstract
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: ParserData.inc,v 1.35 2002/05/25 03:38:53 CelloG Exp $
*/
class parserBase
{
	/**
	* always base
	* @var string
	*/
	var $type = 'base';
	/**
	* set to different things by its descendants
	* @abstract
	* @var mixed
	*/
	var $value = false;
	
	/**
	* line number on file where this thing is found
	*/
	var $linenum = 0;
	
	function setLineNum($l)
	{
		$this->linenum = $l;
	}
	
	/**
	* @return string returns value of {@link $type}
	*/
	function getType()
	{
		return $this->type;
	}
	
	/**
	* @param mixed set the value of this element
	*/
	function setValue($value)
	{
		$this->value = $value;
	}
	
	/**
	* @return mixed get the value of this element (element-dependent)
	*/
	function getValue()
	{
		return $this->value;
	}
}

/**
* Use this element to represent an {@inline tag} like link (see the PHPDocumentor spec for more information about inline tags)
* @see parserStringWithInlineTags
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: ParserData.inc,v 1.35 2002/05/25 03:38:53 CelloG Exp $
*/
class parserInlineTag extends parserBase
{
	/**
	* always 'inlinetag'
	* @var string
	*/
	var $type = 'inlinetag';
	/**
	* the name of the inline tag (like link)
	* @var string
	*/
	var $inlinetype = '';
	
	/**
	* @param string $type tag type (example: link)
	* @param string $value tag value (example: what to link to)
	*/
	function parserInlineTag($type,$value)
	{
		$this->inlinetype = $type;
		$this->value = trim($value);
	}
	
	/**
	* @return integer length of the tag
	*/
	function Strlen()
	{
		return strlen($this->value);
	}
	
	/**
	* @return string always '', used by {@link Parser::handleDocBlock()} to calculate the short description of a DocBlock
	* @see parserStringWithInlineTags::getString()
	* @see parserStringWithInlineTags::trimmedStrlen()
	*/
	function getString()
	{
		return '';
	}
}

/**
* represents inline links
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: ParserData.inc,v 1.35 2002/05/25 03:38:53 CelloG Exp $
*/
class parserLinkInlineTag extends parserInlineTag
{
	/**
	* text to display in the link, can be different from the link for standard links like websites
	* @var string
	*/
	var $linktext = '';
	
	/**
	* @param string $link stored in $value, see {@link parserBase::$value}
	* @param string $text see {@link $linktext}
	*/
	function parserLinkInlineTag($link,$text)
	{
		parserInlineTag::parserInlineTag('link',$link);
		$this->linktext = trim($text);
	}
	
	/**
	* @param Converter converter used to change the abstract link into text for display
	* @return mixed returns the converted link or false if not converted successfully
	*/
	function Convert(&$c)
	{
		if (strpos($this->value,'://'))
		{
			return $c->returnLink($this->value,$this->linktext);
		} else
		{
			$value = $c->getLink($this->value);
			if (is_string($value)) return $value;
			if (is_object($value)) return $c->returnSee($value,$this->linktext);
			// getLink parsed a comma-delimited list of linked thingies, add the commas back in
			if (is_array($value))
			{
				$a = '';
				foreach($value as $i => $bub)
				{
					if (!empty($a)) $a .= ', ';
					if (is_string($value[$i])) $a .= $value[$i];
					if (is_object($value[$i])) $a .= $c->returnSee($value[$i]);
				}
				return $a;
			}
			return false;
		}
	}
}

/**
* Used to represent strings that contain inline tags, so that they can be properly parsed at link time
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: ParserData.inc,v 1.35 2002/05/25 03:38:53 CelloG Exp $
*/
class parserStringWithInlineTags extends parserBase
{
	/**
	* always '_string'
	* @var string
	*/
	var $type = '_string';
	/**
	* array of strings and {@link parserInlineTag}s
	* Format:
	* array(string1,string2,parserInlineTag1,string3,parserInlineTag2,...)
	* @var array
	*/
	var $value = array();
	/**
	* find the inline tags (xml format), parse them out as new objects, set up {@link $value}
	* @param string $value any string that may contain inline tags
	*/
	function parserStringWithInlineTags($value='')
	{
		if (empty($value)) return;
		$tempvalue = explode('<inlinetag',$value);
		$parsedvalue = array($tempvalue[0]);
		for($i=1;$i<count($tempvalue);$i++)
		{
			$a = strpos($tempvalue[$i],'>');
			$parsedvalue[] = new ParsedInlineTag('<inlinetag'.substr($tempvalue[$i],0,$a));
			$parsedvalue[] = substr($tempvalue[$i],$a);
		}
		$this->value = $parsedvalue;
	}

	/**
	* equivalent to the . operator ($a = $b . $c)
	* @param mixed either a string or a {@link parserInlineTag}
	*/
	function add($stringOrInlineTag)
	{
		if (is_string($stringOrInlineTag))
		{
			if (!count($this->value))
			{
				$this->value[] = $stringOrInlineTag;
				return;
			}
			if (is_string($this->value[count($this->value) - 1]))
			{
				$this->value[count($this->value) - 1] .= $stringOrInlineTag;
				return;
			} else
			{
				$this->value[] = $stringOrInlineTag;
				return;
			}
		} else
		{
			$this->value[] = $stringOrInlineTag;
		}
	}
	
	/**
	* equivalent to trim(strlen($string))
	* @return integer length of the string this object represents
	*/
	function trimmedStrlen()
	{
		$a = 0;
		for($i=0;$i<count($this->value);$i++)
		{
			if (is_string($this->value[$i]))
			{
				if ($i == 0)
				{
					$a += strlen(ltrim($this->value[$i]));
				} elseif ($i == count($this->value[$i]) - 1)
				{
					$a += strlen(chop($this->value[$i]));
				}
			} else
			{
				$a += $this->value[$i]->Strlen();
			}
		}
		return $a;
	}
	
	/**
	* return the string unconverted (all inline tags are taken out - this should only be used in pre-parsing to see if any other text
	* is in the string)
	* @return string trimmed value
	*/
	function getString()
	{
		$a = '';
		for($i=0; $i<count($this->value); $i++)
		{
			if (is_string($this->value[$i]))
			{
				$a .= $this->value[$i];
			} else
			{
				$a .= $this->value[$i]->getString();
			}
		}
		return trim($a);
	}
	
	/**
	* Use to convert the string to a real string with all inline tags parsed and linked
	* @param Converter
	*/
	function Convert(&$converter)
	{
		if (is_string($this->value)) return $this->value;
		$a = '';
		for($i=0; $i<count($this->value); $i++)
		{
			if (is_string($this->value[$i]))
			{
				$a .= $this->value[$i];
			} else
			{
				$a .= $this->value[$i]->Convert($converter);
			}
		}
		return trim($a);
	}
}

/**
* represents a short or long description in a DocBlock ({@link parserDocBlock})
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: ParserData.inc,v 1.35 2002/05/25 03:38:53 CelloG Exp $
*/
class parserDesc extends parserStringWithInlineTags
{
	/**
	* always '_desc'
	* @var string
	*/
	var $type = '_desc';

	/**
	* @param mixed like {@link parserStringWithInlineTags::add()}, this can be a string or parserInlineTag, but it can also be a
	*              parserStringWithInlineTags, and the contents will be merged
	*/
	function add($stringOrClass)
	{
		if (is_object($stringOrClass))
		{
			if (get_class($stringOrClass) == 'parserstringwithinlinetags' || get_class($stringOrClass) == 'parserStringWithInlineTags' || is_subclass_of($stringOrClass,'parserStringWithInlineTags') || is_subclass_of($stringOrClass,'parserstringwithinlinetags'))
			{
				for($i=0;$i<count($stringOrClass->value);$i++)
				{
					parserStringWithInlineTags::add($stringOrClass->value[$i]);
				}
			} else
			{
				parserStringWithInlineTags::add($stringOrClass);
			}
		} else return parserStringWithInlineTags::add($stringOrClass);
	}
}

/**
* used to represent standard tags like @access, etc.
* This class is aware of inline tags, and will automatically handle them using inherited functions
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: ParserData.inc,v 1.35 2002/05/25 03:38:53 CelloG Exp $
*/
class parserTag extends parserStringWithInlineTags
{
	/**
	* always '_tag'
	* @var string
	*/
	var $type = '_tag';
	/**
	* tag name (see, access, etc.)
	* @var string
	*/
	var $keyword = '';
	
	/**
	* @param string $keyword tag name
	* @param parserStringWithInlineTags $value
	*/
	function parserTag($keyword, $value)
	{
		$this->keyword = $keyword;
		$this->value = $value;
	}
	
	/**
	* @param Converter
	* @see Converter
	*/
	function Convert(&$converter)
	{
		return $this->value->Convert($converter);
	}
	
	function getString()
	{
		return $this->value->getString();
	}
}

class parserAccessTag extends parserTag
{
	var $keyword = 'access';
	
	var $isvalid = false;
	
	/**
	* @param parserStringWithInlineTags $value
	*/
	function parserAccessTag($value)
	{
		if (!is_string($value))
		{
			if (is_object($value))
			{
				if (method_exists($value,'getstring'))
				{
					$value = $value->getString();
				}
			}
		}
		switch(trim($value))
		{
			case 'private' :
			case 'public' :
				$this->value = $value;
				$this->isvalid = true;
			break;
			default :
			addError(PDERROR_ACCESS_WRONG_PARAM,$value);
				$this->value = 'public';
			break;
		}
	}
	
	function Convert(&$converter)
	{
		return $this->value;
	}
	
	function getString()
	{
		return $this->value;
	}
}

/**
* represents "@return"
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: ParserData.inc,v 1.35 2002/05/25 03:38:53 CelloG Exp $
*/
class parserReturnTag extends parserTag
{
	/**
	* always 'return'
	* @var string
	*/
	var $keyword = 'return';
	/**
	* the type a function returns
	*/
	var $returnType = 'void';
	
	/**
	* @param string
	* @param parserStringWithInlineTags
	*/
	function parserReturnTag($returnType, $value)
	{
		$this->returnType = $returnType;
		$this->value = $value;
	}
}

/**
* represents "@var"
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: ParserData.inc,v 1.35 2002/05/25 03:38:53 CelloG Exp $
*/
class parserVarTag extends parserReturnTag
{
	/**
	* always 'return'
	* @var string
	*/
	var $keyword = 'var';
	/**
	* the type a var has
	*/
	var $returnType = 'mixed';
}

/**
* represents "@link"
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: ParserData.inc,v 1.35 2002/05/25 03:38:53 CelloG Exp $
*/
class parserLinkTag extends parserTag
{
	/**
	* always 'link'
	* @var string
	*/
	var $keyword = 'link';
	
	/**
	* @param string $link
	*/
	function parserLinkTag($link)
	{
		$this->value = $link;
	}
}

/**
* represents "@see"
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: ParserData.inc,v 1.35 2002/05/25 03:38:53 CelloG Exp $
*/
class parserSeeTag extends parserLinkTag
{
	/**
	* always 'see'
	* @var string
	*/
	var $keyword = 'see';

	/**
	* @param Converter
	* @see Converter, HTMLConverter::returnSee()
	*/
	function Convert(&$converter)
	{
		$a = $converter->getLink(trim($this->value->Convert($converter)));
		if (is_string($a)) return $a;
		if (is_object($a)) return $converter->returnSee($a);
		// getLink parsed a comma-delimited list of linked thingies, add the commas back in
		if (is_array($a))
		{
			$b = '';
			foreach($a as $i => $bub)
			{
				if (!empty($b)) $b .= ', ';
				if (is_string($a[$i])) $b .= $a[$i];
				if (is_object($a[$i])) $b .= $converter->returnSee($a[$i]);
			}
			return $b;
		}
		return false;
	}
}

/**
* Represents a docblock and its components, {@link $desc}, {@link $sdesc}, {@link $tags}, and also {@link $params} for functions
* @package phpDocumentor
* @subpackage ParserData
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: ParserData.inc,v 1.35 2002/05/25 03:38:53 CelloG Exp $
*/
class parserDocBlock
{
	/**
	* @var parserDesc
	*/
	var $desc = false;
	/**
	* @var parserDesc
	*/
	var $sdesc = false;
	/**
	* array of {@link parserTag}s
	* @var array
	*/
	var $tags = array();
	/**
	* array of unrecognized {@link parserTag}s
	* @var array
	*/
	var $unknown_tags = array();
	/**
	* array of param data.
	* Format:
	* array(index of param in function parameter list -OR- parameter name =>
	*         parserStringWithInlineTags,...)
	* @var array
	*/
	var $params = array();
	/**
	* This is either a {@link parserReturnTag} or false if no return tag is present
	* @var mixed
	*/
	var $return = false;
	/**
	* This is either a {@link parserVarTag} or false if no var tag is present
	* @var mixed
	*/
	var $var = false;
	var $package = 'default';
	var $subpackage = '';
	var $hasaccess = false;
	
	function parserDocBlock()
	{
		$this->package = $GLOBALS['PHPDocumentor_DefaultPackageName'];
	}
	
	/**
	* @param parserDesc sets {@link $sdesc}
	*/
	function setShortDesc($desc)
	{
		$this->sdesc = $desc;
	}
	
	/**
	* @param parserDesc sets {@link $desc}
	*/
	function setDesc($desc)
	{
		$this->desc = $desc;
	}
	
	/**
	* @param Converter takes {@link $sdesc} and converts it to a string and returns it if present, otherwise returns ''
	* @return string
	*/
	function getSDesc(&$converter)
	{
		if ($this->sdesc) return $this->sdesc->Convert($converter);
		return '';
	}
	
	/**
	* @param Converter takes {@link $desc} and converts it to a string and returns it if present, otherwise returns ''
	* @return string
	*/
	function getDesc(&$converter)
	{
		if ($this->desc) return $this->desc->Convert($converter);
		return '';
	}
	
	/**
	* @param string $paramVar if empty, param is indexed in the order received and set using {@link changeParam()}
	* @param parserStringWithInlineTags $value
	*/
	function addParam($paramVar, $value)
	{
		if (empty($paramVar))
		$this->params[] = $value;
		else
		$this->params[$paramVar] = $value;
	}
	
	/**
	* @param integer $index index of parameter in the {@link $params} array
	* @param string $name name of the parameter to set in the $params array
	*/
	function changeParam($index,$name)
	{
		$this->params[$name] = $this->params[$index];
		unset($this->params[$index]);
	}
	
	/**
	* replaces nameless parameters in the {@link $params} array with their names
	* @param array $params Format: array(parameter index => parameter name,...)
	*/
	function updateParams($params)
	{
		for($i=0;$i<count($params);$i++)
		{
			if (isset($this->params[$i]))
			{
				$this->changeParam($i,$params[$i]);
			}
		}
		if (isset($this->tags))
		unset($this->tags['param']);
	}

	/**
	* @param string $keyword tag name
	* @param parserStringWithInlineTags $value the contents of the tag
	*/
	function addKeyword($keyword, $value)
	{
		if ($keyword == 'package' || $keyword == 'subpackage') return $this->addPackage($keyword, $value);
		if ($keyword == 'access') return $this->addAccess($value);
		if ($keyword == 'link') return $this->addLink($value);
		if ($keyword == 'see') return $this->addSee($value);
		if (!in_array($keyword,$GLOBALS['PHPDOC_TAGS_ALLOWED']))
		$this->addUnknownTag($keyword,$value);
		else
		$this->tags[$keyword][] = new parserTag($keyword, $value);
	}
	
	function addUnknownTag($keyword, $value)
	{
		addWarning(PDERROR_UNKNOWN_TAG,$keyword);
		$this->unknown_tags[$keyword][] = new parserTag($keyword, $value);
	}
	
	function overridePackage($package,$subpackage,$elname,$type)
	{
		if ($this->package != $GLOBALS['PHPDocumentor_DefaultPackageName'])
		addError(PDERROR_OVERRIDDEN_PACKAGE_TAGS,$elname,$type,$this->package);
		if (!empty($this->subpackage))
		addError(PDERROR_OVERRIDDEN_SUBPACKAGE_TAGS,$type,$elname,$this->subpackage);
		$this->package = $GLOBALS['PHPDocumentor_DefaultPackageName'];
		$this->subpackage = '';
		$this->addPackage('package',$package);
		$this->addPackage('subpackage',$subpackage);
	}
	
	/**
	* @param string $keyword tag name (either package or subpackage)
	* @param mixed $value either a string or a parserStringWithInlineTags.  Strips all inline tags and use the text as the package
	*/
	function addPackage($keyword, $value)
	{
		if ($keyword == 'package')
		{
			if ($this->package == $GLOBALS['PHPDocumentor_DefaultPackageName'])
			{
				if (is_string($value))
				{
					preg_match("/^([a-zA-Z0-9\-_\.\[\]]+)$/",$value,$match);
					if (!isset($match[0]))
					{
						// if were a single line and the only bad character is a space then will fix things for them
						preg_match("/^([a-zA-Z0-9\-_\.\[\]\/\\: ]+)$/",$value,$match);
						if (!isset($match[0]))
						{
							addErrorDie(PDERROR_ILLEGAL_PACKAGENAME,'package','package',$value);
						}
						else
						{
							$value = 
								str_replace(array(" ","/","\\",":"),"_",
									trim($value));
						}
					}
					$this->package = trim($value);
				} else
				{
					preg_match("/^([a-zA-Z0-9\-_\.\[\]]+)$/",$value->getString(),$match);
					if (!isset($match[0]))
					{
						// if were a single line and the only bad character is a space then will fix things for them
						preg_match("/^([a-zA-Z0-9\-_\.\[\]\/\\: ]+)$/",$value->getString(),$match);
						if (!isset($match[0]))
						{
							addErrorDie(PDERROR_ILLEGAL_PACKAGENAME,'package','package',$value->getString());
						}
						else
						{
							$value->value[0] = 
								str_replace(array(" ","/","\\",":"),"_",
									trim($value->value[0]));
						}
					}
					$this->package = trim($value->getString());
				}
			} else
			{
				if (is_string($value))
				addError(PDERROR_MULTIPLE_PACKAGE_TAGS,$value);
				else
				addError(PDERROR_MULTIPLE_PACKAGE_TAGS,$value->getString());
			}
		} else
		{
			if (empty($this->subpackage))
			{
				if (is_string($value))
				{
					if (!empty($value))
					{
						preg_match("/^([a-zA-Z0-9\-_\.\[\]]+)$/",$value,$match);
						if (!isset($match[0]))
						{
							// if were a single line and the only bad character is a space then will fix things for them
							preg_match("/^([a-zA-Z0-9\-_\.\[\]\/\\: ]+)$/",$value,$match);
							if (!isset($match[0]))
							{
								addErrorDie(PDERROR_ILLEGAL_PACKAGENAME,'subpackage','subpackage',$value);
							}
							else
							{
								$value = 
									str_replace(array(" ","/","\\",":"),"_",
										trim($value));
							}
						}
					}
					$this->subpackage = trim($value);
				} else
				{
					$x = $value->getString();
					if (empty($x))
					{
						preg_match("/^([a-zA-Z0-9\-_\.\[\]]+)$/",$value->getString(),$match);
						if (!isset($match[0]))
						{
							// if were a single line and the only bad character is a space then will fix things for them
							preg_match("/^([a-zA-Z0-9\-_\.\[\]\/\\: ]+)$/",$value->getString(),$match);
							if (!isset($match[0]))
							{
								addErrorDie(PDERROR_ILLEGAL_PACKAGENAME,'subpackage','subpackage',$value->getString());
							}
							else
							{
								$value->value[0] = 
									str_replace(array(" ","/","\\",":"),"_",
										trim($value->value[0]));
							}
						}
					}
					$this->subpackage = trim($value->getString());
				}
			} else
			{
				if (is_string($value))
				addError(PDERROR_MULTIPLE_SUBPACKAGE_TAGS,$value);
				else
				addError(PDERROR_MULTIPLE_SUBPACKAGE_TAGS,$value->getString());
			}
		}
	}
	
	function addAccess($value)
	{
		if (is_object($value)) $value = $value->getString();
		$value = strtolower($value);
		if (!$this->hasaccess)
		{
			$x = new parserAccessTag($value);
			if ($x->isvalid)
			{
				$this->hasaccess = true;
				$this->tags['access'][] = $x;
			}
		} else
		{
			if (is_string($value))
			addError(PDERROR_MULTIPLE_ACCESS_TAGS,$value);
			else
			addError(PDERROR_MULTIPLE_ACCESS_TAGS,$value->getString());
		}
	}
	
	/**
	* creates a {@link parserLinkTag} and adds it to the {@link $tags} array
	* @param string $link
	*/
	function addLink($link)
	{
		$this->tags['link'][] = new parserLinkTag($link);
	}
	
	/**
	* creates a {@link parserLinkTag} and adds it to the {@link $tags} array
	* @param string $value
	*/
	function addSee($value)
	{
		$this->tags['see'][] = new parserSeeTag($value);
	}
	
	/**
	* creates a {@link parserReturnTag} and adds it to the {@link $tags} array
	* @param string $returnType the one-word name of the return type (mixed should be used if more than one type)
	* @param parserStringWithInlineTags $value
	*/
	function addReturn($returnType, $value)
	{
		// only take the first one
		if (!$this->return)
		{
			$this->return = new parserReturnTag($returnType, $value);
		} else
		{
			addError(PDERROR_MULTIPLE_RETURN_TAGS,$returnType,$value->getString());
		}
	}
	
	/**
	* creates a {@link parserVarTag} and adds it to the {@link $tags} array
	* @param string $varType the one-word name of the variable type (mixed should be used if more than one type)
	* @param parserStringWithInlineTags $value
	*/
	function addVar($varType, $value)
	{
		// only take the first one
		if (!$this->var)
		{
			$this->var = new parserVarTag($varType, $value);
		} else
		{
			addError(PDERROR_MULTIPLE_VAR_TAGS,$varType,$value->getString());
		}
	}
	
	/**
	* @param string
	* @return mixed false if no keyword, unconverted value if one keyword, array of unconverted values if more than one keyword
	*/
	function getKeyword($keyword)
	{
		if (isset($this->tags[$keyword]))
		{
			if (count($this->tags[$keyword]) == 1)
			{
				return $this->tags[$keyword][0];
			} else return $this->tags[$keyword];
		} else return false;
	}
	
	function listParams()
	{
		if (isset($this->params))
		{
			$ret = array();
			foreach($this->params as $key => $val)
			{
				$ret[] = array("var" => ucfirst($key),"data" => $val);
			}
			return $ret;
		} else {
			return array();
		}
	}
	
	function getType()
	{
		return 'docblock';
	}
}

/**
* all elements except {@link parserPackagePage} descend from this abstract class
* @abstract
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: ParserData.inc,v 1.35 2002/05/25 03:38:53 CelloG Exp $
*/
class parserElement extends parserBase
{
	var $docblock = false;
	var $name;
	
	/**
	* location of this element (filename)
	* @var string
	*/
	var $file = '';
	
	/**
	* @param parserDocBlock
	*/
	function setDocBlock($docblock)
	{
		$this->docblock = $docblock;
	}
	
	/**
	* @param string
	*/
	function setName($name)
	{
		$this->name = $name;
	}
	
	function getPackage()
	{
		if ($this->docblock)
		{
			return $this->docblock->package;
		} else return $GLOBALS['PHPDocumentor_DefaultPackageName'];
	}
	
	function setFile($file)
	{
		$this->file = $file;
	}
	
	/**
	* @return string
	*/
	function getName()
	{
		if (!isset($this->name)) return false;
		return $this->name;
	}
}

/**
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: ParserData.inc,v 1.35 2002/05/25 03:38:53 CelloG Exp $
*/
class parserInclude extends parserElement
{
	var $type = 'include';
}

/**
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: ParserData.inc,v 1.35 2002/05/25 03:38:53 CelloG Exp $
*/
class parserFunction extends parserElement
{
	var $type = 'function';
	var $params = false;
	var $returnsreference = false;
	
	/**
	* @param string
	* @param parserStringWithInlineTags
	*/
	function addParam($name, $value)
	{
		$this->params[$name] = $value;
	}

	/**
	* @param string
	*/
	function getParam ($name)
	{
		if (!isset($this->params[$name])) return false;
		return $this->params[$name];
	}

	/**
	* @return array format: array(tagname => unconverted tag value)
	*/
	function listParams ()
	{
		if (isset($this->params))
		{
			$ret = array();
			if ($this->params)
			foreach($this->params as $key => $val)
			{
				$ret[] = array($key,$val);
			}
			return $ret;
		} else {
			return array();
		}
	}
	
	function setReturnsReference()
	{
		$this->returnsreference = true;
	}
	
	function getReturnsReference()
	{
		return $this->returnsreference;
	}
}

/**
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: ParserData.inc,v 1.35 2002/05/25 03:38:53 CelloG Exp $
*/
class parserClass extends parserElement
{
	var $type = 'class';
	var $sourceLocation = '';
	var $extends = false;
	/**
	* Format: array(file, parent) where parent class is found or false if no parent
	* @var mixed
	*/
	var $parent = false;
	
	var $curfile = false;

	/**
	* @param string $p parent class name
	* @param string $f parent class file
	* @param Classes $c {@link Classes} object currently calling setParent
	* @see Classes::setClassParent()
	*/
	
	function setParent($p,$f, &$c)
	{
		$this->parent = array($f, $p);
		$p = $c->getClass($p, $f);
		// inherit package
		if ($this->docblock->package == $GLOBALS['PHPDocumentor_DefaultPackageName'])
		{
			$this->docblock->package = $p->docblock->package;
			$this->docblock->subpackage = $p->docblock->subpackage;
		}
	}
	
	/**
	* @param string $par parent class name (used by {@link Classes::setClassParent()} if parent class not found
	*/
	function setParentNoClass($par)
	{
		$this->parent = $par;
	}
	
	/**
	* retrieve object that represents the parent class
	* @param Converter $c this function will not work before the Conversion stage of parsing
	* @return mixed returns the {@link parserClass} representation of the parent class, or false if no parent class
	*/
	function &getParent(&$c)
	{
		if (!$this->parent) return false;
		if (is_array($this->parent))
		{
			return $c->classes->getClass($this->parent[1],$this->parent[0]);
		} else return $this->parent;
	}
	
	/**
	* @param Converter $c this function will not work before the Conversion stage of parsing
	* @return array returns a simple array of method objects
	*/
	function getMethods(&$c)
	{
		if (!isset($c->classes->methods[$this->docblock->package][$this->name])) return false;
		return array_values($c->classes->methods[$this->docblock->package][$this->name]);
	}
	
	function getMethod(&$c, $name)
	{
		if (!isset($c->methods[$this->docblock->package][$this->name])) return false;
		foreach($c->methods[$this->docblock->package][$this->name] as $method => $obj)
		{
			if ($method == $name) return $obj;
		}
	}
	
	function getVar(&$c, $name)
	{
		if (!isset($c->classes->vars[$this->docblock->package][$this->name])) return false;
		foreach($c->classes->vars[$this->docblock->package][$this->name] as $var => $obj)
		{
			if ($var == $name) return $obj;
		}
	}
	
	/**
	* @param Converter $c this function will not work before the Conversion stage of parsing
	* @return array returns a simple array of method name strings
	*/
	function getMethodNames(&$c)
	{
		if (!isset($c->classes->methodsbyfile[$this->curfile][$this->name])) return array();
		$arr = array();
		$arr1 = $this->getMethods($c);
		for($i=0; $i < count($arr1); $i++)
		{
			$arr[] = $arr1[$i]->name;
		}
		return $arr;
	}
	
	/**
	* @param Converter $c this function will not work before the Conversion stage of parsing
	* @param string $name method name
	* @return boolean whether this class has a method of name $name
	*/
	function hasMethod(&$c,$name)
	{
		if (!isset($c->classes->methodsbyfile[$this->curfile][$this->name])) return false;
		for($i=0; $i<count($c->classes->methodsbyfile[$this->curfile][$this->name]); $i++)
		{
			if ($c->classes->methodsbyfile[$this->curfile][$this->name][$i]->getName() == $name) return true;
		}
		return false;
	}
	
	/**
	* @param Converter $c this function will not work before the Conversion stage of parsing
	* @param string $name var name
	* @return boolean whether this class has a var of name $name
	*/
	function hasVar(&$c,$name)
	{
		if (!isset($c->classes->varsbyfile[$this->curfile][$this->name])) return false;
		for($i=0; $i<count($c->classes->varbyfile[$this->curfile][$this->name]); $i++)
		{
			if ($c->classes->varsbyfile[$this->curfile][$this->name][$i]->getName() == $name) return true;
		}
		return false;
	}
	
	/**
	* @param Converter $c this function will not work before the Conversion stage of parsing
	* @return array returns a simple array of var objects
	*/
	function getVars(&$c)
	{
		if (!isset($c->classes->vars[$this->docblock->package][$this->name])) return false;
		return array_values($c->classes->vars[$this->docblock->package][$this->name]);
	}
	
	/**
	* @param Converter $c this function will not work before the Conversion stage of parsing
	* @return array returns a simple array of var name strings
	*/
	function getVarNames(&$c)
	{
		if (!isset($c->classes->varsbyfile[$this->curfile][$this->name])) return array();
		$arr = array();
		$arr1 = $this->getVars($c);
		for($i=0; $i < count($arr1); $i++)
		{
			$arr[] = $c->classes->varsbyfile[$this->curfile][$this->name][$i]->name;
		}
		return $arr;
	}
	
	/**
	* @param Converter $c this function will not work before the Conversion stage of parsing
	* @param boolean $override determines whether overriden methods should be included in the list of inherited methods
	* @param void $methods private parameter for recursion
	* @return array returns an array of methods by parent classname array(name => array(method1,method2..),name2 => array(method1....))
	*/
	function getInheritedMethods(&$c,$override = false, $methods = false)
	{
		if (is_array($methods)) $methods = array_merge($methods,$this->getMethodNames($c));
		if (!$methods) $methods = $this->getMethodNames($c);
		$arr = array();
		if ($this->parent && is_array($this->parent))
		{
			$par = $c->classes->getClass($this->parent[1],$this->parent[0]);
			// retrieve all parent methods
			$arr = $par->getInheritedMethods($c,$override, $methods);
			$parmethodnames = $par->getMethodNames($c);
			$parmethods = $par->getMethods($c);
			for($i=0; $i<count($parmethodnames); $i++)
			{
				if ($override)
				{
					if (!in_array($parmethodnames[$i],$methods))
					{
						$methods[] = $parmethodnames[$i];
						$arr[$par->getName()]['methods'][] = $parmethods[$i];
						$arr[$par->getName()]['file'] = $par->curfile;
					}
				} else
				{
					$arr[$par->getName()]['methods'][] = $parmethods[$i];
					$arr[$par->getName()]['file'] = $par->curfile;
				}
			}
		}
		return $arr;
	}
	
	/**
	* @param Converter $c this function will not work before the Conversion stage of parsing
	* @param boolean $override determines whether overriden vars should be included in the list of inherited vars
	* @param void $vars private parameter for recursion
	* @return array returns an array of vars by parent classname array(name => array(var1,var1..),name2 => array(var1....))
	*/
	function getInheritedVars(&$c,$override = true, $vars = false)
	{
		if (is_array($vars)) $vars = array_merge($vars,$this->getVarNames($c));
		if (!$vars) $vars = $this->getVarNames($c);
		$arr = array();
		if ($this->parent && is_array($this->parent))
		{
			$par = $c->classes->getClass($this->parent[1],$this->parent[0]);
			// retrieve all parent vars
			$arr = $par->getInheritedVars($c,$override, $vars);
			$parvarnames = $par->getVarNames($c);
			$parvars = $par->getVars($c);
			for($i=0; $i<count($parvarnames); $i++)
			{
				if ($override)
				{
					if (!in_array($parvarnames[$i],$vars))
					{
						$vars[] = $parvarnames[$i];
						$arr[$par->getName()]['vars'][] = $parvars[$i];
						$arr[$par->getName()]['file'] = $par->curfile;
					}
				} else
				{
					$arr[$par->getName()]['vars'][] = $parvars[$i];
					$arr[$par->getName()]['file'] = $par->curfile;
				}
			}
		}
		return $arr;
	}
	
	/**
	* @param Converter $c this function will not work before the Conversion stage of parsing
	* @return array Format: array(parentclassname => parserClass/false if no parent, parentclassname2 => ...)
	*/
	function getParentClassTree(&$c,$arr = false)
	{
		if (!$arr) $arr = array();
		$arr[$this->name] = $this->getParent($c);
		if ($arr[$this->name] && is_object($arr[$this->name]))
		{
			$arr = $arr[$this->name]->getParentClassTree($c,$arr);
		} else
		{
			if (is_string($arr[$this->name])) $arr[$arr[$this->name]] = false;
		}
		return $arr;
	}
	
	/**
	* returns a list of all child classes of this class
	* @param Converter $c this function will not work before the Conversion stage of parsing
	* @return array Format: array(parserClass child1,parserClass child2,...)
	*/
	function getChildClassList(&$c)
	{
		$list = array();
		$kids = $c->classes->getDefiniteChildren($this->name,$this->curfile);
		if ($kids)
		{
			foreach($kids as $chile => $file)
			{
				$list[] = $c->classes->getClass($chile,$file);
			}
		}
		return $list;
	}
	
	/**
	* @param string
	*/
	function setSourceLocation($sl)
	{
		$this->sourceLocation = $sl;
	}
	
	function getSourceLocation()
	{
		if (!isset($this->sourceLocation)) return false;
		return $this->sourceLocation;
	}
	
	/**
	* @param string
	*/
	function setExtends($extends)
	{
		$this->extends = $extends;
	}
	
	function getExtends()
	{
		if (!isset($this->extends)) return false;
		return $this->extends;
	}
}

/**
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: ParserData.inc,v 1.35 2002/05/25 03:38:53 CelloG Exp $
*/
class parserVar extends parserElement
{
	var $type = 'var';
	var $class = '';
	
	/**
	* @param string
	*/
	function parserVar($class)
	{
		$this->class = $class;
	}
}

/**
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: ParserData.inc,v 1.35 2002/05/25 03:38:53 CelloG Exp $
*/
class parserMethod extends parserFunction
{
	var $type = 'method';
	var $isConstructor = false;
	var $class;
	
	/**
	* @param string
	*/
	function parserMethod($class)
	{
		$this->class = $class;
	}
	
	/**
	* Use this method to tell the parser that this method is the class constructor
	*/
	function setConstructor()
	{
		$this->isConstructor = true;
	}
	
	/**
	* @param Converter &$c
	* @return array an array of parserMethods from child classes that override this method
	*/
	function getOverridingMethods(&$c)
	{
		$class = $c->classes->getClass($this->class,$c->curfile);
		$meths = array();
		if (!$class) return $meths;
		$kids = $class->getChildClassList($c);
		for($i=0; $i<count($kids); $i++)
		{
			if ($kids[$i]->hasMethod($c, $this->name))
			{
				$meths[] = $kids[$i]->getMethod($c, $this->name);
			}
		}
		return $meths;
	}
}

/**
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: ParserData.inc,v 1.35 2002/05/25 03:38:53 CelloG Exp $
*/
class parserDefine extends parserElement
{
	var $type = 'define';
}

/**
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: ParserData.inc,v 1.35 2002/05/25 03:38:53 CelloG Exp $
*/
class parserPackagePage extends parserStringWithInlineTags
{
	var $type = 'packagepage';
	var $package = 'default';
	
	/**
	* @param string
	*/
	function parserPackagePage($package)
	{
		$this->package = $package;
	}
}
?>