"""
An agg http://antigrain.com/ backend

Features that are implemented

 * capstyles and join styles
 * dashes
 * linewidth 
 * lines, rectangles, ellipses
 * clipping to a rectangle
 * output to RGBA and PNG
 * alpha blending
 * DPI scaling properly - everything scales properly (dashes, linewidths, etc)
 * draw polygon 
 * freetype1

TODO:

 * use ttf manager to get font - right now I just use Vera
 * freetype2
 
INSTALLING 

  REQUIREMENTs

    python2.2+
    Numeric 22+
    agg2 (see below)
    freetype 1
    libpng
    libz
    
  Install AGG2 (cut and paste below into xterm should work)

    wget http://www.antigrain.com/agg2.tar.gz
    tar xvfz agg2.tar.gz
    cd agg2
    make

    (Optional) if you want to make the examples:
    cd examples/X11
    make

  Installing backend_agg

   Grab the latest CVS:

     cvs -d:pserver:anonymous@cvs.sourceforge.net:/cvsroot/matplotlib login 
     cvs -z3 -d:pserver:anonymous@cvs.sourceforge.net:/cvsroot/matplotlib co matplotlib
     
   Edit setup.py: change aggsrc to point to the agg2 src tree and
   replace if 0: with if 1: in the backend_agg section

   Then just do the usual thing: python setup.py build

   Please let me know if you encounter build problems, and tell me
   platform, gcc version, etc...  Currently the paths in setupext.py
   assume as linux like filesystem (eg X11 include dir, location of
   libttf, etcc) so you may need to tweak these

  Using agg backend

    python somefile.py -dAgg   

  or

    import matplotlib
    matplotlib.use('Agg')
    
TODO:

  * implement ttf font manager

  * allow save to file handle

  * allow load from png


Errors: running python2.2 on bace.bsd

    from _backend_agg import RendererAgg as _RendererAgg, Font
ImportError: /usr/lib/python2.2/site-packages/matplotlib/backends/_backend_agg.so: undefined symbol: __gxx_personality_v0

"""
from __future__ import division

import os, sys
from Numeric import array, Float

from matplotlib import get_data_path
from matplotlib._matlab_helpers import Gcf
from matplotlib.backend_bases import RendererBase,\
     GraphicsContextBase, FigureManagerBase, FigureCanvasBase, error_msg


from matplotlib.artist import DPI
from matplotlib.cbook import enumerate, True, False
from matplotlib.figure import Figure
from matplotlib.transforms import Bound1D, Bound2D, Transform
from ttf_font_manager import fontManager

from _backend_agg import RendererAgg as _RendererAgg, Font


PIXELS_PER_INCH = 96 # a constant used to scale point sizes with dpi

class RendererAgg(RendererBase):
    """
    The renderer handles all the drawing primitives using a graphics
    context instance that controls the colors/styles
    """
    fontd = {}
    offsetd = {}

    def __init__(self, width, height, dpi):
        #print 'init renderer'
        self.dpi = dpi
        self.width = width
        self.height = height
        self._renderer = _RendererAgg(int(width), int(height), dpi.get())

        self.draw_polygon = self._renderer.draw_polygon
        self.draw_rectangle = self._renderer.draw_rectangle
        self.draw_lines = self._renderer.draw_lines
        
    def draw_arc(self, gcEdge, rgbFace, x, y, width, height, angle1, angle2):
        """
        Draw an arc centered at x,y with width and height and angles
        from 0.0 to 360.0

        If rgbFace is not None, fill the rectangle with that color.  gcEdge
        is a GraphicsContext instance

        Currently, I'm only supporting ellipses, ie angle args are
        ignored
        """
        #print 'draw arc'
        self._renderer.draw_ellipse(
            gcEdge, rgbFace, x, y, width/2, height/2)  # ellipse takes radius
        

    def draw_line(self, gc, x1, y1, x2, y2):
        """
        x and y are equal length arrays, draw lines connecting each
        point in x, y
        """
        #print 'draw line'
        x = array([x1,x2], typecode=Float)
        y = array([y1,y2], typecode=Float)
        self._renderer.draw_lines(gc, x, y)


    def draw_point(self, gc, x, y):
        """
        Draw a single point at x,y
        """
        rgbFace = gcEdge.get_rgb()
        self._renderer.draw_ellipse(
            gcEdge, rgbFace, x, y, 0.5, 0.5)
        

    def _get_text_props(self, t):
        """
        return the properties needed to uniquely characterize this text
        """
        fontsize = self.get_text_scale()*t.get_fontsize()
        if t.get_rotation() == 'vertical':  rotation = 90
        else:  rotation = 0
        ttffile = fontManager.findfont(t)
        return ttffile, fontsize, rotation

    def draw_text(self, gc, x, y, t):
        """
        Render the text using the RendererPaint instance
        """
        #print 'draw text'
        props = self._get_text_props(t)

        x, y = t.get_xy_display()
        ox, oy = self.compute_text_offsets(t, props)
        xox = int(x+ox)
        yoy = int(y+oy)
        color = self.get_agg_color(gc.get_rgb())
        font = self._get_agg_font(t, props)
        textwidth, textheight = font.textsize(t.get_text())        
        self._renderer.draw_text(
            font, xox, self.height - (yoy+textheight), color, t.get_text())

    def compute_text_offsets(self, t, props):
        """
        Return the (x,y) offsets to adjust for the alignment
        specifications
        """
        tup = self.offsetd.get(props)
        #if tup is not None: return tup

        font = self._get_agg_font(t, props)
        textwidth, textheight = font.textsize(t.get_text())


        halign = t.get_horizontalalignment()
        valign = t.get_verticalalignment()
        
        if t.get_rotation() == 'vertical':
            if halign=='center': offsetx = -textwidth/2
            elif halign=='right': offsetx = -textwidth
            else: offsetx = 0

            if valign=='center': offsety = -textheight/2
            elif valign=='top': offsety = -textheight
            else: offsety = 0
        else:
            if halign=='center': offsetx = -textwidth/2
            elif halign=='right': offsetx = -textwidth
            else: offsetx = 0

            if valign=='center': offsety = -textheight/2
            elif valign=='top': offsety = -textheight
            else: offsety = 0


        tup = (offsetx, offsety)
        self.offsetd[props] = tup
        return tup

    def _get_agg_font(self, t, props):
        """
        Get the paint font for text instance t, cacheing for efficiency
        """

        #font = self.fontd.get(props)
        #if font is None:
        font = Font(*props)
        self.fontd[props] = font
        return font

    def get_text_extent(self, t):
        """
        Return the ink extent of the text as Bound2D instance
        """
        #print 'get_text_extent'
        props = self._get_text_props(t)
        x, y = t.get_xy_display()
        ox, oy = self.compute_text_offsets(t, props)
        xox = int(x+ox)
        yoy = int(y+oy)

        font = self._get_agg_font(t, props)
        width, height = font.textsize(t.get_text())

        return Bound2D(xox, yoy, width, height)
    

    def get_agg_color(self, rgb):
        """returns an agg color object based on the given rgb tuple"""
        r,g,b = rgb
        return self._renderer.rgb(int(r*255),int(g*255),int(b*255))

    def get_text_scale(self):
        """
        Return the scale factor for fontsize taking screendpi and pixels per
        inch into account
        """
        return self.dpi.get()/PIXELS_PER_INCH


    def points_to_pixels(self, points):
        """
        convert point measures to pixes using dpi and the pixels per
        inch of the display
        """
        #print 'points to pixels'
        return points*PIXELS_PER_INCH/72.0*self.dpi.get()/72.0
    
def new_figure_manager(num, *args):
    """
    Create a new figure manager instance
    """
    #print 'new figure manager'
    thisFig = Figure(*args)
    canvas = FigureCanvasAgg(thisFig)
    manager = FigureManagerBase(canvas, num)
    return manager


class FigureCanvasAgg(FigureCanvasBase):
    """
    The canvas the figure renders into.  Calls the draw and print fig
    methods, creates the renderers, etc...

    Public attribute

      figure - A Figure instance
    """

    def draw(self):
        """
        Draw the figure using the renderer
        """
        #print 'draw'
        l,b,w,h = self.figure.get_window_extent().get_bounds()
        self.renderer = RendererAgg(w, h, self.figure.dpi)
        self.figure.draw(self.renderer)

    def print_figure(self, filename, dpi=150,
                     facecolor='w', edgecolor='w',
                     orientation='portrait'):
        """
        Render the figure to hardcopy.  Set the figure patch face and
        edge colors.  This is useful because some of the GUIs have a
        gray figure face color background and you'll probably want to
        override this on hardcopy

        If the extension matches PNG, write a PNG file

        If the extension matches BMP or RAW, write an RGBA bitmap file

        """

        #print 'print_figure'
        
        # store the orig figure dpi, color and size information so we
        # can restore them later.  For image creation alone, this is
        # not important since after the print the figure is done.  But
        # backend_agg may be used as a renderer for a GUI figure, and
        # restoring figure props will be important in that case.
        # TODO: move most of this functionality into backend_bases
        origDPI = self.figure.dpi.get()
        scale = dpi/origDPI
        xmin, xmax = self.figure.bbox.x.bounds()
        ymin, ymax = self.figure.bbox.y.bounds()
        origfacecolor = self.figure.get_facecolor()
        origedgecolor = self.figure.get_edgecolor()

        # now set the figure props for the print
        self.figure.bbox.x.set_max(scale*xmax)
        self.figure.bbox.y.set_max(scale*ymax)
        self.figure.dpi.set(dpi)
        for a in self.figure.get_axes(): a.resize()
        self.figure.set_facecolor(facecolor)
        self.figure.set_edgecolor(edgecolor)

        # render the printed figure
        self.draw()

        # take a look at the extension and choose the print handler
        basename, ext = os.path.splitext(filename)
        if not len(ext):
            ext = '.png'
            filename += ext

        ext = ext.lower()
        if (ext.find('rgb')>=0 or
            ext.find('raw')>=0 or
            ext.find('bmp')>=0 ):
            self.renderer._renderer.write_rgba(filename)
        elif ext.find('png')>=0:
            self.renderer._renderer.write_png(filename)
        else:
            error_msg('Do not know know to handle extension *%s' % ext)

        # restore the original figure properties
        self.figure.bbox.x.set_max(xmax)
        self.figure.bbox.y.set_max(ymax)
        self.figure.dpi.set(origDPI)
        for a in self.figure.get_axes(): a.resize()
        self.figure.set_facecolor(origfacecolor)
        self.figure.set_edgecolor(origedgecolor)

            
            
            
                
    

         
