"""
A gd backend http://newcenturycomputers.net/projects/gdmodule.html
"""


from __future__ import division
import sys, os, math

try: import gd
except ImportError:
    print >>sys.stderr, 'You must first install the gd module http://newcenturycomputers.net/projects/gdmodule.html'
    sys.exit()

def gd_requirements_failed():
    'Print a message about the required gd version and quit'

    error_msg_gd( """\
You must use gd 2.0.21 (or later) from http://www.boutell.com/gd and
gdmodule-0.52 (or later) from
http://newcenturycomputers.net/projects/gdmodule.html
""")
    sys.exit()
        
    
from matplotlib.backend_bases import RendererBase, \
     GraphicsContextBase, FigureManagerBase, FigureCanvasBase,\
     error_msg

from matplotlib._matlab_helpers import Gcf
from matplotlib.cbook import enumerate, True, False, pieces, is_string_like
from matplotlib.colors import colorConverter
from matplotlib.figure import Figure
from matplotlib.transforms import Bound2D
from ttf_font_manager import fontManager
from matplotlib import get_data_path

# support old font names
if (os.environ.has_key('GDFONTPATH') and not
    os.environ.has_key('TTFPATH')):
    os.environ['TTFPATH'] = os.environ['GDFONTPATH']



PIXELS_PER_INCH = 96  # constant GD uses for screen DPI

def error_msg_gd(msg, *args):
    """
    Signal an error condition -- in a GUI, popup a error dialog
    """
    print >>sys.stderr, 'Error:', msg

def round(x):
    return int(math.floor(x+0.5))


class RendererGD(RendererBase):
    """
    The renderer handles all the drawing primitives using a graphics
    context instance that controls the colors/styles
    """

    
    # todo: can gd support cap and join styles?
    def __init__(self, im, dpi):
        "Initialize the renderer with a gd image instance"
        self.im = im
        self._cached = {}  # a map from get_color args to colors

        self.width, self.height = im.size()
        self.dpi = dpi

        
    def compute_text_offsets(self, t):
        """
        Return the (x,y) offsets to adjust for the alignment
        specifications
        """

        text = t.get_text()
        size = t.get_fontsize()
        font = fontManager.findfont(t)

        scale = self.get_text_scale(t)
        try:
            llx, lly, lrx, lry, urx, ury, ulx, uly = \
                 self.im.get_bounding_rect(
                font, scale*size, 0.0, (0,0), text)
        except ValueError:
            error_msg_gd('Could not load font %s.  Try setting TTFFONTPATH to include this font' % fontname)
            
        w = lrx - llx
        h = lly - uly

        #print self._text, llx, lrx, lry, ury

        halign = t.get_horizontalalignment()
        valign = t.get_verticalalignment()

        if t.get_rotation()=='vertical':
            w, h = h, w
            if halign=='center': offsetx = w/2
            elif halign=='right': offsetx = 0
            else: offsetx = w

            if valign=='center': offsety = -h/2
            elif valign=='top': offsety = -h
            else: offsety = 0
        else:
            if halign=='center': offsetx = -w/2
            elif halign=='right': offsetx = -w
            else: offsetx = 0

            if valign=='center': offsety = -h/2
            elif valign=='top': offsety = -h
            else: offsety = 0

        return (offsetx, offsety)



    def draw_arc(self, gc, rgbFace, x, y, width, height, angle1, angle2):
        """
        Draw an arc centered at x,y with width and height and angles
        from 0.0 to 360.0
        """

        center = int(x), self.height-int(y)
        wh = int(width), int(height)
        a1, a2 = int(angle1), int(angle2)
        if rgbFace is not None:
            color = self.get_gd_color( rgbFace )
            self.im.filledEllipse(
                center, wh, color)
        color = self.get_gd_color( gc.get_rgb() )
        self.im.arc(center, wh, a1, a2, color)
        self.flush_clip()

    def draw_line(self, gc, x1, y1, x2, y2):
        """
        Draw a single line from x1,y1 to x2,y2
        """
        self.draw_lines(gc, (x1, x2), (y1, y2))
        
    def draw_lines(self, gc, x, y):
        """
        x and y are equal length arrays, draw lines connecting each
        point in x, y
        """

        style = self._set_gd_style(gc)
        points = [ (int(thisx),self.height-int(thisy))
                   for thisx,thisy in zip(x,y)]
        self.im.lines(points, style)
        self.flush_clip()

    def draw_point(self, gc, x, y):
        """
        Draw a single point at x,y
        """
        self.im.setPixel((int(x),self.height-int(y)),
                         self.get_gd_color( gc.get_rgb() ))
        self.flush_clip()


    
    def draw_polygon(self, gc, rgbFace, points):
        """
        Draw a polygon.  points is a len vertices tuple, each element
        giving the x,y coords a vertex
        """

        edgecolor = self.get_gd_color( gc.get_rgb() )

        points = [(int(x), self.height-int(y)) for x,y in points]


        if rgbFace is not None:
            facecolor = self.get_gd_color( rgbFace )
            self.im.filledPolygon(points, facecolor)
        else: facecolor = None
        if edgecolor != facecolor:
            self.im.polygon(points, edgecolor)
        self.flush_clip()
        
    def draw_rectangle(self, gc, rgbFace, x, y, width, height):
        """
        Draw a rectangle at lower left x,y with width and height
        If filled=True, fill the rectangle with the gc foreground
        gc is a GraphicsContext instance
        """

        lb = int(x), self.height-int(y)
        ur = int(x+width), self.height-int((y+height))
        edgecolor = self.get_gd_color( gc.get_rgb() )
        
        if rgbFace is not None:
            facecolor = self.get_gd_color( rgbFace )
            self.im.filledRectangle(ur, lb, facecolor)
        else: facecolor = None

        if edgecolor != facecolor:
            self.im.rectangle(ur, lb, edgecolor)
        self.flush_clip()
            
    def draw_text(self, gc, x, y, t):
        """
        Render the text using the RendererGD instance
        """
        text = t.get_text()
        size = t.get_fontsize()
        font = fontManager.findfont(t)

        ox, oy = self.compute_text_offsets(t)
        x = int(x+ox)
        y = int(y+oy)
        y = self.height - y

        color = self.get_gd_color( gc.get_rgb() )

        if t.get_rotation()=='vertical': angle=math.pi/2.0
        else: angle = 0

        scale = self.get_text_scale(t)
        self.im.string_ft(font, scale*size, angle,
                          (x, y), text, color)
        self.flush_clip()

    def finish(self):
        pass
        #self.im.writePng( file('xx.png', 'w') )

    
    def flush_clip(self):
        imw, imh = self.im.size()
        lb = 0, 0
        ur = imw, imh
        self.im.setClip(ur, lb)        


    def get_gd_color(self, rgb):
        """
        RGB is a unit RGB tuple, return a gd color
        """

        
        try: return self._cached[rgb]
        except KeyError: pass

        r,g,b = rgb
        arg = (int(r*255),int(g*255),int(b*255))
        color = self.im.colorAllocate(  arg )

        if color==-1:
            print >>sys.stderr, ('Unable to allocate color %1.3f, %1.3f, %1.3f; using nearest neighbor' % rgb)
            color = self.im.colorClosest(arg)

        self._cached[rgb] = color
        return color



    def get_text_extent(self, t):

        # there is a very small error in the x window extent.  Don't
        # know where it is coming from right now

        text = t.get_text()
        size = t.get_fontsize()
        font = fontManager.findfont(t)

        x,y = t.get_xy_display()
        ox, oy = self.compute_text_offsets(t)
        x = int(x+ox)
        y = int(y+oy)

        scale = self.get_text_scale(t)
        llx, lly, lrx, lry, urx, ury, ulx, uly = self.im.get_bounding_rect(
            font, scale*size, 0, (int(x),int(y)), text) 
        w = lrx - llx
        h = lly - uly
        if t.get_rotation()=='vertical':
            w,h = h,w
            x = x - w
        bbox = Bound2D(x, y, w, h)
        return bbox
    

    def get_text_scale(self, t):
        """
        Return the scale factor for fontsize taking screendpi and pixels per
        inch into account
        """
        return t.dpi.get()/PIXELS_PER_INCH
    
    def new_gc(self):
        """
        Return an instance of a GraphicsContextGD
        """
        return GraphicsContextGD( self.im, self )

    def _set_gd_style(self, gc):
        color = self.get_gd_color( gc.get_rgb() )
        offset, dashes = gc.get_dashes()

        if dashes is not None:
            pixels = self.points_to_pixels(dashes)
            #print dashes, pixels
            style = []
            for on, off in pieces(pixels):
                #print 'before:', on, off
                if on<1: on = 1
                else: on = round(on)
                if off<1: off = 1
                else: off = round(off)
                #print 'after:', on, off

                style.extend([color]*on)
                style.extend([gd.gdTransparent]*off)
            #print style
            #print
            self.im.setStyle(style)
            return gd.gdStyled
        else:
            if gc.get_antialiased():
                self.im.setAntiAliased(color)
                return gd.gdAntiAliased
            else:
                self.im.setStyle([color])
                return gd.gdStyled


    def points_to_pixels(self, points):
        """
        convert point measures to pixes using dpi and the pixels per
        inch of the display
        """
        return points*PIXELS_PER_INCH/72.0*self.dpi.get()/72.0

        
class GraphicsContextGD(GraphicsContextBase):
    """
    The graphics context provides the color, line styles, etc...  See
    the gtk and postscript backends for examples of mapping the
    graphics context attributes (cap styles, join styles, line widths,
    colors) to a particular backend.      """
    def __init__(self, im, renderer):
        """
        Initialize with a gd image
        """
        GraphicsContextBase.__init__(self)
        self.im = im
        self.renderer = renderer
        

    def set_clip_rectangle(self, rectangle):
        GraphicsContextBase.set_clip_rectangle(self, rectangle)
        x,y,w,h = rectangle
        imw, imh = self.im.size()
        lb = int(x), imh-int(y)
        ur = int(x+w), imh-int(y+h)
        self.im.setClip(ur, lb)        

    def set_linestyle(self, style):
        GraphicsContextBase.set_linestyle(self, style)
        offset, dashes = self._dashd[style]
        self.set_dashes(offset, dashes)

    def set_linewidth(self, lw):
        GraphicsContextBase.set_linewidth(self, lw)
        pixels = self.renderer.points_to_pixels(lw)
        if pixels<1: pixels = 1
        else: pixels = round(pixels)
        self.im.setThickness(pixels)
              
########################################################################
#    
# The following functions and classes are for matlab compatibility
# mode (matplotlib.matlab) and implement figure managers, etc...
#
########################################################################


def new_figure_manager(num, *args):
    """
    Add a new figure num (default autoincrement).  For GUI
    backends, you'll need to instantiate a new window and embed
    the figure in it.
    """
    thisFig = Figure(*args)
    canvas = FigureCanvasGD(thisFig)
    manager = FigureManagerGD(canvas, num)
    return manager


class FigureCanvasGD(FigureCanvasBase):
    
    def print_figure(self, filename, dpi=150,
                     facecolor='w', edgecolor='w',
                     orientation='portrait'):

        """
        Render the figure to hardcopy using self.renderer as the
        renderer if neccessary

        filename can be a string filename or writable file instance

        """

        origDPI = self.figure.dpi.get()
        scale = dpi/origDPI
        xmin, xmax = self.figure.bbox.x.bounds()
        ymin, ymax = self.figure.bbox.y.bounds()
        self.figure.bbox.x.set_max(scale*xmax)
        self.figure.bbox.y.set_max(scale*ymax)
        self.figure.dpi.set(dpi)
        for a in self.figure.get_axes(): a.resize()

        origfacecolor = self.figure.get_facecolor()
        origedgecolor = self.figure.get_edgecolor()
        self.figure.set_facecolor(facecolor)
        self.figure.set_edgecolor(edgecolor)

        im = self.draw()

        if is_string_like(filename):
            basename, ext = os.path.splitext(filename)
            if not len(ext): filename += '.png'
 
        im.writePng( filename )

        self.figure.bbox.x.set_max(xmax)
        self.figure.bbox.y.set_max(ymax)
        self.figure.dpi.set(origDPI)
        for a in self.figure.get_axes(): a.resize()
        self.figure.set_facecolor(origfacecolor)
        self.figure.set_edgecolor(origedgecolor)

    def draw(self):
        """
        Draw to a gd image and return the image instance
        
        """

        left, bottom, width, height = self.figure.bbox.get_bounds()
        im = gd.image((int(width), int(height)))

        if not hasattr(im, 'setAntiAliased'):
            gd_requirements_failed()
        renderer = RendererGD(im, self.figure.dpi)
        for a in self.figure.axes:  a.resize()

        self.figure.draw(renderer)
        renderer.finish()

        return im

class FigureManagerGD(FigureManagerBase):
    """
    This class manages all the figures for matlab mode
    """
    pass



########################################################################
#    
# Now just provide the standard names that backend.__init__ is expecting
# 
########################################################################

FigureManager = FigureManagerGD
error_msg = error_msg_gd
         
