"""
A class for managing and finding fonts across platforms

This module provides a single FontManager fontManager that can be
shared across backends.  It returns the best *.ttf file in the TTFPATH
that matches the matplotlib.text.Text properties
"""

import os, sys

from ttfquery import ttffiles, describe
from ttfquery._scriptregistry import registry, registryFile
from matplotlib import get_data_path
class FontManager:

    paths = [get_data_path()]

    
    if os.environ.has_key('TTFPATH'):
        ttfpath = os.environ['TTFPATH']
        if ttfpath.find(';')>0: #win32 style
            paths.extend(ttfpath.split(';'))
        elif ttfpath.find(':')>0: # unix style
            paths.extend(ttfpath.split(':'))
        else:
            paths.append(ttfpath)

    #print 'PATHS', paths
    new, failed = registry.scan( paths, printErrors = 1, force = 0)
    registry.save(registryFile)
    cache = {}

    def findfont(self, text):
        """
        text is a Text instance.  Return the proper font, using
        cacheing for efficiency
        """

        fontname = text.get_fontname()
        weight = text.get_fontweight()
        italics = text.get_fontangle()

        origname = fontname
        fontname = fontname.lower()

        if italics.find('ital')>=0: italics = 1
        elif italics.find('oblique')>=0: italics = 1 # todo fixme for oblique
        else: italics = None

        try: return self.cache[ (fontname, weight, italics) ]
        except KeyError: pass
        
        for name, table in registry.fonts.items():
            if name.lower().find(fontname)<0: continue

            items = table.items()
            items.sort()

            if weight is not None:
                weight = describe.weightNumber( weight )
                items = [item for item in items if item[0][0]==weight]
            if italics is not None:
                items = [item for item in items if item[0][1]==italics]
            if len(items):
                match = items[0][1][0]
                fullpath = registry.fontFile( match )
                head, tail = os.path.split(fullpath)
                fname, ext = os.path.splitext(tail)
                self.cache[ (fontname, weight, italics) ] = fullpath
                return fullpath
        default = self.get_default_font(origname)
        self.cache[ (fontname, weight, italics) ] = default

        return default

    def get_default_font(self, fontname):
        
        print >>sys.stderr, 'Could not find requested font %s' % fontname 
        print >>sys.stderr, 'Please set environment var TTFPATH to point to your true type fonts (*.ttf)'

        if fontname.lower()=='vera':
            print >>sys.stderr, 'Could not find any fonts or the default Vera' 
            print >>sys.stderr, 'Please check your TTFPATH'
            sys.exit()
        else:
            print >>sys.stderr, 'Could not find %s; falling back on Vera' % fontname
            return os.path.join(get_data_path(),  'Vera.ttf')
    

class FontManagerFixed(FontManager):
    """
    Always return the same font for performace    
    """
    def __init__(self, name):
        self.name = name
    def findfont(self, *arg, **kwargs):
        return self.name


fontManager = FontManager()
