"""
A gd backend http://newcenturycomputers.net/projects/gdmodule.html
"""


from __future__ import division
import sys, os, math

try: import gd
except ImportError:
    print >>sys.stderr, 'You must first install the gd module http://newcenturycomputers.net/projects/gdmodule.html'
    sys.exit()

def gd_requirements_failed():
    'Print a message about the required gd version and quit'

    error_msg_gd( """\
You must use gd 2.0.21 (or later) from http://www.boutell.com/gd and
gdmodule-0.52 (or later) from
http://newcenturycomputers.net/projects/gdmodule.html
""")
    sys.exit()
        
    
from matplotlib.backend_bases import RendererBase, \
     GraphicsContextBase, FigureManagerBase, FigureCanvasBase,\
     error_msg, arg_to_rgb

from matplotlib._matlab_helpers import Gcf
from matplotlib.cbook import enumerate, True, False, pieces
from matplotlib.figure import Figure
from matplotlib.transforms import Bound2D
from matplotlib import get_data_path

PIXELS_PER_INCH = 96  # constant GD uses for screen DPI

def error_msg_gd(msg, *args):
    """
    Signal an error condition -- in a GUI, popup a error dialog
    """
    print >>sys.stderr, 'Error:', msg

def round(x):
    return int(math.floor(x+0.5))


class RendererGD(RendererBase):
    """
    The renderer handles all the drawing primitives using a graphics
    context instance that controls the colors/styles
    """

    # todo: can gd support cap and join styles?
    def __init__(self, im, dpi):
        "Initialize the renderer with a gd image instance"
        self.im = im
        self._cached = {}  # a map from get_color args to colors

        self.width, self.height = im.size()
        self.dpi = dpi


        
    def compute_text_offsets(self, t):
        """
        Return the (x,y) offsets to adjust for the alignment
        specifications
        """

        fontname = t.get_fontname()
        fontweight = t.get_fontweight()
        fontangle = t.get_fontangle()
        text = t.get_text()
        size = t.get_fontsize()
        font = fontManager.findfont(fontname, fontweight, fontangle)

        scale = self.get_text_scale(t)
        try:
            llx, lly, lrx, lry, urx, ury, ulx, uly = \
                 self.im.get_bounding_rect(
                font, scale*size, 0.0, (0,0), text)
        except ValueError:
            error_msg_gd('Could not load font %s.  Try setting GDFONTPATH to include this font' % fontname)
            
        w = lrx - llx
        h = lly - uly

        #print self._text, llx, lrx, lry, ury

        halign = t.get_horizontalalignment()
        valign = t.get_verticalalignment()

        if t.get_rotation()=='vertical':
            w, h = h, w
            if halign=='center': offsetx = w/2
            elif halign=='right': offsetx = 0
            else: offsetx = w

            if valign=='center': offsety = -h/2
            elif valign=='top': offsety = -h
            else: offsety = 0
        else:
            if halign=='center': offsetx = -w/2
            elif halign=='right': offsetx = -w
            else: offsetx = 0

            if valign=='center': offsety = -h/2
            elif valign=='top': offsety = -h
            else: offsety = 0

        return (offsetx, offsety)



    def draw_arc(self, gc, faceColor, x, y, width, height, angle1, angle2):
        """
        Draw an arc centered at x,y with width and height and angles
        from 0.0 to 360.0
        """
        color = self.get_gd_color( gc.get_rgb() )
        center = int(x), self.height-int(y)
        wh = int(width), int(height)
        a1, a2 = int(angle1), int(angle2)
        
        self.im.arc(center, wh, a1, a2, color)

        if faceColor is not None:
            color = self.get_gd_color( arg_to_rgb(faceColor) )
            self.im.fill( center, color )
        self.flush_clip()

    def draw_line(self, gc, x1, y1, x2, y2):
        """
        Draw a single line from x1,y1 to x2,y2
        """
        style = self._set_gd_style(gc)        
        self.im.line((int(x1),self.height-int(y1)),
                     (int(x2), self.height-int(y2)),
                     style)
        self.flush_clip()
        
    def draw_lines(self, gc, x, y):
        """
        x and y are equal length arrays, draw lines connecting each
        point in x, y
        """

        style = self._set_gd_style(gc)
        points = [ (int(thisx),self.height-int(thisy))
                   for thisx,thisy in zip(x,y)]
        self.im.lines(points, style)
        self.flush_clip()



    def draw_point(self, gc, x, y):
        """
        Draw a single point at x,y
        """
        self.im.setPixel((int(x),self.height-int(y)),
                         self.get_gd_color( gc.get_rgb() ))
        self.flush_clip()


    
    def draw_polygon(self, gc, faceColor, points):
        """
        Draw a polygon.  points is a len vertices tuple, each element
        giving the x,y coords a vertex
        """

        color = self.get_gd_color( gc.get_rgb() )

        points = [(int(x), self.height-int(y)) for x,y in points]


        if faceColor is not None:
            faceColor = self.get_gd_color( arg_to_rgb(faceColor) )
            self.im.filledPolygon(points, faceColor)

        if color != faceColor:
            self.im.polygon(points, color)
        self.flush_clip()
        
    def draw_rectangle(self, gc, faceColor, x, y, width, height):
        """
        Draw a rectangle at lower left x,y with width and height
        If filled=True, fill the rectangle with the gc foreground
        gc is a GraphicsContext instance
        """

        lb = int(x), self.height-int(y)
        ur = int(x+width), self.height-int((y+height))
        color = self.get_gd_color( gc.get_rgb() )
        
        if faceColor is not None:
            rgb = arg_to_rgb(faceColor)
            faceColor = self.get_gd_color( rgb )
            self.im.filledRectangle(ur, lb, faceColor)

        if color != faceColor:
            self.im.rectangle(ur, lb, color)
        self.flush_clip()
            
    def draw_text(self, gc, x, y, t):
        """
        Render the text using the RendererGD instance
        """
        fontname = t.get_fontname()
        fontweight = t.get_fontweight()
        fontangle = t.get_fontangle()
        text = t.get_text()
        size = t.get_fontsize()
        font = fontManager.findfont(fontname, fontweight, fontangle)

        ox, oy = self.compute_text_offsets(t)
        x = int(x+ox)
        y = int(y+oy)
        y = self.height - y

        color = self.get_gd_color( gc.get_rgb() )

        if t.get_rotation()=='vertical': angle=math.pi/2.0
        else: angle = 0

        scale = self.get_text_scale(t)
        self.im.string_ft(font, scale*size, angle,
                            (x, y), text, color)
        self.flush_clip()

    def finish(self):
        pass
        #self.im.writePng( file('xx.png', 'w') )

    
    def flush_clip(self):
        imw, imh = self.im.size()
        lb = 0, 0
        ur = imw, imh
        self.im.setClip(lb, ur)        


    def get_gd_color(self, rgb):
        """
        RGB is a unit RGB tuple, return a gd color
        """

        
        try: return self._cached[rgb]
        except KeyError: pass

        r,g,b = rgb
        color = self.im.colorAllocate( (int(r*255),int(g*255),int(b*255)) )

        def dist(p1, p2):
            return math.sqrt( (p1[0]-p2[0])**2 + (p1[1]-p2[1])**2 + (p1[2]-p2[2])**2)
        if color==-1:
            
            error_msg_gd('Unable to allocate color %1.3f, %1.3f, %1.3f; using nearest neighbor' % (r, g, b))
            items = self._cached.items()
            distances = [ (dist(rgb, tup), color) for tup, color  in items]
            distances.sort()
            distance, color = distances[0]
        self._cached[rgb] = color
        return color



    def get_text_extent(self, t):

        # there is a very small error in the x window extent.  Don't
        # know where it is coming from right now

        fontname = t.get_fontname()
        fontweight = t.get_fontweight()
        fontangle = t.get_fontangle()
        text = t.get_text()
        size = t.get_fontsize()
        font = fontManager.findfont(fontname, fontweight, fontangle)

        x,y = t.get_xy_display()
        ox, oy = self.compute_text_offsets(t)
        x = int(x+ox)
        y = int(y+oy)

        scale = self.get_text_scale(t)
        llx, lly, lrx, lry, urx, ury, ulx, uly = self.im.get_bounding_rect(
            font, scale*size, 0, (int(x),int(y)), text) 
        w = lrx - llx
        h = lly - uly
        if t.get_rotation()=='vertical':
            w,h = h,w
            x = x - w
        bbox = Bound2D(x, y, w, h)
        return bbox
    

    def get_text_scale(self, t):
        """
        Return the scale factor for fontsize taking screendpi and pixels per
        inch into account
        """
        return t.dpi.get()/PIXELS_PER_INCH
    
    def new_gc(self):
        """
        Return an instance of a GraphicsContextGD
        """
        return GraphicsContextGD( self.im, self.dpi )

    def _set_gd_style(self, gc):
        color = self.get_gd_color( gc.get_rgb() )
        offset, dashes = gc.get_dashes()

        if dashes is not None:
            pixels = gc.points_to_pixels(dashes)
            #print dashes, pixels
            style = []
            for on, off in pieces(pixels):
                #print 'before:', on, off
                if on<1: on = 1
                else: on = round(on)
                if off<1: off = 1
                else: off = round(off)
                #print 'after:', on, off

                style.extend([color]*on)
                style.extend([gd.gdTransparent]*off)
            #print style
            #print
            self.im.setStyle(style)
            return gd.gdStyled
        else:
            if gc.get_antialiased():
                self.im.setAntiAliased(color)
                return gd.gdAntiAliased
            else:
                self.im.setStyle([color])
                return gd.gdStyled



        
class GraphicsContextGD(GraphicsContextBase):
    """
    The graphics context provides the color, line styles, etc...  See
    the gtk and postscript backends for examples of mapping the
    graphics context attributes (cap styles, join styles, line widths,
    colors) to a particular backend.      """
    def __init__(self, im, dpi):
        """
        Initialize with a gd image
        """
        GraphicsContextBase.__init__(self)
        self.im = im
        self.dpi = dpi
        
    def points_to_pixels(self, points):
        return points*PIXELS_PER_INCH/72.0*self.dpi.get()/72.0
        

    def set_clip_rectangle(self, rectangle):
        GraphicsContextBase.set_clip_rectangle(self, rectangle)
        x,y,w,h = rectangle
        imw, imh = self.im.size()
        lb = int(x), imh-int(y)
        ur = int(x+w), imh-int(y+h)
        self.im.setClip(lb, ur)        

    def set_linestyle(self, style):
        GraphicsContextBase.set_linestyle(self, style)
        offset, dashes = self._dashd[style]
        self.set_dashes(offset, dashes)

    def set_linewidth(self, lw):
        GraphicsContextBase.set_linewidth(self, lw)
        pixels = self.points_to_pixels(lw)
        if pixels<1: pixels = 1
        else: pixels = round(pixels)
        self.im.setThickness(pixels)
              
########################################################################
#    
# The following functions and classes are for matlab compatibility
# mode (matplotlib.matlab) and implement figure managers, etc...
#
########################################################################


def new_figure_manager(num, figsize, dpi):
    """
    Add a new figure num (default autoincrement).  For GUI
    backends, you'll need to instantiate a new window and embed
    the figure in it.
    """
    thisFig = Figure(figsize, dpi)
    canvas = FigureCanvasGD(thisFig)
    manager = FigureManagerGD(canvas, num)
    return manager


class FigureCanvasGD(FigureCanvasBase):
    
    def print_figure(self, filename, dpi, facecolor='w', edgecolor='w'):
        """
        Render the figure to hardcopy using self.renderer as the
        renderer if neccessary

        """
        origfacecolor = self.figure.get_facecolor()
        origedgecolor = self.figure.get_edgecolor()
        self.figure.set_facecolor(facecolor)
        self.figure.set_edgecolor(edgecolor)

        im = self.draw_image(dpi)

        basename, ext = os.path.splitext(filename)
        if not len(ext): filename += '.png'
        im.writePng( file(filename, 'wb') )
        self.figure.set_facecolor(origfacecolor)
        self.figure.set_edgecolor(origedgecolor)

    def draw(self): pass

    def draw_image(self, dpi):
        """
        Draw to a gd image and return the image instance
        
        """

        oldDPI = self.figure.dpi.get()
        scale = dpi/oldDPI
        xmin, xmax = self.figure.bbox.x.bounds()
        ymin, ymax = self.figure.bbox.y.bounds()


        self.figure.bbox.x.set_max(scale*xmax)
        self.figure.bbox.y.set_max(scale*ymax)
        
        self.figure.dpi.set(dpi)
        left, bottom, width, height = self.figure.bbox.get_bounds()
        im = gd.image((int(width), int(height)))

        if not hasattr(im, 'setAntiAliased'):
            gd_requirements_failed()
        renderer = RendererGD(im, self.figure.dpi)
        for a in self.figure.axes:  a.resize()

        self.figure.draw(renderer)
        renderer.finish()
        self.figure.bbox.x.set_max(xmax)
        self.figure.bbox.y.set_max(ymax)

        return im

class FigureManagerGD(FigureManagerBase):
    """
    This class manages all the figures for matlab mode
    """
    pass

class FontManager:
    from ttfquery import ttffiles, describe
    from ttfquery._scriptregistry import registry, registryFile

    paths = [get_data_path()]
 
    if os.environ.has_key('GDFONTPATH'):
        gdpath = os.environ['GDFONTPATH']
        if gdpath.find(';')>0: #win32 style
            paths.extend(gdpath.split(';'))
        elif gdpath.find(':')>0: # unix style
            paths.extend(gdpath.split(':'))
        else:
            paths.append(gdpath)
    #print 'PATHS', paths
    new, failed = registry.scan( paths, printErrors = 1, force = 0)
    registry.save(registryFile)

    cache = {}

    def findfont(self, fontname, weight, italics):
        origname = fontname
        fontname = fontname.lower()

        if italics.find('ital')>=0: italics = 1
        elif italics.find('oblique')>=0: italics = 1 # todo fixme for oblique
        else: italics = None
        
        try: return self.cache[ (fontname, weight, italics) ]
        except KeyError: pass
        
        for name, table in self.registry.fonts.items():
            if name.lower().find(fontname)<0: continue

            items = table.items()
            items.sort()

            if weight is not None:
                weight = self.describe.weightNumber( weight )
                items = [item for item in items if item[0][0]==weight]
            if italics is not None:
                items = [item for item in items if item[0][1]==italics]
            if len(items):
                match = items[0][1][0]
                fullpath = self.registry.fontFile( match )
                head, tail = os.path.split(fullpath)
                fname, ext = os.path.splitext(tail)
                self.cache[ (fontname, weight, italics) ] = fullpath
                return fullpath
        default = self.get_default_font(origname)
        self.cache[ (fontname, weight, italics) ] = default
        return default

    def get_default_font(self, fontname):
        error_msg_gd('Could not find requested font %s\n' % fontname + 
                     'Please set environment var GDFONTPATH to point to your true type fonts (*.ttf)')

        if fontname.lower()=='vera':
            error_msg_gd('Could not find any fonts or the default Vera\n' +
                      'Please check your GDFONTPATH')
        error_msg_gd('Could not find %s; falling back on Vera' % fontname)
        return os.path.join(get_data_path(),  'Vera.ttf')


fontManager = FontManager()

########################################################################
#    
# Now just provide the standard names that backend.__init__ is expecting
# 
########################################################################

FigureManager = FigureManagerGD
error_msg = error_msg_gd
         
