from artist import Artist, DPI
from patches import Rectangle
from text import Text, _process_text_args

from transforms import Bound1D, Bound2D, Transform

class Figure(Artist):
    def __init__(self, figsize, dpi, facecolor=0.75, edgecolor='w'):
        """
        paper size is a w,h tuple in inches
        DPI is dots per inch 
        """

        bbox = Bound2D(0, 0, figsize[0]*dpi, figsize[1]*dpi)
        Artist.__init__(self, DPI(dpi), bbox)
        self.set_clip_on(False)

        self.figsize = figsize


        
        self.axes = []
        self._text=[]
        
        self._figurePatch = Rectangle(
            self.dpi, self.bbox, 
            (0,0), 1, 1,
            facecolor=facecolor, edgecolor=edgecolor,
            transx = Transform( Bound1D(0,1), self.bbox.x),
            transy = Transform( Bound1D(0,1), self.bbox.y),
            )

    def get_edgecolor(self):
        'Get the edge color of the Figure rectangle' # 
        return self._figurePatch.get_edgecolor()

    def get_facecolor(self):
        'Get the face color of the Figure rectangle'
        return self._figurePatch.get_facecolor()

    def set_edgecolor(self, color):
        'Set the edge color of the Figure rectangle'
        self._figurePatch.set_edgecolor(color)

    def set_facecolor(self, color):
        'Set the face color of the Figure rectangle'
        self._figurePatch.set_facecolor(color)


    def add_axis(self, a):
        self.axes.append(a)
        a.set_clip_on(False)

    def clear(self):
        self.axes = []
        self._text = []
        self.draw()

    def _draw(self, renderer, *args, **kwargs):
        """
        Render the figure using RendererGD instance renderer
        """
        # draw the figure bounding box, perhaps none for white figure

        self._figurePatch.draw(renderer)

        # render the axes
        for a in self.axes:
            a.draw(renderer)

        # render the figure text
        for t in self._text:
            t.draw(renderer)
        

    def get_axes(self):
        return self.axes

    def get_window_extent(self, renderer=None):
        'Return the window extent of the figure'
        xmin, xmax = self.bbox.x.bounds()
        w = xmax-xmin
        ymin, ymax = self.bbox.y.bounds()
        h = ymax-ymin
        return Bound2D(xmin, ymin, w, h)
        

    def text(self, x, y, s, *args, **kwargs):
        """
        Add text to figure at location x,y (relative 0-1 coords) See
        the help for Axis text for the meaning of the other arguments
        """

        override = _process_text_args({}, *args, **kwargs)
        t = Text(
            self.dpi, self.bbox,
            x=x, y=y, text=s,
            transx = Transform(Bound1D(0,1), self.bbox.x),
            transy = Transform(Bound1D(0,1), self.bbox.y),
            **override)
        self._text.append(t)
        return t

    def copy_properties(self, other):
        other.figsize = self.figsize
        other.bbox = self.bbox
        other.dpi = self.dpi
        other._figurePatch = self._figurePatch
        other.axes = self.axes
        other._text = self._text


