import sys
from artist import Artist
from axes import Axes, Subplot
from cbook import flatten, True, False
from matplotlib import rcParams
from patches import Rectangle
from text import Text, _process_text_args
from legend import Legend
from transforms import Bbox, Value, Point, get_bbox_transform, unit_bbox

class Figure(Artist):
    
    def __init__(self,
                 figsize   = rcParams['figure.figsize'],
                 dpi       = rcParams['figure.dpi'],
                 facecolor = rcParams['figure.facecolor'],
                 edgecolor = rcParams['figure.edgecolor'],
                 frameon = True,
                 ):
        """
        paper size is a w,h tuple in inches
        DPI is dots per inch 
        """
        Artist.__init__(self)
        self.set_figure(self)
        
        self.dpi = Value(dpi)
        self.figwidth = Value(figsize[0])
        self.figheight = Value(figsize[1])
        self.ll = Point( Value(0), Value(0) )
        self.ur = Point( self.figwidth*self.dpi,
                         self.figheight*self.dpi )
        self.bbox = Bbox(self.ll, self.ur)
        self.frameon = frameon
        
        self.transFigure = get_bbox_transform( unit_bbox(), self.bbox) 

        self.axes = []
        self._text=[]
        self._legends = []

        
        self._figurePatch = Rectangle(
            xy=(0,0), width=1, height=1,
            facecolor=facecolor, edgecolor=edgecolor,
            )
        self._set_artist_props(self._figurePatch)

    def set_figsize_inches(self, w, h):
        'set the figure size in inches'
        self.figwidth.set(w)
        self.figheight.set(h)

    def get_size_inches(self):
        return self.figwidth.get(), self.figheight.get()

    def get_edgecolor(self):
        'Get the edge color of the Figure rectangle' # 
        return self._figurePatch.get_edgecolor()

    def get_facecolor(self):
        'Get the face color of the Figure rectangle'
        return self._figurePatch.get_facecolor()

    def set_edgecolor(self, color):
        'Set the edge color of the Figure rectangle'
        self._figurePatch.set_edgecolor(color)

    def set_facecolor(self, color):
        'Set the face color of the Figure rectangle'
        self._figurePatch.set_facecolor(color)

    def add_axis(self, *args, **kwargs):
        print >> sys.stderr, """\
matplotlib changed its axes creation API in 0.54.
Please see http://matplotlib.sourceforge.net/API_CHANGES for
instructions on how to port your code.
"""

        sys.exit()
        
    def add_axes(self, rect, axisbg=rcParams['axes.facecolor'], frameon=True):
        """
        Add an a axes with axes rect [left, bottom, width, height]
        where all quantities are in fractions of figure width and
        height.

        The Axes instance will be returned
        """
        
        a = Axes(self, rect, axisbg, frameon)
        self.axes.append(a)
        return a

    def add_subplot(self, *args, **kwargs):
        """
        Add an a subplot, eg
        add_subplot(111) or add_subplot(212, axisbg='r')

        The Axes instance will be returned
        """
        a = Subplot(self, *args, **kwargs)
        self.axes.append(a)
        return a
    
    def clf(self):
        """
        Clear the figure
        """
        self.axes = []
        self._text = []

    def clear(self):
        """
        Clear the figure
        """
        self.clf()
        
    def draw(self, renderer):
        """
        Render the figure using RendererGD instance renderer
        """
        # draw the figure bounding box, perhaps none for white figure


        self.transFigure.freeze()  # eval the lazy objects
        if self.frameon: self._figurePatch.draw(renderer)

        # render the axes
        for a in self.axes: a.draw(renderer)

        # render the figure text
        for t in self._text: t.draw(renderer)

        for legend in self._legends:
            legend.draw(renderer)

        self.transFigure.thaw()  # release the lazy objects

    def get_axes(self):
        return self.axes

    def legend(self, handles, labels, loc):
        """
        Place a legend in the figure.  Labels are a sequence of
        strings, handles is a sequence of line or patch instances, and
        loc can be a string or an integer specifying the legend
        location

        USAGE: 
          legend( (line1, line2, line3),
                  ('label1', 'label2', 'label3'),
                  'upper right')

        The LOC location codes are

          'best' : 0,          (currently not supported, defaults to upper right)
          'upper right'  : 1,  (default)
          'upper left'   : 2,
          'lower left'   : 3,
          'lower right'  : 4,
          'right'        : 5,
          'center left'  : 6,
          'center right' : 7,
          'lower center' : 8,
          'upper center' : 9,
          'center'       : 10,

        The legend instance is returned
        """


        handles = flatten(handles)
        l = Legend(handles, labels, loc)
        self._set_artist_props(l)
        self._legends.append(l)
        return l
    
    def text(self, x, y, s, *args, **kwargs):
        """
        Add text to figure at location x,y (relative 0-1 coords) See
        the help for Axis text for the meaning of the other arguments
        """

        override = _process_text_args({}, *args, **kwargs)
        t = Text(
            x=x, y=y, text=s,
            )

        t.update_properties(override)
        self._set_artist_props(t)
        self._text.append(t)
        return t

    def _set_artist_props(self, a):
        a.set_figure(self)
        a.set_transform(self.transFigure)

