//----------------------------------------------------------------------------
// Anti-Grain Geometry - Version 2.0 
// Copyright (C) 2002 Maxim Shemanarev (McSeem)
//
// Permission to copy, use, modify, sell and distribute this software 
// is granted provided this copyright notice appears in all copies. 
// This software is provided "as is" without express or implied
// warranty, and with no claim as to its suitability for any purpose.
//
//----------------------------------------------------------------------------
// Contact: mcseem@antigrain.com
//          mcseemagg@yahoo.com
//          http://www.antigrain.com
//----------------------------------------------------------------------------
//
// Polygon clipping converter
// There an optimized Liang-Basky algorithm is used. 
// The algorithm doesn't optimize the degenerate edges, i.e. it will never
// break a closed polygon into two or more ones, instead, there will be 
// degenerate edges coinciding with the respective clipping boundaries.
// This is a sub-optimal solution, because that optimization would require 
// extra, rather expensive math while the rasterizer tolerates it quite well, 
// without any considerable overhead.
//
//----------------------------------------------------------------------------
#ifndef AGG_CONV_CLIP_POLYGON_INCLUDED
#define AGG_CONV_CLIP_POLYGON_INCLUDED

#include "agg_basics.h"
#include "agg_clip_liang_barsky.h"
#include "agg_vertex_iterator.h"

namespace agg
{

    //------------------------------------------------------------------------
    template<class VertexSource> struct conv_clip_polygon
    {
        enum status
        {
            initial,
            line_to,
            close_polygon,
            finalize,
            stop
        };

    public:
        typedef VertexSource source_type;

        conv_clip_polygon(VertexSource& vs) : 
            m_source(&vs),
            m_clip_box(0, 0, 1, 1),
            m_start_x(0),
            m_start_y(0),
            m_prev_x(0),
            m_prev_y(0),
            m_clip_flags(0),
            m_num_vertices(0),
            m_vertex(0),
            m_cmd(path_cmd_move_to),
            m_polygon_flags(0),
            m_status(initial)
        {
        }

        void clip_box(double x1, double y1, double x2, double y2)
        {
            m_clip_box.x1 = x1;
            m_clip_box.y1 = y1;
            m_clip_box.x2 = x2;
            m_clip_box.y2 = y2;
            m_clip_box.normalize();
        }


        double x1() const { return m_clip_box.x1; }
        double y1() const { return m_clip_box.y1; }
        double x2() const { return m_clip_box.x2; }
        double y2() const { return m_clip_box.y2; }

        // Vertex Source Interface
        void     rewind(unsigned id);
        unsigned vertex(double* x, double* y);

        // Iterator
        typedef conv_clip_polygon self_type;
        typedef vertex_iterator<self_type> iterator;
        iterator begin(unsigned id) { return iterator(*this, id); }
        iterator end() { return iterator(path_cmd_stop); }

    private:
        unsigned clipping_flags(double x, double y);
        void clip(double x, double y);

    private:
        source_type*  m_source;
        rect_d        m_clip_box;
        double        m_start_x;
        double        m_start_y;
        double        m_prev_x;
        double        m_prev_y;
        unsigned      m_clip_flags;
        double        m_x[4];
        double        m_y[4];
        unsigned      m_num_vertices;
        unsigned      m_vertex;
        unsigned      m_cmd;
        unsigned      m_polygon_flags;
        status        m_status;
    };


    //------------------------------------------------------------------------
    // Determine the clipping code of the vertex according to the 
    // Cyrus-Beck line clipping algorithm
    //
    //        |        |
    //  0110  |  0010  | 0011
    //        |        |
    // -------+--------+-------- clip_box.y2
    //        |        |
    //  0100  |  0000  | 0001
    //        |        |
    // -------+--------+-------- clip_box.y1
    //        |        |
    //  1100  |  1000  | 1001
    //        |        |
    //  clip_box.x1  clip_box.x2
    //
    // 
    template<class VertexSource> 
    inline unsigned conv_clip_polygon<VertexSource>::clipping_flags(double x, double y)
    {
        if(x < m_clip_box.x1) 
        {
            if(y > m_clip_box.y2) return 6;
            if(y < m_clip_box.y1) return 12;
            return 4;
        }

        if(x > m_clip_box.x2) 
        {
            if(y > m_clip_box.y2) return 3;
            if(y < m_clip_box.y1) return 9;
            return 1;
        }

        if(y > m_clip_box.y2) return 2;
        if(y < m_clip_box.y1) return 8;

        return 0;
    }

    //----------------------------------------------------------------------------
    template<class VertexSource> 
    inline void conv_clip_polygon<VertexSource>::clip(double x, double y)
    {
        m_vertex = 0;
        m_num_vertices = 0;
        unsigned flags = clipping_flags(x, y);
        if(m_clip_flags == flags)
        {
            if(flags == 0)
            {
                m_x[0] = x;
                m_y[0] = y;
                m_num_vertices = 1;
            }
        }
        else
        {
            m_num_vertices = clip_liang_barsky(m_prev_x, m_prev_y, 
                                               x, y, 
                                               m_clip_box, 
                                               m_x, m_y);
        }
        m_clip_flags = flags;
    }


    //----------------------------------------------------------------------------
    template<class VertexSource> 
    void conv_clip_polygon<VertexSource>::rewind(unsigned path_id)
    {
        m_source->rewind(path_id);
        m_start_x       = 0;
        m_start_y       = 0;
        m_prev_x        = 0;
        m_prev_y        = 0;
        m_clip_flags    = 0;
        m_num_vertices  = 0;
        m_vertex        = 0;
        m_cmd           = path_cmd_move_to;
        m_polygon_flags = 0;
        m_status        = initial;
    }


    //----------------------------------------------------------------------------
    template<class VertexSource> 
    unsigned conv_clip_polygon<VertexSource>::vertex(double* x, double* y)
    {
        unsigned cmd = path_cmd_stop;
        for(;;)
        {
            if(m_vertex < m_num_vertices)
            {
                *x = m_x[m_vertex];
                *y = m_y[m_vertex];
                ++m_vertex;
                cmd = m_cmd;
                m_cmd = path_cmd_line_to;
                break;
            }
            else
            {
                if(m_status == stop)
                {
                    break;
                }

                if(m_status == finalize)
                {
                    m_status = stop;
                    cmd = path_cmd_end_poly | path_flags_close | m_polygon_flags;
                    break;
                }

                if(m_status == close_polygon)
                {
                    m_status = initial;
                    m_cmd = path_cmd_move_to;
                    cmd = path_cmd_end_poly | path_flags_close | m_polygon_flags;
                    m_polygon_flags = 0;
                    break;
                }

                double tx, ty;
                cmd = m_source->vertex(&tx, &ty);

                if(is_stop(cmd))
                {
                    if(m_cmd == path_cmd_line_to)
                    {
                        clip(m_start_x, m_start_y);
                        m_status = finalize;
                    }
                    else
                    {
                        break;
                    }
                }
                else
                {
                    if(is_vertex(cmd))
                    {
                        if(is_move_to(cmd))
                        {
                            if(m_status == initial)
                            {
                                m_clip_flags = clipping_flags(tx, ty);
                                m_status = line_to;
                            }
                            else
                            {
                                if(m_cmd == path_cmd_line_to)
                                {
                                    clip(m_start_x, m_start_y);
                                    m_status = close_polygon;
                                }
                                else
                                {
                                    m_clip_flags = clipping_flags(tx, ty);
                                    m_status = line_to;
                                }
                            }
                            m_start_x = tx;
                            m_start_y = ty;
                        }
                        else
                        {
                            clip(tx, ty);
                            m_status = line_to;
                        }
                        m_prev_x = tx;
                        m_prev_y = ty;
                    }
                    else
                    {
                        clip(m_start_x, m_start_y);
                        m_prev_x = m_start_x;
                        m_prev_y = m_start_y;
                        m_status = close_polygon;
                    }
                }
            }
        }
        return cmd;
    }


}

#endif
