"""
The dates module is built around the assumption that users will want
to specify dates in a variety of ways.

The goal of the converter classes defined here is to hide all the
conversions required to plot data on the screen from you, and to
provide an abstract interface to matplotlib plotting routines so you
can use date instances of your choice.  You pass these converter
classes to date plotting functions and these functions will do the
rest.

The following converters are provided

    * EpochConverter      - dates are seconds since the epoch

    * PyDatetimeConverter - dates are python2.3 date/datetime instances

    * MxDatetimeConverter - dates are egenix mx.Datetime instances

You can provide your own converter by deriving from the base class
DateConverter and overriding the methods epoch and from_epoch.  If you
define one for a widely used date class, or extend the ones here,
please contribute the code back!  All conversions are done to and from
seconds since the epoch locatime.

Here is an example of how you could plot dates with python datetime
instances in the eastern time zone::

  from matplotlib.dates import PyDatetimeConverter, Eastern
  from matplotlib.matlab import *
  
  dates = [ dt1, dt2, dt2, ...]     # the datetime instances
  vals  = [ val1, val2, val3, ...]  # the y values
  converter = PyDatetimeConverter(Eastern)  # dates are in Eastern tz
  ax = subplot(111)
  plot_date(dates, vals, converter)

The matplotlib.ticker module provides many custom tick locators and
formatters for you so you don't have to work at the level of
manipulating epoch seconds to set your ticks.  Eg, to set the ticks on
the years, you would do::

    from matplotlib.ticker import YearLocator

    ... your plot here ...
    years = YearLocator(5)   # every 5 years
    ax.xaxis.set_major_locator(years)

and the date tick locator will try and pick the best axis range and
tick locations for you, ie, on multiples of 5 years.  Although the
default choices will probably be good, if you are unhappy with the
range choices the locator makes for you, you can use your date
instances and converter to specify them yourself, as in::

  dmin = converter( datetime(2002,01,15) )
  dmax = converter( datetime(2004,01,15) )
  ax.set_xlim( (dmin, dmax) )

Use date formatters to specify major and minor tick formats.  Minor
ticks are off by default, but you can turn them on by passing a date
locator and date formatter to the axis.  The relevant methods are::

  ax.xaxis.set_major_locator( majLocator )
  ax.xaxis.set_major_formatter( majFormatter )

  ax.xaxis.set_minor_locator( minLocator )
  ax.xaxis.set_minor_formatter( minFormatter )

where majLocator and minLocator are matplotlib.ticker.Locator instance
and majFormatter and minFormatter are matplotlib.ticker.Formatter
instances.  Many date locators are provided: MinuteLocator,
HourLocator, DayLocator, WeekdayLocator, MonthLocator, YearLocator.
Most of these take an argument to, for example, select only hours or
years that are multiples of some number.  The DateFormatter class
takes a strftime format string to format the dates on the major and
minor ticker.  The minor ticker is off by default in all plots so you
will need to set the minor locator if you want to turn them on, and
the minor formatter if you want to label them.

Here is an example setting the major date formatter
strftime string, as in::

    formatter = DateFormatter('%Y')
    ax.xaxis.set_major_formatter(formatter)

See the examples/dates*.py in the matplotlib src distribution for
complete demos, and the matplotlib.ticker help for more information on
tick locators and formatters.

"""

from __future__ import division

import sys, os, math
from urllib import urlopen
import time


SEC_PER_MIN = 60
SEC_PER_HOUR = 3600
SEC_PER_DAY = SEC_PER_HOUR * 24
SEC_PER_WEEK = SEC_PER_DAY * 7


MONDAY, TUESDAY, WEDNESDAY, THURSDAY, FRIDAY, SATURDAY, SUNDAY = range(7)
WEEKDAYS = (MONDAY, TUESDAY, WEDNESDAY, THURSDAY, FRIDAY, SATURDAY, SUNDAY)


try: import datetime
except ImportError: pass
else:
    # timezone code is from the python library datetime module reference
    from datetime import tzinfo, timedelta, datetime

    ZERO = timedelta(0)
    HOUR = timedelta(hours=1)

    # A UTC class.

    class UTC(tzinfo):
        """UTC"""

        def utcoffset(self, dt):
            return ZERO

        def tzname(self, dt):
            return "UTC"

        def dst(self, dt):
            return ZERO

    utc = UTC()

    # A class building tzinfo objects for fixed-offset time zones.
    # Note that FixedOffset(0, "UTC") is a different way to build a
    # UTC tzinfo object.

    class FixedOffset(tzinfo):
        """Fixed offset in minutes east from UTC."""

        def __init__(self, offset, name):
            self.__offset = timedelta(minutes = offset)
            self.__name = name

        def utcoffset(self, dt):
            return self.__offset

        def tzname(self, dt):
            return self.__name

        def dst(self, dt):
            return ZERO

    # A class capturing the platform's idea of local time.

    import time as _time

    STDOFFSET = timedelta(seconds = -_time.timezone)
    if _time.daylight:
        DSTOFFSET = timedelta(seconds = -_time.altzone)
    else:
        DSTOFFSET = STDOFFSET

    DSTDIFF = DSTOFFSET - STDOFFSET

    class LocalTimezone(tzinfo):

        def utcoffset(self, dt):
            if self._isdst(dt):
                return DSTOFFSET
            else:
                return STDOFFSET

        def dst(self, dt):
            if self._isdst(dt):
                return DSTDIFF
            else:
                return ZERO

        def tzname(self, dt):
            return _time.tzname[self._isdst(dt)]

        def _isdst(self, dt):
            tt = (dt.year, dt.month, dt.day,
                  dt.hour, dt.minute, dt.second,
                  dt.weekday(), 0, -1)
            stamp = _time.mktime(tt)
            tt = _time.localtime(stamp)
            return tt.tm_isdst > 0

    Local = LocalTimezone()

    # A complete implementation of current DST rules for major US time zones.

    def first_sunday_on_or_after(dt):
        days_to_go = 6 - dt.weekday()
        if days_to_go:
            dt += timedelta(days_to_go)
        return dt

    # In the US, DST starts at 2am (standard time) on the first Sunday in April.
    DSTSTART = datetime(1, 4, 1, 2)
    # and ends at 2am (DST time; 1am standard time) on the last Sunday of Oct.
    # which is the first Sunday on or after Oct 25.
    DSTEND = datetime(1, 10, 25, 1)

    class USTimeZone(tzinfo):

        def __init__(self, hours, reprname, stdname, dstname):
            self.stdoffset = timedelta(hours=hours)
            self.reprname = reprname
            self.stdname = stdname
            self.dstname = dstname

        def __repr__(self):
            return self.reprname

        def tzname(self, dt):
            if self.dst(dt):
                return self.dstname
            else:
                return self.stdname

        def utcoffset(self, dt):
            return self.stdoffset + self.dst(dt)

        def dst(self, dt):
            if dt is None or dt.tzinfo is None:
                # An exception may be sensible here, in one or both cases.
                # It depends on how you want to treat them.  The default
                # fromutc() implementation (called by the default astimezone()
                # implementation) passes a datetime with dt.tzinfo is self.
                return ZERO
            assert dt.tzinfo is self

            # Find first Sunday in April & the last in October.
            start = first_sunday_on_or_after(DSTSTART.replace(year=dt.year))
            end = first_sunday_on_or_after(DSTEND.replace(year=dt.year))

            # Can't compare naive to aware objects, so strip the timezone from
            # dt first.
            if start <= dt.replace(tzinfo=None) < end:
                return HOUR
            else:
                return ZERO

    Eastern  = USTimeZone(-5, "Eastern",  "EST", "EDT")
    Central  = USTimeZone(-6, "Central",  "CST", "CDT")
    Mountain = USTimeZone(-7, "Mountain", "MST", "MDT")
    Pacific  = USTimeZone(-8, "Pacific",  "PST", "PDT")


        
class DateConverter:
    """
    An abstract class to allow users to specify dates however they
    want.  The class provides a standard set of calls which take the
    users date instance and convert to to the output indicated.  This
    allows you to store dates as mx datetimes, python datetimes,
    epochs etc.
    """

    def __call__(self, x):
        'convert userland datetime instance x to epoch'        
        return self.epoch(x)

    def epoch(self, x):
        'convert userland datetime instance x to epoch'        
        raise NotImplementedError('Derived must override')

    def from_epoch(self, e):
        'return an instance of your date from the epoch'
        raise NotImplementedError('Derived must override')

    def ymdhms_to_epoch(self, year, month, day, hour, minute, second):
        return time.mktime(
            ( year, month, day, hour, minute, second, 0, 0, -1))
    
    def day_of_week(self, x):
        'return day of week; monday is 0'        
        e = self.epoch(x)
        y,month,d,h,m,s,wd,jd,ds = time.localtime(e) 
        
        return wd

    def strftime(self, fmt, x):
        'format x using strftime format string fmt'
        e = self.epoch(x)
        return time.strftime(fmt, time.localtime(e))        

    def strptime(self, datestr, fmt):
        'return an instance of your date from a date string and fouermat'
        e = time.mktime( time.strptime(datestr, fmt))
        return  self.from_epoch(e)

    def add_month(self, x, direction=1):
        'return now plus one month as epoch'
        e = self.epoch(x)        
        y,month,d,h,m,s,wd,jd,ds = time.localtime(e)        
        if direction==-1 and month==1:
            y-=1
            month = 12
        elif direction==1 and month==12:
            y+=1
            month = 1
        else:
            month += direction
            

        return time.mktime((y,month,d,h,m,s,0,0,-1))

    def add_year(self, x, direction=1):
        'return now plus one year as epoch'
        e = self.epoch(x)        
        y,month,d,h,m,s,wd,jd,ds = time.localtime(e) 
                                   
        return time.mktime((y+direction,month,d,h,m,s,0,0,-1))
    
    def floor_minute(self, x):
        'return the floored minute'
        e = self.epoch(x)
        return (e // SEC_PER_MIN) * SEC_PER_MIN

    def ceil_minute(self, x):
        'return the ceiled minute'
        e = self.epoch(x)
        return int(math.ceil(e / SEC_PER_MIN) * SEC_PER_MIN)

    def floor_hour(self, x):
        'return the floored hour as epoch'
        e = self.epoch(x)
        return (e // SEC_PER_HOUR) * SEC_PER_HOUR

    def ceil_hour(self, x):
        'return the ceiled hour as epoch'
        e = self.epoch(x)
        return int(math.ceil(e / SEC_PER_HOUR) * SEC_PER_HOUR)


    def floor_day(self, x):
        'return the floored day'
        e = self.epoch(x)
        return (e // SEC_PER_DAY) * SEC_PER_DAY

    def ceil_day(self, x):
        'return the ceiled day'
        e = self.epoch(x)
        return int(math.ceil(e / SEC_PER_DAY) * SEC_PER_DAY)

    def floor_month(self, x):
        'return the floored month'
        e = self.epoch(x)
        y,month,d,h,m,s,wd,jd,ds = time.localtime(e)        
        if y<1970: raise RuntimeError('You are before the epoch!')
            
        return time.mktime((y,month,1,0,0,0,0,0,-1))

    def ceil_month(self, x):
        'return the ceiled month'
        e = self.epoch(x)        
        y,month,d,h,m,s,wd,jd,ds = time.localtime(e)

        if d==1 and h==0 and m==0 and s==0:
            return e
        if month==12:
            y+=1
            month = 1
        else:
            month +=1
        return time.mktime((y,month,1,0,0,0,0,0,-1))


    def floor_year(self, x):
        'return the floored year'
        e = self.epoch(x)        
        y,month,d,h,m,s,wd,jd,ds = time.localtime(e)        
        return time.mktime((y,1,1,0,0,0,0,0,-1))

    def ceil_year(self, x):
        'return the ceiled year'
        e = self.epoch(x)        
        y,month,d,h,m,s,wd,jd,ds = time.localtime(e)        
        if month==1 and d==1 and h==0 and m==0 and s==0:
            return time.mktime((y,1,1,0,0,0,0,0,-1))
        else:
            return time.mktime((y+1,1,1,0,0,0,0,0,-1))




    def ymd(self, x):
        'return the year month day. month and day are indexed from 1'
        e = self.epoch(x)
        y,month,d,h,m,s,wd,jd,ds = time.localtime(e)        
        return y, month, day

    def hms(self, x):
        'return the hour, minute, second'
        e = self.epoch(x)
        y,month,d,h,m,s,wd,jd,ds = time.localtime(e)        
        return h,m,s

        
class EpochConverter(DateConverter):
    """
    Represent time in the epoch
    """


    def epoch(self, x):
        'convert userland datetime instance x to epoch'        
        return x

    def from_epoch(self, e):
        'return an instance of your date from the epoch'
        return e

    def day_of_week(self, x):
        'return day of week; monday is 0'        
        y,month,d,h,m,s,wd,jd,ds = time.localtime(x)        
        return wd

    def strftime(self, fmt, x):
        'format x using strftime format string fmt'
        return time.strftime(fmt, time.localtime(x))    

    def strptime(self, datestr, fmt):
        'return an instance of your date from a date string and format'
        return  time.mktime(time.strptime(datestr, fmt))
    
    def floor_day(self, x):
        'return the floored day'
        return (x // SEC_PER_DAY) * SEC_PER_DAY

    def ceil_day(self, x):
        'return the ceiled day'
        return int(math.ceil(x / SEC_PER_DAY) * SEC_PER_DAY)

    def floor_hour(self, x):
        'return the floored hour'
        return (x // SEC_PER_HOUR) * SEC_PER_HOUR

    def ceil_hour(self, x):
        'return the ceiled hour'
        return int(math.ceil(x / SEC_PER_HOUR) * SEC_PER_HOUR)

    def floor_minute(self, x):
        'return the floored minute'
        return (x // SEC_PER_MIN) * SEC_PER_MIN

    def ceil_minute(self, x):
        'return the ceiled minute'
        return int(math.ceil(x / SEC_PER_MIN) * SEC_PER_MIN)


    def ymd(self, x):
        'return the year month day. month and day are indexed from 1'
        y,month,d,h,m,s,wd,jd,ds = time.localtime(x)        
        return y, month, d

    def hms(self, x):
        'return the hour, minute, second'
        y,month,d,h,m,s,wd,jd,ds = time.localtime(x)
        return h,m,s

try: from datetime import datetime
except ImportError: pass
else:

    class PyDatetimeConverter(DateConverter):
        """
        Convert python2.3 date or datetime instances
        """


        def epoch(self, x):
            'convert userland datetime instance x to epoch'                    
            return time.mktime(x.timetuple())

        def from_epoch(self, e):
            'return an instance of your date from the epoch'
            y,month,d,h,m,s,wd,jd,ds = time.localtime(e)
            return datetime(y,month,d,h,m,s)            

        def day_of_week(self, x):
            'return day of week; monday is 0'        
            return x.weekday()

        def strftime(self, fmt, x):
            'format x using strftime format string fmt'
            x.strftime(fmt)

        def strptime(self, datestr, fmt):
            'return an instance of your date from a date string and format'
            y,month,d,h,m,s,wd,jd,ds = time.strptime(datestr, fmt)
            #dt = datetime(y, month, d, h, m, s, tzinfo=self.tz)
            dt = datetime(y, month, d, h, m, s)            
            return dt

        def ymd(self, x):
            'return the year month day. month and day are indexed from 1'
            return x.year, x.month, x.day

        def hms(self, x):
            'return the hour, minute, second'
            try: return x.hour, x.minute, x.second
            except AttributeError: return 0,0,0


    class intdate(int, DateConverter):
         '''Subclasses int for use as dates.'''
         def __init__(self, ordinal):
             int.__init__(self, ordinal)
             self.__date = datetime.date.fromtimestamp(ordinal)

         day = property(fget=lambda self:self.__date.day)
         month = property(fget=lambda self:self.__date.month)
         year = property(fget=lambda self:self.__date.year)

         def isoformat(self): return self.__date.isoformat()

         def timetuple(self): return self.__date.timetuple()

         def date(self): return self.__date

         def epoch(self, x):
             'convert userland datetime instance x to epoch'
             return x

         def from_epoch(self, e):
             'return an instance of your date from the epoch'
             return intdate(e)


try:
    import mx.DateTime
except ImportError:
    pass
else:
    class MxDatetimeConverter(DateConverter):
        """
        Convert egenix mx.Datetime instances
        """

        def epoch(self, x):
            'convert userland datetime instance x to epoch'
            return x.ticks()

        def from_epoch(self, e):
            'return an instance of your date from the epoch'
            return  mx.DateTime.localtime(e)
