<?php
/**
* Base class for all output converters.
* @package Converters
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: Converter.inc,v 1.103.2.3 2002/11/07 21:56:10 CelloG Exp $
*/
/**
* Base class for all output converters.
* A Converter takes output from the {@link IntermediateParser} and converts it to output.  A converter for the standard phpDocumentor
* template, {@link HTMLdefaultConverter}, is provided with this release.  A <b>pre-alpha</b> PDF converter (pre-alpha means it is
* pretty lame but will improve), {@link PDFdefaultConverter}, is also supplied with this release.  Future releases will have support for other
* formats and templates, including DocBook, XML, and possibly other HTML templates.  The converter takes output directly from
* {@link IntermediateParser} and using {@link walk()} it "walks" over an array of phpDocumentor elements, as represented by descendants
* of {@link parserElement}
*
* a converter must define the abstract function Convert (an example is {@link HTMLdefaultConverter::Convert()}), a function that
* takes a passed object and uses it to generate structures for an output template, or directly generates output.  Since all
* elements have their DocBlocks linked directly, this allows for more logical parsing than earlier versions of phpDocumentor.
*
* A Converter is passed several data structures in its constructor, all of which are optional in output and may be handled in any way
* the converter would like:
* <ul>
*	<li>array of all packages found in documentation (see {@link Converter::$all_packages})</li>
*	<li>array of package parents, i.e. class X is in package parent, and class Y extends X is in package child</li>
*	<li>a {@link Classes} object that contains all class information adjusted for proper inheritance</li>
*	<li><b>NEW v1.1</b> {@link ProceduralPages} class, containing all procedural elements</li>
*	<li>array of packages to document (-po command-line) (see {@link IntermediateParser::$packageoutput})</li>
*	<li>boolean option, set to true to parse all elements marked @access private (see {@link IntermediateParser::$parsePrivate})</li>
*	<li>boolean option, set to true to stop informative output while parsing (good for cron jobs) (see {@link IntermediateParser::$quietMode})</li>
*	<li>string target directory, the directory all output should be placed in</li>
*	<li>string template directory, path to all template files relative to phpDocumentor location</li>
*	<li>string title of generated documentation</li>
* </ul>
* Creating a converter is relatively simple.  You need to override one data structure:
* <ul>
*	<li>{@link $leftindex}</li>
* </ul>
* To tell the Converter which of the individual element indexes your Converter will use.  You will also need to override
* a few methods for the Converter to work.  The most important relate to linking and output.
* 
* You must override these methods:
* <ul>
*	<li>{@link Convert()} - take any descendant of parserElement or a parserPackagePage and convert it into output</li>
*	<li>{@link returnSee()} - takes a abstract link and returns a string that links to an element's documentation</li>
*	<li>{@link returnLink()} - takes a URL and text to display and returns an internet-enabled link</li>
*	<li>{@link Output()} - generate output (optionally - see {@link HTMLdefaultConverter} paradigm)</li>
* </ul>
* You should override these methods:
* <ul>
*	<li>{@link ConvertErrorLog()} - formats errors and warnings from {@link $phpDocumentor_errors}. see {@link HTMLdefaultConverter::ConvertErrorLog()}</li>
*	<li>{@link formatIndex()} - format the {@link $elements} array into an index see {@link HTMLdefaultConverter::generateElementIndex()}</li>
*	<li>{@link formatPkgIndex()} - format the {@link $pkg_elements} array into an index see {@link HTMLdefaultConverter::generatePkgElementIndex()}</li>
*	<li>{@link formatLeftIndex()} - format the {@link $elements} array into an index see {@link HTMLdefaultConverter::formatLeftIndex()}</li>
*	<li>{@link endPage()} - do any post-processing of procedural page elements, possibly output documentation for the page</li>
*	<li>{@link endClass()} - do any post-processing of class elements, possibly output documentation for the class</li>
*	<li>{@link getFunctionLink()} - for all of the functions below, see {@link HTMLdefaultConverter::getFunctionLink()} for an example </li>
*	<li>{@link getClassLink()}</li>
*	<li>{@link getDefineLink()}</li>
*	<li>{@link getGlobalLink()}</li>
*	<li>{@link getMethodLink()}</li>
*	<li>{@link getVarLink()}</li>
*	<li><b>Do not override</b>, but see {@link getSortedClassTreeFromClass()} for information on generating class trees by package</li>
* </ul>
*
* You may override these methods, but they are not absolutely required:
* <ul>
*	<li>{@link checkState()} - used by the {@link parserStringWithInlineTags::Convert()} cache to determine whether a cache hit or miss occurs</li>
*	<li>{@link getState()} - used by the {@link parserStringWithInlineTags::Convert()} cache to save state for the next Convert() call</li>
* </ul>
* @package Converters
* @abstract
* @see parserDocBlock, parserInclude, parserPage, parserClass, parserDefine, parserFunction, parserMethod, parserVar, parserPackagePage
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: Converter.inc,v 1.103.2.3 2002/11/07 21:56:10 CelloG Exp $
*/
class Converter
{
	/**
	* output format of this converter
	* @var string
	*/
	var $outputformat = 'Generic';
	/**
	* package name
	* @var string
	*/
	var $package = 'default';
	/**
	* subpackage name
	* @var string
	*/
	var $subpackage = '';
	/**
	* set to a classname if currently parsing a class, false if not
	* @var mixed
	*/
	var $class = false;
	/**
	* the workhorse of linking.
	* This array is an array of link objects of format:
	* [package][subpackage][eltype][elname] = descendant of {@link abstractLink}
	* eltype can be page|function|define|class|method|var
	* if eltype is method or var, the array format is:
	* [package][subpackage][eltype][class][elname]
	* @var array
	* @see functionLink, pageLink, classLink, defineLink, methodLink, varLink, globalLink
	*/
	var $links = array();
	
	/**
	* the workhorse of linking, with allowance for support of multiple elements in different files.
	* This array is an array of link objects of format:
	* [package][subpackage][eltype][file][elname] = descendant of {@link abstractLink}
	* eltype can be function|define|class|method|var
	* if eltype is method or var, the array format is:
	* [package][subpackage][eltype][file][class][elname]
	* @var array
	* @see functionLink, pageLink, classLink, defineLink, methodLink, varLink, globalLink
	*/
	var $linkswithfile = array();
	/**
	* set to value of -po commandline
	* @var mixed
	*/
	var $package_output;
	/**
	* alphabetical index of all elements
	* @var array Format: array(first letter of element name => array({@link parserElement} or {@link parserPage},...))
	* @see formatIndex(), HTMLdefaultConverter::formatIndex()
	*/
	var $elements = array();
	/**
	* alphabetized index of procedural pages by package
	* @see $leftindex
	* @var array Format: array(package => array(subpackage => array({@link pageLink} 1,{@link pageLink} 2,...)
	*/
	var $page_elements = array();
	/**
	* alphabetized index of defines by package
	* @see $leftindex
	* @var array Format: array(package => array(subpackage => array({@link defineLink} 1,{@link defineLink} 2,...)
	*/
	var $define_elements = array();
	/**
	* alphabetized index of classes by package
	* @see $leftindex
	* @var array Format: array(package => array(subpackage => array({@link defineLink} 1,{@link defineLink} 2,...)
	*/
	var $class_elements = array();
	/**
	* alphabetized index of global variables by package
	* @see $leftindex
	* @var array Format: array(package => array(subpackage => array({@link globalLink} 1,{@link globalLink} 2,...)
	*/
	var $global_elements = array();
	/**
	* alphabetized index of functions by package
	* @see $leftindex
	* @var array Format: array(package => array(subpackage => array({@link functionLink} 1,{@link functionLink} 2,...)
	*/
	var $function_elements = array();
	/**
	* alphabetical index of all elements
	* @var array Format: array(first letter of element name => array({@link parserElement} or {@link parserPage},...))
	* @see formatPkgIndex(), HTMLdefaultConverter::formatPkgIndex()
	*/
	var $pkg_elements = array();
	
	/**
	* alphabetical index of all elements on a page by package/subpackage
	*
	* The page itself has a link under ###main
	* @var array Format: array(package => array(subpackage => array(path => array({@link abstractLink} descendant 1, ...)))
	* @see formatLeftIndex()
	*/
	var $page_contents = array();
	
	/**
	* This determines whether the {@link $page_contents} array should be sorted by element type as well as alphabetically by name
	* @see sortPageContentsByElementType()
	* @var boolean
	*/
	var $sort_page_contents_by_type = false;
	/**
	* alphabetical index of all methods and vars in a class by package/subpackage
	*
	* The class itself has a link under ###main
	* @var array Format: array(package => array(subpackage => array(path => array(class => array({@link abstractLink} descendant 1, ...))))
	* @see formatLeftIndex()
	*/
	var $class_contents = array();
	/**
	* controls processing of elements marked private with @access private
	*
	* defaults to false.  Set with command-line --parseprivate or -pp
	* @var bool
	*/
	var $parseprivate;
	/**
	* controls display of progress information while parsing.
	*
	* defaults to false.  Set to true for cron jobs or other situations where no visual output is necessary
	* @var bool
	*/
	var $quietmode;
	
	/**
	* directory that output is sent to. -t command-line sets this.
	*/
	var $targetDir = '';
	
	/**
    * Directory that the template is in, relative to phpDocumentor root directory
    * @var string
	*/
	var $templateDir = '';
	
	/**
    * Name of the template
	* @var string
	*/
	var $templateName = '';
	
	/**
	* name of the current file being converted
	*/
	var $curfile;
	
	/**
	* @var Classes
	*/
	var $classes;
	
	/**
	* @var array
	* @see IntermediateParser::$package_parents
	*/
	var $package_parents;
	
	/**
	* @var array
	* @see IntermediateParser::$all_packages
	*/
	var $all_packages;
	
	/**
	* Controls which of the one-element-only indexes are generated.
	*
	* Generation of these indexes for large packages is time-consuming.  This is an optimization feature.  An
	* example of how to use this is in {@link HTMLdefaultConverter::$leftindex}, and in {@link HTMLdefaultConverter::formatLeftIndex()}.
	* These indexes are intended for use as navigational aids through documentation, but can be used for anything by converters.
	* @see $class_elements, $page_elements, $function_elements, $define_elements, $global_elements
	* @see formatLeftIndex()
	* @var array
	*/
	var $leftindex = array('classes' => true, 'pages' => true, 'functions' => true, 'defines' => true, 'globals' => true);
	
	/** @access private */
	var $killclass = false;
	/**
	* @var string
	* @see IntermediateParser::$title
	*/
	var $title = 'Generated Documentation';
	
	/**
	* Initialize Converter data structures
	* @param array {@link $all_packages} value
	* @param array {@link $package_parents} value
	* @param Classes {@link $classes} value
	* @param ProceduralPages {@link $proceduralpages} value
	* @param array {@link $package_output} value
	* @param boolean {@link $parseprivate} value
	* @param boolean {@link $quietmode} value
	* @param string {@link $targetDir} value
	* @param string {@link $templateDir} value
	* @param string (@link $title} value
	*/
	function Converter(&$allp, &$packp, &$classes, &$procpages, $po, $pp, $qm, $targetDir, $template, $title)
	{
		$this->all_packages = $allp;
		$this->package_parents = $packp;
		$this->package = $GLOBALS['phpDocumentor_DefaultPackageName'];
		$this->proceduralpages = &$procpages;
		$this->package_output = $po;
		if (is_array($po))
		{
			$a = $po[0];
			$this->all_packages = array_flip($po);
			$this->all_packages[$a] = 1;
		}
		$this->parseprivate = $pp;
		$this->quietmode = $qm;
		$this->classes = &$classes;
		$this->roots = $classes->getRoots();
		$this->title = $title;
		$this->setTemplateDir($template);
		$this->setTargetdir($targetDir);
	}
	
	/**
	* Called by {@link walk()} while converting, when the last class element has been parsed.
	*
	* A Converter can use this method in any way it pleases.  HTMLdefaultConverter uses it to complete
	* the template for the class and to output its documentation
	* @see HTMLdefaultConverter::endClass()
	* @abstract
	*/
	function endClass()
	{
	}
	
	/**
	* Called by {@link walk()} while converting, when the last procedural page element has been parsed.
	*
	* A Converter can use this method in any way it pleases.  HTMLdefaultConverter uses it to complete
	* the template for the procedural page and to output its documentation
	* @see HTMLdefaultConverter::endClass()
	* @abstract
	*/
	function endPage()
	{
	}
	
	/**
	* Called by {@link walk()} while converting.
	*
	* This method is intended to be the place that {@link $pkg_elements} is formatted for output.
	* @see HTMLdefaultConverter::formatPkgIndex()
	* @abstract
	*/
	function formatPkgIndex()
	{
	}
	
	/**
	* Called by {@link walk()} while converting.
	*
	* This method is intended to be the place that {@link $elements} is formatted for output.
	* @see HTMLdefaultConverter::formatIndex()
	* @abstract
	*/
	function formatIndex()
	{
	}
	
	/**
	* Called by {@link walk()} while converting.
	*
	* This method is intended to be the place that any of {@link $class_elements, $function_elements, $page_elements},
	* {@link $define_elements}, and {@link $global_elements} is formatted for output, depending on the value of
	* {@link $leftindex}
	* @see HTMLdefaultConverter::formatLeftIndex()
	* @abstract
	*/
	function formatLeftIndex()
	{
	}
	
	/**
	* Called by {@link parserSourceInlineTag::Convert()} to allow converters to format the source code the way they'd like.
	*
	* default returns it unchanged (html with xhtml tags)
	* @param string output from highlight_string() - use this function to reformat the returned data for
	* Converter-specific output
	*/
	function unmangle($sourcecode)
	{
		return $sourcecode;
	}
	
	/**
	* called by {@link IntermediateParser::Convert()} to traverse the array of pages and their elements, converting them to the output format
	*
	* The walk() method should be flexible enough such that it never needs modification.  walk() sets up all of the indexes, and sorts
	* everything in logical alphabetical order.  It then passes each element individually to {@link Convert()}, a method that <b>must</b>
	* be overridden in a child Converter.
	*
	* walk() first creates all of the indexes {@link $elements, $pkg_elements} and the left indexes specified by {@link $leftindexes},
	* and then sorts them by calling {@link sortIndexes()}.  After this, it passes all package-level docs to Convert().  Then, it
	* calls the index sorting functions {@link formatPkgIndex(), formatIndex()} and {@link formatLeftIndex()}.
	*
	* Finally, it converts each procedural page in alphabetical order.  This stage passes elements from the physical file to Convert()
	* in alphabetical order.  First, procedural page elements {@link parserDefine, parserInclude, parserGlobal}, and {@link parserFunction}
	* are passed to Convert().  Then, class elements are passed in this order: {@link parserClass}, then all of the {@link parserVar}s
	* in the class and all of the {@link parserMethod}s in the class.  Classes are in alphabetical order, and both vars and methods are
	* in alphabetical order.
	*
	* Finally, {@link ConvertErrorLog()} is called and the data walk is complete.
	* @param array Format: array(fullpath => {@link parserData} structure with full {@link parserData::$elements}
	*										 and {@link parserData::$class_elements}.
	* @param array Format: array({@link parserPackagePage} 1, {@link parserPackagePage} 2,...)
	*/
	function walk(&$pages,&$package_pages)
	{
		if (empty($pages))
		{
			die("<b>ERROR</b>: nothing parsed");
		}
		if (empty($this->elements))
		{
			$this->elements = array();
			$this->pkg_elements = array();
			$this->links = array();
			phpdoc_out('Building indexes...');
			foreach($pages as $j => $flub)
			{
				if (PHPDOC_EXCEPTIONS)
				{
					addException(!is_object($flub),PDEXCEPTION_NOT_A_CLASS,__FILE__,__LINE__,"pages[j]",$flub,"j",$j);
					addException(!is_object($pages[$j]->parent),PDEXCEPTION_NOT_A_CLASS,__FILE__,__LINE__,"pages[j]->parent",$pages[$j]->parent,"pages[j]",$pages[$j],"pages",$pages);
				}
				$this->package = $pages[$j]->parent->package;
				$this->subpackage = $pages[$j]->parent->subpackage;
				$this->class = false;
				$this->curfile = $pages[$j]->parent->getFile();
				$this->curname = $pages[$j]->parent->getName();
				$this->curpath = $pages[$j]->parent->getPath();
				$use = true;
				if ($this->package_output)
				{
					if (in_array($this->package,$this->package_output))
					{
						$this->addElement($pages[$j]->parent);
					} else
					{
						list(,$pages[$j]->parent->package) = each($this->package_output);
						reset($this->package_output);
						$pages[$j]->parent->subpackage = '';
						$this->addElement($pages[$j]->parent);
					}
				} else
				{
					$this->addElement($pages[$j]->parent);
				}
				if (PHPDOC_EXCEPTIONS)
				{
					addException(!is_array($pages[$j]->elements),PDEXCEPTION_NOT_AN_ARRAY,__FILE__,__LINE__,"pages[j]->elements",$pages[$j]->elements,"pages[j]",$pages[$j]);
				}
				if ($use)
				for($i=0; $i<count($pages[$j]->elements); $i++)
				{
					if (PHPDOC_EXCEPTIONS)
					{
						addException(!is_object($pages[$j]->elements[$i]),PDEXCEPTION_NOT_A_CLASS,__FILE__,__LINE__,"pages[j]->elements[i]",$pages[$j]->elements[$i],"i",$i,"pages[j]",$pages[$j]);
					}
					if (PHPDOC_EXCEPTIONS)
					{
						$test = get_class($pages[$j]->elements[$i]);
						addException($test != 'parserinclude' && $test != 'parserdefine' && $test != 'parserglobal' && $test != 'parserfunction',PDEXCEPTION_NOT_A_PROCPAGEELEMENT,__FILE__,__LINE__,"pages[j]->elements[i]",$pages[$j]->elements[$i],"i",$i,"pages[j]",$pages[$j]);
					}
					$pages[$j]->elements[$i]->docblock->package = $this->package;
					$pages[$j]->elements[$i]->docblock->subpackage = $this->subpackage;
					$this->proceduralpages->replaceElement($pages[$j]->elements[$i]);
					$this->addElement($pages[$j]->elements[$i]);
				}
				if (PHPDOC_EXCEPTIONS)
				{
					addException(!is_array($pages[$j]->classelements),PDEXCEPTION_NOT_AN_ARRAY,__FILE__,__LINE__,"pages[j]->classelements",$pages[$j]->classelements,"pages[j]",$pages[$j]);
				}
				for($i=0; $i<count($pages[$j]->classelements); $i++)
				{
					if (PHPDOC_EXCEPTIONS)
					{
						$test = get_class($pages[$j]->classelements[$i]);
						addException(!is_object($pages[$j]->classelements[$i]),PDEXCEPTION_NOT_A_CLASS,__FILE__,__LINE__,"pages[j]->classelements[i]",$pages[$j]->classelements[$i],"i",$i,"pages[j]",$pages[$j]);
					}
					if (PHPDOC_EXCEPTIONS)
					{
						$test = get_class($pages[$j]->classelements[$i]);
						addException($test != 'parserclass' && $test != 'parsermethod' && $test != 'parservar',PDEXCEPTION_NOT_A_CLASSELEMENT,__FILE__,__LINE__,"pages[j]->classelements[i]",$pages[$j]->classelements[$i],"i",$i,"pages[j]",$pages[$j]);
					}
					if ($this->class)
					{
						if ($pages[$j]->classelements[$i]->type == 'class')
						{
							if ($this->checkKillClass($pages[$j]->classelements[$i]->getName(),$pages[$j]->classelements[$i]->getPath())) continue;
							$this->package = $pages[$j]->classelements[$i]->docblock->package;
							if ($this->package_output) if (!in_array($this->package,$this->package_output)) continue;
							$this->subpackage = $pages[$j]->classelements[$i]->docblock->subpackage;
							$this->class = $pages[$j]->classelements[$i]->name;
						} else
						{
							if ($this->killclass) continue;
							// force all contained elements to have parent package/subpackage
							$pages[$j]->classelements[$i]->docblock->package = $this->package;
							$pages[$j]->classelements[$i]->docblock->subpackage = $this->subpackage;
						}
					}
					if ($pages[$j]->classelements[$i]->type == 'class')
					{
						if ($this->checkKillClass($pages[$j]->classelements[$i]->getName(),$pages[$j]->classelements[$i]->getPath())) continue;
						$this->package = $pages[$j]->classelements[$i]->docblock->package;
						if ($this->package_output) if (!in_array($this->package,$this->package_output)) continue;
						$this->subpackage = $pages[$j]->classelements[$i]->docblock->subpackage;
						$this->class = $pages[$j]->classelements[$i]->name;
					}
					if (!$this->killclass) $this->addElement($pages[$j]->classelements[$i]);
				}
			}
			phpdoc_out("done\n");
		}
		$this->sortIndexes();
		if ($this->sort_page_contents_by_type) $this->sortPageContentsByElementType($pages);
		foreach($package_pages as $i => $perp)
		{
			if ($this->package_output)
			{
				if (!in_array($package_pages[$i]->package,$this->package_output)) continue;
			}
			phpdoc_out('Converting package page for package '.$package_pages[$i]->package.'... ');
			flush();
			$this->package = $package_pages[$i]->package;
			$this->subpackage = '';
			$this->class = false;
			$this->Convert($package_pages[$i]);
			phpdoc_out("done\n");
		}
		phpdoc_out("Formatting Package Indexes...");
		flush();
		$this->formatPkgIndex();
		phpdoc_out("done\n");
		flush();
		phpdoc_out("Formatting Index...");
		flush();
		$this->formatIndex();
		phpdoc_out("done\n\n");
		flush();
		phpdoc_out("Formatting Left Quick Index...");
		flush();
		$this->formatLeftIndex();
		phpdoc_out("done\n\n");
		flush();
		foreach($pages as $j => $flub)
		{
			phpdoc_out('Converting '.$pages[$j]->parent->getPath());
			flush();
			$this->package = $pages[$j]->parent->package;
			$this->subpackage = $pages[$j]->parent->subpackage;
			$this->class = false;
			$this->curfile = $pages[$j]->parent->getFile();
			$this->curname = $pages[$j]->parent->getName();
			$this->curpath = $pages[$j]->parent->getPath();
			$use = true;
			if ($this->package_output)
			{
				if (in_array($this->package,$this->package_output))
				{
					$this->Convert($pages[$j]);
				} else
				{
					$use = false;
				}
			} else
			{
				$this->Convert($pages[$j]);
			}
			phpdoc_out(" Procedural Page Elements...");
			flush();
			if ($use)
			for($i=0; $i<count($pages[$j]->elements); $i++)
			{
				$a = $pages[$j]->elements[$i]->docblock->getKeyword('access');
				if ($a) $a = $a->getString();
				if (!$this->parseprivate && ($a == 'private'))
					continue;
//				phpdoc_out("    ".$pages[$j]->elements[$i]->name."\n");
				$pages[$j]->elements[$i]->docblock->package = $this->package;
				$pages[$j]->elements[$i]->docblock->subpackage = $this->subpackage;
				$this->Convert($pages[$j]->elements[$i]);
			}
			phpdoc_out(" Classes...");
			flush();
			for($i=0; $i<count($pages[$j]->classelements); $i++)
			{
				if ($this->class)
				{
					if ($pages[$j]->classelements[$i]->type == 'class')
					{
						if (!$this->killclass) $this->endClass();
						$this->killclass = false;
						if ($this->checkKillClass($pages[$j]->classelements[$i]->getName(),$pages[$j]->classelements[$i]->getPath())) continue;
						$this->package = $pages[$j]->classelements[$i]->docblock->package;
						if ($this->package_output) if (!in_array($this->package,$this->package_output)) continue;
						$this->subpackage = $pages[$j]->classelements[$i]->docblock->subpackage;
						$this->class = $pages[$j]->classelements[$i]->name;
					} else
					{
						$a = $pages[$j]->classelements[$i]->docblock->getKeyword('access');
						if ($a) $a = $a->getString();
						if (!$this->parseprivate && ($a == 'private'))
							continue;
						if ($this->killclass) continue;
						// force all contained elements to have parent package/subpackage
						$pages[$j]->classelements[$i]->docblock->package = $this->package;
						$pages[$j]->classelements[$i]->docblock->subpackage = $this->subpackage;
					}
				}
				if ($pages[$j]->classelements[$i]->type == 'class')
				{
					$this->killclass = false;
					if ($this->checkKillClass($pages[$j]->classelements[$i]->getName(),$pages[$j]->classelements[$i]->getPath())) continue;
					$this->package = $pages[$j]->classelements[$i]->docblock->package;
					if ($this->package_output) if (!in_array($this->package,$this->package_output)) continue;
					$this->subpackage = $pages[$j]->classelements[$i]->docblock->subpackage;
					$this->class = $pages[$j]->classelements[$i]->name;
				}
				if ($this->killclass) continue;
//				phpdoc_out("    ".$pages[$j]->classelements[$i]->name."\n");
				$this->Convert($pages[$j]->classelements[$i]);
			}
			if (count($pages[$j]->classelements) && !$this->killclass) $this->endClass();
			phpdoc_out(" done\n");
			flush();
			$this->endPage();
		}
		phpdoc_out("\nConverting Error Log...");
		flush();
		$this->ConvertErrorLog();
		phpdoc_out("done\n");
		flush();
	}
	
	/**
	* @abstract
	* @see HTMLdefaultConverter::ConvertErrorLog()
	*/
	function ConvertErrorLog()
	{
	}
	
	/**
	* Sorts all indexes - do not override or modify this function
	*/
	function sortIndexes()
	{
		phpdoc_out("\nSorting Indexes...");
		flush();
		uksort($this->elements,'strnatcasecmp');
		if ($this->leftindex['classes'])
		{
			foreach($this->class_elements as $package => $o1)
			{
				foreach($o1 as $subpackage => $links)
				{
					usort($this->class_elements[$package][$subpackage],array($this,'compareLink'));
				}
			}
		}
		if ($this->leftindex['pages'])
		{
			foreach($this->page_elements as $package => $o1)
			{
				uksort($this->page_elements[$package],'strnatcasecmp');
				foreach($o1 as $subpackage => $links)
				{
					usort($this->page_elements[$package][$subpackage],array($this,'compareLink'));
				}
			}
		}
		if ($this->leftindex['defines'])
		{
			foreach($this->define_elements as $package => $o1)
			{
				uksort($this->define_elements[$package],'strnatcasecmp');
				foreach($o1 as $subpackage => $links)
				{
					usort($this->define_elements[$package][$subpackage],array($this,'compareLink'));
				}
			}
		}
		if ($this->leftindex['globals'])
		{
			foreach($this->global_elements as $package => $o1)
			{
				uksort($this->global_elements[$package],'strnatcasecmp');
				foreach($o1 as $subpackage => $links)
				{
					usort($this->global_elements[$package][$subpackage],array($this,'compareLink'));
				}
			}
		}
		if ($this->leftindex['functions'])
		{
			foreach($this->function_elements as $package => $o1)
			{
				uksort($this->function_elements[$package],'strnatcasecmp');
				foreach($o1 as $subpackage => $links)
				{
					usort($this->function_elements[$package][$subpackage],array($this,'compareLink'));
				}
			}
		}
		foreach($this->elements as $letter => $nothuing)
		{
			uasort($this->elements[$letter],array($this,"elementCmp"));
		}
		foreach($this->pkg_elements as $package => $els)
		{
			uksort($this->pkg_elements[$package],'strnatcasecmp');
			foreach($this->pkg_elements[$package] as $subpackage => $els)
			{
				if (empty($els)) continue;
				uksort($this->pkg_elements[$package][$subpackage],'strnatcasecmp');
				foreach($els as $letter => $yuh)
				{
					usort($this->pkg_elements[$package][$subpackage][$letter],array($this,"elementCmp"));
				}
			}
		}
		phpdoc_out("done\n");
		flush();
	}
	
	/**
	* sorts {@link $page_contents} by element type as well as alphabetically
	* @see $sort_page_contents_by_element_type
	*/
	function sortPageContentsByElementType(&$pages)
	{
		foreach($this->page_contents as $package => $els)
		{
			foreach($this->page_contents[$package] as $subpackage => $els)
			{
				if (empty($els)) continue;
				foreach($this->page_contents[$package][$subpackage] as $path => $stuff)
				{
					if (!count($pages[$path]->elements)) continue;
					usort($pages[$path]->elements,array($this,'eltypecmp'));
					usort($this->page_contents[$package][$subpackage][$path],array($this,'eltypecmp'));
					if (isset($this->page_contents[$package][$subpackage][$path][0]))
					$this->page_contents[$package][$subpackage][$path]['###main'] = $this->page_contents[$package][$subpackage][$path][0];
					unset($this->page_contents[$package][$subpackage][$path][0]);
				}
			}
		}
	}
	
	/**
	* @access private
	* @see Converter::sortIndexes()
	*/
	function compareLink($a, $b)
	{
 		return strnatcasecmp($a->name,$b->name);
	}
	
	/**
	* @access private
	* @see Converter::sortPageContentsByElementType()
	*/
	function eltypecmp($a, $b)
	{
		if ($a->type == 'page') return -1;
		if ($b->type == 'page') return 1;
 		return strnatcasecmp($a->type.$a->name,$b->type.$a->name);
	}
	
	/**
	* does a nat case sort on the specified second level value of the array
	*
	* @param	mixed	$a
	* @param	mixed	$b
	* @return	int
	* @access private
	*/
	function elementCmp ($a, $b)
	{
		if (PHPDOC_EXCEPTIONS)
		{
			addException(!is_object($a),PDEXCEPTION_NOT_A_CLASS,__FILE__,__LINE__,"a",$a);
			addException(!is_object($b),PDEXCEPTION_NOT_A_CLASS,__FILE__,__LINE__,"a",$b);
		}
		return strnatcasecmp($a->getName(), $b->getName());
	}

	/**
	* Used to stop conversion of @ignored or private @access classes
	* @access private
	*/
	function checkKillClass($class, $path)
	{
		$this->killclass = false;
		if (isset($this->classes->killclass[$class]) && isset($this->classes->killclass[$class][$path])) $this->killclass = true;
		if ($this->package_output)
		{
			$a = $this->classes->getClass($class, $path);
			if (!in_array($a->docblock->package,$this->package_output)) $this->killclass = true;
		}
		if (DEBUG && $this->killclass) debug("$class $path killed");
		return $this->killclass;
	}
	
	/**
	* Adds all elements to the {@link $elements, $pkg_elements, $links, $linkswithfile} and left indexes - Do not modify or override
	* @access private
	* @staticvar string path of current page, used for {@link $page_contents} setup
	*/
	function addElement(&$element)
	{
		if ($this->package_output)
		{
			if (!in_array($this->package, $this->package_output)) return;
		}
		if (isset($element->docblock))
		{
			$a = $element->docblock->getKeyword('access');
			if ($a) $a = $a->getString();
			if (!$this->parseprivate && ($a == 'private'))
				return;
		}
		$i = 0;
		static $curpath = '';
		switch($element->type)
		{
			case 'page' :
				$link = $this->addLink($element);
				$curpath = $element->getPath();
				if ($this->leftindex['pages'])
				$this->page_elements[$element->package][$element->subpackage][] = $link;
				$this->page_contents[$element->package][$element->subpackage][$curpath]['###main'] = $link;
			break;
			case 'class' :
				$link = $this->addLink($element);
				if ($this->leftindex['classes'])
				$this->class_elements[$element->docblock->package][$element->docblock->subpackage][] = $link;
				$this->class_contents[$element->docblock->package][$element->docblock->subpackage][$this->class]['###main'] = $link;
			break;
			case 'include' :
				$link = $this->addLink($element);
			break;
			case 'define' :
				$link = $this->addLink($element);
				if ($this->leftindex['defines'])
				$this->define_elements[$element->docblock->package][$element->docblock->subpackage][] = $link;
				$this->page_contents[$element->docblock->package][$element->docblock->subpackage][$curpath][] = $link;
			break;
			case 'global' :
				$link = $this->addLink($element);
				$i++;
				if ($this->leftindex['globals'])
				$this->global_elements[$element->docblock->package][$element->docblock->subpackage][] = $link;
				$this->page_contents[$element->docblock->package][$element->docblock->subpackage][$curpath][] = $link;
			break;
			case 'var' :
				$link = $this->addLink($element);
				$i++;
				$this->class_contents[$element->docblock->package][$element->docblock->subpackage][$this->class][] = $link;
			break;
			case 'method' :
				$link = $this->addLink($element);
				$this->class_contents[$element->docblock->package][$element->docblock->subpackage][$this->class][] = $link;
			break;
			case 'function' :
				$link = $this->addLink($element);
				if ($this->leftindex['functions'])
				$this->function_elements[$element->docblock->package][$element->docblock->subpackage][] = $link;
				$this->page_contents[$element->docblock->package][$element->docblock->subpackage][$curpath][] = $link;
			break;
			default :
				if (PHPDOC_EXCEPTIONS)
				{
					addException(true,PDEXCEPTION_ELEMENT_NOT_PARSERELEMENT,"element",$element);
				}
			break;
		}
		if ($element->getType() != 'include')
		{
			if ($element->getType() == 'var' || $element->getType() == 'method')
			{
				$this->links[$this->package][$this->subpackage][$element->getType()][$element->class][$element->getName()] = $link;
				$this->linkswithfile[$this->package][$this->subpackage][$element->getType()][$element->getPath()][$element->class][$element->getName()] = $link;
			} else
			{
				if ($element->type == 'page')
				{
					$this->links[$this->package][$this->subpackage][$element->getType()][$element->getFile()] = $link;
					$this->linkswithfile[$this->package][$this->subpackage][$element->getType()][$element->getPath()][$element->getFile()] = $link;
				} else
				{
					$this->links[$this->package][$this->subpackage][$element->getType()][$element->getName()] = $link;
					$this->linkswithfile[$this->package][$this->subpackage][$element->getType()][$element->getPath()][$element->getName()] = $link;
				}
			}
		}
		if ($element->type == 'page')
		{
			$this->elements[substr(strtolower($element->getFile()),$i,1)][] = $element;
			$this->pkg_elements[$this->package][$this->subpackage][substr(strtolower($element->getFile()),$i,1)][] = $element;
		} else
		{
			$this->elements[substr(strtolower($element->getName()),$i,1)][] = $element;
			$this->pkg_elements[$this->package][$this->subpackage][substr(strtolower($element->getName()),$i,1)][] = $element;
		}
	}
	
	/**
	* returns an abstract link to element.  Do not modify or override
	* @return abstractLink abstractLink descendant
	* @access private
	* @param parserElement element to add a new link (descended from {@link abstractLink})to the {@link $links} array
	* @param string classname for elements that are class-based (this may be deprecated in the future, as the classname
	*               should be contained within the element.  if $element is a page, this parameter is a package name
	* @param string subpackage name for page elements
	*/
	function addLink(&$element)
	{
		switch($element->type)
		{
			case 'function':
				$x = new functionLink;
				$x->addLink($this->curpath, $this->curname, $element->name, $element->docblock->package, $element->docblock->subpackage);
				return $x;
			break;
			case 'define':
				$x = new defineLink;
				$x->addLink($this->curpath, $this->curname, $element->name, $element->docblock->package, $element->docblock->subpackage);
				return $x;
			break;
			case 'global':
				$x = new globalLink;
				$x->addLink($this->curpath, $this->curname, $element->name, $element->docblock->package, $element->docblock->subpackage);
				return $x;
			break;
			case 'class':
				$x = new classLink;
				$x->addLink($this->curpath, $this->curname, $element->name, $element->docblock->package, $element->docblock->subpackage);
				return $x;
			break;
			case 'method':
				$x = new methodLink;
				$x->addLink($this->class, $this->curpath, $this->curname, $element->name, $element->docblock->package, $element->docblock->subpackage);
				return $x;
			break;
			case 'var':
				$x = new varLink;
				$x->addLink($this->class, $this->curpath, $this->curname, $element->name, $element->docblock->package, $element->docblock->subpackage);
				return $x;
			break;
			case 'page':
				$x = new pageLink;
				$x->addLink($element->getPath(),$element->name,$element->file,$element->package, $element->subpackage);
				return $x;
			break;
		}
	}

	/**
	* Return a tree of all classes that extend this class
	*
	* The data structure returned is designed for a non-recursive algorithm, and is somewhat complex.
	* In most cases, the array returned is:
	* array('#root' => array('link' => {@link classLink} to $class,
	*                        'parent' => false,
	*                        'children' => array(array('class' => 'childclass1', 'package' => 'child1package'),
	*                                            array('class' => 'childclass2', 'package' => 'child2package'),...
	*                                           )
	*                       ),
	*       'child1package#childclass1' => array('link' => {@link classLink} to childclass1,
	*                                            'parent' => '#root',
	*                                            'children' => array(array('class' => 'kidclass', 'package' => 'kidpackage'),...
	*                                                               )
	*                                           ),
	*       'kidpackage#kidclass' => array('link' => {@link classLink} to kidclass,
	*                                      'parent' => 'child1package#childclass1',
	*                                      'children' => array() // no children
	*                                     ),
	*      ....
	*      )
	*
	* To describe this format using language, every class in the tree has an entry in the first level of the array.  The index for all child
	* classes that extend the root class is childpackage#childclassname.  Each entry in the array has 3 elements: link, parent, and children.
	* <ul>
	*	<li>link - a {@link classLink} to the current class</li>
	*	<li>parent - a {@link classLink} to the class's parent, or false (except for one special case described below)</li>
	*	<li>children - an array of arrays, each entry has a 'class' and 'package' index to the child class,
	* used to find the entry in the big array</li>
	* </ul>
	*
	* special cases are when the #root class has a parent in another package, or when the #root class extends a class not found
	* by phpDocumentor.  In the first case, parent will be a classLink to the parent class.  In the second, parent will be the
	* extends clause, as in:
	* <code>
	* class X extends Y
	* {
	* ...
	* }
	* </code>
	* in this case, the #root entry will be array('link' => classLink to X, 'parent' => 'Y', children => array(...))
	*
	* The fastest way to design a method to process the array returned is to copy HTMLdefaultConverter::getRootTree() into
	* your converter and to modify the html to whatever output format you are goign to use
	* @see HTMLdefaultConverter::getRootTree()
	* @param string class name
	* @param string
	* @param string
	* @return array Format: see docs 
	*/
	function getSortedClassTreeFromClass($class,$package,$subpackage)
	{
		$my_tree = array();
		$root = $this->classes->getClassByPackage($class,$package);
		if (!$root) return false;
		$class_children = $this->classes->getDefiniteChildren($class,$root->curfile);
		if (!$class_children)
		{
			// special case: parent class is found, but is not part of this package, class has no children
			if (is_array($root->parent))
			{
				$x = $root->getParent($this);
				if (PHPDOC_EXCEPTIONS)
				{
					addException(!is_object($x),PDEXCEPTION_NOT_A_CLASS,__FILE__,__LINE__,"class",$root->parent[1],"file",$root->parent[0],"parent of class",$root);
				}
				if ($x->docblock->package != $package)
				{
					$v = Converter::getClassLink($root->getName(),$package,$root->getPath());
					return array('#root' => array('link' => $v,'parent' => Converter::getClassLink($x->getName(),$x->docblock->package,$x->getPath()), 'children' => array()));
				}
			} else
			{ // class has normal situation, no children
				if (is_string($root->getParent($this)))
				return array('#root' => array('link' => Converter::getClassLink($root->getName(),$package,$root->getPath()), 'parent' => $root->getExtends(),'children' => array()));
				else
				return array('#root' => array('link' => Converter::getClassLink($root->getName(),$package,$root->getPath()), 'parent' => false, 'children' => array()));
			}
		}
		// special case: parent class is found, but is not part of this package, class has children
		if (is_array($root->parent))
		{
			$x = $root->getParent($this);
			if (PHPDOC_EXCEPTIONS)
			{
				addException(!is_object($x),PDEXCEPTION_NOT_A_CLASS,__FILE__,__LINE__,"class",$root->parent[1],"file",$root->parent[0],"parent of class",$root);
			}
			if ($x->docblock->package != $package)
			{
				$v = Converter::getClassLink($root->getName(),$package,$root->getPath());
				$my_tree = array('#root' => array('link' => $v, 'parent' => Converter::getClassLink($x->getName(),$x->docblock->package,$x->getPath()), 'children' => array()));
			} else
			{
				if (PHPDOC_EXCEPTIONS)
				{
					addException(true,PDEXCEPTION_ROOT_THAT_AINT,__FILE__,__LINE__,"root",$root,"parent",$x);
				}
			}
		} else
		$my_tree = array('#root' => array('link' => Converter::getClassLink($root->getName(),$package,$root->getPath()), 'parent' => false, 'children' => array()));
		// location of tree walker
		$cur = '#root';
		$lastcur = array(array(false,0));
		$childpos = 0;
		if (isset($class_children))
		{
			do
			{
				if (!$class_children)
				{
					list($cur, $childpos) = array_pop($lastcur);
					if (isset($my_tree[$cur]['children'][$childpos + 1]))
					{
						array_push($lastcur, array($cur, $childpos + 1));
						$par = $cur;
						$cur = $my_tree[$cur]['children'][$childpos + 1];
						$x = $this->classes->getClassByPackage($cur['class'],$cur['package']);
						$childpos = 0;
						$cur = $cur['package'] . '#' . $cur['class'];
						$my_tree[$cur]['link'] = Converter::getClassLink($x->getName(),$x->docblock->package,$x->getPath());
						$my_tree[$cur]['parent'] = $par;
						$my_tree[$cur]['children'] = array();
						$class_children = $this->classes->getDefiniteChildren($x->getName(), $x->curfile);
						continue;
					} else
					{
						$class_children = false;
						continue;
					}
				}
				foreach($class_children as $chileclass => $chilefile)
				{
					$ch = $this->classes->getClass($chileclass,$chilefile);
					if (PHPDOC_EXCEPTIONS)
					{
						addException(!is_object($ch),PDEXCEPTION_NOT_A_CLASS,__FILE__,__LINE__,"class",$chileclass,"file",$chilefile,"child of class",$class);
					}
					$my_tree[$cur]['children'][] = array('class' => $ch->getName(), 'package' => $ch->docblock->package);
				}
				usort($my_tree[$cur]['children'],'rootcmp');
				if (isset($my_tree[$cur]['children'][$childpos]))
				{
					array_push($lastcur, array($cur, $childpos));
					$par = $cur;
					$cur = $my_tree[$cur]['children'][$childpos];
					$x = $this->classes->getClassByPackage($cur['class'],$cur['package']);
					$cur = $cur['package'] . '#' . $cur['class'];
					$my_tree[$cur]['link'] = Converter::getClassLink($x->getName(),$x->docblock->package,$x->getPath());
					$my_tree[$cur]['parent'] = $par;
					$my_tree[$cur]['children'] = array();
					$childpos = 0;
					$class_children = $this->classes->getDefiniteChildren($x->getName(), $x->curfile);
				} else
				{
					list($cur, $childpos) = array_pop($lastcur);
				}
			} while ($cur);
		}
		return $my_tree;
	}
	
	/**
	* do not override
	* @return bool true if a link to this class exists in package $package and subpackage $subpackage
	* @param string $expr class name
	* @param string $package package to search in
	* @param string $subpackage subpackage to search in
	*/
	function isLinkedClass($expr,$package,$subpackage,$file=false)
	{
		if ($file)
		return isset($this->linkswithfile[$package][$subpackage]['class'][$file][$expr]);
		return isset($this->links[$package][$subpackage]['class'][$expr]);
	}
	
	/**
	* do not override
	* @return bool true if a link to this function exists in package $package and subpackage $subpackage
	* @param string $expr function name
	* @param string $package package to search in
	* @param string $subpackage subpackage to search in
	*/
	function isLinkedFunction($expr,$package,$subpackage,$file=false)
	{
		if ($file)
		return isset($this->linkswithfile[$package][$subpackage]['function'][$file][$expr]);
		return isset($this->links[$package][$subpackage]['function'][$expr]);
	}
	
	/**
	* do not override
	* @return bool true if a link to this define exists in package $package and subpackage $subpackage
	* @param string $expr define name
	* @param string $package package to search in
	* @param string $subpackage subpackage to search in
	*/
	function isLinkedDefine($expr,$package,$subpackage,$file=false)
	{
		if ($file)
		return isset($this->linkswithfile[$package][$subpackage]['define'][$file][$expr]);
		return isset($this->links[$package][$subpackage]['define'][$expr]);
	}
	
	/**
	* do not override
	* @return bool true if a link to this define exists in package $package and subpackage $subpackage
	* @param string $expr define name
	* @param string $package package to search in
	* @param string $subpackage subpackage to search in
	*/
	function isLinkedGlobal($expr,$package,$subpackage,$file=false)
	{
		if ($file)
		return isset($this->linkswithfile[$package][$subpackage]['global'][$file][$expr]);
		return isset($this->links[$package][$subpackage]['global'][$expr]);
	}
	
	/**
	* do not override
	* @return bool true if a link to this procedural page exists in package $package and subpackage $subpackage
	* @param string $expr procedural page name
	* @param string $package package to search in
	* @param string $subpackage subpackage to search in
	*/
	function isLinkedPage($expr,$package,$subpackage,$path=false)
	{
		if ($path)
		return isset($this->linkswithfile[$package][$subpackage]['page'][$path][$expr]);
		return isset($this->links[$package][$subpackage]['page'][$expr]);
	}
	
	/**
	* do not override
	* @return bool true if a link to this method exists in package $package, subpackage $subpackage and class $class
	* @param string $expr method name
	* @param string $class class name
	* @param string $package package to search in
	* @param string $subpackage subpackage to search in
	*/
	function isLinkedMethod($expr,$package,$subpackage,$class,$file=false)
	{
		if ($file)
		return isset($this->linkswithfile[$package][$subpackage]['method'][$file][$class][$expr]);
		return isset($this->links[$package][$subpackage]['method'][$class][$expr]);
	}
	
	/**
	* do not override
	* @return bool true if a link to this method exists in package $package, subpackage $subpackage and class $class
	* @param string $expr method name
	* @param string $class class name
	* @param string $package package to search in
	* @param string $subpackage subpackage to search in
	*/
	function isLinkedVar($expr,$package,$subpackage,$class,$file=false)
	{
		if ($file)
		return isset($this->linkswithfile[$package][$subpackage]['var'][$file][$class][$expr]);
		return isset($this->links[$package][$subpackage]['var'][$class][$expr]);
	}
	
	/**
	* return false or a {@link classLink} to $expr
	* @param string $expr class name
	* @param string $package package name
	* @return mixed returns a {@link classLink} or false if the element is not found in package $package
	* @see classLink
	*/
	function getClassLink($expr,$package,$file=false, $text = false)
	{
		if (!isset($this->links[$package])) return false;
		foreach($this->links[$package] as $subpackage => $notused)
		{
			if ($this->isLinkedClass($expr,$package,$subpackage,$file))
			{
				if ($file)
				{
					if (PHPDOC_EXCEPTIONS)
					{
						addException(!isset($this->linkswithfile[$package][$subpackage]['class'][$file]), PDEXCEPTION_FILENOTFOUND, __FILE__, __LINE__, "class", $expr, "file", $file, "package", $package, "subpackage", $subpackage,  "this->linkswithfile[package][subpackage]['class']",$this->linkswithfile[$package][$subpackage]['class']);
					}
					return $this->linkswithfile[$package][$subpackage]['class'][$file][$expr];
				}
				return $this->links[$package][$subpackage]['class'][$expr];
			}
		}
		return false;
	}

	/**
	* return false or a {@link functionLink} to $expr
	* @param string $expr function name
	* @param string $package package name
	* @return mixed returns a {@link functionLink} or false if the element is not found in package $package
	* @see functionLink
	*/
	function getFunctionLink($expr,$package,$file=false, $text = false)
	{
		if (!isset($this->links[$package])) return false;
		foreach($this->links[$package] as $subpackage => $notused)
		{
			if ($this->isLinkedFunction($expr,$package,$subpackage,$file))
			{
				if ($file)
				{
					if (PHPDOC_EXCEPTIONS)
					{
						addException(!isset($this->linkswithfile[$package][$subpackage]['function'][$file]), PDEXCEPTION_FILENOTFOUND, __FILE__, __LINE__, "function", $expr, "file", $file, "package", $package, "subpackage", $subpackage,  "this->linkswithfile[package][subpackage]['function']",$this->linkswithfile[$package][$subpackage]['function']);
					}
					return $this->linkswithfile[$package][$subpackage]['function'][$file][$expr];
				}
				return $this->links[$package][$subpackage]['function'][$expr];
			}
		}
		return false;
	}

	/**
	* return false or a {@link defineLink} to $expr
	* @param string $expr constant name
	* @param string $package package name
	* @return mixed returns a {@link defineLink} or false if the element is not found in package $package
	* @see defineLink
	*/
	function getDefineLink($expr,$package,$file=false, $text = false)
	{
		if (!isset($this->links[$package])) return false;
		foreach($this->links[$package] as $subpackage => $notused)
		{
			if ($this->isLinkedDefine($expr,$package,$subpackage,$file))
			{
				if ($file)
				{
					if (PHPDOC_EXCEPTIONS)
					{
						addException(!isset($this->linkswithfile[$package][$subpackage]['define'][$file]), PDEXCEPTION_FILENOTFOUND, __FILE__, __LINE__, "define", $expr, "file", $file, "package", $package, "subpackage", $subpackage,  "this->linkswithfile[package][subpackage]['define']",$this->linkswithfile[$package][$subpackage]['define']);
					}
					return $this->linkswithfile[$package][$subpackage]['define'][$file][$expr];
				}
				return $this->links[$package][$subpackage]['define'][$expr];
			}
		}
		return false;
	}

	/**
	* return false or a {@link globalLink} to $expr
	* @param string $expr global variable name (with leading $)
	* @param string $package package name
	* @return mixed returns a {@link defineLink} or false if the element is not found in package $package
	* @see defineLink
	*/
	function getGlobalLink($expr,$package,$file=false, $text = false)
	{
		if (!isset($this->links[$package])) return false;
		foreach($this->links[$package] as $subpackage => $notused)
		{
			if ($this->isLinkedGlobal($expr,$package,$subpackage,$file))
			{
				if ($file)
				{
					if (PHPDOC_EXCEPTIONS)
					{
						addException(!isset($this->linkswithfile[$package][$subpackage]['global'][$file]), PDEXCEPTION_FILENOTFOUND, __FILE__, __LINE__, "global variable", $expr, "file", $file, "package", $package, "subpackage", $subpackage,  "this->linkswithfile[package][subpackage]['global']",$this->linkswithfile[$package][$subpackage]['global']);
					}
					return $this->linkswithfile[$package][$subpackage]['global'][$file][$expr];
				}
				return $this->links[$package][$subpackage]['global'][$expr];
			}
		}
		return false;
	}
	
	/**
	* return false or a {@link pageLink} to $expr
	* @param string $expr procedural page name
	* @param string $package package name
	* @return mixed returns a {@link pageLink} or false if the element is not found in package $package
	* @see pageLink
	*/
	function getPageLink($expr,$package,$path = false, $text = false, $packages = false)
	{
		if (!isset($this->links[$package])) return false;
		foreach($this->links[$package] as $subpackage => $notused)
		{
			if ($this->isLinkedPage($expr,$package,$subpackage,$path))
			{
				if ($path)
				{
					if (PHPDOC_EXCEPTIONS)
					{
						addException(!isset($this->linkswithfile[$package][$subpackage]['page'][$path]), PDEXCEPTION_FILENOTFOUND, __FILE__, __LINE__, "page", $expr, "path", $path, "package", $package, "subpackage", $subpackage,  "this->linkswithfile[package][subpackage]['page']",$this->linkswithfile[$package][$subpackage]['page']);
					}
					return $this->linkswithfile[$package][$subpackage]['page'][$path][$expr];
				}
				return $this->links[$package][$subpackage]['page'][$expr];
			}
		}
		return false;
	}

	/**
	* return false or a {@link methodLink} to $expr in $class
	* @param string $expr method name
	* @param string $class class name
	* @param string $package package name
	* @return mixed returns a {@link methodLink} or false if the element is not found in package $package, class $class
	* @see methodLink
	*/
	function getMethodLink($expr,$class,$package,$file=false, $text = false)
	{
		$expr = trim($expr);
		$class = trim($class);
		if (!isset($this->links[$package])) return false;
		foreach($this->links[$package] as $subpackage => $notused)
		{
			if ($this->isLinkedMethod($expr,$package,$subpackage,$class,$file))
			{
				if ($file)
				{
					if (PHPDOC_EXCEPTIONS)
					{
						addException(!isset($this->linkswithfile[$package][$subpackage]['method'][$file]), PDEXCEPTION_FILENOTFOUND, __FILE__, __LINE__, "class", $class, "method", $expr, "file", $file, "package", $package, "subpackage", $subpackage,  "this->linkswithfile[package][subpackage]['method']",$this->links[$package][$subpackage]['method']);
					}
					return $this->linkswithfile[$package][$subpackage]['method'][$file][$class][$expr];
				}
				return $this->links[$package][$subpackage]['method'][$class][$expr];
			}
		}
		return false;
	}

	/**
	* return false or a {@link varLink} to $expr in $class
	* @param string $expr var name
	* @param string $class class name
	* @param string $package package name
	* @return mixed returns a {@link varLink} or false if the element is not found in package $package, class $class
	* @see varLink
	*/
	function getVarLink($expr,$class,$package,$file=false, $text = false)
	{
		$expr = trim($expr);
		$class = trim($class);
		if (!isset($this->links[$package])) return false;
		foreach($this->links[$package] as $subpackage => $notused)
		{
			if ($this->isLinkedVar($expr,$package,$subpackage,$class,$file))
			{
				if ($file)
				{
					if (PHPDOC_EXCEPTIONS)
					{
						addException(!isset($this->linkswithfile[$package][$subpackage]['var'][$file]), PDEXCEPTION_FILENOTFOUND, __FILE__, __LINE__, "class", $class, "var", $expr, "file", $file, "package", $package, "subpackage", $subpackage,  "this->linkswithfile[package][subpackage]['var']",$this->links[$package][$subpackage]['var']);
					}
					return $this->linkswithfile[$package][$subpackage]['var'][$file][$class][$expr];
				}
				return $this->links[$package][$subpackage]['var'][$class][$expr];
			}
		}
		return false;
	}

	/**
	* The meat of the @see tag and inline { @link} tag
	*
	* $expr is a string with many allowable formats:
	* <ol>
	*	<li>proceduralpagename.ext</li>
	*	<li>constant_name</li>
	*	<li>classname::function()</li>
	*	<li>classname::$variablename</li>
	*	<li>classname</li>
	*	<li>function functionname()</li>
	*	<li>packagename#expr where expr is any of the above</li>
	* </ol>
	*
	* New in version 1.1, you can explicitly specify a package to link to that is different from the current package.  Use the # operator
	* to specify a new package, as in tests#bug-540368.php (which should appear as a link like: "{@link tests#bug-540368.php}").  This
	* example links to the procedural page bug-540368.php in package tests.  Also, the "function" operator is now used to specifically
	* link to a function instead of a method in the current class.
	* <code>
	* class myclass
	* {
	*	// from inside the class definition, use "function conflict()" to refer to procedural function "conflict()"
	*	function conflict()
	*   {
	*   }
	* }
	* 
	* function conflict()
	* {
	* }
	* </code>
	*
	* If classname:: is not present, and the see tag is in a documentation block within a class, then the function uses the classname to
	* search for $expr as a function or variable within classname, or any of its parent classes.
	* given an $expr without '$', '::' or '()' getLink first searches for classes, procedural pages, constants, and then searches for
	* methods and variables within the default class, and finally for any function
	*
	* @param string $expr expression to search for a link
	* @param string $package package to start searching in
	* @param array $packages list of all packages to search in
	* @return mixed getLink returns a descendant of {@link abstractLink} if it finds a link, otherwise it returns a string
	* @see parserDocBlock, Render::HandleEvent(), getPageLink(), getDefineLink(), getVarLink(), getFunctionLink(), getClassLink()
	* @see pageLink, functionLink, defineLink, classLink, methodLink, varLink
	*/
	function &getLink($expr, $package = false, $packages = false)
	{
		if (strpos($expr,'#'))
		{
			$a = explode('#',$expr);
			if (count($a) == 2)
			{ // can have exactly 1 package override, otherwise it's ignored
				return Converter::getLink($a[1],$a[0],array());
			}
		}
		if (!$package) $package = $this->package;
		if (!isset($this->all_packages[$package])) return $expr;
		elseif (isset($packages[$package])) unset($packages[$package]);
		// is $expr a comma-delimited list?
		if (strpos($expr,','))
		{
			$a = explode(',',$expr);
			$b = array();
			for($i=0;$i<count($a);$i++)
			{
				// if so return each component with a link
				$b[] = Converter::getLink(trim($a[$i]));
			}
			return $b;
		}
		$links = &$this->links;
		$class = $this->class;
		if (strpos($expr,'function') === 0)
		{ // asking for a function, not a method
			if ($test = Converter::getFunctionLink(str_replace('function','',str_replace('()','',$expr)), $package)) return $test;
			else return $expr;
		}
		if (strpos($expr,'global') === 0)
		{ // asking for a function, not a method
			if ($test = Converter::getGlobalLink(str_replace('global','',$expr), $package)) return $test;
			else return $expr;
		}
		// are we in a class?
		if ($class)
		{
			// is $expr simply a word? if so, check to see if it is a method or variable of this class tree
			if (!strpos($expr,'::'))
			{
				// if get is neither get() nor $get, assume get is a function, add () to make get()
				if (strpos($expr,'$') !== 0 && !strpos($expr,'()')) //$get = $get.'()';
				{
					if ($a = $this->getLinkMethod($expr,$class,$package)) return $a;
					if ($a = $this->getLinkVar('$'.$expr,$class,$package)) return $a;
				}
				if (strpos($expr,'()')) if ($a = $this->getLinkMethod($expr,$class,$package)) return $a;
				if (is_numeric(strpos($expr,'$'))) if ($a = $this->getLinkVar($expr,$class,$package)) return $a;
			}
		}
		if ($test = Converter::getClassLink($expr,$package)) return $test;
		if ($test = Converter::getPageLink($expr,$package)) return $test;
		if ($test = Converter::getDefineLink($expr,$package)) return $test;
		if ($test = Converter::getGlobalLink($expr,$package)) return $test;
//		if (strpos($expr,'.'))
		// package specified

		do
		{
			if (!is_array($packages))
			{
				$packages = $this->all_packages;
				if (isset($packages[$package])) unset($packages[$package]);
			}
			if ($test = Converter::getClassLink($expr,$package)) return $test;
			if ($test = Converter::getPageLink($expr,$package)) return $test;
			if ($test = Converter::getDefineLink($expr,$package)) return $test;
			if ($test = Converter::getGlobalLink($expr,$package)) return $test;
			// is $expr in class::method() or class::$variable format?
			if (strpos($expr,'function') === 0)
			{ // asking for a function, not a method
				if ($test = Converter::getFunctionLink(str_replace('function','',str_replace('()','',$expr)), $package)) return $test;
				else return $expr;
			}
			if (@strpos($expr,'::'))
			{
				$class_method = explode('::',$expr);
				if ($class_method[0] == 'parent')
				{
					$cl = $this->classes->getClassByPackage($class,$package);
					if (PHPDOC_EXCEPTIONS)
					{
						addException(!$cl, PDEXCEPTION_NOT_A_CLASS, __FILE__, __LINE__, "class", $class, "package", $package, "this->classes", $this->classes);
					}
					$par = $cl->getParent($this);
					$phpparent = false;
					if (is_object($par))
					{
						$package = $par->docblock->package;
						$phpparent = $par->getName();
					} else
					{
						addWarning(PDERROR_CLASS_PARENT_NOT_FOUND,$class,$package,$class_method[1]);
						return $expr;
					}
					if ($phpparent) $class_method[0] = $phpparent;
				}
				if (strpos($class_method[1],'()'))
				{
					// strip everything but the function name, return a link
					if ($test = Converter::getMethodLink(str_replace('()','',$class_method[1]), $class_method[0], $package)) return $test;
				}
				if ($test = Converter::getVarLink($class_method[1], $class_method[0], $package)) return $test;
			}
			// $expr does not have ::
			if (is_numeric(@strpos('$',$expr)))
			{
				// default to current class, whose name is contained in $this->render->parent
				if ($test = Converter::getVarLink($expr, $class, $package)) return $test;
			}
			// $expr is a function? (non-method)
			if (@strpos($expr,'()'))
			{
				// otherwise, see if it is a method
				if ($class)
				{
					if ($test = Converter::getMethodLink(str_replace('function ','',str_replace('()','',$expr)), $class, $package)) return $test;
				}
				// extract the function name, use it to retrieve the file that the function is in
	//			$page = $this->func_page[str_replace('function ','',str_replace('()','',$expr))];
				// return the link
				if ($test = Converter::getFunctionLink(str_replace('function ','',str_replace('()','',$expr)), $package)) return $test;
			}
			// $expr is just a word.  First, test to see if it is a function of the current package
			if ($test = Converter::getFunctionLink(str_replace('function ','',str_replace('()','',$expr)), $package)) return $test;
			// try other packages
			// look in parent package first, if found
			if (isset($this->package_parents[$package]))
			{
				$p1 = $package;
				$package = $this->package_parents[$package];
				if ($package)
				{
					if (isset($packages[$package])) unset($packages[$package]);
				}
			}
			// no parent package, so start with the first one that's left
			list($package,) = @each($packages);
			if ($package)
			{
				unset($packages[$package]);
			}
		} while (count($packages) || $package);
		// no links found
		return $expr;
	}

	/**
	* do not use or override, used by getLink
	* @access private
	*/
	function &getLinkMethod($expr, $class, $package)
	{
		$links = &$this->links;
		do
		{
			// is $expr in class::method() or class::$variable format?
			if (@strpos($expr,'::'))
			{
				$class_method = explode('::',$expr);
				if ($class_method[0] == 'parent')
				{
					$cl = $this->classes->getClassByPackage($class,$package);
					if (PHPDOC_EXCEPTIONS)
					{
						addException(!$cl, PDEXCEPTION_NOT_A_CLASS, __FILE__, __LINE__, "class", $class, "package", $package, "this->classes", $this->classes);
					}
					$par = $cl->getParent($this);
					$phpparent = false;
					if (is_object($par))
					{
						$package = $par->docblock->package;
						$phpparent = $par->getName();
					} else addWarning(PDERROR_CLASSPARENT_NOTFOUND,$class,$package,$class_method[1]);
					if ($phpparent) $class_method[0] = $phpparent;
				} else
				{
					$cl = $this->classes->getClassByPackage($class,$package);
				}
				if (strpos($class_method[1],'()'))
				{
					// strip everything but the function name, return a link
					if ($test = Converter::getMethodLink(str_replace('function ','',str_replace('()','',$class_method[1])), $class_method[0], $package)) return $test;
				}
			}
			if ($test = Converter::getMethodLink(str_replace('()','',$expr), $class, $package)) return $test;
			$cl = $this->classes->getClassByPackage($class,$package);
			if ($cl)
			{
				$par = $cl->getParent($this);
				if (is_object($par))
				{
					$package = $par->docblock->package;
					$class = $par->getName();
				} else $class = $par;
			} else $class = false;
		} while ($class);
		// no links found
		return false;
	}
	
	/**
	* do not use or override, used by getLink
	* @access private
	*/
	function &getLinkVar($expr, $class, $package)
	{
		$links = &$this->links;
		do
		{
			// is $expr in class::method() or class::$variable format?
			if (@strpos($expr,'::'))
			{
				$class_method = explode('::',$expr);
				if ($class_method[0] == 'parent')
				{
					$cl = $this->classes->getClassByPackage($class,$package);
					if (PHPDOC_EXCEPTIONS)
					{
						addException(!$cl, PDEXCEPTION_NOT_A_CLASS, __FILE__, __LINE__, "class", $class, "package", $package, "this->classes", $this->classes);
					}
					$phpparent = false;
					$par = $cl->getParent($this);
					if (is_object($par))
					{
						$package = $par->docblock->package;
						$phpparent = $par->getName();
					} else addWarning(PDERROR_CLASSPARENT_NOTFOUND,$class,$package,$class_method[1]);
					if ($phpparent) $class_method[0] = $phpparent;
				} else
				{
					$cl = $this->classes->getClassByPackage($class,$package);
				}
				if ($test = Converter::getVarLink($class_method[1], $class_method[0], $package)) return $test;
				if ($test = Converter::getVarLink('$'.$class_method[1], $class_method[0], $package)) return $test;
			}
			if ($test = Converter::getVarLink($expr, $class, $package)) return $test;
			if ($test = Converter::getVarLink('$'.$expr, $class, $package)) return $test;
			$cl = $this->classes->getClassByPackage($class,$package);
			if ($cl)
			{
				$par = $cl->getParent($this);
				if (is_object($par))
				{
					$package = $par->docblock->package;
					$class = $par->getName();
				} else $class = $par;
			} else $class = false;
		} while ($class);
		// no links found
		return false;
	}
	
	/**
	* take URL $link and text $text and return a link in the format needed for the Converter
	* @param string URL
	* @param string text to display
	* @return string link to $link
	* @abstract
	*/
	function returnLink($link,$text)
	{
	}

	/**
	* take {@link abstractLink} descendant and text $eltext and return a link in the format needed for the Converter
	* @param abstractLink
	* @param string
	* @return string link to $element
	* @abstract
	*/
	function returnSee(&$element, $eltext = false)
	{
	}
	
	/**
	* Convert all elements to output format
	* @param mixed {@link parserElement} descendant or {@link parserPackagePage} or {@link parserData}
	* @see Converter
	* @abstract
	*/
	function Convert(&$element)
	{
	}
	
	/**
	* do all necessary output
	* @see Converter
	* @abstract
	*/
	function Output($title)
	{
		phpdoc_out("WARNING: Generic Converter was used, no output will be generated");
	}
	
	/**
	* sets the template directory based on the {@link $outputformat} and {@link $name}
    * Also sets {@link $templateName} to the $dir parameter
	* @param string subdirectory
	*/
	function setTemplateDir($dir)
	{
        $this->templateName = substr($dir,0,strlen($dir) - 1);
		$this->templateDir = "Converters/" . $this->outputformat . "/" . $this->name . "/templates/" . $dir;
	}
	
	/**
	* Parse a global variable's default value for class initialization.
	*
	* If a global variable's default value is "new class" as in:
	* <code>
	* $globalvar = new Parser
	* </code>
	* This method will document it not as "new Parser" but instead as "new {@link Parser}".    For examples, see {@link phpdoc.inc}.
	* Many global variables are classes, and phpDocumentor links to their documentation
	* @return string default global variable value with link to class if it's "new Class"
	* @param string default value of a global variable.
	*/
	function getGlobalValue($value)
	{
		if (strpos($value,'new') === 0)
		{
			preg_match('/new([^(]*)(.*)/',$value,$newval);
			if (isset($newval[1]))
			{
				$a = Converter::getLink(trim($newval[1]));
				if (!isset($newval[2])) $newval[2] = '';
				if ($a && get_class($a) == 'classlink') $value = 'new '.$this->returnSee($a).$newval[2];
			}
		}
		return $value;
	}
	
	/**
	* Parse an include's file to see if it is a file documented in this project
	*
	* Although not very smart yet, this method will try to look for the included file file.ext:
	* <code>
	* include ("file.ext");
	* </code>
	* If it finds it, it will return a link to the file's documentation.  For examples, see {@link phpdoc.inc} includes.
	* Every include auto-links to the documentation for the file that is included
	* @return string included file with link to docs for file, if found
	* @param string file included by include statement.
	*/
	function getIncludeValue($value)
	{
		preg_match('/"\.\/(.*\..*)"/',$value,$match);
		if (!isset($match[1]))
		preg_match('/"(.*\..*)"/',$value,$match);
		if (isset($match[1]))
		{
			$per = Converter::getLink($match[1]);
			if (is_object($per) && get_class($per) == 'pagelink')
			$value = $this->returnSee($per);
		}
		return $value;
	}

	/**
	* Sets the output directory for generated documentation
	* @param string $dir the output directory
	*/
	function setTargetDir($dir)
	{
		if (strlen($dir) > 0) 
		{
			$this->targetDir = $dir;
			// if directory does exist create it, this should have more error checking in the future
			if (!file_exists($dir))
			{
				$tmp = str_replace(array("/","\\"),SMART_PATH_DELIMITER,$dir);
				if (substr($tmp,-1) == SMART_PATH_DELIMITER)
				{
					$tmp = substr($tmp,0,(strlen($tmp)-1));
				}
				$tmp = explode(SMART_PATH_DELIMITER,$tmp);
				array_pop($tmp);
				$parent = implode(SMART_PATH_DELIMITER,$tmp);
				if ($parent != '' && !file_exists($parent))
				{
					echo "Creating Parent Directory $parent\n";
					mkdir($parent,0775);
				}
				echo "Creating Directory $dir\n";
				mkdir($dir,0775);
			} 
			 else if (!is_dir($dir))
			{
				echo "Output path: '$dir' is not a directory\n";
				die();
			}
		} else {
			echo "a target directory must be specified\n try phpdoc -h\n";
			die();
		}
	}

	/**
	* Writes a file to target dir
	* @param string
	* @param string
	*/
	function writeFile($file,$data)
	{
		if (!file_exists($this->targetDir))
		{
			mkdir($this->targetDir,0775);
		}
		phpdoc_out("    Writing ".$this->targetDir . PATH_DELIMITER . $file . "\n");
		$fp = fopen($this->targetDir . PATH_DELIMITER . $file,"w");
		set_file_buffer( $fp, 0 );
		fwrite($fp,$data,strlen($data));
		fclose($fp);
	}
	
	/**
	* Copies a file from the template directory to the target directory
	* thanks to Robert Hoffmann for this fix
	* @param string
	*/
	function copyFile($file)
	{
		if (!file_exists($this->targetDir))
		{
			mkdir($this->targetDir,0775); 
		}
		copy($this->templateDir . $file, $this->targetDir . PATH_DELIMITER . $file); 
	} 

	/**
	* Return parserStringWithInlineTags::Convert() cache state
	* @see parserStringWithInlineTags::Convert()
	* @abstract
	*/
	function getState()
	{
		return true;
	}

	/**
	* Compare parserStringWithInlineTags::Convert() cache state to $state
	* @param mixed
	* @see parserStringWithInlineTags::Convert()
	* @abstract
	*/
	function checkState($state)
	{
		return true;
	}

}

/**
* @access private
* @see Converter::getSortedClassTreeFromClass()
*/
function rootcmp($a, $b)
{
	return strnatcasecmp($a['class'],$b['class']);
}

/**
* smart htmlentities, doesn't entity the allowed tags list
* Since version 1.1, this function uses htmlspecialchars instead of htmlentities, for international support
* @param string $s
* @return string browser-displayable page
*/
function adv_htmlentities($s)
{
	global $___html,$PHPDOC_HTML_ALLOWED;
	$result = htmlspecialchars($s);
	$entities = array_flip(get_html_translation_table(HTML_SPECIALCHARS));
	$result = strtr($result,$___html);
	$matches = array();
	preg_match_all('/(&lt;img.*&gt;)/U',$result,$matches);
	for($i=0;$i<count($matches[1]);$i++)
	{
		$result = str_replace($matches[1][$i],strtr($matches[1][$i],array_flip(get_html_translation_table(HTML_SPECIALCHARS))),$result);
	}
	preg_match_all('/(&lt;font.*&gt;)/U',$result,$matches);
	for($i=0;$i<count($matches[1]);$i++)
	{
		$result = str_replace($matches[1][$i],strtr($matches[1][$i],array_flip(get_html_translation_table(HTML_SPECIALCHARS))),$result);
	}
	preg_match_all('/(&lt;ol.*&gt;)/U',$result,$matches);
	for($i=0;$i<count($matches[1]);$i++)
	{
		$result = str_replace($matches[1][$i],strtr($matches[1][$i],array_flip(get_html_translation_table(HTML_SPECIALCHARS))),$result);
	}
	preg_match_all('/(&lt;ul.*&gt;)/U',$result,$matches);
	for($i=0;$i<count($matches[1]);$i++)
	{
		$result = str_replace($matches[1][$i],strtr($matches[1][$i],array_flip(get_html_translation_table(HTML_SPECIALCHARS))),$result);
	}
	preg_match_all('/(&lt;li.*&gt;)/U',$result,$matches);
	for($i=0;$i<count($matches[1]);$i++)
	{
		$result = str_replace($matches[1][$i],strtr($matches[1][$i],array_flip(get_html_translation_table(HTML_SPECIALCHARS))),$result);
	}
	preg_match_all('/(&lt;a .*&gt;)/U',$result,$matches);
	for($i=0;$i<count($matches[1]);$i++)
	{
		$result = str_replace($matches[1][$i],strtr($matches[1][$i],array_flip(get_html_translation_table(HTML_SPECIALCHARS))),$result);
	}
	return $result;
}
?>
