<?php
/**
* HTML output converter.
* This Converter takes output from the {@link Parser} and converts it to HTML-ready output for use with {@link Template}.
*
* @package Converters
* @subpackage HTMLdefault
* @see parserDocBlock, parserInclude, parserPage, parserClass, parserDefine, parserFunction, parserMethod, parserVar
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: HTMLdefaultConverter.inc,v 1.75.2.4 2002/11/07 21:56:10 CelloG Exp $
*/
/**
* HTML output converter.
* This Converter takes output from the {@link Parser} and converts it to HTML-ready output for use with {@link Template}.
*
* @package Converters
* @subpackage HTMLdefault
* @see parserDocBlock, parserInclude, parserPage, parserClass, parserDefine, parserFunction, parserMethod, parserVar
* @author Greg Beaver <cellog@users.sourceforge.net>
* @author Joshua Eichorn <jeichorn@phpdoc.org>
* @since 1.0rc1
* @version $Id: HTMLdefaultConverter.inc,v 1.75.2.4 2002/11/07 21:56:10 CelloG Exp $
*/
class HTMLdefaultConverter extends Converter
{
	/** @var string */
	var $outputformat = 'HTML';
	/** @var string */
	var $name = 'default';
	/**
	* indexes of elements by package that need to be generated
	* @var array
	*/
	var $leftindex = array('classes' => true, 'pages' => true, 'functions' => true, 'defines' => false, 'globals' => false);
	/**
	* whether a @see is going to be in the {@link $base_dir}, or in a package/subpackage subdirectory of $base_dir
	* @var boolean
	*/
	var $local = true;
	
	/**
	* name of current page being converted
	* @var string
	*/
	var $page;
	
	/**
	* name of current class being converted
	* @var string
	*/
	var $class;
	
	/**
	* template for the procedural page currently being processed
	* @var Template
	*/
	var $page_data;
	
	/**
	* output directory for the current procedural page being processed
	* @var string
	*/
	var $page_dir;
	
	/**
	* target directory passed on the command-line.
	* {@link $targetDir} is malleable, always adding package/ and package/subpackage/ subdirectories onto it.
	* @var string
	*/
	var $base_dir;
	
	/**
	* output directory for the current class being processed
	* @var string
	*/
	var $class_dir;
	
	/**
	* template for the class currently being processed
	* @var Template
	*/
	var $class_data;
	
	/**
	* array of converted package page names.
	* Used to link to the package page in the left index
	* @var array Format: array(package => 1)
	*/
	var $package_pages = array();
	
	/**
	* controls formatting of parser informative output
	* 
	* Converter prints:
	* "Converting /path/to/file.php... Procedural Page Elements... Classes..."
	* Since HTMLdefaultConverter outputs files while converting, it needs to send a \n to start a new line.  However, if there
	* is more than one class, output is messy, with multiple \n's just between class file output.  This variable prevents that
	* and is purely cosmetic
	* @var boolean
	*/
	var $juststarted = false;
	
	/**
	* contains all of the template procedural page element loop data needed for the current template
	* @var array
	*/
	var $current;
	
	/**
	* contains all of the template class element loop data needed for the current template
	* @var array
	*/
	var $currentclass;
	/**
	* sets {@link $base_dir} to $targetDir
	* @see Converter()
	*/
	function HTMLdefaultConverter(&$allp, &$packp, &$classes, &$procpages, $po, $pp, $qm, $targetDir, $templateDir, $title)
	{
		Converter::Converter($allp, $packp, $classes, $procpages,$po, $pp, $qm, $targetDir, $templateDir, $title);
		$this->base_dir = $targetDir;
	}
	
	/**
	* Writes out the template file of {@link $class_data} and unsets the template to save memory
	* @see registerCurrentClass()
	* @see parent::endClass()
	*/
	function endClass()
	{
		$this->registerCurrentClass();
		if ($this->juststarted)
		{
			$this->juststarted = false;
			phpdoc_out("\n");
		}
		$this->setTargetDir($this->base_dir . PATH_DELIMITER . $this->class_dir);
		$this->writefile($this->class . '.html',$this->class_data->ret());
		unset($this->class_data);
	}
	
	/**
	* Writes out the template file of {@link $page_data} and unsets the template to save memory
	* @see registerCurrent()
	* @see parent::endPage()
	*/
	function endPage()
	{
		$this->registerCurrent();
		$this->setTargetDir($this->base_dir . PATH_DELIMITER . $this->page_dir);
		$this->writefile($this->page . '.html',$this->page_data->ret());
		unset($this->page_data);
	}
	
	/**
	* @param string
	* @param string
	* @return string &lt;a href="'.$link.'">'.$text.'</a&gt;
	*/
	function returnLink($link,$text)
	{
		return '<a href="'.$link.'">'.$text.'</a>';
	}
	
	/**
	* HTMLdefaultConverter chooses to format both package indexes and the complete index here
	*
	* This function formats output for the elementindex.html and pkgelementindex.html template files.  It then
	* writes them to the target directory
	* @see generateElementIndex(), generatePkgElementIndex()
	*/
	function formatPkgIndex()
	{
		list($elindex,$mletters) = $this->generateElementIndex();
		$template = new Template($this->templateDir,"elementindex.html");
		$template->register("elementindex",$elindex);
		$template->register("letters",$mletters);
		$template->register("title","Element Index");
		$template->register("page","elementindex.html");
		$template->register("date",date("r",time()));
		list($package_indexes,$packages,$mletters) = $this->generatePkgElementIndexes();
		$pnames = array();
		for($i=0;$i<count($packages);$i++)
		{
			$pnames[$packages[$i]['package']] = 1;
		}
		$template->register("packageindexes",$packages);
		phpdoc_out("\n");
		$this->setTargetDir($this->base_dir);
		$this->writefile('elementindex.html',$template->ret());
		for($i=0;$i<count($package_indexes);$i++)
		{
			$template = new Template($this->templateDir,"pkgelementindex.html");

			$template->register("pindex",$package_indexes[$i]['inner_loop']['pindex']);
			$template->register("package",$package_indexes[$i]['package']);
			$template->register("packageindexes",$package_indexes[$i]['inner_loop']['packageindexes']);
			$template->register("letters",$mletters[$package_indexes[$i]['package']]);
			$template->register("title","Package ".$package_indexes[$i]['package']." Element Index");
			$template->register("date",date("r",time()));
			$this->writefile('elementindex_'.$package_indexes[$i]['package'].'.html',$template->ret());
		}
		phpdoc_out("\n");
	}
	
	/**
	* @see parent::unmangle()
	*/
	function unmangle($sourcecode)
	{
		$sourcecode = str_replace('<br />',"\n",$sourcecode);
		$sourcecode = str_replace('&nbsp;',' ',$sourcecode);
		return $sourcecode;
	}
	
	/**
	* HTMLdefaultConverter uses this function to format template index.html and packages.html
	*
	* This function generates the package list from {@link $all_packages}, eliminating any
	* packages that don't have any entries in their package index (no files at all, due to @ignore
	* or other factors).  Then it uses the default package name as the first package index to display.
	* It sets the right pane to be either a blank file with instructions on making package-level docs,
	* or the package-level docs for the default package.
	* @global string Used to set the starting package to display
	*/
	function formatIndex()
	{
		global $phpDocumentor_DefaultPackageName;
		// Created index.html
		foreach($this->all_packages as $key => $val)
		{
			if (isset($this->pkg_elements[$key]))
			{
				if (!isset($start)) $start = $key;
				$package_index[$key] = array('link' => "li_$key.html", 'title' => $key);
			}
		}
		if (isset($package_index[$phpDocumentor_DefaultPackageName])) $start = $phpDocumentor_DefaultPackageName;
		if (!isset($package_index)) $package_index = array();
		if (!isset($start)) $start = '';
		uksort($package_index,"strnatcasecmp");
		$index = new Template($this->templateDir,"index.html");
		
		$index->register("date",date("r",time()));
		$index->register("title",$this->title);
		$index->register("start","li_$start.html");
		if (isset($this->package_pages[$start]))
		{
			$index->register("blank",$start. PATH_DELIMITER . 'package_'.$start);
		}
		else
		{
			$index->register("blank","blank");
			$blank = <<< EOF
<html>
<head>
	<title>Generated Documentation</title>
</head>
<body>
<b>Welcome to phpDocumentor!</b><br>
<br>
To change this page, create a package-level doc file packagename.html where packagename is the name of the main package
</body>
</html>
EOF;
			$this->setTargetDir($this->base_dir);
			$this->writefile("blank.html",$blank);
		}
		phpdoc_out("\n");
		$this->setTargetDir($this->base_dir);
		$this->writefile("index.html",$index->ret());
		unset($index);

		// Create package index
		$package = new Template($this->templateDir,"packages.html");
		$package->register("date",date("r",time()));
		$package->register("packages",$package_index);
		$this->writefile("packages.html",$package->ret());
		unset($package);

	}
	
	/**
	* Generate indexes for li_package.html and classtree output files
	*
	* This function generates the li_package.html files from the template file left.html.  It does this by
	* iterating through each of the $page_elements, $class_elements and  $function_elements arrays to retrieve
	* the pre-sorted {@link abstractLink} descendants needed for index generation.  Conversion of these links to
	* text is done by {@link returnSee()}.  The {@link $local} parameter is set to false to ensure that paths are correct.
	* 
	* Then it uses {@link generateFormattedClassTrees()} to create class trees from the template file classtrees.html.  Output
	* filename is classtrees_packagename.html.  This function also unsets {@link $elements} and {@link $pkg_elements} to free
	* up the considerable memory these two class vars use
	* @see $page_elements, $class_elements, $function_elements
	*/
	function formatLeftIndex()
	{
		phpdoc_out("\n");
		$this->setTargetDir($this->base_dir);
		$this->local = false;
		static $recur = 0;
		foreach($this->page_elements as $package => $o1)
		{
			foreach($o1 as $subpackage => $links)
			{
				for($i=0;$i<count($links);$i++)
				{
					$left[$package][$subpackage]['files'][] =
						array("link" => $this->returnSee($links[$i], false, false, false), "title" => $links[$i]->name);
				}
			}
		}
		foreach($this->class_elements as $package => $o1)
			foreach($o1 as $subpackage => $links)
			{
				for($i=0;$i<count($links);$i++)
				{
					$left[$package][$subpackage]['classes'][] =
						array("link" => $this->returnSee($links[$i], false, false, false), "title" => $links[$i]->name);
				}
			}
		foreach($this->function_elements as $package => $o1)
		{
			foreach($o1 as $subpackage => $links)
			{
				for($i=0;$i<count($links);$i++)
				{
					$left[$package][$subpackage]['functions'][] =
						array("link" => $this->returnSee($links[$i], false, false, false), "title" => $links[$i]->name);
				}
			}
		}
		if (!isset($left))
		{
			debug("Nothing parsed, check the command-line");
			die();
		}
		foreach($left as $package => $rest)
		{
			$nonclass_index = false;
			$class_index = false;
			$function_index = false;
			$subpackage_index = false;
			foreach($rest as $subpackage => $els)
			{
				if (!isset($els['functions'])) $els['functions'] = array();
				if (!isset($els['classes'])) $els['classes'] = array();
				if (!isset($els['files'])) $els['files'] = array();
				if (empty($subpackage))
				{
					$nonclass_index = $els['files'];
					$class_index = $els['classes'];
					$function_index = $els['functions'];
				} else
				{
					$subpackage_index[] = array("subpackage" => $subpackage, 
						"inner_loop" => array("classes" => $els['classes'],"procedural" => $els['files'],"function" => $els['functions']));
				}
			}
			$template = new Template($this->templateDir,"left.html");
			$template->register("date",date("r",time()));
			$template->register("nonclass",$nonclass_index);
			$template->register("class",$class_index);
			$template->register("function",$function_index);
			$template->register("subpackages",$subpackage_index);
			$template->register("packagename",$package);
			$template->register("phpdocversion",PHPDOC_VER);
			$template->register("phpdocwebsite",PHPDOC_WEBSITE);
			if (isset($this->package_pages[$package]))
			{
				$template->register("packagedoc",'<a target="right" href="' . $package . PATH_DELIMITER . 'package_' . $package . '.html">'.$package.' Package-level Docs</a><br>');
			} else
			{
				$template->register("packagedoc","");
			}
			$template->register("classtreepage","classtrees_$package");
			$template->register("elementindex","elementindex_$package");
			$this->writefile("li_$package.html",$template->ret());
			unset($template);
	
			// Create class tree page
			$template = new Template($this->templateDir,"classtrees.html");
			$template->register("classtrees",$this->generateFormattedClassTrees($package));
			$template->register("package",$package);
			$template->register("date",date("r",time()));
			$template->register("title","Class Trees for Package $package");
			$this->writefile("classtrees_$package.html",$template->ret());
			phpdoc_out("\n");
		}
		// free up considerable memory
		unset($this->elements);
		unset($this->pkg_elements);
		$this->local = true;
	}
	
	/**
	* This function takes an {@link abstractLink} descendant and returns an html link
	*
	* @param abstractLink a descendant of abstractlink should be passed, and never text
	* @param string text to display in the link
	* @param boolean this parameter is not used, and it set to $this->local.  See {@link $local}
	* @param boolean determines whether the returned text is enclosed in an &lt;a> tag
	*/
	function returnSee(&$element, $eltext = false, $local = true, $with_a = true)
	{
		if (PHPDOC_EXCEPTIONS)
		{
			addException(!is_object($element),PDEXCEPTION_NOT_A_CLASS,__FILE__,__LINE__,"element",$element,'eltext',$eltext,'local',$local,'with_a',$with_a);
		}
		$local = $this->local;
		if (!$element) return false;
		if (!$eltext)
		{
			$eltext = '';
			switch($element->type)
			{
				case 'method' :
				case 'var' :
				$eltext .= $element->class.'::';
				case 'page' :
				case 'define' :
				case 'class' :
				case 'function' :
				case 'global' :
				default :
				$eltext .= $element->name;
				if ($element->type == 'function' || $element->type == 'method') $eltext .= '()';
				break;
			}
		}
		$a = 1;
		$b = '';
		$c = '';
		if (!empty($this->subpackage))
		{
			$a++;
		}
		if (!empty($element->subpackage))
		{
			$c = '/'.$element->subpackage;
		}
		if ($local)
		{
			for($i=0;$i<$a;$i++) $b .= '../';
		}
		switch ($element->type)
		{
			case 'page' :
			if ($with_a)
			return '<a href="'.$b.$element->package.$c.'/_'.$element->fileAlias.'.html">'.$eltext.'</a>';
			else
			return $b.$element->package.$c.'/_'.$element->fileAlias.'.html';
			break;
			case 'define' :
			if ($with_a)
			return '<a href="'.$b.$element->package.$c.'/_'.$element->fileAlias.'.html#'.urlencode($element->name).'">'.$eltext.'</a>';
			else
			return $b.$element->package.$c.'/_'.$element->fileAlias.'.html#'.$element->name;
			break;
			case 'global' :
			if ($with_a)
			return '<a href="'.$b.$element->package.$c.'/_'.$element->fileAlias.'.html#'.urlencode($element->name).'">'.$eltext.'</a>';
			else
			return $b.$element->package.$c.'/_'.$element->fileAlias.'.html#'.$element->name;
			break;
			case 'class' :
			if ($with_a)
			return '<a href="'.$b.$element->package.$c.'/'.$element->name.'.html">'.$eltext.'</a>';
			else
			return $b.$element->package.$c.'/'.$element->name.'.html';
			break;
			case 'function' :
			if ($with_a)
			return '<a href="'.$b.$element->package.$c.'/_'.$element->fileAlias.'.html#'.$element->name.'">'.$eltext.'</a>';
			else
			return $b.$element->package.$c.'/_'.$element->fileAlias.'.html#'.$element->name;
			break;
			case 'method' :
			if ($with_a)
			return '<a href="'.$b.$element->package.$c.'/'.$element->class.'.html#'.$element->name.'">'.$eltext.'</a>';
			else
			return $b.$element->package.$c.'/'.$element->class.'.html#'.$element->name;
			break;
			case 'var' :
			if ($with_a)
			return '<a href="'.$b.$element->package.$c.'/'.$element->class.'.html#'.$element->name.'">'.$eltext.'</a>';
			else
			return $b.$element->package.$c.'/'.$element->class.'.html#'.$element->name;
			break;
		}
	}
	
	/**
	* Convert abstract $element for template
	*
	* calls helper function based on element type
	* @param mixed any descendant of {@link parserElement}, or {@link parserPage} or {@link parserPackagePage}
	* @see convertMethod(), convertFunction(), convertVar(), convertClass(), convertDefine(), convertGlobal()
	* @see convertInclude(), convertPage(), convertPackagePage()
	*/
	function Convert(&$element)
	{
		switch($element->type)
		{
			case 'method' :
				$this->convertMethod($element);
			break;
			case 'function' :
				$this->convertFunction($element);
			break;
			case 'var' :
				$this->convertVar($element);
			break;
			case 'class' :
				$this->convertClass($element);
			break;
			case 'define' :
				$this->convertDefine($element);
			break;
			case 'global' :
				$this->convertGlobal($element);
			break;
			case 'include' :
				$this->convertInclude($element);
			break;
			case 'page' :
				$this->convertPage($element);
			break;
			case 'packagepage' :
				$this->convertPackagePage($element);
			break;
		}
	}

	/**
	* Create errors.html template file output
	*
	* This method takes all parsing errors and warnings and spits them out ordered by file and line number.  It does not
	* display any phpDocumentor internal errors, that is handled by exceptions, which always terminate execution.
	* @see PHPDOC_EXCEPTIONS
	* @global ErrorTracker We'll be using it's output facility
	*/
	function ConvertErrorLog()
	{
		global $phpDocumentor_errors;
		$allfiles = array();
		$files = array();
		$warnings = $phpDocumentor_errors->returnWarnings();
		$errors = $phpDocumentor_errors->returnErrors();
		$template = new Template($this->templateDir,"errors.html");
		$template->register("date",date("r",time()));
		foreach($warnings as $warning)
		{
			$file = '##none';
			$linenum = 'Warning';
			if ($warning->file)
			{
				$file = $warning->file;
				$allfiles[$file] = 1;
				$linenum .= ' on line '.$warning->linenum;
			}
			$files[$file]['warnings'][] = array('name' => $linenum, 'listing' => $warning->data);
		}
		foreach($errors as $error)
		{
			$file = '##none';
			$linenum = 'Error';
			if ($error->file)
			{
				$file = $error->file;
				$allfiles[$file] = 1;
				$linenum .= ' on line '.$error->linenum;
			}
			$files[$file]['errors'][] = array('name' => $linenum, 'listing' => $error->data);
		}
		$i=1;
		$af = array();
		foreach($allfiles as $file => $num)
		{
			$af[$i++] = $file;
		}
		$allfiles = $af;
		usort($allfiles,'strnatcasecmp');
		$allfiles[0] = "Post-parsing";
		foreach($allfiles as $i => $a)
		{
			$allfiles[$i] = array('file' => $a);
		}
		$out = array();
		foreach($files as $file => $data)
		{
			if ($file == '##none') $file = 'Post-parsing';
			$out[] = array('file' => $file, 'inner_loop' => $data);
		}
		$template->register("files",$allfiles);
		$template->register("all",$out);
		$template->register("title","phpDocumentor Parser Errors and Warnings");
		$this->setTargetDir($this->base_dir);
		$this->writefile("errors.html",$template->ret());
		unset($template);
		phpdoc_out("\n\nTo view errors and warnings, look at ".$this->base_dir. PATH_DELIMITER . "errors.html\n");
	}
	
	/**
	* convert the element's DocBlock for output
	*
	* This function converts all tags and descriptions for output
	* @return array Format: array(short description, long description, array(array('keyword' => tagname,'data' => converted tag),...))
	* @param mixed any descendant of {@link parserElement}, or {@link parserData}
	* @param boolean set to false for pages and classes, the only elements allowed to specify @package
	*/
	function prepareDocBlock(&$element,$nopackage = true)
	{
		$tagses = $element->docblock->listTags();
		$tags = array();
		$names = array('staticvar' => 'Static Variable','deprec' => 'Deprecated','abstract' => 'Abstract Element','todo' => 'TODO');
		if (!$nopackage)
		{
			$tags[] = array('keyword' => 'Package','data' => $element->docblock->package);
			if (!empty($element->docblock->subpackage)) $tags[] = array('keyword' => 'Sub-Package','data' => $element->docblock->subpackage);
		}
		if ($element->docblock->var)
		{
			$a = $element->docblock->var->Convert($this);
			if (!empty($a))
			$tags[] = array('keyword' => 'Var', 'data' => adv_htmlentities($a));
		}
		if ($element->docblock->return)
		{
			$a = $element->docblock->return->Convert($this);
			if (!empty($a))
			$tags[] = array('keyword' => 'Return', 'data' => adv_htmlentities($a));
		}
		if ($element->docblock->funcglobals)
		foreach($element->docblock->funcglobals as $global => $val)
		{
			if ($a = $this->getGlobalLink($global,$element->docblock->package))
			{
				$global = $a;
			}
			$b = Converter::getLink($val[0]);
			if (is_object($b) && get_class($b) == 'classlink')
			{
				$val[0] = $this->returnSee($b);
			}
			$tags[] = array('keyword' => 'Global Variable Used','data' => $val[0].' '.$global.': '.adv_htmlentities($val[1]->Convert($this)));
		}
		if ($element->docblock->statics)
		foreach($element->docblock->statics as $static => $val)
		{
			$a = $val->Convert($this);
			$tags[] = array('keyword' => 'Static Variable Used','data' => $val->converted_returnType.' '.$static.': '.adv_htmlentities($a));
		}
		foreach($tagses as $tag)
		{
			if (isset($names[$tag->keyword])) $tag->keyword = $names[$tag->keyword];
			$tags[] = array("keyword" => ucfirst($tag->keyword),"data" => adv_htmlentities($tag->Convert($this)));
		}
		$sdesc = '';
		$desc = '';
		if ($element->docblock->sdesc) $sdesc = adv_htmlentities($element->docblock->sdesc->Convert($this));
		if ($element->docblock->desc) $desc = adv_htmlentities($element->docblock->desc->Convert($this));
		return array($sdesc,$desc,$tags);
	}
	
	/**
	* Converts package page and sets its package as used in {@link $package_pages}
	* @param parserPackagePage
	*/
	function convertPackagePage(&$element)
	{
		phpdoc_out("\n");
		$this->setTargetDir($this->base_dir. PATH_DELIMITER . $element->package);
		$this->writeFile('package_'.$element->package.'.html',$element->Convert($this));
		$this->package_pages[$element->package] = true;
	}
	
	/**
	* Converts class variables for template output.
	* @see prepareDocBlock(), getFormattedOverrides()
	* @param parserVar
	*/
	function convertVar(&$element)
	{
		$docblock = $this->prepareDocBlock($element);
		$b = 'mixed';
		if ($element->docblock->var)
		{
			$b = $element->docblock->var->converted_returnType;
		}
		$this->currentclass['var'][] = 
			array(
				"var_name" => $element->getName(), 
				"var_default" => htmlspecialchars($element->getValue()), 
				"var_sdesc" => $docblock[0]
			);
		$this->currentclass['var_detail'][] = 
			array(
				"var_name" => $element->getName(), 
				"var_default" => htmlspecialchars($element->getValue()), 
				"var_type" => $b,
				"var_overrides" => $this->getFormattedOverrides($element),
				"sdesc" => $docblock[0],
				"desc" => $docblock[1],
				"inner_loop" => array("docblock" => $docblock[2])
			);
	}
	
	/**
	* Converts class for template output
	* @see prepareDocBlock(), generateChildClassList(), generateFormattedClassTree(), getFormattedConflicts()
	* @see getFormattedInheritedMethods(), getFormattedInheritedVars()
	* @param parserClass
	*/
	function convertClass(&$element)
	{
		$this->class = $element->getName();
		$this->class_dir = $element->docblock->package;
		if (!empty($element->docblock->subpackage)) $this->class_dir .= PATH_DELIMITER . $element->docblock->subpackage;
		$this->class_data = new Template($this->templateDir,"class.html");
		$a = '../classtrees_';
		if ($element->docblock->subpackage != '') $a = "../$a";
		$docblock = $this->prepareDocBlock($element,false);
		
		$this->class_data->register("title","Docs For Class " . $element->getName());
		$this->class_data->register("page",$element->getName() . '.html');
		$this->class_data->register("class_name",$element->getName());
		$this->class_data->register("sdesc",$docblock[0]);
		$this->class_data->register("type","Method");
		$this->class_data->register("cpackage",$element->docblock->package);
		$a = '../';
		if (!empty($element->docblock->subpackage)) $a .= $a;
		$this->class_data->register("relloc",$a);
		$this->class_data->register("package",$a.$element->docblock->package);
		$this->class_data->register("desc",$docblock[1]);
		$this->class_data->register("docblock",$docblock[2]);

		$this->class_data->register("children", $this->generateChildClassList($element));
		$this->class_data->register("class_tree", $this->generateFormattedClassTree($element));
		$this->class_data->register("conflicts", $this->getFormattedConflicts($element,"classes"));
		
		$this->class_data->register("source_location",$element->getSourceLocation());
		$this->class_data->register("date",date("r",time()));

		$inherited_methods = $this->getFormattedInheritedMethods($element);
		if (!empty($inherited_methods))
		{
			$this->class_data->register("inherited_functions",$inherited_methods);
		}
		$inherited_vars = $this->getFormattedInheritedVars($element);
		if (!empty($inherited_vars))
		{
			$this->class_data->register("vars_inherited",$inherited_vars);
		}
	}
	
	/**
	* Converts method for template output
	* @see prepareDocBlock(), parserMethod::getFunctionCall(), getFormattedDescMethods(), getFormattedOverrides()
	* @param parserMethod
	*/
	function convertMethod(&$element)
	{
		$fname = $element->getName();
		if ($element->isConstructor)
		{
			$fname = 'constructor '.$element->getName();
		}
		$docblock = $this->prepareDocBlock($element);
		$returntype = 'void';
		if ($element->docblock->return)
		{
			$a = $element->docblock->return->Convert($this);
			$returntype = $element->docblock->return->converted_returnType;
		}
		$params = array();
		if (count($element->docblock->params))
		foreach($element->docblock->params as $param => $val)
		{
			$a = $val->Convert($this);
			$params[] = array("var" => ucfirst($param),"datatype" => $val->converted_returnType,"data" => adv_htmlentities($a));
		}

		$this->currentclass['function'][] = 
			array(
				"function_name" 	=> $fname,
				"function_sdesc" 	=> $docblock[0],
				"function_return"	=> $returntype,
				"function_call" 	=> $element->getFunctionCall()
			);
		$this->currentclass['function_detail'][] = 
			array(
				"function_name" 	=> $fname,
				"sdesc"		 	=> $docblock[0],
				"function_call" 	=> $element->getFunctionCall(),
				"function_return"	=> $returntype,
				"function_conflicts"	=> '',
				"descmethod"	=> $this->getFormattedDescMethods($element),
				"method_overrides"	=> $this->getFormattedOverrides($element),
				"desc" 			=> $docblock[1],
				"inner_loop"		=> array(
								"docblock" => $docblock[2],
								"params" => $params
								)
			);
	}
	
	/**
	* Converts function for template output
	* @see prepareDocBlock(), parserFunction::getFunctionCall(), getFormattedConflicts()
	* @param parserFunction
	*/
	function convertFunction(&$element)
	{
		$docblock = $this->prepareDocBlock($element);
		$fname = $element->getName();
		$params = array();
		if (count($element->docblock->params))
		foreach($element->docblock->params as $param => $val)
		{
			$a = $val->Convert($this);
			$params[] = array("var" => ucfirst($param),"datatype" => $val->converted_returnType,"data" => adv_htmlentities($a));
		}
		$returntype = 'void';
		if ($element->docblock->return)
		{
			$a = $element->docblock->return->Convert($this);
			$returntype = $element->docblock->return->converted_returnType;
		}

		$this->current['function'][] = 
			array(
				"function_name" 	=> $fname,
				"function_sdesc" 	=> $docblock[0],
				"function_return"	=> $element->docblock->return->returnType,
				"function_call" 	=> $element->getFunctionCall()
			);
		$this->current['function_detail'][] = 
			array(
				"function_name" 	=> $fname,
				"sdesc"		 	=> $docblock[0],
				"function_return"	=> $returntype,
				"function_conflicts"	=> $this->getFormattedConflicts($element,"functions"),
				"descmethod"	=> '',
				"method_overrides"	=>	'',
				"function_call" 	=> $element->getFunctionCall(),
				"desc" 			=> $docblock[1],
				"inner_loop"		=> array(
								"docblock" => $docblock[2],
								"params" => $params
								)
			);
	}
	
	/**
	* Converts include elements for template output
	* @see prepareDocBlock()
	* @param parserInclude
	*/
	function convertInclude(&$element)
	{
		$docblock = $this->prepareDocBlock($element);
		$per = $this->getIncludeValue($element->getValue());
		

		$this->current['include'][] = 
			array(
				"include_name" => $element->getName(), 
				"include_value" => $per,
				"include_file" => '_'.strtr($element->getValue(),array('"' => '', "'" => '','.' => '_')),
				"include_sdesc" => $docblock[0]
			);
		$this->current['include_detail'][] = 
			array(
				"include_name" 	=> $element->getName(), 
				"include_value" 	=> $per, 
				"include_file" => '_'.strtr($element->getValue(),array('"' => '', "'" => '','.' => '_')),
				"sdesc" 		=> $docblock[0],
				"desc" 			=> $docblock[1],
				"inner_loop"		=> array("docblock" => $docblock[2])
			);
	}
	
	/**
	* Converts defines for template output
	* @see prepareDocBlock(), getFormattedConflicts()
	* @param parserDefine
	*/
	function convertDefine(&$element)
	{
		$docblock = $this->prepareDocBlock($element);
		$this->current['constant'][] = 
			array(
				"constant_name" => $element->getName(), 
				"constant_link" => urlencode($element->getName()),
				"constant_value" => $element->getValue(),
				"constant_sdesc" => $docblock[0]
			);
		$this->current['constant_detail'][] = 
			array(
				"constant_name" 	=> $element->getName(), 
				"constant_value" 	=> $element->getValue(), 
				"constant_link" => urlencode($element->getName()),
				"define_conflicts"	=> $this->getFormattedConflicts($element,"defines"),
				"sdesc" 		=> $docblock[0],
				"desc" 			=> $docblock[1],
				"inner_loop"		=> array("docblock" => $docblock[2])
			);
	}
	
	/**
	* Converts global variables for template output
	* @param parserGlobal
	* @see prepareDocBlock(), getFormattedConflicts()
	*/
	function convertGlobal(&$element)
	{
		$docblock = $this->prepareDocBlock($element);
		$value = $this->getGlobalValue($element->getValue());
		$this->current['global'][] = 
			array(
				"global_name" => $element->getName(),
				"global_link" => urlencode($element->getName()),
				"global_value" => adv_htmlentities($value),
				"global_sdesc" => $docblock[0]
			);
		$this->current['global_detail'][] = 
			array(
				"global_name" 	=> $element->getName(), 
				"global_link" => urlencode($element->getName()),
				"global_value" 	=> adv_htmlentities($value), 
				"global_conflicts"	=> $this->getFormattedConflicts($element,"global variables"),
				"sdesc" 		=> $docblock[0],
				"desc" 			=> $docblock[1],
				"inner_loop"		=> array("docblock" => $docblock[2])
			);
	}
	
	/**
	* converts procedural pages for template output
	* @see prepareDocBlock(), getClassesOnPage()
	* @param parserData
	*/
	function convertPage(&$element)
	{
		$this->juststarted = true;
		$this->page_data = new Template($this->templateDir,"page.html");
		$this->page = '_'.$element->parent->getName();
		$this->page_dir = $element->parent->package;
		if (!empty($element->parent->subpackage)) $this->page_dir .= PATH_DELIMITER . $element->parent->subpackage;
		// registering stuff on the template
		$this->page_data->register("source_location",$element->parent->getSourceLocation());
		$this->page_data->filename = $element->parent->getFile();
		$this->page_data->register("date",date("r",time()));
		$this->page_data->register("type","Function");
		$this->page_data->register("classes",$this->getClassesOnPage($element));
		$this->page_data->register("cpackage",$element->parent->package);
		$a = '../';
		if (!empty($element->parent->subpackage)) $a .= $a;
		$this->page_data->register("relloc",$a);
		$this->page_data->register("page",'_'.$element->parent->getName() . '.html');
		$this->page_data->register("title","Docs for page ".$element->parent->getFile());
		if ($element->docblock)
		{
			$docblock = $this->prepareDocBlock($element, false);
			$this->page_data->register("sdesc",$docblock[0]);
			$this->page_data->register("desc",$docblock[1]);
			$this->page_data->register("docblock",$docblock[2]);
		} else
		{
			$this->page_data->register("sdesc",'');
			$this->page_data->register("desc",'');
			$this->page_data->register("docblock",array());
		}
		$this->page_data->register("name", $element->parent->getFile());
	}
	
	/**
	* Registers all of the procedural page elements on a page for variable substitution and output
	*
	* registers loops: constant, constant_detail, global, global_detail, function, include, include_detail, function_detail
	* @see Template::register(), $current, endPage()
	*/
	function registerCurrent()
	{
		if (!empty($this->current['constant']))
		{
			$this->rcnatcmpkey = "constant_name";
//			uasort($this->current['constant'],array($this,"rcNatCmp"));
			$this->page_data->register("constants",$this->current['constant']);
		}
		if (!empty($this->current['constant_detail']))
		{
			$this->rcnatcmpkey = "constant_name";
//			uasort($this->current['constant_detail'],array($this,"rcNatCmp"));
			$this->page_data->register("constant_detail",$this->current['constant_detail']);
		}
		if (!empty($this->current['global']))
		{
			$this->rcnatcmpkey = "global_name";
//			uasort($this->current['global'],array($this,"rcNatCmp"));
			$this->page_data->register("globals",$this->current['global']);
		}
		if (!empty($this->current['global_detail']))
		{
			$this->rcnatcmpkey = "global_name";
//			uasort($this->current['global_detail'],array($this,"rcNatCmp"));
			$this->page_data->register("global_detail",$this->current['global_detail']);
		}
		if (!empty($this->current['function']))
		{
			$this->rcnatcmpkey = "function_name";
//			uasort($this->current['function'],array($this,"rcNatCmp"));
			$this->page_data->register("functions",$this->current['function']);
		}
		if (!empty($this->current['include']))
		{
			$this->page_data->register("includes",$this->current['include']);
		}
		if (!empty($this->current['include_detail']))
		{
			$this->page_data->register("include_detail",$this->current['include_detail']);
		}
		if (!empty($this->current['function_detail']))
		{
			$this->rcnatcmpkey = "function_name";
//			uasort($this->current['function_detail'],array($this,"rcNatCmp"));
//			print_r($this->current['function_detail']);
			$this->page_data->register("function_detail",$this->current['function_detail']);
		}
		unset($this->current);
	}

	/**
	* Registers all of the class elements in a class for variable substitution and output
	*
	* registers loops: function, function_detail, var, var_detail
	* @see Template::register(), $currentclass, endClass()
	*/
	function registerCurrentClass()
	{
		if (!empty($this->currentclass['function']))
		{
			$this->rcnatcmpkey = "function_name";
			uasort($this->currentclass['function'],array($this,"rcNatCmp1"));
			$this->class_data->register("functions",$this->currentclass['function']);
		}
		if (!empty($this->currentclass['function_detail']))
		{
			$this->rcnatcmpkey = "function_name";
			uasort($this->currentclass['function_detail'],array($this,"rcNatCmp1"));
		//	print_R($this->current['function_detail']);
			$this->class_data->register("function_detail",$this->currentclass['function_detail']);
		}
		if (!empty($this->currentclass['var']))
		{
			$this->rcnatcmpkey = "var_name";
//			uasort($this->currentclass['var'],array($this,"rcNatCmp"));
			$this->class_data->register("vars",$this->currentclass['var']);
		}
		if (!empty($this->currentclass['var_detail']))
		{
			$this->rcnatcmpkey = "var_name";
//			uasort($this->currentclass['var_detail'],array($this,"rcNatCmp"));
			$this->class_data->register("var_detail",$this->currentclass['var_detail']);
		}
		unset($this->currentclass);
	}

	/**
	* gets a list of all classes declared on a procedural page represented by $element, a {@link parserData} class
	* @param parserData &$element
	* @return string comma-delimited list of links to each classes documentation
	*/
	function getClassesOnPage(&$element)
	{
		$a = $element->getClasses($this);
		$r = '';
        $data = array();
		foreach($a as $package => $classes)
		{
			for($i=0; $i<count($classes); $i++)
			{
                // hack for this release, all templates will be updated for the next release
                if ($this->templateName == 'phphtmllib')
                {
    				$desc = adv_htmlentities($classes[$i]->docblock->getDesc($this));
    				if (strlen($desc) == 0)
                    {
    					$desc = "n/a";
    				}
                    $data[] = array("class_link" => $this->getClassLink($classes[$i]->getName(),$package,$classes[$i]->getPath()),
    								"class_desc" => $desc);
                } else
                {
    				if (!empty($r)) $r .= ', ';
    				$r .= $this->getClassLink($classes[$i]->getName(),$package,$classes[$i]->getPath());
                }
			}
		}
        if ($this->templateName == 'phphtmllib')
        {
    		$template = new Template($this->templateDir,"classes_summary.inc.html");
    		$template->register("classes",$data);
    		return $template->ret();
        } else return $r;
	}
	
	/**
	* @param mixed {@link parserClass, parserFunction, parserDefine} or {@link parserGlobal}
	* @param string type to display.  either 'class','function','define' or 'global variable'
	* @return string full div of comma-separated links to conflicting elements, or empty string
	* @see parserClass::getConflicts(), parserFunction::getConflicts(), parserDefine::getConflicts(), parserGlobal::getConflicts()
	*/
	function getFormattedConflicts(&$element,$type)
	{
		$conflicts = $element->getConflicts($this);
		$r = '';
		if (!$conflicts) return $r;
		foreach($conflicts as $package => $class)
		{
			if (!empty($r)) $r .= ', ';
			$r .= $class->getLink($this,$class->docblock->package);
		}
        // hack for this release
		if (!empty($r))
        {
            if ($this->templateName == 'phphtmllib')
            {
                $r = "<div align=\"left\"><span class=\"font10bold\" style=\"color:#FF0000\">Warning:</span> Conflicts with $type in packages $r<br></div>";
            } else
            {
                $r = "<div align=\"left\"><b>Warning:</b> Conflicts with $type in packages $r<br></div>";
            }
        }
		return $r;
	}
	
	/**
	* @return string empty string or 'Overridden in child class(es) as: link to class1, link to class2...'
	* @see parserMethod::getOverridingMethods()
	* @param parserMethod
	*/
	function getFormattedDescMethods(&$element)
	{
		$meths = $element->getOverridingMethods($this);
		$r = '';
		for($i=0; $i<count($meths); $i++)
		{
			if (!empty($r)) $r .= ', ';
			$r .= $meths[$i]->getLink($this);//$this->getMethodLink($meths[$i]->getName(),$meths[$i]->class,$meths[$i]->docblock->package);
		}
		if (!empty($r)) $r = "Overridden in child class(es) as: $r<br>";
		return $r;
	}
	
	/**
	* @return string empty string or 'Overrides {link to overridden method}'
	* @see parserMethod::getOverrides()
	* @param parserMethod
	*/
	function getFormattedOverrides(&$element)
	{
		$ovr = $element->getOverrides($this);
		if (!$ovr) return '';
        if ($this->templateName == 'phphtmllib')
        return '<span class="font10bold" style="padding-left: 10px;">Overrides :</span> '.$ovr->getLink($this).'';
		return '<p>Overrides: '.$ovr->getLink($this).'</p>';
	}
	
	/**
	* returns a string containing a comma-delimited list of child classes
	*
	* @param parserClass class variable
	* @see parserClass::getChildClassList()
	*/
	
	function generateChildClassList($class)
	{
		$kids = $class->getChildClassList($this);
		$list = '';
		if (count($kids))
		{
			for($i=0; $i<count($kids); $i++)
			{
				if (!empty($list)) $list .= ', ';
				$list .= $this->getClassLink($kids[$i]->getName(),$kids[$i]->docblock->package, false, false, true);
			}
		}
		return $list;
	}

	/**
	* returns a string containing the class inheritance tree from the root object to the class
	*
	* @param parserClass	class variable
	* @see parserClass::getParentClassTree()
	*/
	
	function generateFormattedClassTree($class)
	{
		$tree = $class->getParentClassTree($this);
		$out = '';
		if (count($tree) - 1)
		{
			$result = array($class->getName());
			$parent = $tree[$class->getName()];
			while ($parent)
			{
				$subpackage = $parent->docblock->subpackage;
				$package = $parent->docblock->package;
				$x = $parent;
				if (is_object($parent))
				$x = $parent->getLink($this);
				if (!$x) $x = $parent->getName();
				$result[] = 
					$x."\n" .
					"%s|\n" .
					"%s--";
				if (is_object($parent))
				$parent = $tree[$parent->getName()];
				elseif (isset($tree[$parent]))
				$parent = $tree[$parent];
			}
			$nbsp = '   ';
			for($i=count($result) - 1;$i>=0;$i--)
			{
				$my_nbsp = '';
				for($j=0;$j<count($result) - $i;$j++) $my_nbsp .= $nbsp;
				$out .= sprintf($result[$i],$my_nbsp,$my_nbsp);
			}
			return "<pre>$out</pre>";
		} else
		{
			return $class->getName();
		}
	}
	
	/** @access private */
	function sortVar($a, $b)
	{
		return strnatcasecmp($a->getName(),$b->getName());
	}
	
	/** @access private */
	function sortMethod($a, $b)
	{
		if ($a->isConstructor) return -1;
		if ($b->isConstructor) return 1;
		return strnatcasecmp($a->getName(),$b->getName());
	}
	
	/**
	* Return template-enabled list of inherited variables
	*
	* uses parserVar helper function getInheritedVars and generates a template-enabled list using getClassLink()
	* @param parserVar $child class method
	* @see getClassLink(), parserVar::getInheritedVars()
	* @return array {@link Template}-ready array
	*/
	
	function getFormattedInheritedVars($child)
	{
		$package = $child->docblock->package;
		$subpackage = $child->docblock->subpackage;
		$ivars = $child->getInheritedVars($this);
		$results = array();
		if (!count($ivars)) return $results;
		foreach($ivars as $parent => $vars)
		{
			$file = $vars['file'];
			$vars = $vars['vars'];
			$par = $this->classes->getClass($parent,$file);
			if (PHPDOC_EXCEPTIONS)
			{
				addException(!is_object($par),PDEXCEPTION_NOT_A_CLASS,__FILE__,__LINE__,"class",$parent,"file",$file,"parent of class",$child);
			}
			$package = $par->docblock->package;
			usort($vars,array($this,"sortVar"));
			$result['parent_class'] = $this->getClassLink($parent,$package);
			foreach($vars as $var)
			{
				$info = array();
				if ($b = $this->getClassLink($parent,$package,false , false, true, false))
				{
					$info['ipath'] = $b;
				}
				$info['ivar_name'] = adv_htmlentities($var->getName());
				$info['ivar_default'] = adv_htmlentities($var->getValue());
				if ($var->docblock)
				$info['ivar_sdesc'] = adv_htmlentities($var->docblock->getSDesc($this));
				else
				$info['ivar_sdesc'] = '';
				$info['ipage'] = $this->class . '.html';
				$result['inner_loop']["ivars"][] = $info;
			}
			$results[] = $result;
			$result = array();
		}
		return $results;
	}
	
	/**
	* Return template-enabled list of inherited methods
	*
	* uses parserMethod helper function getInheritedMethods and generates a template-enabled list using getClassLink()
	* @param parserMethod $child class method
	* @see getClassLink(), parserMethod::getInheritedMethods()
	* @return array {@link Template}-ready array
	*/

	function getFormattedInheritedMethods($child)
	{
		$package = $child->docblock->package;
		$subpackage = $child->docblock->subpackage;
		$imethods = $child->getInheritedMethods($this);
		$results = array();
		if (!count($imethods)) return $results;
		foreach($imethods as $parent => $methods)
		{
			$file = $methods['file'];
			$methods = $methods['methods'];
			$par = $this->classes->getClass($parent,$file);
			if (PHPDOC_EXCEPTIONS)
			{
				addException(!is_object($par),PDEXCEPTION_NOT_A_CLASS,__FILE__,__LINE__,"class",$parent,"file",$file,"parent of class",$child);
			}
			$package = $par->docblock->package;
			usort($methods,array($this,"sortMethod"));
			$result['parent_class'] = $this->getClassLink($parent,$package);
			foreach($methods as $method)
			{
				$info = array();
				if ($b = $this->getClassLink($parent,$package,false , false, true, false))
				{
					$info['ipath'] = $b;
				}
				$qq = '';
				if ($method->isConstructor) $qq = 'constructor ';
				$info['ifunction_name'] = $qq.$method->getName();
				if ($method->docblock)
				$info['ifunction_sdesc'] = adv_htmlentities($method->docblock->getSDesc($this));
				else
				$info['ifunction_sdesc'] = '';
				$function_call = $qq.$method->getName() . " ( ";
				$tmp = 0;
				foreach($method->listParams() as $param)
				{
					if ($tmp == 0)
					{
						$tmp = 1;
					} else {
						$function_call .= ", ";
					}
					if ($param[1] !== '')
					{
						$function_call .= "[$param[0] = $param[1]]";
					} else {
						$function_call .= $param[0];
					}
				}
				$function_call .= " )";
				$info['ifunction_call'] = $function_call;
				$info['ipage'] = $this->class . '.html';
				$result['inner_loop']["ifunctions"][] = $info;
			}
			$results[] = $result;
			$result = array();
		}
		return $results;
	}

	/**
	* returns a template-enabled array of class trees
	* 
	* @param	string	$package	package to generate a class tree for
	* @see $roots, HTMLConverter::getRootTree()
	*/
	function generateFormattedClassTrees($package)
	{
		$this->local = false;
		if (!isset($this->roots[$package])) return array();
		$roots = $trees = array();
		$roots = $this->roots[$package];
		for($i=0;$i<count($roots);$i++)
		{
			$trees[] = array('class' => $roots[$i],'class_tree' => "<ul>\n".$this->getRootTree($this->getSortedClassTreeFromClass($roots[$i],$package,''),$package)."</ul>\n");
		}
		$this->local = true;
		return $trees;
	}
	
	/**
	* return formatted class tree for the Class Trees page
	*
	* @param array $tree output from {@link getSortedClassTreeFromClass()}
	* @see Classes::$definitechild, generateFormattedClassTrees()
	* @return string
	*/
	function getRootTree($tree,$package)
	{
		if (!$tree) return '';
		$my_tree = '';
		$cur = '#root';
		$lastcur = array(false);
		$kids = array();
		$dopar = false;
		if ($tree[$cur]['parent'])
		{
			$dopar = true;
			if (!is_object($tree[$cur]['parent']))
			{
//				debug("parent ".$tree[$cur]['parent']." not found");
				$my_tree .= '<li>' . $tree[$cur]['parent'] .'<ul>';
			}
			else
			{
//						debug("parent ".$this->returnSee($tree[$cur]['parent'], false, false)." in other package");
				$my_tree .= '<li>' . $this->returnSee($tree[$cur]['parent'], false, false);
				if ($tree[$cur]['parent']->package != $package) $my_tree .= ' <b>(Different package)</b><ul>';
			}
		}
		do
		{
//			fancy_debug($cur,$lastcur,$kids);
			if (count($tree[$cur]['children']))
			{
//				debug("$cur has children");
				if (!isset($kids[$cur]))
				{
//					debug("set $cur kids");
					$kids[$cur] = 1;
					$my_tree .= '<li>'.$this->returnSee($tree[$cur]['link'], false, false);
					$my_tree .= '<ul>'."\n";
				}
				array_push($lastcur,$cur);
				list(,$cur) = each($tree[$cur]['children']);
//				var_dump('listed',$cur);
				if ($cur)
				{
					$cur = $cur['package'] . '#' . $cur['class'];
//					debug("set cur to child $cur");
//					$my_tree .= '<li>'.$this->returnSee($tree[$cur]['link'], false, false);
					continue;
				} else
				{
//					debug("end of children for $cur");
					$cur = array_pop($lastcur);
					$cur = array_pop($lastcur);
					$my_tree .= '</ul></li>'."\n";
					if ($dopar && ($cur == '#root' || !$cur)) $my_tree .= '</ul></li>';
				}
			} else 
			{
//				debug("$cur has no children");
				$my_tree .= '<li>'.$this->returnSee($tree[$cur]['link'], false, false)."</li>";
				if ($dopar && $cur == '#root') $my_tree .= '</ul></li>';
				$cur = array_pop($lastcur);
			}
		} while ($cur);
		return $my_tree;
	}
	/**
	* Generate alphabetical index of all elements
	*
	* @see $elements, walk()
	*/
	function generateElementIndex()
	{
		$this->local = false;
		$elementindex = array();
		$letters = array();
		$i = 0;
		foreach($this->elements as $letter => $nutoh)
		{
			$letters[]['letter'] = $letter;
			$elindex['letter'] = $letter;
			foreach($this->elements[$letter] as $i => $yuh)
			{
				switch($this->elements[$letter][$i]->type)
				{
					case 'class':
						$aa = '';
						$aa = adv_htmlentities($this->elements[$letter][$i]->docblock->getSDesc($this));
						if (!empty($aa)) $aa = "<br>&nbsp;&nbsp;&nbsp;&nbsp;$aa";
						$oo['name'] = $this->elements[$letter][$i]->getName();
						$oo['listing'] =
							'in file '.$this->elements[$letter][$i]->file.', class '.$this->getClassLink($this->elements[$letter][$i]->getName(),
												$this->elements[$letter][$i]->docblock->package,
												$this->elements[$letter][$i]->getPath(),
												$this->elements[$letter][$i]->getName()
												, false
												, true)."$aa";
						$elindex['inner_loop']['index'][] = $oo; 
					break;
					case 'define':
						$aa = '';
						$aa = adv_htmlentities($this->elements[$letter][$i]->docblock->getSDesc($this));
						if (!empty($aa)) $aa = "<br>&nbsp;&nbsp;&nbsp;&nbsp;$aa";
						$oo['name'] = $this->elements[$letter][$i]->getName();
						$oo['listing'] =
							'in file '.$this->elements[$letter][$i]->file.', constant '.$this->getDefineLink($this->elements[$letter][$i]->getName(),
												 $this->elements[$letter][$i]->docblock->package,
												 $this->elements[$letter][$i]->getPath(),
												 $this->elements[$letter][$i]->getName()
												 , false)."$aa";
						$elindex['inner_loop']['index'][] = $oo; 
					break;
					case 'global':
						$aa = '';
						$aa = adv_htmlentities($this->elements[$letter][$i]->docblock->getSDesc($this));
						if (!empty($aa)) $aa = "<br>&nbsp;&nbsp;&nbsp;&nbsp;$aa";
						$oo['name'] = $this->elements[$letter][$i]->getName();
						$oo['listing'] =
							'in file '.$this->elements[$letter][$i]->file.', global variable '.$this->getGlobalLink($this->elements[$letter][$i]->getName(),
												 $this->elements[$letter][$i]->docblock->package,
												 $this->elements[$letter][$i]->getPath(),
												 $this->elements[$letter][$i]->getName()
												 , false)."$aa";
						$elindex['inner_loop']['index'][] = $oo; 
					break;
					case 'function':
						$aa = '';
						$aa = adv_htmlentities($this->elements[$letter][$i]->docblock->getSDesc($this));
						if (!empty($aa)) $aa = "<br>&nbsp;&nbsp;&nbsp;&nbsp;$aa";
						$oo['name'] = $this->elements[$letter][$i]->getName();
						$oo['listing'] =
							'in file '.$this->elements[$letter][$i]->file.', function '.$this->getFunctionLink($this->elements[$letter][$i]->getName(),
												   $this->elements[$letter][$i]->docblock->package,
												   $this->elements[$letter][$i]->getPath(),
												   $this->elements[$letter][$i]->getName().'()'
												   , false)."$aa";
						$elindex['inner_loop']['index'][] = $oo; 
					break;
					case 'method':
						$aa = '';
						$aa = adv_htmlentities($this->elements[$letter][$i]->docblock->getSDesc($this));
						if (!empty($aa)) $aa = "<br>&nbsp;&nbsp;&nbsp;&nbsp;$aa";
						$oo['name'] = $this->elements[$letter][$i]->getName();
						$oo['listing'] =
							'in file '.$this->elements[$letter][$i]->file.', method '.$this->getMethodLink($this->elements[$letter][$i]->getName(),
												 $this->elements[$letter][$i]->class,
												 $this->elements[$letter][$i]->docblock->package,
												 $this->elements[$letter][$i]->getPath(),
												 $this->elements[$letter][$i]->class.'::'.$this->elements[$letter][$i]->getName().'()'
												 , false)."$aa";
						$elindex['inner_loop']['index'][] = $oo; 
					break;
					case 'var':
						$aa = '';
						$aa = adv_htmlentities($this->elements[$letter][$i]->docblock->getSDesc($this));
						if (!empty($aa)) $aa = "<br>&nbsp;&nbsp;&nbsp;&nbsp;$aa";
						$oo['name'] = $this->elements[$letter][$i]->getName();
						$oo['listing'] =
							'in file '.$this->elements[$letter][$i]->file.', variable '.$this->getVarLink($this->elements[$letter][$i]->getName(),
											  $this->elements[$letter][$i]->class,
											  $this->elements[$letter][$i]->docblock->package,
											  $this->elements[$letter][$i]->getPath(),
											  $this->elements[$letter][$i]->class.'::'.$this->elements[$letter][$i]->getName()
											  , false)."$aa";
						$elindex['inner_loop']['index'][] = $oo; 
					break;
					case 'page':
						$oo['name'] = $this->elements[$letter][$i]->getFile();
						$oo['listing'] =
							'procedural page '.$this->getPageLink($this->elements[$letter][$i]->getFile(),
											   $this->elements[$letter][$i]->package,
											   $this->elements[$letter][$i]->getPath(),
											   $this->elements[$letter][$i]->getFile()
											   , false);
						$elindex['inner_loop']['index'][] = $oo; 
					break;
				}
			}
			if (isset($elindex['inner_loop']))
			{
				$elementindex[] = $elindex;
			} else
			{
				unset($letters[count($letters) - 1]);
			}
			$elindex = array();
		}
		$this->local = true;
		return array($elementindex,$letters);
	}
	
	/**
	* calls the converter setTargetDir, and then copies any template images and the stylesheet if they haven't been copied
	* @see Converter::setTargetDir()
	*/
	function setTargetDir($dir)
	{
		Converter::setTargetDir($dir);
		static $wroteto = array();
		static $template_images = array();
		if (!count($template_images))
		{
			$d = dir($this->templateDir); 
			$template_images = array();
			while($entry = $d->read())
			{
				$sp = explode(".", $entry); 
				if ( preg_match("/\.(gif|jpg|png)$/i", $entry) )
				{
					$template_images[] = $entry;
				}
			}
		}
		if (!isset($wroteto[$dir]))
		{
			$wroteto[$dir] = 1;
			phpdoc_out("Writing stylesheet.css\n");
			$this->copyFile("stylesheet.css");
			phpdoc_out("Copying Any Template Images\n");
			foreach($template_images as $image)
			{
				if (file_exists($this->templateDir.$image))
				{
					phpdoc_out("Writing $image\n");
					$this->copyFile($image);
				}
			}
		}
	}
	
	/**
	* Generate alphabetical index of all elements by package and subpackage
	*
	* @param string $package name of a package
	* @see $pkg_elements, walk(), generatePkgElementIndexes()
	*/
	function generatePkgElementIndex($package)
	{
		$this->local = false;
		$elementindex = array();
		$letters = array();
		$letterind = array();
		$used = array();
		$subp = '';
		foreach($this->pkg_elements[$package] as $subpackage => $els)
		{
			if (empty($els)) continue;
			if (!empty($subpackage)) $subp = " (<b>subpackage:</b> $subpackage)"; else $subp = '';
			foreach($els as $letter => $yuh)
			{
				if (!isset($used[$letter]))
				{
					$letters[]['letter'] = $letter;
					$letterind[$letter] = count($letters) - 1;
					$used[$letter] = 1;
				}
				$elindex[$letter]['letter'] = $letter;
				foreach($els[$letter] as $i => $yuh)
				{
					switch($els[$letter][$i]->type)
					{
						case 'class':
							$aa = '';
							$aa = adv_htmlentities($els[$letter][$i]->docblock->getSDesc($this));
							if (!empty($aa)) $aa = "<br>&nbsp;&nbsp;&nbsp;&nbsp;$aa";
							$oo['name'] = $els[$letter][$i]->getName();
							$oo['listing'] =
								'in file '.$els[$letter][$i]->file.', class '.$this->getClassLink($els[$letter][$i]->getName(),
													$els[$letter][$i]->docblock->package,
													$els[$letter][$i]->getPath(),
													$els[$letter][$i]->getName()
													, false
													, true)."$subp$aa";
							$elindex[$letter]['inner_loop']['index'][] = $oo; 
						break;
						case 'define':
							$aa = '';
							$aa = adv_htmlentities($els[$letter][$i]->docblock->getSDesc($this));
							if (!empty($aa)) $aa = "<br>&nbsp;&nbsp;&nbsp;&nbsp;$aa";
							$oo['name'] = $els[$letter][$i]->getName();
							$oo['listing'] =
								'in file '.$els[$letter][$i]->file.', constant '.$this->getDefineLink($els[$letter][$i]->getName(),
													 $els[$letter][$i]->docblock->package,
													 $els[$letter][$i]->getPath(),
													 $els[$letter][$i]->getName()
													 , false)."$subp$aa";
							$elindex[$letter]['inner_loop']['index'][] = $oo; 
						break;
						case 'global':
							$aa = '';
							$aa = adv_htmlentities($els[$letter][$i]->docblock->getSDesc($this));
							if (!empty($aa)) $aa = "<br>&nbsp;&nbsp;&nbsp;&nbsp;$aa";
							$oo['name'] = $els[$letter][$i]->getName();
							$oo['listing'] =
								'in file '.$els[$letter][$i]->file.', global variable '.$this->getGlobalLink($els[$letter][$i]->getName(),
													 $els[$letter][$i]->docblock->package, 
													 $els[$letter][$i]->getPath(),
													 $els[$letter][$i]->getName()
													 ,false)."$subp$aa";
							$elindex[$letter]['inner_loop']['index'][] = $oo; 
						break;
						case 'function':
							$aa = '';
							$aa = adv_htmlentities($els[$letter][$i]->docblock->getSDesc($this));
							if (!empty($aa)) $aa = "<br>&nbsp;&nbsp;&nbsp;&nbsp;$aa";
							$oo['name'] = $els[$letter][$i]->getName();
							$oo['listing'] =
								'in file '.$els[$letter][$i]->file.', function '.$this->getFunctionLink($els[$letter][$i]->getName(),
													   $els[$letter][$i]->docblock->package,
													   $els[$letter][$i]->getPath(),
													   $els[$letter][$i]->getName().'()'
													   , false)."$subp$aa";
							$elindex[$letter]['inner_loop']['index'][] = $oo; 
						break;
						case 'method':
							$aa = '';
							$aa = adv_htmlentities($els[$letter][$i]->docblock->getSDesc($this));
							if (!empty($aa)) $aa = "<br>&nbsp;&nbsp;&nbsp;&nbsp;$aa";
							$oo['name'] = $els[$letter][$i]->getName();
							$oo['listing'] =
								'in file '.$els[$letter][$i]->file.', method '.$this->getMethodLink($els[$letter][$i]->getName(),
													 $els[$letter][$i]->class,
													 $els[$letter][$i]->docblock->package, 
													 $els[$letter][$i]->getPath(),
													 $els[$letter][$i]->class.'::'.$els[$letter][$i]->getName().'()'
													 , false)."$subp$aa";
							$elindex[$letter]['inner_loop']['index'][] = $oo; 
						break;
						case 'var':
							$aa = '';
							$aa = adv_htmlentities($els[$letter][$i]->docblock->getSDesc($this));
							if (!empty($aa)) $aa = "<br>&nbsp;&nbsp;&nbsp;&nbsp;$aa";
							$oo['name'] = $els[$letter][$i]->getName();
							$oo['listing'] =
								'in file '.$els[$letter][$i]->file.', variable '.$this->getVarLink($els[$letter][$i]->getName(),
												  $els[$letter][$i]->class,
												  $els[$letter][$i]->docblock->package,
												  $els[$letter][$i]->getPath(),
												  $els[$letter][$i]->class.'::'.$els[$letter][$i]->getName()
												  , false)."$subp$aa";
							$elindex[$letter]['inner_loop']['index'][] = $oo; 
						break;
						case 'page':
							$oo['name'] = $els[$letter][$i]->getFile();
							$oo['listing'] =
								'procedural page '.$this->getPageLink($els[$letter][$i]->getFile(),
												   $els[$letter][$i]->package,
												   $els[$letter][$i]->getPath(),
												   $els[$letter][$i]->getFile()
												   , false)."$subp";
							$elindex[$letter]['inner_loop']['index'][] = $oo; 
						break;
					}
				}
			}
		}
		ksort($elindex);
		usort($letters,'lettersort');
		if (isset($elindex))
		{
			while(list($letter,$tempel) = each($elindex))
			{
				if (!isset($tempel['inner_loop']))
				{
					unset($letters[$letterind[$tempel['letter']]]);
				} else
				$elementindex[] = $tempel;
			}
		} else $letters = array();
		$this->local = true;
		return array($elementindex,$letters);
	}
	
	/**
	*
	* @see generatePkgElementIndex()
	*/
	function generatePkgElementIndexes()
	{
		$packages = array();
		$package_names = array();
		$pkg = array();
		$letters = array();
		foreach($this->pkg_elements as $package => $trash)
		{
			$pkgs['package'] = $package;
			$pkg['package'] = $package;
			list($pkg['inner_loop']['pindex'],$letters[$package]) = $this->generatePkgElementIndex($package);
			if (count($pkg['inner_loop']['pindex']))
			{
				$packages[] = $pkg;
				$package_names[] = $pkgs;
			}
			unset($pkgs);
			unset($pkg);
		}
		foreach($packages as $i => $package)
		{
			$pnames = array();
			for($j=0;$j<count($package_names);$j++)
			{
				if ($package_names[$j]['package'] != $package['package']) $pnames[] = $package_names[$j];
			}
			$packages[$i]['inner_loop']['packageindexes'] = $pnames;
		}
		return array($packages,$package_names,$letters);
	}
	
	/**
	* @param string name of class
	* @param string package name
	* @param string full path to look in (used in index generation)
	* @param boolean deprecated
	* @param boolean return just the URL, or enclose it in an html a tag
	* @return mixed false if not found, or an html a link to the class's documentation
	* @see parent::getClassLink()
	*/
	function getClassLink($expr,$package, $file = false,$text = false, $local = true, $with_a = true)
	{
		$a = Converter::getClassLink($expr,$package,$file);
		if (!$a) return false;
		return $this->returnSee($a, $text, $local, $with_a);
	}

	/**
	* @param string name of function
	* @param string package name
	* @param string full path to look in (used in index generation)
	* @param boolean deprecated
	* @param boolean return just the URL, or enclose it in an html a tag
	* @return mixed false if not found, or an html a link to the function's documentation
	* @see parent::getFunctionLink()
	*/
	function getFunctionLink($expr,$package, $file = false,$text = false, $local = true)
	{
		$a = Converter::getFunctionLink($expr,$package,$file);
		if (!$a) return false;
		return $this->returnSee($a, $text, $local);
	}

	/**
	* @param string name of define
	* @param string package name
	* @param string full path to look in (used in index generation)
	* @param boolean deprecated
	* @param boolean return just the URL, or enclose it in an html a tag
	* @return mixed false if not found, or an html a link to the define's documentation
	* @see parent::getDefineLink()
	*/
	function getDefineLink($expr,$package, $file = false,$text = false, $local = true)
	{
		$a = Converter::getDefineLink($expr,$package,$file);
		if (!$a) return false;
		return $this->returnSee($a, $text, $local);
	}

	/**
	* @param string name of global variable
	* @param string package name
	* @param string full path to look in (used in index generation)
	* @param boolean deprecated
	* @param boolean return just the URL, or enclose it in an html a tag
	* @return mixed false if not found, or an html a link to the global variable's documentation
	* @see parent::getGlobalLink()
	*/
	function getGlobalLink($expr,$package, $file = false,$text = false, $local = true)
	{
		$a = Converter::getGlobalLink($expr,$package,$file);
		if (!$a) return false;
		return $this->returnSee($a, $text, $local);
	}

	/**
	* @param string name of procedural page
	* @param string package name
	* @param string full path to look in (used in index generation)
	* @param boolean deprecated
	* @param boolean return just the URL, or enclose it in an html a tag
	* @return mixed false if not found, or an html a link to the procedural page's documentation
	* @see parent::getPageLink()
	*/
	function getPageLink($expr,$package, $path = false,$text = false, $local = true)
	{
		$a = Converter::getPageLink($expr,$package,$path);
		if (!$a) return false;
		return $this->returnSee($a, $text, $local);
	}

	/**
	* @param string name of method
	* @param string class containing method
	* @param string package name
	* @param string full path to look in (used in index generation)
	* @param boolean deprecated
	* @param boolean return just the URL, or enclose it in an html a tag
	* @return mixed false if not found, or an html a link to the method's documentation
	* @see parent::getMethodLink()
	*/
	function getMethodLink($expr,$class,$package, $file = false,$text = false, $local = true)
	{
		$a = Converter::getMethodLink($expr,$class,$package,$file);
		if (!$a) return false;
		return $this->returnSee($a, $text, $local);
	}

	/**
	* @param string name of var
	* @param string class containing var
	* @param string package name
	* @param string full path to look in (used in index generation)
	* @param boolean deprecated
	* @param boolean return just the URL, or enclose it in an html a tag
	* @return mixed false if not found, or an html a link to the var's documentation
	* @see parent::getVarLink()
	*/
	function getVarLink($expr,$class,$package, $file = false,$text = false, $local = true)
	{
		$a = Converter::getVarLink($expr,$class,$package,$file);
		if (!$a) return false;
		return $this->returnSee($a, $text, $local);
	}
	
	/**
	* returns {@link $local}
	* @see Converter::getState()
	*/
	function getState()
	{
		return $this->local;
	}

	/**
	* checks $state to see if it is the same as {@link $local}
	* @param boolean value to compare to $local
	* @see Converter::getState(), $local
	*/
	function checkState($state)
	{
		return $state == $this->local;
	}

	/**
	* does a nat case sort on the specified second level value of the array
	*
	* @param	mixed	$a
	* @param	mixed	$b
	* @return	int
	*/
	function rcNatCmp ($a, $b)
	{
		$aa = strtoupper($a[$this->rcnatcmpkey]);
		$bb = strtoupper($b[$this->rcnatcmpkey]);
		
		return strnatcasecmp($aa, $bb);
	}
	
	/**
	* does a nat case sort on the specified second level value of the array.
	* this one puts constructors first
	*
	* @param	mixed	$a
	* @param	mixed	$b
	* @return	int
	*/
	function rcNatCmp1 ($a, $b)
	{
		$aa = strtoupper($a[$this->rcnatcmpkey]);
		$bb = strtoupper($b[$this->rcnatcmpkey]);
		
		if (strpos($aa,'CONSTRUCTOR') === 0)
		{
			return -1;
		}
		if (strpos($bb,'CONSTRUCTOR') === 0)
		{
			return 1;
		}
		if (strpos($aa,strtoupper($this->class)) === 0)
		{
			return -1;
		}
		if (strpos($bb,strtoupper($this->class)) === 0)
		{
			return -1;
		}
		return strnatcasecmp($aa, $bb);
	}
	
	/**
	* This function is not used by HTMLdefaultConverter, but is required by Converter
	*/
	function Output()
	{
	}
}

/** @access private */
function lettersort($a, $b)
{
	return strnatcasecmp($a['letter'],$b['letter']);
}
?>
