<?php
//
// phpDocumentor, a program for creating javadoc style documentation from php code
// Copyright (C) 2000-2002 Joshua Eichorn
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//

//
// Copyright 2000-2002 Joshua Eichorn
// Email jeichorn@phpdoc.org
// Web 		http://phpdoc.org/
// Mirror 	http://phpdocu.sourceforge.net/
// Project    	http://sourceforge.net/projects/phpdocu/
//

/** The IntermediateParser Class
 *
 *  The Intermediary Data Parser (intermediate between Parse and Converter)
 *
 *  @author Gregory Beaver
 *  @version 0.1
 *  @copyright 2002 Gregory Beaver
 *  @package 	phpDocumentor
 */
/** The IntermediateParser Class
 *
 * This class performs the work of organizing raw data from the parser in the format of descendants
 * of the {@link parserElement} class.  This is also where processing of package pages occurs,
 * in {@link IntermediateParser::handleClass()} for class-level packages and {@link IntermediateParser::handleDocBlock()}
 * for page-level packages.  Most of the work of this parser goes to matching up DocBlocks with the elements
 * that they are documenting.  Since DocBlocks are passed before the element they document, the last DocBlock
 * is stored in {@link IntermediateParser::$last} and then placed into the $docblock parameter of the parserElement
 * descendant object.
 *
 * 
 *
 *
 *  @author Gregory Beaver
 *  @version $Id: IntermediateParser.inc,v 1.81.2.2 2002/11/07 20:45:59 CelloG Exp $
 *  @copyright 2002 Gregory Beaver
 *  @package 	phpDocumentor
 */
class IntermediateParser
{
	/**
	* @var parserDocBlock
	*/
	var $last	= "";
	
	/**
	* type of the last parser Element handled
	*
	* This is used in handleDocBlock to determine whether a DocBlock is a page-level DocBlock
	* in conjunction with the {@link parserData::$clean} var.  A page-level DocBlock is always
	* the first DocBlock in a file, and must be followed by another DocBlock.  The first test
	* is handled by parserData::$clean, which is set to false on the first encounter of an
	* element, and the second test is handled by this variable, which must be equal to "docblock"
	* @see handleDocBlock()
	* @var string
	*/
	var $lasttype = '';
	
	/**
	* Name of the class currently being parsed.
	* It is only used (and only valid) when IntermediateParser is parsing a class
	* @var string
	*/
	var $cur_class = '';
	
	/**
	* type of the current parser Element being handled
	* 
	* This is used by {@link HandleEvent()} to set the {@link $lasttype} var, which is
	* used to detect page-level DocBlocks
	* @var string
	*/
	var $type = '';
	
	/**
	* set in {@link phpdoc.inc} to the value of the parserprivate commandline option.
	* If this option is true, elements with an @access private tag will be parsed and displayed
	* @var bool
	*/
	var $parsePrivate = false;
	
	/**
	* this variable is used to prevent parsing of elements with an @ignore tag
	* @see $packageoutput
	* @see $parsePrivate
	*/
	var $private_class = false;
	
	/**
	* used to set the output directory
	* @see setTargetDir()
	*/
	var $targetDir;
	
	/**
	* array of parsed package pages
	*
	* used by {@link Convert()} to convert all package pages into output
	* @var array
	*/
	var $package_pages = array();
	
	/**
	* @var array array of all {@link parserData} containing page information
	*/
	var $pages = array();
	/**
	* When a page has @access private in its DocBlock, it is placed here instead of in
	* {@link $pages}, to allow for proper Class parsing.  Since classes and pages
	* are parsed as if they were separate, this array allows public classes on
	* private pages to retrieve information needed about the page that holds the class
	* and to {@link addPageIfNecessary()} to the $pages array
	*/
	var $privatepages = array();
	/**
	* Keeps track of packages of classes that have parent classes in another package.  Used in automatic linking.
	*
	* This array is updated by {@link addPackageParent()}, which is called in {@link Classes::processChild()}
	* to keep track of classes that descend from classes in different packages.  In other words, if class foo
	* is in package one, and class bar is in package two, an entry $package_parents['two'] = 'one' will be made.
	* @var array Format: packagename => parentpackagename
	* @see Converter::getLink()
	*/
	var $package_parents = array();
	
	/**
	* list of all packages encountered while documenting.  Used in automatic linking.
	* 
	* Converter::getLink() first checks if an ambiguous link is found in the current package.
	* If not, it then checks in parent packages, and if still not found, uses this array to
	* check in the rest of the packages before giving up
	* @var array Format: array(packagename => 1, packagename => 1,...)
	* @see Converter::getLink()
	*/
	var $all_packages = array();
	
	/**
	* array of packages to parser and output documentation for, if not all packages should be documented
	* Format:
	* array(package1,package2,...)
	*   or false if not set
	* Use this option to limit output similar to ignoring files.  If you have some temporary files that you don't want to specify by name
	* but don't want included in output, set a package name for all the elements in your project, and set packageoutput to that name.
	* the default package will be ignored.  Parsing speed does not improve.  If you want to ignore files for speed reasons, use the ignore
	* command-line option
	* @see Io
	* @var mixed
	*/
	var $packageoutput = false;
	
	/**
	* the functions which handle output from the {@link Parser}
	* @see handleEvent(), handleDocBlock(), handlePage(), handleClass(), handleDefine(), handleFunction(), handleMethod(), handleVar(),
	*      handlePackagePage(), handleInclude()
	*/
	var $event_handlers = array(
			'docblock' => 'handleDocBlock',
			'page' => 'handlePage',
			'class' => 'handleClass',
			'define' => 'handleDefine',
			'function' => 'handleFunction',
			'method' => 'handleMethod',
			'var' => 'handleVar',
			'packagepage' => 'handlePackagePage',
			'include' => 'handleInclude',
			'global' => 'handleGlobal',
			);
	
	/**
	* $data contains parsed structures for the current page being parsed
	*
	* In version 1.1+, $data is only used to store the current page information.  All handling of documented elements
	* is handled by the {@link ProceduralPages} and {@link Classes} classes.
	* @var parserData
	* @see parserData
	*/
	var $data;
	
	/**
	* set in {@link phpdoc.inc} to the value of the quitemode commandline option.
	* If this option is true, informative output while parsing will not be displayed (documentation is unaffected)
	* @var bool
	*/
	var $quietMode = false;
	
	/**
	* used to keep track of inheritance at the smartest level possible for a dumb computer
	* @var Classes
	*/
	var $classes = false;
	
	/**
	* used to keep track of all elements in a procedural page.  Handles name conflicts with elegance
	* @since 1.1
	* @var ProceduralPages
	*/
	var $proceduralpages = false;
	
	/**
	* an array of template names indexed by converter name
	*
	* For example, if the default HTMLdefaultConverter is using the DOM/l0l33t template,
	* the array will be $converters['default'] = 'DOM/l0l33t'
	* @var array Format: array(Convertername1 => templatename)
	* @see Converter
	*/
	var $converters = false;
	/** 
	* @var string Title of generated documentation, passed to Converters
	*/
	var $title = '';

	/**
	* sets up basic data structures
	* @param string Title of generated documentation, passed to Converters
	* @see $title, $data, $classes, $proceduralpages
	*/
	function IntermediateParser($title='Generated Documentation')
	{
		$this->title = $title;
		$this->data = new parserData;
		$this->classes = new Classes;
		$this->proceduralpages = new ProceduralPages;
	}
	
	/**
	* handles post-parsing of include/require/include_once/require_once
	*
	* This function sets {@link $data}->clean to false to tell the IntermediateParser that
	* a page-level DocBlock can't be found after this point on this page.  It then sets the package
	* to be the same as the page, and adds itself to the {@link ProceduralPages} class
	* @param integer $event Event number from {@link Parser.inc}
	* @param parserInclude $data
	*/
	function handleInclude($event,$data)
	{
		$this->data->clean = false;
		// page was @ignored
		if ($this->private_page)
		{
			unset($this->last);
			return;
		}
		if (empty($this->last))
		{
			// we don't have a docblock, create an empty one to get rid of errors
			$this->last = new parserDocblock();
		}
		if ($this->last->getKeyword('ignore'))
		{
			addWarning(PDERROR_IGNORE_TAG_IGNORED,'include',$data->getName().'('.$data->getValue().')');
		}

		$this->last->overridePackage($this->package,$this->subpackage,$data->getName(),'include');
		$data->setDocBlock($this->last);
		$this->proceduralpages->addInclude($data);
		$this->last = false;
	}
	
	/**
	* handles post-parsing of global variables
	*
	* This function sets {@link $data}->clean to false to tell the IntermediateParser that
	* a page-level DocBlock can't be found after this point on this page.  It then sets the package
	* to be the same as the page, and adds itself to the {@link ProceduralPages} class
	* @param integer $event Event number from {@link Parser.inc}
	* @param parserGlobal $data
	*/
	function handleGlobal($event,$data)
	{
		$this->data->clean = false;
		if ($this->private_page)
		{
			unset($this->last);
			return;
		}
		if (empty($this->last))
		{
			// we don't have a docblock, create an empty one to get rid of errors
			$this->last = new parserDocblock();
		}
		if ($this->last->getKeyword('ignore'))
		{
			addWarning(PDERROR_IGNORE_TAG_IGNORED,'global variable - just don\'t document the',$data->getName());
			unset($this->last);
			return;
		}

		$this->last->overridePackage($this->package,$this->subpackage,$data->getName(),'global');
		$data->setDocBlock($this->last);
		if ($data->docblock->getKeyword('name'))
		{
			$a = $data->docblock->getKeyword('name');
			$data->setName($a->value);
		}
		$this->proceduralpages->addGlobal($data);
		$this->last = false;
	}
	
	/**
	* handles post-parsing of Package-level documentation pages.
	*
	* sets the {@link $package_pages}[$data->package] to $data
	* @param integer $event Event number from {@link Parser.inc}
	* @param parserPackagePage $data
	*/
	function handlePackagePage($event,$data)
	{
		$type = 'packagepage';
		$this->package_pages[$data->package] = &$data;
		$this->last = false;
	}
	
	/**
	* handles post-parsing of class vars
	*
	* This function sets up a @var tag if none is found, and aligns $data's $path var and packages to match
	* the parent object
	* @param integer $event Event number from {@link Parser.inc}
	* @param parserVar $data
	*/
	function handleVar($event,$data)
	{
		if ($this->private_class)
		{
			unset($this->last);
			return;
		}
		if (empty($this->last))
		{
			// we don't have a docblock, create an empty one to get rid of errors
			$this->last = new parserDocblock();
		}
		$this->last->overridePackage($this->package,$this->subpackage,$data->getName(),'var');

		if ($this->last->getKeyword('ignore'))
		{
			addWarning(PDERROR_IGNORE_TAG_IGNORED,'var',$this->cur_class.'::'.$data->getName());
		}
		if (!$this->last->var)
		{
			$this->last->addVar('mixed',new parserStringWithInlineTags);
		}
		
		$data->setDocBlock($this->last);
		$data->path = $this->data->parent->path;
		$this->classes->addVar($data);
		$this->last = false;
	}
	
	/**
	* handles post-parsing of class methods
	*
	* This function first aligns $data's path and package to match the parent object, and also
	* aligns the docblock's @param, @global, and @staticvar tags with the information parsed
	* from the method source code.  It also checks to see if the method is a constructor
	* and sets the $isConstructor flag.  Finally, it adds the method to the {@link Classes} class.
	* @param integer $event Event number from {@link Parser.inc}
	* @param parserMethod $data
	*/
	function handleMethod($event,$data)
	{
		if ($this->private_class)
		{
			unset($this->last);
			return;
		}

		if (empty($this->last))
		{
			// we don't have a docblock, create an empty one to get rid of errors
			$this->last = new parserDocblock();
		}
		if ($this->last->getKeyword('ignore'))
		{
			addWarning(PDERROR_IGNORE_TAG_IGNORED,'method',$this->cur_class.'::'.$data->getName());
		}
		$this->last->overridePackage($this->package,$this->subpackage,$data->getName(),'method');
		foreach($data->listParams() as $param)
		{
			$update_params[] = $param[0];
		}
		foreach($data->listGlobals() as $param)
		{
			$update_globals[] = $param[1];
		}
		foreach($data->listStatics() as $param)
		{
			$update_statics[] = $param[0];
		}
		if (isset($update_params))
		$this->last->updateParams($update_params);
		if (isset($update_globals))
		$this->last->updateGlobals($update_globals);
		if (isset($update_statics))
		$this->last->updateStatics($update_statics);
		unset($update_params);
		unset($update_globals);
		unset($update_statics);

		if ($data->getName() == $this->cur_class) $data->setConstructor();

		$data->setDocBlock($this->last);
		$data->path = $this->data->parent->path;
		$this->classes->addMethod($data);
		$this->last = false;
	}

	/**
	* handles post-parsing of functions
	*
	* This function sets {@link $data}->clean to false to tell the IntermediateParser that
	* a page-level DocBlock can't be found after this point on this page.  It then sets the package
	* to be the same as the page, aligns the docblock's @param, @global, and @staticvar tags with the information parsed
	* from the function source code, and adds itself to the {@link ProceduralPages} class
	* @param integer $event Event number from {@link Parser.inc}
	* @param parserFunction $data
	*/
	function handleFunction($event,$data)
	{
		$this->data->clean = false;
		if ($this->private_page)
		{
			unset($this->last);
			return;
		}

		if (empty($this->last))
		{
			// we don't have a docblock, create an empty one to get rid of errors
			$this->last = new parserDocblock();
		}
		if ($this->last->getKeyword('ignore'))
		{
			unset($this->last);
			return;
		}
		$this->last->overridePackage($this->package,$this->subpackage,$data->getName(),'function');

		foreach($data->listParams() as $param)
		{
			$update_params[] = $param[0];
		}
		foreach($data->listGlobals() as $param)
		{
			$update_globals[] = $param[1];
		}
		foreach($data->listStatics() as $param)
		{
			$update_statics[] = $param[0];
		}
		if (isset($update_params))
		$this->last->updateParams($update_params);
		if (isset($update_globals))
		$this->last->updateGlobals($update_globals);
		if (isset($update_statics))
		$this->last->updateStatics($update_statics);
		unset($update_params);
		unset($update_globals);
		unset($update_statics);

		if ($data->hasSource())
		{
			$this->last->setSource($data->getSource());
		}
		if (count($this->last->params) == 1 && !count($data->listParams()))
		{
			// if the function has no parameters, and 1 @param, add it to the list as optional, default value is description from @param
			$pars = $this->last->listParams();
			$data->addParam($pars[0]['var'],$pars[0]['data']->getString());
		}
		$data->setDocBlock($this->last);
		$this->proceduralpages->addFunction($data);
		$this->last = false;
	}
	
	/**
	* handles post-parsing of defines
	*
	* This function sets {@link $data}->clean to false to tell the IntermediateParser that
	* a page-level DocBlock can't be found after this point on this page.  It then sets the package
	* to be the same as the page and adds itself to the {@link ProceduralPages} class
	* @param integer $event Event number from {@link Parser.inc}
	* @param parserDefine $data
	*/
	function handleDefine($event,$data)
	{
		$this->data->clean = false;
		if ($this->private_page)
		{
			unset($this->last);
			return;
		}
		if (empty($this->last))
		{
			// we don't have a docblock, create an empty one to get rid of errors
			$this->last = new parserDocblock();
		}
		if ($this->last->getKeyword('ignore'))
		{
			unset($this->last);
			return;
		}

		$this->last->overridePackage($this->package,$this->subpackage,$data->getName(),'define');
		$data->setDocBlock($this->last);
		$this->proceduralpages->addDefine($data);
		$this->last = false;
	}
	
	/**
	* handles post-parsing of classes
	*
	* This function sets {@link $data}->clean to false to tell the IntermediateParser that
	* a page-level DocBlock can't be found after this point on this page.  It sets
	* {@link $cur_class} to its name, and if an @ignore tag is found in the DocBlock,
	* it sets {@link $private_class} to true, to prevent post-parsing of any of the class's
	* vars or methods.  Then it checks for the existence of a package page for the class's package
	* 
	* @param integer $event Event number from {@link Parser.inc}
	* @param parserClass $data
	*/
	function handleClass($event,$data)
	{
		$this->data->clean = false;
		if (empty($this->last))
		{
			// we don't have a docblock, create an empty one to get rid of errors
			$this->last = new parserDocblock();
		}

		$data->setDocBlock($this->last);
		$this->cur_class = $name = $data->getName();
		if ($this->last->getKeyword('ignore'))
		{
			$this->private_class = true;
			unset($this->last);
			return;
		}
		$data->path = $this->data->parent->path;
		$this->classes->addClass($data);
		$this->private_class = false;
		if ($this->last->package)
		{
			if (!isset($this->package_pages[$this->last->package]))
			{
				if (file_exists(dirname($this->data->parent->getPath()) . SMART_PATH_DELIMITER . $this->last->package . '.html'))
				{
					if ($this->quietMode === false)
					{
						phpdoc_out("Reading package-level file ".$this->last->package . '.html');
						flush();
					}
					$fp = fopen(dirname($this->data->parent->getPath()) . SMART_PATH_DELIMITER . $this->last->package . '.html',"r");
					$ret = fread($fp,filesize(dirname($this->data->parent->getPath()) . SMART_PATH_DELIMITER . $this->last->package . '.html'));
					fclose($fp);
					unset($fp);
					if ($this->quietMode === false)
					{
						phpdoc_out(" -- Parsing File\n");
						flush();
					}
					$pageParser = new ppageParser;
					$tempp = $this->package;
					$lp = $this->last;
					$this->package = $this->last->package;
					$pageParser->subscribe('*',$this);
					$pageParser->parse($ret,$this->last->package);
					$this->package = $tempp;
					$this->last = $lp;
					unset($tempp);
					unset($pageParser);
				}
			}
		}
		$this->last = false;
	}
	
	/**
	* handles post-parsing of procedural pages
	*
	* this event is called at the start of a new page, before the Parser knows whether the
	* page will contain any procedural pages or not
	* @param integer $event Event number from {@link Parser.inc}
	* @param parserPage $data
	*/
	function handlePage($event,$data)
	{
		$type = 'page';
		$this->private_page = false;
		$this->data = new parserData;
		$this->package = $GLOBALS['phpDocumentor_DefaultPackageName'];
		$this->subpackage = '';
		$this->proceduralpages->addPage($data);
		$this->data->setParent($data);
		$this->pages[$data->getPath()] = $this->data;
		$this->classes->nextFile($data->getPath());
		$this->packageoutput = $data->getPackageOutput();
	}
	
	/**
	* handles post-parsing of DocBlocks
	*
	* This function sets {@link $last} to the DocBlock represented by $data, to allow the next documentable element passed
	* to IntermediateParser to link the DocBlock into its $docblock property.  This function also checks for two special
	* cases of DocBlocks:
	* <ol>
	*	<li>First DocBlock in the file contains a @package tag</li>
	*	<li>First DocBlock in the file is immediately followed by another DocBlock</li>
	* </ol>
	* In the first case, the function extracts this tag and uses it as the page-level package.  This is only here to support
	* the erratic use of @package in PEAR.  If the @package tag is in the DocBlock of an element (function, global variable,
	* whatever) that isn't a page-level DocBlock, a warning will be raised to notify the author that a @package tag
	* belongs in a page-level DocBlock.
	*
	* In the second case, the function recognizes the DocBlock as a page-level DocBlock and processes it as such.  First,
	* It checks for an @ignore tag, and if found, calls {@link ProceduralPages::ignorePage()}.  An @ignore tag in a page-level
	* DocBlock will ignore all functions, defines, global variables, and includes.  It will not ignore classes!  The function
	* next checks for an @access private, and if --parseprivate is off, performs the same actions as @ignore, raising a
	* warning for the unsuspecting user.  Next, it checks for the @name tag, which is used to rename the page.  This is also
	* a PEAR compatibility issue, and may not be very useful in the long run.  Documentation is best when it refers to real
	* entities in the package, and not to aliases.
	*
	* Finally, the function replaces the old parserPage in {@link parserData::$data}->parent with the new one containing
	* information from the DocBlock by calling {@link addPage()}, and checks for package-level docs.
	* @param integer $event Event number from {@link Parser.inc}
	* @param parserDocBlock $data
	*/
	function handleDocBlock($event,$data)
	{
		$type = 'docblock';
		// 1st docblock in file, check for @package
		if ($this->data->isClean())
		{
            if ($data->getExplicitPackage())
            {
    			$this->package = $this->data->parent->package = $data->package;
    			$this->subpackage = $this->data->parent->subpackage = $data->subpackage;
            }
			$doc = new parserDocBlock;
			$doc->package = $this->package;
			$doc->subpackage = $this->subpackage;
			$this->data->setDocBlock($doc);
			$this->proceduralpages->addPagePackage($this->data->parent->getPath(),$this->package,$this->subpackage);
			unset($doc);
		}
		// 2nd docblock in a row, and it's at the top of the file, page-level docblock
		if ($this->lasttype == "docblock" && $this->data->isClean())
		{
			// can only have 1 package-level docblock, others are ignored
			$this->data->clean = false;
			if ($this->last->getKeyword('ignore'))
			{
				$this->proceduralpages->ignorePage($this->data->parent);
				$this->private_page = true;
				unset($this->last);
				$this->privatepages[$this->data->parent->getPath()] = $this->data;
				unset($this->pages[$this->data->parent->getPath()]);
				return;
			}
			$this->data->setDocBlock($this->last);
			$this->package = $this->data->parent->package = $this->last->package;
			$this->subpackage = $this->data->parent->subpackage = $this->last->subpackage;
			$this->proceduralpages->addPagePackage($this->data->parent->getPath(),$this->package,$this->subpackage);
			if ($access = $this->last->getKeyword('access'))
			{
				if (($access->getString() == 'private') && (!$this->parsePrivate))
				{
					addWarning(PDERROR_PARSEPRIVATE, $this->data->parent->getPath());
					$this->proceduralpages->ignorePage($this->data->parent);
					$this->private_page = true;
					unset($this->last);
					$this->privatepages[$this->data->parent->getPath()] = $this->data;
					unset($this->pages[$this->data->parent->getPath()]);
					return;
				}
			}
			if ($this->last->getKeyword('name'))
			{
				$a = $this->last->getKeyword('name');
				$this->data->parent->setFile($a->value);
				$this->proceduralpages->setName($a->value);
			}
			$this->addPage($this->data->parent, $this->data->parent->getPath());
			if ($this->package)
			{
				if (!isset($this->package_pages[$this->package]))
				{
					if (file_exists(dirname($this->data->parent->getPath()) . SMART_PATH_DELIMITER . $this->package . '.html'))
					{
						if ($this->quietMode === false)
						{
							phpdoc_out("Reading package-level file ".$this->package . '.html');
							flush();
						}
						$fp = fopen(dirname($this->data->parent->getPath()) . SMART_PATH_DELIMITER . $this->package . '.html',"r");
						$ret = fread($fp,filesize(dirname($this->data->parent->getPath()) . SMART_PATH_DELIMITER . $this->package . '.html'));
						fclose($fp);
						unset($fp);
						if ($this->quietMode === false)
						{
							phpdoc_out(" -- Parsing File\n");
							flush();
						}
						$pageParser = new ppageParser;
						$tempp = $this->package;
						$lp = $this->last;
						$this->package = $this->package;
						$pageParser->subscribe('*',$this);
						$pageParser->parse($ret,$this->package);
						$this->package = $tempp;
						$this->last = $lp;
						unset($tempp);
						unset($pageParser);
					}
				}
			}
		}
		$this->all_packages[$data->package] = 1;
		$this->last = $data;
	}
	
	/**
	* called via {@link Parser::parse()} and Parser's inherited method {@link Publisher::publishEvent()}
	*
	* $event is one of the PHPDOC constants from Parser.inc.  If it is not PHPDOC_EVENT_NEWSTATE, then a function name
	* is retrieved from the {@link $event_handlers} array and called to handle the $data
	* @param integer $event event number from {@link Parser.inc}
	* @param mixed $data if $event is {@link PHPDOC_EVENT_NEWSTATE}, $data is a {@link PHP_DOC_EVENT_END_PAGE} or {@link STATE_END_CLASS},
	*                    otherwise $data is either a {@link parserDocBlock}, {@link parserPage} or descendant of {@link parserElement}
	*/
	function HandleEvent($event,$data)
	{
		if ($event == PHPDOC_EVENT_NEWSTATE)
		{
			if ($data == STATE_END_CLASS)
			{
			} elseif ($data == PHPDOC_EVENT_END_PAGE)
			{
				if (!$this->private_page)
				{
					$this->pages[$this->data->parent->getPath()] = $this->data;
					$this->all_packages[$this->package] = 1;
				}
				$this->private_page = false;
				$this->private_class = false;
			}
			//echo $this->state_lookup[$data] . "\n";
			//echo $data."\n";
		} 
		 else 
		{
			$this->lasttype = $this->type;
			$type = $data->getType();
//			fancy_debug($type,$data);
			if (($type != 'page') && ($type != 'docblock') && ($type != 'packagepage'))
			{
				$data->setFile($this->data->parent->getFile());
			}
			$this->type = $type;
			//echo $type . "\n";
			
			if (isset($this->event_handlers[$type]))
			{
				$handle = $this->event_handlers[$type];
				$this->$handle($event,$data);
			}
		}
	}
	
	/**
	* Replaces the {@link parserPage} represented by $this->pages[$path] with $page
	*
	* Called by {@link addPageIfNecessary(), handleDocBlock()} and {@link ProceduralPages::setupPages()}, this method
	* first checks to see if the page has been added.  If not, it assumes that the page has either been @ignored or
	* set with @access private with --parseprivate off, and returns {@link addPrivatePage()}.  Otherwise, it sets
	* the pages[$path] to be the parserPage $page and sets the package and subpackage to that of $page
	* @see $pages
	*/
	function addPage($page, $path)
	{
		if (!isset($this->pages[$path])) return $this->addPrivatePage($page, $path);
		$this->pages[$path]->setParent($page);
		if ($page->package != $GLOBALS['phpDocumentor_DefaultPackageName'])
		{
			if (!$this->pages[$path]->docblock)
			{
				$docblock = new parserDocBlock;
				$docblock->package = $page->package;
				$docblock->subpackage = $page->subpackage;
				$this->pages[$path]->docblock = $docblock;
			} else
			{
				$this->pages[$path]->docblock->package = $page->package;
				$this->pages[$path]->docblock->subpackage = $page->subpackage;
			}
		}
	}
	
	/**
	* add a new {@link parserPage} to the $pages array if none is found
	*
	* This method is used when a page has been @ignored or marked with @access private, and a public class is
	* in the page (a class with no @access private in its DocBlock).  The method first creates a new page in the {@link $pages}
	* array and then copies path information, and calls {@link addPage()} to set up packages
	*/
	function addPageIfNecessary($path)
	{
		if (!$this->parsePrivate)
		{
			if (!isset($this->pages[$path]))
			{
				$this->pages[$path] = new parserData;
				$this->pages[$path]->docblock = new parserDocBlock;
				$this->pages[$path]->docblock->package = $this->privatepages[$path]->docblock->package;
				$this->pages[$path]->docblock->subpackage = $this->privatepages[$path]->docblock->subpackage;
				$par = $this->privatepages[$path]->parent;
				$this->pages[$path]->setParent($par);
				$this->proceduralpages->addPage($par);
			}
		}
	}
	
	/**
	* Adds a {@link parserPage} element to the {@link parserData} element in this->privatepages[$path]
	*
	* Performs a similar function to addPage, but adds to the {@link $privatePages} array
	* @param parserPage $page
	* @param string $path full path to the page
	* @see addPage()
	*/
	function addPrivatePage($page, $path)
	{
		$this->privatepages[$path]->setParent($page);
		if ($page->package != $GLOBALS['phpDocumentor_DefaultPackageName'])
		{
			if (!$this->privatepages[$path]->docblock)
			{
				$docblock = new parserDocBlock;
				$docblock->package = $page->package;
				$docblock->subpackage = $page->subpackage;
				$this->privatepages[$path]->docblock = $docblock;
			} else
			{
				$this->privatepages[$path]->docblock->package = $page->package;
				$this->privatepages[$path]->docblock->subpackage = $page->subpackage;
			}
		}
	}
	
	/**
	* adds a processed descendant of {@link parserElement} to the {@link $pages} array or {@link $privatepages} array
	*
	* This function expects the page to exist in either $pages or $privatepages, and will raise an exception if
	* exceptions are enabled (by default they are off to offer tremendous speed improvement.  See {@link PHPDOC_EXCEPTIONS}).
	* It calls the {@link parserData::addElement()} method to add $element to the page.
	* @param parserElement $element this will actually be a descendant of parserElement
	* @param string $path
	*/
	function addElementToPage($element, $path)
	{
		if (PHPDOC_EXCEPTIONS)
		{
			addException(!is_object($element),PDEXCEPTION_NOT_A_CLASS,__FILE__,__LINE__,"element",$element,"path",$path);
			addException(!is_subclass_of($element,'parserElement'),PDEXCEPTION_NOT_AN_ELEMENT,__FILE__,__LINE__,"element",$element,"path",$path);
		}
		if (isset($this->privatepages[$path]))
		{
			if (PHPDOC_EXCEPTIONS)
			{
				addException(get_class($this->privatepages[$path]) != 'parserdata',PDEXCEPTION_NOT_A_PARSERDATA,__FILE__,__LINE__,"this->privatepages[path]",$this->privatepages[$path],"path",$path);
			}
			if (isset($this->pages[$path]))
			{
				if ($element->type == 'class' || $element->type == 'method' || $element->type == 'var')
				{
					$this->pages[$path]->addElement($element);
				} else
				$this->privatepages[$path]->addElement($element);
			} else
			$this->privatepages[$path]->addElement($element);
		} else
		{
			if (isset($this->pages[$path]))
			$this->pages[$path]->addElement($element);
			elseif (PHPDOC_EXCEPTIONS)
			{
				addException(true,PDEXCEPTION_NOT_A_CLASS,__FILE__,__LINE__,"this->pages[path]",$this->pages[$path],"path",$path,"this->pages",$this->pages);
			}

		}
	}
	
	/**
	* Interface to the Converter
	*
	* This function simply passes {@link $pages} and {@link package_pages} to the walk() method, and then calls
	* the Output() method.  Note that Output() is not required to do anything, and in fact doesn't in HTMLdefaultConverter.
	* @see Converter::walk()
	* @see Converter::Output()
	*/
	function Convert($title, $converter)
	{
		$converter->walk($this->pages, $this->package_pages);
		$converter->Output($title);
	}
	
	/**
	* @access private
	*/
	function fixClasses()
	{
		$this->classes->Inherit($this);
	}
	
	/**
	* @access private
	*/
	function fixProcPages()
	{
		$this->proceduralpages->setupPages($this);
//		var_dump($this->proceduralpages);
	}
	
	/**
	* If the parent class of $class is in a different package, adds it to the {@link $package_parents} array
	* @param parserClass &$class
	*/
	function addPackageParent(&$class)
	{
		if (!is_array($class->parent)) return;
		$par = $this->classes->getClass($class->parent[1], $class->parent[0]);
		if (PHPDOC_EXCEPTIONS)
		{
			addException(!is_object($par),PDEXCEPTION_NOT_A_CLASS,__FILE__,__LINE__,"class",$class->parent[1],"file",$class->parent[0],"parent of class",$class);
		}
		if ($class->docblock->package == $par->docblock->package) return;
		$this->package_parents[$class->docblock->package] = $par->docblock->package;
		if (!isset($this->package_parents[$par->docblock->package]) || !$this->package_parents[$par->docblock->package]) $this->package_parents[$par->docblock->package] = false;
	}
	
	/**
	* Add a converter name to use to the list of converters
	
	* Sets up the {@link $converters} array
	* @param string $output output format (HTML, PDF, XML).  Must be all caps
	* @param string $name Converter name (default, for example, is the name of HTMLdefaultConverter)
	* @param string $template template to use, should be a relative path to the templates dir (like DOM/default)
	*/
	function addConverter($output,$name,$template)
	{
		if (PHPDOC_EXCEPTIONS)
		include_once('Converters' . PATH_DELIMITER . $output . PATH_DELIMITER . $name . PATH_DELIMITER . $output . $name . "Converter" . ".inc");
		else
		@include_once('Converters' . PATH_DELIMITER . $output . PATH_DELIMITER . $name . PATH_DELIMITER . $output . $name . "Converter" . ".inc");
		if (class_exists($output . $name . 'Converter'))
		{
			$this->converters[$output][$output . $name . "Converter"] = $template;
		} else
		{
			addError(PDERROR_CONVERTER_NOT_FOUND,$name);
		}
	}

	/**
	* does a natural case sort on two {@link parserElement} descendants
	*
	* @param	mixed	$a
	* @param	mixed	$b
	* @return	int
	* @see		generateElementIndex()
	*/
	function elementCmp ($a, $b)
	{
		if (PHPDOC_EXCEPTIONS)
		{
			addException(!is_object($a),PDEXCEPTION_NOT_A_CLASS,__FILE__,__LINE__,"a",$a);
			addException(!is_object($b),PDEXCEPTION_NOT_A_CLASS,__FILE__,__LINE__,"a",$b);
		}
		return strnatcasecmp($a->getName(), $b->getName());
	}
	
	/**
	* does a natural case sort on two class elements (either {@link parserClass, parserMethod} or {@link parserVar}
	*
	* @param	mixed	$a
	* @param	mixed	$b
	* @return	int
	* @see		generateElementIndex()
	*/
	function ClasselementCmp ($a, $b)
	{
		if (PHPDOC_EXCEPTIONS)
		{
			addException(!is_object($a),PDEXCEPTION_NOT_A_CLASS,__FILE__,__LINE__,"a",$a);
			addException(!is_object($b),PDEXCEPTION_NOT_A_CLASS,__FILE__,__LINE__,"a",$b);
		}
		if (get_class($a) == 'parserclass') $atest = $a->name; else $atest = $a->class;
		if (get_class($b) == 'parserclass') $btest = $b->name; else $btest = $b->class;
		
		if(($c = strnatcasecmp($atest, $btest)) != 0) return $c;
		if ((get_class($a) == 'parservar') || (get_class($a) == 'parsermethod')) $atest .= $a->name;
		if ((get_class($b) == 'parservar') || (get_class($b) == 'parsermethod')) $btest .= $b->name;
		return strnatcasecmp($atest,$btest);
	}
	
	/**
	* call this method once parsing has completed.
	*
	* This method calls the private methods fixClasses and fixProcPages, both of which adjust inheritance
	* and package information based on complicated post-parsing rules described in {@link ProceduralPages::setupPages()}
	* and {@link Classes::Inherit()}.  Then, it sorts elements of the $pages array and calls Convert for each
	* Converter in the $converters array
	* @see $converters
	* @see $pages
	* @see Convert()
	*/
	function Output ($title = "Generated Documentation")
	{
		$GLOBALS['phpDocumentor_errors']->curfile = false;
		$this->fixClasses();
		$this->fixProcPages();
		phpdoc_out("\nSorting page elements...");
		flush();
		uasort($this->pages,'pagesort');
		foreach($this->pages as $i => $page)
		{
			usort($this->pages[$i]->elements,array($this,'elementCmp'));
			usort($this->pages[$i]->classelements,array($this,'ClasselementCmp'));
		}
		phpdoc_out("done\n");
		flush();
		if (is_array($this->converters))
		{
			foreach($this->converters as $converter => $blah)
			{
				if (is_array($blah))
				{
					foreach($blah as $converter => $template)
					{
						$a = new $converter($this->all_packages, $this->package_parents, $this->classes, $this->proceduralpages, $this->packageoutput, $this->parsePrivate, $this->quietMode, $this->targetDir, $template, $this->title);
						$this->Convert($title, $a);
					}
				}
			}
		} else
		{
			addErrorDie(PDERROR_NO_CONVERTERS);
		}
	}

	/**
	* Sets the output directory
	*
	* @param string $dir the output directory
	*/
	function setTargetDir($dir)
	{
		$this->targetDir = $dir;
	}

	/**
	 * set parsing information output mode (quiet or verbose)
	 *
	 * If set to false, no parsing information (parsing /php/file/thisfile.php, Converting etc.) will be displayed.
	 * Useful for cron jobs
	 * @param	bool $quietMode
	 */
	function setQuietMode($quietMode) {
	$this->quietMode = $quietMode;
	}
	
	/**
	* set display of elements marked with @access private
	*
	* If set to true, elements will be displayed
	* @param	bool $parse
	*/
	function setParsePrivate($parse)
	{
		$this->parsePrivate = $parse;
	}
}

/** @access private */
function pagesort($a, $b)
{
	return strnatcasecmp($a->parent->file,$b->parent->file);
}
?>
