<?php
/**
* Parser Data Structures
* @package phpDocumentor
* @subpackage ParserData
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: ParserData.inc,v 1.57.2.4 2002/10/31 06:11:12 CelloG Exp $
*/
/**
* @package phpDocumentor
* @subpackage ParserData
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: ParserData.inc,v 1.57.2.4 2002/10/31 06:11:12 CelloG Exp $
*/
class parserPage
{
	/**
	* Type is used by many functions to skip the hassle of if get_class($blah) == 'parserBlah'
	* @var string
	*/
	var $type = 'page';
	/**
	* not implemented in this version, will be used to link xml output pages
	* @var string
	*/
	var $id = '';
	/**
	* filename.ext (no path)
	* @var string
	*/
	var $file = '';
	/**
	* relative source location
	* @var string
	*/
	var $sourceLocation = '';
	/**
	* phpdoc-safe name (only letters, numbers and _)
	* @var string
	*/
	var $name = '';
	/**
	* @var string
	*/
	var $package = 'default';
	/**
	* @var string
	*/
	var $subpackage = '';
	/**
	* @var string
	*/
	var $parserVersion = PHPDOC_VER;
	/**
	* not implemented yet
	* file modification date, will be used for makefiles
	* @var string
	*/
	var $modDate = '';
	/**
	* @var string full path this page represents
	*/
	var $path = '';
	/**
	* Used to limit output, contains contents of --packageoutput commandline.  Does not increase parsing time.  Use --ignore for that
	* @see IntermediateParser::$packageoutput, Converter::$package_output
	* @var mixed either false or an array of packages
	*/
	var $packageOutput = false;
	
	/**
	* sets package to default package
	* @global string default package name
	*/
	function parserPage()
	{
		global $phpDocumentor_DefaultPackageName;
		$this->package = $GLOBALS['phpDocumentor_DefaultPackageName'];
	}
	
	/**
	* @return string always "page"
	*/
	function getType()
	{
		return 'page';
	}
	
	/**
	* Sets the name to display in documentation (can be an alias set with @name)
	* @param string $file
	*/
	function setFile($file)
	{
		$this->file = $file;
	}
	
	/**
	* @return string filename.ext or @name alias
	*/
	function getFile()
	{
		if (!isset($this->file)) return false;
		return $this->file;
	}
	
	/**
	* @param string $path full path to file
	*/
	function setPath($path)
	{
		// look for special windows case
		if(SMART_PATH_DELIMITER === '\\')
			$this->path = strtr($path,'/','\\');
		else
			$this->path = $path;
	}
	
	/**
	* @return string fully delimited path (OS-dependent format)
	*/
	function getPath()
	{
		if (!isset($this->path)) return false;
		return $this->path;
	}
	
	/**
	* @param array $packages array of packages to display in documentation (package1,package2,...)
	* @see IntermediateParser::$packageoutput
	*/
	function setPackageOutput($packages)
	{
		$this->packageOutput = $packages;
	}
	
	/**
	* @return array array of packages (package1,package2,...)
	* @see IntermediateParser::$packageoutput
	*/
	function getPackageOutput()
	{
		return $this->packageOutput;
	}
	
	/**
	* @param string $name phpdoc-safe name (only _, numbers and letters) set by Parser::parse()
	* @see Parser::parse()
	*/
	function setName($name)
	{
		$this->name = $name;
	}
	
	/**
	* @return string phpdoc-safe name (only _, numbers and letters)
	*/
	function getName()
	{
		if (!isset($this->name)) return false;
		return $this->name;
	}
	
	/**
	* @param string $source path of this file relative to program root
	*/
	function setSourceLocation($source)
	{
		$this->sourceLocation = $source;
	}
	
	/**
	* @return string path of this file relative to program root
	*/
	function getSourceLocation ()
	{
		if (!isset($this->sourceLocation)) return false;
		return $this->sourceLocation;
	}
	/**
	* Not implemented in this version
	* @return boolean tell the parser whether to parse the file, otherwise
	* 				  this function will retrieve the parsed data from external file
	*/
	function getParseData()
	{
		return true;
	}
}

/**
* Contains an in-memory representation of all documentable elements ({@link parserPage}, {@link parserFunction}, {@link parserDefine},
* {@link parserInclude}, {@link parserClass}, {@link parserMethod}, {@link parserVar}) and their DocBlocks ({@link parserDocBlock}).
*
* This class works in coordination with {@link IntermediateParser} to take output from {@link Parser::handleEvent()} and create indexes,
* links, and other assorted things (all documented in IntermediateParser and {@link Converter})
* @package phpDocumentor
* @subpackage ParserData
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: ParserData.inc,v 1.57.2.4 2002/10/31 06:11:12 CelloG Exp $
*/
class parserData
{
	/**
	* {@link parserPage} element that is this parserData's parent, or false if not set
	* @var mixed
	*/
	var $parent = false;
	/**
	* array of parsed elements
	* @var array
	*/
	var $elements = array();
	/**
	* array of parsed elements with @access private
	* @var array
	*/
	var $privateelements = array();
	/**
	* array of parsed class elements
	* @var array
	*/
	var $classelements = array();
	/**
	* array of parsed class elements with @access private
	* @var array
	*/
	var $privateclasselements = array();
	/**
	* array of links descended from {@link abstractLink}
	* @var array
	* @see pageLink, defineLink, classLink, functionLink, methodLink, varLink
	*/
	var $links = array();
	/**
	* used by {@link IntermediateParser::handleDocBlock()} to determine whether a docblock is a page-level docblock or not.
	* $clean is true as long as only 0 or 1 docblock has been parsed, and no element other than parserPage has been parsed
	* @var bool
	*/
	var $clean = true;
	/**
	* DocBlock ({@link parserDocBlock}) for this page, or false if not set
	* @var mixed
	*/
	var $docblock = false;
	/**
	* Type is used by many functions to skip the hassle of if get_class($blah) == 'parserBlah'
	* always 'page', used in element indexing and conversion functions found in {@link Converter}
	* @var string
	*/
	var $type = 'page';
	
	/**
	* @param parserElement add a parsed element to the {@link $elements} array, also sets {@link $clean} to false
	*/
	function addElement(&$element)
	{
		$element->setPath($this->parent->path);
		if ($element->getType() == 'class' || $element->getType() == 'method' || $element->getType() == 'var')
		{
			$this->classelements[] = $element;
		} else
		{
			$this->elements[] = $element;
		}
		$this->clean = false;
	}
	
	/**
	* @param parserElement element to add a new link (descended from {@link abstractLink})to the {@link $links} array
	* @param string classname for elements that are class-based (this may be deprecated in the future, as the classname
	*               should be contained within the element.  if $element is a page, this parameter is a package name
	* @param string subpackage name for page elements
	*/
	function addLink(&$element,$classorpackage = '', $subpackage = '')
	{
		switch($element->type)
		{
			case 'function':
				$x = new functionLink;
				$x->addLink($this->parent->path, $this->parent->name, $element->name, $element->docblock->package, $element->docblock->subpackage);
				return $x;
			break;
			case 'define':
				$x = new defineLink;
				$x->addLink($this->parent->path, $this->parent->name, $element->name, $element->docblock->package, $element->docblock->subpackage);
				return $x;
			break;
			case 'global':
				$x = new globalLink;
				$x->addLink($this->parent->path, $this->parent->name, $element->name, $element->docblock->package, $element->docblock->subpackage);
				return $x;
			break;
			case 'class':
				$x = new classLink;
				$x->addLink($this->parent->path, $this->parent->name, $element->name, $element->docblock->package, $element->docblock->subpackage);
				return $x;
			break;
			case 'method':
				$x = new methodLink;
				$x->addLink($classorpackage, $this->parent->path, $this->parent->name, $element->name, $element->docblock->package, $element->docblock->subpackage);
				return $x;
			break;
			case 'var':
				$x = new varLink;
				$x->addLink($classorpackage, $this->parent->path, $this->parent->name, $element->name, $element->docblock->package, $element->docblock->subpackage);
				return $x;
			break;
			case 'page':
				if (empty($classorpackage)) $classorpackage = $GLOBALS['phpDocumentor_DefaultPackageName'];
				$x = new pageLink;
				$x->addLink($element->path,$element->name,$element->file,$classorpackage, $subpackage);
				return $x;
			break;
		}
	}
	
	/**
	* returns a list of all classes declared in a file
	* @param Converter &$c
	* @return array Format: array(packagename => parserClass,packagename => parserClass,...)
	*/
	function getClasses(&$c)
	{
		$r = $c->classes->getClassesInPath($this->parent->path);
		$rr = array();
		if ($r)
		foreach($r as $class => $obj)
		{
			$rr[$obj->docblock->package][] = $obj;
		}
		return $rr;
	}
	
	/**
	* @param parserPage parent element of this parsed data
	*/
	function setParent(&$parent)
	{
		$this->parent = $parent;
	}
	
	/**
	* @return bool returns the value of {@link $clean}
	*/
	function isClean()
	{
		return $this->clean;
	}
	
	/**
	* @param parserDocBlock
	* @see parserDocBlock
	*/
	function setDocBlock(&$docblock)
	{
		$this->docblock = $docblock;
	}
}

/**
* base class for all elements
* @package phpDocumentor
* @subpackage ParserData
* @abstract
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: ParserData.inc,v 1.57.2.4 2002/10/31 06:11:12 CelloG Exp $
*/
class parserBase
{
	/**
	* Type is used by many functions to skip the hassle of if get_class($blah) == 'parserBlah'
	* always base
	* @var string
	*/
	var $type = 'base';
	/**
	* set to different things by its descendants
	* @abstract
	* @var mixed
	*/
	var $value = false;
	
	/**
	* line number on file where this thing is found
	*/
	var $linenum = 0;
	
	function setLineNum($l)
	{
		$this->linenum = $l;
	}
	
	/**
	* @return string returns value of {@link $type}
	*/
	function getType()
	{
		return $this->type;
	}
	
	/**
	* @param mixed set the value of this element
	*/
	function setValue($value)
	{
		$this->value = $value;
	}
	
	/**
	* @return mixed get the value of this element (element-dependent)
	*/
	function getValue()
	{
		return $this->value;
	}
}

/**
* Use this element to represent an {@inline tag} like link (see the phpDocumentor spec for more information about inline tags)
* @see parserStringWithInlineTags
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: ParserData.inc,v 1.57.2.4 2002/10/31 06:11:12 CelloG Exp $
*/
class parserInlineTag extends parserBase
{
	/**
	* Type is used by many functions to skip the hassle of if get_class($blah) == 'parserBlah'
	* always 'inlinetag'
	* @var string
	*/
	var $type = 'inlinetag';
	/**
	* the name of the inline tag (like link)
	* @var string
	*/
	var $inlinetype = '';
	
	/**
	* @param string $type tag type (example: link)
	* @param string $value tag value (example: what to link to)
	*/
	function parserInlineTag($type,$value)
	{
		$this->inlinetype = $type;
		$this->value = trim($value);
	}
	
	/**
	* @return integer length of the tag
	*/
	function Strlen()
	{
		return strlen($this->value);
	}
	
	/**
	* @return string always '', used by {@link Parser::handleDocBlock()} to calculate the short description of a DocBlock
	* @see parserStringWithInlineTags::getString()
	* @see parserStringWithInlineTags::trimmedStrlen()
	*/
	function getString()
	{
		return '';
	}
}

/**
* represents inline links
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: ParserData.inc,v 1.57.2.4 2002/10/31 06:11:12 CelloG Exp $
*/
class parserLinkInlineTag extends parserInlineTag
{
	/**
	* text to display in the link, can be different from the link for standard links like websites
	* @var string
	*/
	var $linktext = '';
	
	/**
	* @param string $link stored in $value, see {@link parserBase::$value}
	* @param string $text see {@link $linktext}
	*/
	function parserLinkInlineTag($link,$text)
	{
		parserInlineTag::parserInlineTag('link',$link);
		$this->linktext = trim($text);
	}
	
	/**
	* @param Converter converter used to change the abstract link into text for display
	* @return mixed returns the converted link or false if not converted successfully
	*/
	function Convert(&$c)
	{
		if (strpos($this->value,'://') || (strpos($this->value,'mailto:') === 0))
		{
			return $c->returnLink($this->value,$this->linktext);
		} else
		{
			$value = $c->getLink($this->value);
			if (is_string($value)) return $value;
			if (is_object($value)) return $c->returnSee($value,$this->linktext);
			// getLink parsed a comma-delimited list of linked thingies, add the commas back in
			if (is_array($value))
			{
				$a = '';
				foreach($value as $i => $bub)
				{
					if (!empty($a)) $a .= ', ';
					if (is_string($value[$i])) $a .= $value[$i];
					if (is_object($value[$i])) $a .= $c->returnSee($value[$i]);
				}
				return $a;
			}
			return false;
		}
	}
}

/**
* represents inline source tag
*/
class parserSourceInlineTag extends parserInlineTag
{
	var $inlinetype = 'source';
	var $value;
	var $start = 0;
	var $end = '*';
	var $source = false;
	function parserSourceInlineTag($value)
	{
		parserInlineTag::parserInlineTag('source','');
		preg_match('/^([0-9]*)\W([0-9]*)$/',trim($value), $match);
		if (!count($match))
		{
			preg_match('/^([0-9]*)$/',trim($value),$match);
			if (count($match))
			{
				$this->start = (int) $match[1];
			}
		} else
		{
			$this->start = (int) $match[1];
			$this->end = (int) $match[2];
		}
	}
	
	/**
	* only used to determine blank lines.  { @source } will not be blank, probably
	*/
	function Strlen()
	{
		return 1;
	}
	
	function setSource($source)
	{
		$source = strstr($source,'function');
		$pos = strrpos($source,'}');
		$this->source = substr($source,0,$pos + 1);
	}
	
	function Convert(&$c)
	{
		$source = highlight_string('<?php '.$this->source.' ?>', true);
		$source = '<code>'.substr($source,strlen('<code><font color="#000000">
<font color="#0000CC">&lt;?php&nbsp;</font>') - 1);
		$source = str_replace('}&nbsp;</font><font color="#0000CC">?&gt;</font>','}</font></code>',$source);
		if ($this->start || ($this->end != '*'))
		{
			$source = explode('<br />',$source);
			$start = $this->start;
			if ($this->end != '*')
			{
				$source = array_slice($source,$start - 1,$this->end - $start + 1);
			} else
			{
				$source = array_slice($source,$start);
			}
			$source = implode($source,'<br />');
			if ($start > 0) $source = "<code>$source";
			if ($this->end != '*') $source = "$source</code>";
		}
		$source = $c->unmangle($source);
		return $source;
	}
}

/**
* Used to represent strings that contain inline tags, so that they can be properly parsed at link time
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: ParserData.inc,v 1.57.2.4 2002/10/31 06:11:12 CelloG Exp $
*/
class parserStringWithInlineTags extends parserBase
{
	/**
	* Type is used by many functions to skip the hassle of if get_class($blah) == 'parserBlah'
	* always '_string'
	* @var string
	*/
	var $type = '_string';
	var $cache = false;
	/**
	* array of strings and {@link parserInlineTag}s
	* Format:
	* array(string1,string2,parserInlineTag1,string3,parserInlineTag2,...)
	* @var array
	*/
	var $value = array();
	/**
	* find the inline tags (xml format), parse them out as new objects, set up {@link $value}
	* @param string $value any string that may contain inline tags
	*/
	function parserStringWithInlineTags($value='')
	{
		if (empty($value)) return;
		$tempvalue = explode('<inlinetag',$value);
		$parsedvalue = array($tempvalue[0]);
		for($i=1;$i<count($tempvalue);$i++)
		{
			$a = strpos($tempvalue[$i],'>');
			$parsedvalue[] = new ParsedInlineTag('<inlinetag'.substr($tempvalue[$i],0,$a));
			$parsedvalue[] = substr($tempvalue[$i],$a);
		}
		$this->value = $parsedvalue;
	}

	/**
	* equivalent to the . operator ($a = $b . $c)
	* @param mixed either a string or a {@link parserInlineTag}
	*/
	function add($stringOrInlineTag)
	{
		if (is_string($stringOrInlineTag))
		{
			if (!count($this->value))
			{
				$this->value[] = $stringOrInlineTag;
				return;
			}
			if (is_string($this->value[count($this->value) - 1]))
			{
				$this->value[count($this->value) - 1] .= $stringOrInlineTag;
				return;
			} else
			{
				$this->value[] = $stringOrInlineTag;
				return;
			}
		} else
		{
			$this->value[] = $stringOrInlineTag;
		}
	}
	
	function setSource($source)
	{
		for($i=0;$i<count($this->value);$i++)
		{
			if (get_class($this->value[$i]) == 'parsersourceinlinetag')
			{
				$this->value[$i]->setSource($source);
			}
		}
	}

	/**
	* equivalent to trim(strlen($string))
	* @return integer length of the string this object represents
	*/
	function trimmedStrlen()
	{
		$a = 0;
		for($i=0;$i<count($this->value);$i++)
		{
			if (is_string($this->value[$i]))
			{
				if ($i == 0)
				{
					$a += strlen(ltrim($this->value[$i]));
				} elseif ($i == count($this->value[$i]) - 1)
				{
					$a += strlen(chop($this->value[$i]));
				}
			} else
			{
				$a += $this->value[$i]->Strlen();
			}
		}
		return $a;
	}
	
	/**
	* return the string unconverted (all inline tags are taken out - this should only be used in pre-parsing to see if any other text
	* is in the string)
	* @return string trimmed value
	*/
	function getString()
	{
		$a = '';
		for($i=0; $i<count($this->value); $i++)
		{
			if (is_string($this->value[$i]))
			{
				$a .= $this->value[$i];
			} else
			{
				$a .= $this->value[$i]->getString();
			}
		}
		return trim($a);
	}
	
	/**
	* Use to convert the string to a real string with all inline tags parsed and linked
	* @see Converter::returnSee()
	* @param Converter
	*/
	function Convert(&$converter)
	{
		if ($this->cache)
		{
			if ($converter->name == $this->cache['name'] && $converter->outputformat == $this->cache['output'] && $converter->checkState($this->cache['state'])) return $this->cache['contents'];
		}
		if (is_string($this->value)) return $this->value;
		$a = '';
		for($i=0; $i<count($this->value); $i++)
		{
			if (is_string($this->value[$i]))
			{
				$a .= $this->value[$i];
			} else
			{
				$a .= $this->value[$i]->Convert($converter);
				flush();
			}
		}
		$this->cache = array('name' => $converter->name,'output' => $converter->outputformat, 'contents' => trim($a), 'state' => $converter->getState());
		return trim($a);
	}
}

?>