<?php
/**
* DocBlock Parser Classes
* @package phpDocumentor
* @subpackage ParserDocBlock
*/
/**
* represents a short or long description in a DocBlock ({@link parserDocBlock})
* @package phpDocumentor
* @subpackage ParserDocBlock
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: ParserDocBlock.inc,v 1.9.2.2 2002/11/07 20:46:00 CelloG Exp $
*/
class parserDesc extends parserStringWithInlineTags
{
	/**
	* Type is used by many functions to skip the hassle of if get_class($blah) == 'parserBlah'
	* always '_desc'
	* @var string
	*/
	var $type = '_desc';
	
	/**
	* @param mixed like {@link parserStringWithInlineTags::add()}, this can be a string or parserInlineTag, but it can also be a
	*              parserStringWithInlineTags, and the contents will be merged
	*/
	function add($stringOrClass)
	{
		if (is_object($stringOrClass))
		{
			if (get_class($stringOrClass) == 'parserstringwithinlinetags' || get_class($stringOrClass) == 'parserStringWithInlineTags' || is_subclass_of($stringOrClass,'parserStringWithInlineTags') || is_subclass_of($stringOrClass,'parserstringwithinlinetags'))
			{
				for($i=0;$i<count($stringOrClass->value);$i++)
				{
					parserStringWithInlineTags::add($stringOrClass->value[$i]);
				}
			} else
			{
				parserStringWithInlineTags::add($stringOrClass);
			}
		} else return parserStringWithInlineTags::add($stringOrClass);
	}
}

/**
* used to represent standard tags like @access, etc.
* This class is aware of inline tags, and will automatically handle them using inherited functions
* @package phpDocumentor
* @subpackage ParserDocBlock
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: ParserDocBlock.inc,v 1.9.2.2 2002/11/07 20:46:00 CelloG Exp $
*/
class parserTag extends parserStringWithInlineTags
{
	/**
	* Type is used by many functions to skip the hassle of if get_class($blah) == 'parserBlah'
	* always '_tag'
	* @var string
	*/
	var $type = '_tag';
	/**
	* tag name (see, access, etc.)
	* @var string
	*/
	var $keyword = '';
	
	/**
	* @param string $keyword tag name
	* @param parserStringWithInlineTags $value
	*/
	function parserTag($keyword, $value)
	{
		$this->keyword = $keyword;
		$this->value = $value;
	}
	
	/**
	* @param Converter
	* @see Converter
	*/
	function Convert(&$converter)
	{
		return $this->value->Convert($converter);
	}
	
	/**
	* @return string returns the text minus any inline tags
	* @see parserStringWithInlineTags::getString()
	*/
	function getString()
	{
		return $this->value->getString();
	}
}

class parserNameTag extends parserTag
{
	/**
	* tag name
	* @var string
	*/
	var $keyword = 'name';
	
	/**
	* @see parserStringWithInlineTags::Convert()
	* @param Converter
	* @return string converted value of the tag
	*/
	function Convert(&$c)
	{
		return $this->value;
	}
}

class parserAccessTag extends parserTag
{
	/**
	* tag name
	* @var string
	*/
	var $keyword = 'access';
	
	/**
	* set to true if the returned tag has a value type of private or public, false otherwise
	* @var boolean
	*/
	var $isvalid = false;
	
	/**
	* checks $value to make sure it is private or public, otherwise it's not a valid @access tag
	* @see $isvalid
	* @param parserStringWithInlineTags $value
	*/
	function parserAccessTag($value)
	{
		if (!is_string($value))
		{
			if (is_object($value))
			{
				if (method_exists($value,'getstring'))
				{
					$value = $value->getString();
				}
			}
		}
		switch(trim($value))
		{
			case 'private' :
			case 'public' :
				$this->value = $value;
				$this->isvalid = true;
			break;
			default :
			addError(PDERROR_ACCESS_WRONG_PARAM,$value);
				$this->value = 'public';
			break;
		}
	}
	
	/**
	* @see parserStringWithInlineTags::Convert()
	* @param Converter
	* @return string converted value of the tag
	*/
	function Convert(&$converter)
	{
		return $this->value;
	}
	
	/**
	* No inline tags are possible, returns 'public' or 'private'
	* @return string returns the text minus any inline tags
	*/
	function getString()
	{
		return $this->value;
	}
}

/**
* represents "@return"
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: ParserDocBlock.inc,v 1.9.2.2 2002/11/07 20:46:00 CelloG Exp $
*/
class parserReturnTag extends parserTag
{
	/**
	* always 'return'
	* @var string
	*/
	var $keyword = 'return';
	/**
	* the type a function returns
	*/
	var $returnType = 'void';
	
	/**
	* contains a link to the documentation for a class passed as a type in @return, @var or @param
	* Example:
	* <code>
	* class myclass
	* {
	* ...
	* }
	* /** @return myclass blahblahblah
	* ...
	* </code>
	* In this case, $converted_returnType will contain a link to myclass instead of the string 'myclass'
	* @var mixed either the same as $returnType or a link to the docs for a class
	* @see $returnType
	*/
	var $converted_returnType = false;
	
	/**
	* @param string
	* @param parserStringWithInlineTags
	*/
	function parserReturnTag($returnType, $value)
	{
		$this->returnType = $returnType;
		$this->value = $value;
	}
	
	/**
	* sets up $converted_returnType
	* @see parserStringWithInlineTags::Convert(), $converted_returnType
	* @param Converter
	* @return string converted value of the tag
	*/
	function Convert(&$converter)
	{
		$a = $converter->getLink($this->returnType);
		if (is_object($a) && get_class($a) == 'classlink')
		{
			$this->converted_returnType = $converter->returnSee($a);
		} else
		{
			$this->converted_returnType = $this->returnType;
		}
		return parserTag::Convert($converter);
	}
}

/**
* represents "@var"
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: ParserDocBlock.inc,v 1.9.2.2 2002/11/07 20:46:00 CelloG Exp $
*/
class parserVarTag extends parserReturnTag
{
	/**
	* always 'var'
	* @var string
	*/
	var $keyword = 'var';
	/**
	* the type a var has
	* @var string
	*/
	var $returnType = 'mixed';
}

class parserParamTag extends parserVarTag
{
	/**
	* always 'param'
	* @var string
	*/
	var $keyword = 'param';
}

class parserStaticvarTag extends parserParamTag
{
	/**
	* always 'staticvar'
	* @var string
	*/
	var $keyword = 'staticvar';
}

/**
* represents "@link"
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: ParserDocBlock.inc,v 1.9.2.2 2002/11/07 20:46:00 CelloG Exp $
*/
class parserLinkTag extends parserTag
{
	/**
	* always 'link'
	* @var string
	*/
	var $keyword = 'link';
	
	/**
	* URL to link to
	* @param string $link
	*/
	function parserLinkTag($link)
	{
		$this->value = $link;
	}
}

/**
* represents "@see"
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: ParserDocBlock.inc,v 1.9.2.2 2002/11/07 20:46:00 CelloG Exp $
*/
class parserSeeTag extends parserLinkTag
{
	/**
	* always 'see'
	* @var string
	*/
	var $keyword = 'see';

	/**
	* @param Converter
	* @see parserStringWithInlineTags::Convert()
	*/
	function Convert(&$converter)
	{
		$a = $converter->getLink(trim($this->value->Convert($converter)));
		if (is_string($a)) return $a;
		if (is_object($a)) return $converter->returnSee($a);
		// getLink parsed a comma-delimited list of linked thingies, add the commas back in
		if (is_array($a))
		{
			$b = '';
			foreach($a as $i => $bub)
			{
				if (!empty($b)) $b .= ', ';
				if (is_string($a[$i])) $b .= $a[$i];
				if (is_object($a[$i])) $b .= $converter->returnSee($a[$i]);
			}
			return $b;
		}
		return false;
	}
}

/**
* Represents a docblock and its components, {@link $desc}, {@link $sdesc}, {@link $tags}, and also {@link $params} for functions
* @package phpDocumentor
* @subpackage ParserDocBlock
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: ParserDocBlock.inc,v 1.9.2.2 2002/11/07 20:46:00 CelloG Exp $
*/
class parserDocBlock
{
	/**
	* @var parserDesc
	*/
	var $desc = false;
	/**
	* @var parserDesc
	*/
	var $sdesc = false;
	/**
	* array of {@link parserTag}s
	* @var array
	*/
	var $tags = array();
	/**
	* array of unrecognized {@link parserTag}s
	* @var array
	*/
	var $unknown_tags = array();
	/**
	* array of param data.
	* Format:
	* array(index of param in function parameter list -OR- parameter name =>
	*         parserStringWithInlineTags,...)
	* @var array
	*/
	var $params = array();
	/**
	* array of global variable data.
	* Format:
	* array(index of global variable in @global tag list -OR- global variable name =>
	*         array(datatype,parserStringWithInlineTags),...)
	* @var array
	*/
	var $funcglobals = array();
	
	/**
	* array of static variable data.
	* Format:
	* array(index of static variable in @global tag list -OR- static variable name =>
	*         {@link parserStaticvarTag},...)
	* @var array
	*/
	var $statics = array();
	/**
	* This is either a {@link parserReturnTag} or false if no return tag is present
	* @var mixed
	*/
	var $return = false;
	/**
	* This is either a {@link parserVarTag} or false if no var tag is present
	* @var mixed
	*/
	var $var = false;
    /**
    * fix for bug 591396
    * @var boolean
    */
    var $explicitpackage = false;
	/** @var string */
	var $package = 'default';
	/** @var string */
	var $subpackage = '';
	/**
	* whether this DocBlock has an @access tag
	* @var boolean */
	var $hasaccess = false;
	/**
	* whether this DocBlock has a @name tag
	* @var boolean */
	var $hasname = false;
	/**
	* description of package parsed from @package tag
	* Unused in this version
	* @var string
	*/
	var $packagedescrip = '';
	/**
	* description of subpackage parsed from @package tag
	* Unused in this version
	* @var string
	*/
	var $subpackagedescrip = '';
	
	/**
	* sets package to default
	* @global string default package name
	*/
	function parserDocBlock()
	{
		global $phpDocumentor_DefaultPackageName;
		$this->package = $GLOBALS['phpDocumentor_DefaultPackageName'];
	}
	
	/**
	* @param parserDesc sets {@link $sdesc}
	*/
	function setShortDesc($desc)
	{
		$this->sdesc = $desc;
	}
	
	function setSource($source)
	{
		if ($this->desc) $this->desc->setSource($source);
	}
	
	/**
	* @param parserDesc sets {@link $desc}
	*/
	function setDesc($desc)
	{
		$this->desc = $desc;
	}
	
	/**
	* @param Converter takes {@link $sdesc} and converts it to a string and returns it if present, otherwise returns ''
	* @return string
	*/
	function getSDesc(&$converter)
	{
		if ($this->sdesc) return $this->sdesc->Convert($converter);
		return '';
	}
	
	/**
	* @param Converter takes {@link $desc} and converts it to a string and returns it if present, otherwise returns ''
	* @return string
	*/
	function getDesc(&$converter)
	{
		if ($this->desc) return $this->desc->Convert($converter);
		return '';
	}
	
	/**
	* @param string $paramVar if empty, param is indexed in the order received and set using {@link changeParam()}
	* @param parserStringWithInlineTags $value
	*/
	function addParam($paramVar, $paramType, $value)
	{
		if (empty($paramVar))
		$this->params[] = new parserParamTag($paramType,$value);
		else
		$this->params[$paramVar] = new parserParamTag($paramType,$value);
	}
	
	/**
	* @param integer $index index of parameter in the {@link $params} array
	* @param string $name name of the parameter to set in the $params array
	*/
	function changeParam($index,$name)
	{
		$this->params[$name] = $this->params[$index];
		unset($this->params[$index]);
	}
	
	/**
	* replaces nameless parameters in the {@link $params} array with their names
	* @param array $params Format: array(parameter index => parameter name,...)
	*/
	function updateParams($params)
	{
		for($i=0;$i<count($params);$i++)
		{
			if (isset($this->params[$i]))
			{
				$this->changeParam($i,$params[$i]);
			}
		}
		if (isset($this->tags))
		unset($this->tags['param']);
	}

	/**
	* @param string $keyword tag name
	* @param parserStringWithInlineTags $value the contents of the tag
	*/
	function addKeyword($keyword, $value)
	{
		if ($keyword == 'package' || $keyword == 'subpackage') return $this->addPackage($keyword, $value);
		if ($keyword == 'access') return $this->addAccess($value);
		if ($keyword == 'link') return $this->addLink($value);
		if ($keyword == 'see') return $this->addSee($value);
		if ($keyword == 'name') return $this->addName($value);
		if (!in_array($keyword,$GLOBALS['PHPDOC_TAGS_ALLOWED']))
		$this->addUnknownTag($keyword,$value);
		else
		$this->tags[$keyword][] = new parserTag($keyword, $value);
	}
	
	/**
	* adds an unknown tag to the {@link $unknown_tags} array for use by custom converters
	* @param string tag name
	* @param string tag value
	*/
	function addUnknownTag($keyword, $value)
	{
		addWarning(PDERROR_UNKNOWN_TAG,$keyword);
		$this->unknown_tags[$keyword][] = new parserTag($keyword, $value);
	}
	
	/**
	* set the element's package to the passed values.  Used in {@link IntermediateParser} to align package of
	* elements inside a class or procedural page to the package of the class/procedural page
	* @param string
	* @param string
	* @param string element name
	* @param string element type (include, define, var, method, global, function)
	*/
	function overridePackage($package,$subpackage,$elname,$type)
	{
		if ($this->package != $GLOBALS['phpDocumentor_DefaultPackageName'])
		addError(PDERROR_OVERRIDDEN_PACKAGE_TAGS,$elname,$type,$this->package);
		if (!empty($this->subpackage))
		addError(PDERROR_OVERRIDDEN_SUBPACKAGE_TAGS,$type,$elname,$this->subpackage);
		$this->package = $GLOBALS['phpDocumentor_DefaultPackageName'];
		$this->subpackage = '';
		$this->addPackage('package',$package);
		$this->addPackage('subpackage',$subpackage);
	}
    
    function setExplicitPackage()
    {
        $this->explicitpackage = true;
    }
    
    function getExplicitPackage()
    {
        return $this->explicitpackage;
    }
	
	/**
	* @param string $keyword tag name (either package or subpackage)
	* @param mixed $value either a string or a parserStringWithInlineTags.  Strips all inline tags and use the text as the package
	*/
	function addPackage($keyword, $value)
	{
		if ($keyword == 'package')
		{
			if ($this->package == $GLOBALS['phpDocumentor_DefaultPackageName'])
			{
				if (!is_string($value))
				$value = $value->getString();
				$rest = '';
				$value = explode(' ',$value);
				if (count($value) - 1)
				{
					$rest = $value;
					$value = $value[0];
					unset($rest[0]);
					$rest = implode($rest,' ');
				} else
				{
					$value = explode("\t",$value[0]);
					if (count($value) - 1)
					{
						$rest = $value;
						$value = $value[0];
						unset($rest[0]);
						$rest = implode($rest,"\t");
					} else $value = $value[0];
				}
				preg_match("/^([a-zA-Z0-9\-_\.\[\]]+)$/",$value,$match);
				if (!isset($match[0]))
				{
					// if were a single line and the only bad character is a space then will fix things for them
					preg_match("/^([a-zA-Z0-9\-_\.\[\]\/\\: ]+)$/",$value,$match);
					if (!isset($match[0]))
					{
						addError(PDERROR_ILLEGAL_PACKAGENAME,'package','package',$value);
						$value = $GLOBALS['phpDocumentor_DefaultPackageName'];
					}
					else
					{
						$value = 
							str_replace(array(" ","/","\\",":"),"_",
								trim($value));
					}
				}
				$this->packagedescrip = $this->package = trim($value);
				if (!empty($rest)) $this->packagedescrip = $rest;
			} else
			{
				if (is_string($value))
				addError(PDERROR_MULTIPLE_PACKAGE_TAGS,$value);
				else
				addError(PDERROR_MULTIPLE_PACKAGE_TAGS,$value->getString());
			}
		} else
		{
			if (empty($this->subpackage))
			{
				if (!is_string($value))
				$value = $value->getString();
				$rest = '';
				$value = explode(' ',$value);
				if (count($value) - 1)
				{
					$rest = $value;
					$value = $value[0];
					unset($rest[0]);
					$rest = implode($rest,' ');
				} else
				{
					$value = explode("\t",$value[0]);
					if (count($value) - 1)
					{
						$rest = $value;
						$value = $value[0];
						unset($rest[0]);
						$rest = implode($rest,"\t");
					} else $value = $value[0];
				}
				if (!empty($value))
				{
					preg_match("/^([a-zA-Z0-9\-_\.\[\]]+)$/",$value,$match);
					if (!isset($match[0]))
					{
						// if were a single line and the only bad character is a space then will fix things for them
						preg_match("/^([a-zA-Z0-9\-_\.\[\]\/\\: ]+)$/",$value,$match);
						if (!isset($match[0]))
						{
							addError(PDERROR_ILLEGAL_PACKAGENAME,'subpackage','subpackage',$value);
							$value = '';
						}
						else
						{
							$value = 
								str_replace(array(" ","/","\\",":"),"_",
									trim($value));
						}
					}
				}
				$this->subpackagedescrip = $this->subpackage = trim($value);
				if (!empty($rest)) $this->subpackagedescrip = $rest;
			} else
			{
				if (is_string($value))
				addError(PDERROR_MULTIPLE_SUBPACKAGE_TAGS,$value);
				else
				addError(PDERROR_MULTIPLE_SUBPACKAGE_TAGS,$value->getString());
			}
		}
	}
	
	/**
	* Adds a @name tag to the tag list
	* @param string new name of element
	*/
	function addName($value)
	{
		if (is_object($value)) $value = $value->getString();
		if (!$this->hasname)
		{
			$x = new parserNameTag('name',$value);
			$this->hasname = true;
			$this->tags['name'][] = $x;
		} else
		{
			addError(PDERROR_MULTIPLE_NAME_TAGS,$value);
		}
	}
	
	/**
	* @param string if empty, staticvar is indexed in the order received and set using {@link changeStatic()}
	* @param string data type
	* @param parserStringWithInlineTags
	*/
	function addStaticVar($staticvar, $type, $descrip)
	{
		if (empty($staticvar))
		$this->statics[] = new parserStaticvarTag($type,$descrip);
		else
		$this->statics[$staticvar] = new parserStaticvarTag($type,$descrip);
	}
	
	/**
	* adds a function declaration of @global to the {@link $funcglobals} array
	* @param string global type
	* @param string description of how the global is used in the function
	*/
	function addFuncGlobal($type,$value)
	{
		$this->funcglobals[] = array($type,$value);
	}
	
	/**
	* @param integer $index index of parameter in the {@link $funcglobals} array
	* @param string $name name of the parameter to set in the $funcglobals array
	*/
	function changeGlobal($index,$name)
	{
		$this->funcglobals[$name] = $this->funcglobals[$index];
		unset($this->funcglobals[$index]);
	}

	/**
	* @param integer $index index of parameter in the {@link $statics} array
	* @param string $name name of the parameter to set in the $statics array
	*/
	function changeStatic($index,$name)
	{
		$this->statics[$name] = $this->statics[$index];
		unset($this->statics[$index]);
	}

	/**
	* replaces nameless global variables in the {@link $funcglobals} array with their names
	* @param parserFunction $func
	*/
	function updateGlobals($funcs)
	{
		for($i=0;$i<count($funcs);$i++)
		{
			if (isset($this->funcglobals[$i]))
			{
				$this->changeGlobal($i,$funcs[$i]);
			}
		}
	}

	/**
	* replaces nameless global variables in the {@link $funcglobals} array with their names
	* @param parserFunction $func
	*/
	function updateStatics($funcs)
	{
		for($i=0;$i<count($funcs);$i++)
		{
			if (isset($this->statics[$i]))
			{
				$this->changeStatic($i,$funcs[$i]);
			}
		}
	}

	/**
	* add an @access tag to the {@link tags} array
	* @param string should be either public or private
	*/
	function addAccess($value)
	{
		if (is_object($value)) $value = $value->getString();
		$value = strtolower($value);
		if (!$this->hasaccess)
		{
			$x = new parserAccessTag($value);
			if ($x->isvalid)
			{
				$this->hasaccess = true;
				$this->tags['access'][] = $x;
			}
		} else
		{
			if (is_string($value))
			addError(PDERROR_MULTIPLE_ACCESS_TAGS,$value);
			else
			addError(PDERROR_MULTIPLE_ACCESS_TAGS,$value->getString());
		}
	}
	
	/**
	* creates a {@link parserLinkTag} and adds it to the {@link $tags} array
	* @param string $link
	*/
	function addLink($link)
	{
		$this->tags['link'][] = new parserLinkTag($link);
	}
	
	/**
	* creates a {@link parserLinkTag} and adds it to the {@link $tags} array
	* @param string $value
	*/
	function addSee($value)
	{
		$this->tags['see'][] = new parserSeeTag($value);
	}
	
	/**
	* creates a {@link parserReturnTag} and adds it to the {@link $tags} array
	* @param string $returnType the one-word name of the return type (mixed should be used if more than one type)
	* @param parserStringWithInlineTags $value
	*/
	function addReturn($returnType, $value)
	{
		// only take the first one
		if (!$this->return)
		{
			$this->return = new parserReturnTag($returnType, $value);
		} else
		{
			addError(PDERROR_MULTIPLE_RETURN_TAGS,$returnType,$value->getString());
		}
	}
	
	/**
	* creates a {@link parserVarTag} and adds it to the {@link $tags} array
	* @param string $varType the one-word name of the variable type (mixed should be used if more than one type)
	* @param parserStringWithInlineTags $value
	*/
	function addVar($varType, $value)
	{
		// only take the first one
		if (!$this->var)
		{
			$this->var = new parserVarTag($varType, $value);
		} else
		{
			addError(PDERROR_MULTIPLE_VAR_TAGS,$varType,$value->getString());
		}
	}
	
	/**
	* @param string
	* @return mixed false if no keyword, unconverted value if one keyword, array of unconverted values if more than one keyword
	*/
	function getKeyword($keyword)
	{
		if (isset($this->tags[$keyword]))
		{
			if (count($this->tags[$keyword]) == 1)
			{
				return $this->tags[$keyword][0];
			} else return $this->tags[$keyword];
		} else return false;
	}
	
	/**
	* @return array Format: array('var' => tag name, 'data' => unconverted tag value)
	*/
	function listParams()
	{
		if (isset($this->params))
		{
			$ret = array();
			foreach($this->params as $key => $val)
			{
				$ret[] = array("var" => ucfirst($key),"data" => $val);
			}
			return $ret;
		} else {
			return array();
		}
	}
	
	/**
	* @param Converter
	*/
	function listTags()
	{
		$tags = array();
		foreach($this->tags as $keyword => $vals)
		{
			foreach($vals as $val)
			{
				$tags[] = $val;
			}
		}
		usort($tags,'tagsort');
		return $tags;
	}
	
	/** @return string always 'docblock' */
	function getType()
	{
		return 'docblock';
	}
}

/**
* @access private
*/
function tagsort($a, $b)
{
	switch(get_class($a))
	{
		case 'parsertag' :
			switch ($a->keyword)
			{
				case 'author' :
					$o = 3;
				case 'version' :
					$o = 4;
				case 'deprecated' :
				case 'deprec' :
					$o = 7;
				case 'todo' :
				case 'TODO' :
					$o = 8;
				case 'abstract' :
					$o = 9;
			}
		case 'parseraccesstag' :
			$o = 10;
		case 'parsernametag' :
			$o = 11;
		case 'parserseetag' :
			$o = 5;
		case 'parserlinktag' :
			$o = 6;
		case 'parserreturntag' :
			$o = 0;
		case 'parservartag' :
			$o = 1;
		case 'parserstaticvartag' :
			$o = 2;
		default :
			$o = 12;
	}
	switch(get_class($b))
	{
		case 'parsertag' :
			switch ($b->keyword)
			{
				case 'author' :
					$p = 3;
				case 'version' :
					$p = 4;
				case 'deprecated' :
				case 'deprec' :
					$p = 7;
				case 'todo' :
				case 'TODO' :
					$p = 8;
				case 'abstract' :
					$p = 9;
			}
		case 'parseraccesstag' :
			$p = 10;
		case 'parsernametag' :
			$p = 11;
		case 'parserseetag' :
			$p = 5;
		case 'parserlinktag' :
			$p = 6;
		case 'parserreturntag' :
			$p = 0;
		case 'parservartag' :
			$p = 1;
		case 'parserstaticvartag' :
			$p = 2;
		default :
			$p = 12;
	}
	if ($o == $p) return 0;
	if ($o < $p) return -1;
	if ($o > $p) return 1;
}
?>