<?php
/**
* Parser Elements, all classes representing documentable elements
* @package phpDocumentor
* @subpackage ParserElements
* @since 1.1
* @version $Id: ParserElements.inc,v 1.13.2.2 2002/11/07 20:46:00 CelloG Exp $
*/

/**
* all elements except {@link parserPackagePage} descend from this abstract class
* @abstract
* @subpackage ParserElements
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: ParserElements.inc,v 1.13.2.2 2002/11/07 20:46:00 CelloG Exp $
*/
class parserElement extends parserBase
{
	/**
	* @var mixed either false or a {@link parserDocBlock}
	*/
	var $docblock = false;
	/**
	* name of this element, or include type if element is a {@link parserInclude}
	*/
	var $name;
	
	/**
	* @var mixed either false or an array of paths to files with conflicts
	*/
	var $conflicts = false;
	
	/**
	* location of this element (filename)
	* @var string
	*/
	var $file = '';
	
	/**
	* full path location of this element (filename)
	* @var string
	*/
	var $path = '';
	
	/**
	* @param parserDocBlock
	*/
	function setDocBlock($docblock)
	{
		$this->docblock = $docblock;
	}
	
	/**
	* @param string
	*/
	function setName($name)
	{
		$this->name = trim($name);
	}
	
	/** @return string package containing this element */
	function getPackage()
	{
		if ($this->docblock)
		{
			return $this->docblock->package;
		} else return $GLOBALS['phpDocumentor_DefaultPackageName'];
	}
	
	/** @param string */
	function setFile($file)
	{
		$this->file = $file;
	}
	
	/** @param string */
	function setPath($file)
	{
		// look for special windows case
		if(SMART_PATH_DELIMITER === '\\')
			$this->path = strtr($file,'/','\\');
		else
			$this->path = $file;
		$this->path = $file;
	}
	
	/**
	* @return string
	*/
	function getName()
	{
		if (!isset($this->name)) return false;
		return $this->name;
	}
	
	/**
	* @return string
	*/
	function getFile()
	{
		if (!isset($this->file)) return false;
		return $this->file;
	}
	
	/**
	* @return string
	*/
	function getPath()
	{
		if (!isset($this->path)) return false;
		return $this->path;
	}
}

/**
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: ParserElements.inc,v 1.13.2.2 2002/11/07 20:46:00 CelloG Exp $
*/
class parserInclude extends parserElement
{
	/**
	* Type is used by many functions to skip the hassle of if get_class($blah) == 'parserBlah'
	* @var string always 'include'
	*/
	var $type = 'include';
}

/**
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.1
* @version $Id: ParserElements.inc,v 1.13.2.2 2002/11/07 20:46:00 CelloG Exp $
*/
class parserGlobal extends parserElement
{
	/**
	* Type is used by many functions to skip the hassle of if get_class($blah) == 'parserBlah'
	* @var string always 'global'
	*/
	var $type = 'global';

	/**
	* quick way to link to this element
	* @return mixed converter-specific link to this global variable
	* @param Converter
	* @param string text to display for the link or false for default text
	*/
	function getLink(&$c, $text = false)
	{
		return $c->getGlobalLink($this->name, $this->docblock->package, $this->path, $text);
	}

	/**
	* Returns all global variables in other packages that have the same name as this global variable
	* @return mixed false or an array Format: (package => {@link parserGlobal} of conflicting global variable)
	* @param Converter
	*/
	function getConflicts(&$c)
	{
		$a = $c->proceduralpages->getGlobalConflicts($this->name);
		unset($a[$this->docblock->package]);
		return $a;
	}
	

}

/**
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: ParserElements.inc,v 1.13.2.2 2002/11/07 20:46:00 CelloG Exp $
*/
class parserFunction extends parserElement
{
	/**
	* Type is used by many functions to skip the hassle of if get_class($blah) == 'parserBlah'
	* @var string always 'function'
	*/
	var $type = 'function';
	/**
	* parameters parsed from function definition.
	*
	* param name may be null, in which case, updateParams() must be called from the Converter
	* @var array Format: array(param name => default value parsed from function definition)
	* @see updateParams()
	*/
	var $params = false;
	/**
	* set to true if function is declared as:
	* <code>
	* function &func(...
	* </code>
	* @var boolean
	*/
	var $returnsreference = false;
	/**
	* global declarations parsed from function definition
	*
	* @var array Format: array(globalname1, globalname2,....)
	*/
	var $globals = false;
	/**
	* static variable declarations parsed from function definition
	* @var array Format: array(array('name' => staticvar1,'val' => '' or default val of staticvar1),...)
	*/
	var $statics = false;
	
	var $source = '';
	/**
	* @param string
	* @param string default value parsed from function definition
	*/
	function addParam($name, $value)
	{
		$this->params[$name] = $value;
	}
	
	function addSource($source)
	{
		$this->source = $source;
	}
	
	function hasSource()
	{
		return strlen($this->source);
	}
	
	function getSource()
	{
		return $this->source;
	}
	
	/**
	* quick way to link to this element
	* @return mixed converter-specific link to this function
	* @param Converter
	* @param string text to display for the link or false for default text
	*/
	function getLink($c, $text = false)
	{
		return $c->getFunctionLink($this->name, $this->docblock->package, $this->path, $text);
	}

	/**
	* Returns all functions in other packages that have the same name as this function
	* @return mixed false or an array Format: (package => {@link parserFunction} of conflicting functions)
	* @param Converter
	*/
	function getConflicts(&$c)
	{
		$a = $c->proceduralpages->getFuncConflicts($this->name);
		unset($a[$this->docblock->package]);
		return $a;
	}
	
	/**
	* Add all "global $var, $var2" declarations to this function
	* @param array $globals Format: array(globalname1, globalname2,....)
	*/
	function addGlobals($globals)
	{
		$this->globals = $globals;
	}

	/**
	* Add all "static $var, $var2 = 6" declarations to this function
	* @param array Format: array(varname1, varname2,...)
	* @param array Format: array(default val of var 1, default val of var 2,...) if var 1 has no default, array(default val of var 2,...)
	*/
	function addStatics($static,$vals)
	{
		if (count($static))
		{
			$this->statics = array();
			for($i=0;$i<count($static);$i++)
			{
				if (isset($static[$i]))
				{
					$a = '';
					if (isset($vals[$i])) $a = $vals[$i];
					$this->statics[] = array('name' => $static[$i],'val' => $a);
				}
			}
		}
	}

	/**
	* @return string default value of param $name
	* @param string
	*/
	function getParam ($name)
	{
		if (!isset($this->params[$name])) return false;
		return $this->params[$name];
	}

	/**
	* @return array format: array(array(paramname, default value),...)
	*/
	function listParams ()
	{
		if (isset($this->params))
		{
			$ret = array();
			if ($this->params)
			foreach($this->params as $key => $val)
			{
				$ret[] = array($key,$val);
			}
			return $ret;
		} else {
			return array();
		}
	}
	
	/**
	* @return array format: array(array(index, globalname),...)
	*/
	function listGlobals ()
	{
		if (isset($this->globals))
		{
			$ret = array();
			if ($this->globals)
			foreach($this->globals as $key => $val)
			{
				$ret[] = array($key,$val);
			}
			return $ret;
		} else {
			return array();
		}
	}
	
	/**
	* @return array format: array(array(static var name, static var default value),...)
	*/
	function listStatics ()
	{
		if (isset($this->statics))
		{
			$ret = array();
			if ($this->statics)
			foreach($this->statics as $key => $val)
			{
				$ret[] = array($val['name'],$val['val']);
			}
			return $ret;
		} else {
			return array();
		}
	}
	
	/**
	* sets {@link $returnsreference} to true
	*/
	function setReturnsReference()
	{
		$this->returnsreference = true;
	}
	
	/**
	* @return boolean returns value of {@link $returnsreference}
	*/
	function getReturnsReference()
	{
		return $this->returnsreference;
	}
	
	/**
	* takes declaration like:
	* <code>
	* /** @returns string ... {rest of docblock}
	* function &func($param1, $param2 = 6,
	*				$param3 = array('20',9 => "heroo"))
	* {...}
	* </code>
	* and returns:
	* string &func( $param1, [$param2 = 6], [$param3 = array('20',9 => "heroo")] )
	* @return string stylized function declaration
	*/
	function getFunctionCall()
	{
		$a = '';
		if ($this->getReturnsReference()) $a = '&';
		$function_call = $a.$this->getName() . " ( ";
		$tmp = 0;
		foreach($this->listParams() as $param)
		{
			if ($tmp == 0)
			{
				$tmp = 1;
			} else {
				$function_call .= ", ";
			}
			if ($param[1] != '')
			{
				$function_call .= "[$param[0] = $param[1]]";
			} else {
				$function_call .= $param[0];
			}
			$update_params[] = $param[0];
		}
		$function_call .= " )";
		return $function_call;
	}
}

/**
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: ParserElements.inc,v 1.13.2.2 2002/11/07 20:46:00 CelloG Exp $
*/
class parserClass extends parserElement
{
	/**
	* Type is used by many functions to skip the hassle of if get_class($blah) == 'parserBlah'
	* @var string always 'class'
	*/
	var $type = 'class';
	/** @var string
	* @see parserPage::$sourceLocation */
	var $sourceLocation = '';
	/**
	* @var mixed false or contents of extends clause in class declaration
	*/
	var $extends = false;
	/**
	* Format: array(file, parent) where parent class is found or false if no parent
	* @var mixed
	*/
	var $parent = false;
	/**
	* Used to determine whether a class should be ignored or not.  Helps maintain integrity of parsing
	* @var boolean
	* @see Classes::getParentClass()
	*/
	var $ignore = false;
	
	/**
	* @var string same as {@link parserElement::$path}
	*/
	var $curfile = false;
	
	/**
	* Returns all classes in other packages that have the same name as this class
	* @return mixed false or an array Format: (package => {@link parserClass} of conflicting classes)
	* @param Converter
	*/
	function getConflicts(&$c)
	{
		$a = $c->classes->getConflicts($this->name);
		unset($a[$this->docblock->package]);
		return $a;
	}
	
	/**
	* quick way to link to this element
	* @return mixed converter-specific link to this class
	* @param Converter
	* @param string text to display for the link or false for default text
	*/
	function getLink($c, $text = false)
	{
		return $c->getClassLink($this->name, $this->docblock->package, $this->curfile, $text);
	}

	/**
	* @param string parent class name
	* @param string parent class file
	* @param Classes {@link Classes} object currently calling setParent
	* @see Classes::setClassParent()
	*/
	
	function setParent($p,$f, &$c)
	{
		$this->parent = array($f, $p);
		$p = $c->getClass($p, $f);
		// inherit package if no @package tag is in the docblock, fixes 591396
		if (!$this->docblock->getExplicitPackage())
		{
			$this->docblock->package = $p->docblock->package;
		}
		if ($this->docblock->package == $p->docblock->package)
		{
			if ($this->docblock->subpackage == '')
			$this->docblock->subpackage = $p->docblock->subpackage;
		}
	}
	
	/**
	* @param string $par parent class name (used by {@link Classes::setClassParent()} if parent class not found
	*/
	function setParentNoClass($par)
	{
		$this->parent = $par;
	}
	
	/**
	* retrieve object that represents the parent class
	* @param Converter this function will not work before the Conversion stage of parsing
	* @return mixed returns the {@link parserClass} representation of the parent class, or false if no parent class
	*/
	function &getParent(&$c)
	{
		if (!$this->parent) return false;
		if (is_array($this->parent))
		{
			if (PHPDOC_EXCEPTIONS)
			{
				addException(!$c->classes->getClass($this->parent[1],$this->parent[0]), PDEXCEPTION_NOT_A_CLASS, __FILE__, __LINE__, "class", $this->parent[1], "file", $this->parent[0], "this", $this);
			}
			return $c->classes->getClass($this->parent[1],$this->parent[0]);
		} else return $this->parent;
	}
	
	/**
	* @param Converter this function will not work before the Conversion stage of parsing
	* @return array returns a simple array of method objects
	*/
	function getMethods(&$c)
	{
		return $c->classes->getMethods($this->name,$this->curfile);
	}
	
	/**
	* @return mixed {@link parserMethod} or false if not found
	* @param Converter this function will not work before the Conversion stage of parsing
	* @param string method name in this class
	*/
	function getMethod(&$c, $name)
	{
		return $c->classes->getMethod($this->name,$this->curfile,$name);
	}
	
	/**
	* @return mixed {@link parserVar} or false if not found
	* @param Converter this function will not work before the Conversion stage of parsing
	* @param string var name in this class
	*/
	function getVar(&$c, $name)
	{
		return $c->classes->getVar($this->name,$this->curfile,$name);
	}
	
	/**
	* @param Converter this function will not work before the Conversion stage of parsing
	* @return array returns a simple array of method name strings
	*/
	function getMethodNames(&$c)
	{
		if (!$c->classes->hasMethods($this->curfile, $this->name)) return array();
		$arr = array();
		$arr1 = $this->getMethods($c);
		if (PHPDOC_EXCEPTIONS)
		{
			addException(!is_array($arr1), PDEXCEPTION_NOT_AN_ARRAY, __FILE__, __LINE__, "getMethods output", $arr1, "converter->classes",$c->classes,"this",$this);
		}
		for($i=0; $i < count($arr1); $i++)
		{
			$arr[] = $arr1[$i]->name;
		}
		return $arr;
	}
	
	/**
	* @param Converter this function will not work before the Conversion stage of parsing
	* @param string method name
	* @return boolean whether this class has a method of name $name
	*/
	function hasMethod(&$c,$name)
	{
		return $c->classes->hasMethod($this->name, $this->curfile, $name);
	}
	
	/**
	* @param Converter this function will not work before the Conversion stage of parsing
	* @param string var name
	* @return boolean whether this class has a var of name $name
	*/
	function hasVar(&$c,$name)
	{
		return $c->classes->hasVar($this->name, $this->curfile, $name);
	}
	
	/**
	* @param Converter this function will not work before the Conversion stage of parsing
	* @return array returns a simple array of var objects
	*/
	function getVars(&$c)
	{
		return $c->classes->getVars($this->name,$this->curfile);
	}
	
	/**
	* @param Converter this function will not work before the Conversion stage of parsing
	* @return array returns a simple array of var name strings
	*/
	function getVarNames(&$c)
	{
		if (!$c->classes->hasVars($this->name, $this->curfile)) return array();
		$arr = array();
		$arr1 = $this->getVars($c);
		if (PHPDOC_EXCEPTIONS)
		{
			addException(!is_array($arr1), PDEXCEPTION_NOT_AN_ARRAY, __FILE__, __LINE__, "getVars output", $arr1, "converter->classes",$c->classes,"this",$this);
		}
		for($i=0; $i < count($arr1); $i++)
		{
			$arr[] = $arr1[$i]->name;
		}
		return $arr;
	}
	
	/**
	* @param Converter this function will not work before the Conversion stage of parsing
	* @param boolean determines whether overriden methods should be included in the list of inherited methods
	* @return array returns an array of methods by parent classname array(name => array(method1,method2..),name2 => array(method1....))
	*/
	function getInheritedMethods(&$c,$override = false)
	{
		$x = $this;
		$methods = array();
		$arr = array();
		while ($x->parent && is_array($x->parent))
		{
			$methods = array_merge($methods,$x->getMethodNames($c));
			$par = $x->getParent($c);
			$parmethodnames = $par->getMethodNames($c);
			$parmethods = $par->getMethods($c);
			for($i=0; $i<count($parmethodnames); $i++)
			{
				if ($override)
				{
					if (!in_array($parmethodnames[$i],$methods))
					{
						// fix for bug 587733
						if ($parmethods[$i]->docblock && $parmethods[$i]->docblock->hasaccess && !$c->parseprivate && $parmethods[$i]->docblock->tags['access'][0]->value == 'private')
						{
							continue;
						}
						$methods[] = $parmethodnames[$i];
						$arr[$par->getName()]['methods'][] = $parmethods[$i];
						$arr[$par->getName()]['file'] = $par->curfile;
					}
				} else
				{
					// fix for bug 587733
					if ($parmethods[$i]->docblock && $parmethods[$i]->docblock->hasaccess && !$c->parseprivate && $parmethods[$i]->docblock->tags['access'][0]->value == 'private')
					{
						continue;
					}
					$arr[$par->getName()]['methods'][] = $parmethods[$i];
					$arr[$par->getName()]['file'] = $par->curfile;
				}
			}
			$x = &$par;
		}
		return $arr;
	}
	
	/**
	* @param Converter this function will not work before the Conversion stage of parsing
	* @param boolean determines whether overriden vars should be included in the list of inherited vars
	* @return array returns an array of vars by parent classname array(name => array(var1,var1..),name2 => array(var1....))
	*/
	function getInheritedVars(&$c,$override = true, $vars = false)
	{
		$x = $this;
		$vars = array();
		$arr = array();
		while ($x->parent && is_array($x->parent))
		{
			$vars = array_merge($vars,$x->getVarNames($c));
			$par = $x->getParent($c);
			$parvarnames = $par->getVarNames($c);
			$parvars = $par->getVars($c);
			for($i=0; $i<count($parvarnames); $i++)
			{
				if ($override)
				{
					if (!in_array($parvarnames[$i],$vars))
					{
						// fix for bug 587733
						if ($parvars[$i]->docblock && $parvars[$i]->docblock->hasaccess && !$c->parseprivate && $parvars[$i]->docblock->tags['access'][0]->value == 'private')
						{
							continue;
						}
						$vars[] = $parvarnames[$i];
						$arr[$par->getName()]['vars'][] = $parvars[$i];
						$arr[$par->getName()]['file'] = $par->curfile;
					}
				} else
				{
					// fix for bug 587733
					if ($parvars[$i]->docblock && $parvars[$i]->docblock->hasaccess && !$c->parseprivate && $parvars[$i]->docblock->tags['access'][0]->value == 'private')
					{
						continue;
					}
					$arr[$par->getName()]['vars'][] = $parvars[$i];
					$arr[$par->getName()]['file'] = $par->curfile;
				}
			}
			$x = &$par;
		}
		return $arr;
	}
	
	/**
	* @param Converter this function will not work before the Conversion stage of parsing
	* @return array Format: array(parentclassname => parserClass/false if no parent, parentclassname2 => ...)
	*/
	function getParentClassTree(&$c)
	{
		$result = array();
		$result[$this->name] = $arr = $this->getParent($c);
		if (is_string($arr)) $result[$arr] = false;
		while ($arr && is_object($arr))
		{
			$result[$arr->name] = $arr->getParent($c);
			$arr = $arr->getParent($c);
			if (is_string($arr)) $result[$arr] = false;
		}
		return $result;
	}
	
	/**
	* returns a list of all child classes of this class
	* @param Converter this function will not work before the Conversion stage of parsing
	* @return array Format: array(parserClass child1,parserClass child2,...)
	*/
	function getChildClassList(&$c)
	{
		$list = array();
		$kids = $c->classes->getDefiniteChildren($this->name,$this->curfile);
		if ($kids)
		{
			foreach($kids as $chile => $file)
			{
				$list[] = $c->classes->getClass($chile,$file);
			}
		}
		return $list;
	}
	
	/**
	* @param string
	* @see $sourceLocation
	*/
	function setSourceLocation($sl)
	{
		$this->sourceLocation = $sl;
	}
	
	/**
	* @return string
	* @see $sourceLocation
	*/
	function getSourceLocation()
	{
		if (!isset($this->sourceLocation)) return false;
		return $this->sourceLocation;
	}
	
	/**
	* @param string
	* @see $extends
	*/
	function setExtends($extends)
	{
		$this->extends = $extends;
	}
	
	/**
	* @return boolean
	* @see $extends
	*/
	function getExtends()
	{
		if (!isset($this->extends)) return false;
		return $this->extends;
	}
}

/**
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: ParserElements.inc,v 1.13.2.2 2002/11/07 20:46:00 CelloG Exp $
*/
class parserVar extends parserElement
{
	/**
	* Type is used by many functions to skip the hassle of if get_class($blah) == 'parserBlah'
	* @var string always 'var'
	*/
	var $type = 'var';
	/** @var string class that contains this var */
	var $class = '';
	
	/**
	* @param string
	*/
	function parserVar($class)
	{
		$this->class = $class;
	}

	/**
	* quick way to link to this element
	* @return mixed converter-specific link to this var
	* @param Converter $c
	* @param string $text text to display for the link or false for default text
	*/
	function getLink($c, $text = false)
	{
		return $c->getVarLink($this->name, $this->class, $this->docblock->package, false, $text);
	}

	/**
	* @param Converter
	* @return mixed {@link parserVar} representing var this var overrides from the parent class, or false if none
	*/
	function getOverrides(&$c)
	{
		$class = $c->classes->getClass($this->class,$this->path);
		if (PHPDOC_EXCEPTIONS)
		{
			addException(!is_object($class), PDEXCEPTION_NOT_A_CLASS, __FILE__, __LINE__, "this->class", $this->class, "this->path",$this->path,"c->classes",$c->classes,"this",$this);
		}
		$par = $class->getParent($c);
		if (is_object($par))
		{
			if (!$par->hasVar($c,$this->name)) return false;
			return $par->getVar($c,$this->name);
		} else
		{
			return false;
		}
	}
}

/**
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: ParserElements.inc,v 1.13.2.2 2002/11/07 20:46:00 CelloG Exp $
*/
class parserMethod extends parserFunction
{
	/**
	* Type is used by many functions to skip the hassle of if get_class($blah) == 'parserBlah'
	* @var string always 'method'
	*/
	var $type = 'method';
	/** @var boolean whether this method is a constructor */
	var $isConstructor = false;
	/** @var string class that contains this method */
	var $class = '';
	
	/**
	* @param string
	*/
	function parserMethod($class)
	{
		$this->class = $class;
	}
	
	/**
	* adds "constructor " to start of function call if {@link $isConstructor} is true
	* @return string
	* @see parent::getFunctionCall()
	*/
	function getFunctionCall()
	{
		$a = parserFunction::getFunctionCall();
		if ($this->isConstructor) $a = "constructor $a";
		return $a;
	}
	
	/**
	* @param Converter
	* @return mixed {@link parserMethod} representing method this method overrides from the parent class, or false if none
	*/
	function getOverrides(&$c)
	{
		$class = $c->classes->getClass($this->class,$this->path);
		if (PHPDOC_EXCEPTIONS)
		{
			addException(!is_object($class), PDEXCEPTION_NOT_A_CLASS, __FILE__, __LINE__, "this->class", $this->class, "this->path",$this->path,"c->classes",$c->classes,"this",$this);
		}
		$par = $class->getParent($c);
		if (is_object($par))
		{
			if (!$par->hasMethod($c,$this->name)) return false;
			return $par->getMethod($c,$this->name);
		} else
		{
			return false;
		}
	}
	
	/**
	* quick way to link to this element
	* @return mixed converter-specific link to this method
	* @param Converter $c
	* @param string $text text to display for the link or false for default text
	*/
	function getLink($c, $text = false)
	{
		return $c->getMethodLink($this->name, $this->class, $this->docblock->package, false, $text);
	}

	/**
	* Use this method to tell the parser that this method is the class constructor
	*/
	function setConstructor()
	{
		$this->isConstructor = true;
	}
	
	/**
	* @param Converter
	* @return array an array of parserMethods from child classes that override this method
	*/
	function getOverridingMethods(&$c)
	{
		$class = $c->classes->getClass($this->class,$this->path);
		if (PHPDOC_EXCEPTIONS)
		{
			addException(!is_object($class), PDEXCEPTION_NOT_A_CLASS, __FILE__, __LINE__, "this->class", $this->class, "this->path",$this->path,"c->classes",$c->classes,"this",$this);
		}
		$meths = array();
		if (!$class) return $meths;
		$kids = $class->getChildClassList($c);
		for($i=0; $i<count($kids); $i++)
		{
			if ($kids[$i]->hasMethod($c, $this->name))
			{
				$meths[] = $kids[$i]->getMethod($c, $this->name);
			}
		}
		return $meths;
	}
}

/**
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @version $Id: ParserElements.inc,v 1.13.2.2 2002/11/07 20:46:00 CelloG Exp $
*/
class parserDefine extends parserElement
{
	/**
	* Type is used by many functions to skip the hassle of if get_class($blah) == 'parserBlah'
	* @var string always 'define'
	*/
	var $type = 'define';

	/**
	* quick way to link to this element
	* @return mixed converter-specific link to this define
	* @param Converter $c
	* @param string $text text to display for the link or false for default text
	*/
	function getLink($c, $text = false)
	{
		return $c->getDefineLink($this->name, $this->docblock->package, false, $text);
	}

	/**
	* Returns all defines in other packages that have the same name as this define
	* @return mixed false or an array Format: (package => {@link parserDefine} of conflicting defines)
	* @param Converter
	*/
	function getConflicts(&$c)
	{
		$a = $c->proceduralpages->getDefineConflicts($this->name);
		unset($a[$this->docblock->package]);
		return $a;
	}
	
}

/**
* @author Greg Beaver <cellog@users.sourceforge.net>
* @since 1.0rc1
* @subpackage ParserElements
* @version $Id: ParserElements.inc,v 1.13.2.2 2002/11/07 20:46:00 CelloG Exp $
*/
class parserPackagePage extends parserStringWithInlineTags
{
	/**
	* Type is used by many functions to skip the hassle of if get_class($blah) == 'parserBlah'
	* @var string always 'packagepage'
	*/
	var $type = 'packagepage';
	/** @var string */
	var $package = 'default';
	
	/**
	* @param string
	*/
	function parserPackagePage($package)
	{
		$this->package = $package;
	}
}
?>